/*
 * DB: Data Source Definitions
 * Copyright (C) 2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __DB_DATASOURCE_H__
#define __DB_DATASOURCE_H__

/*
 * Definitions used by the chooser support functions.
 */

typedef struct data_source_chooser_key {
    UInt16 cardNo;
    char name[dmDBNameLength];
} DataSourceChooserKey;


/*
 * Schema Definitions
 */

typedef enum {
    FIELD_TYPE_SKIP,     /* uneditable field type that must be skipped. */
    FIELD_TYPE_STRING,   /* null-terminated string */
    FIELD_TYPE_INTEGER,  /* 32-bit signed integer */
    FIELD_TYPE_BOOLEAN,  /* true/false */
    FIELD_TYPE_DATE,     /* date */
    FIELD_TYPE_TIME      /* time */
} DataSourceFieldType;

typedef struct data_source_field_info {
    char * name;               /* name of this field */
    DataSourceFieldType type;  /* type of data stored in the field */
} DataSourceFieldInfo;

typedef struct data_source_schema {
    UInt16 numFields;
    DataSourceFieldInfo * fields;
} DataSourceSchema;


/*
 * View Definitions
 *
 * A data source can store one or more "views" of the data. A view
 * defines the columns displayed in the list view as well as the width
 * of each column.
 */

typedef struct data_source_view_column {
    UInt16 fieldNum;
    UInt16 width;
} DataSourceViewColumn;

typedef struct data_source_view {
    Char name[32];
    int numCols;
    DataSourceViewColumn * cols;
} DataSourceView;


/*
 * Capability definitions
 *
 * A data source can various operations performed on it. However, not
 * all data source drivers support all operations. Thus, the following
 * enumeration allows a caller to ask a data source what operations it
 * can do. Note: This state is dynamic esp. if the data source is
 * opened in read-only mode.
 */

typedef enum data_source_capability {
    DS_CAPABLE_LISTVIEW_EDIT,
    DS_CAPABLE_LISTVIEW_INSERT,
    DS_CAPABLE_LISTVIEW_DELETE,
    DS_CAPABLE_LISTVIEW_NAME_EDIT,
    DS_CAPABLE_LISTVIEW_FIELD_EDIT,
    DS_CAPABLE_LISTVIEW_FIELD_INSERT,
    DS_CAPABLE_LISTVIEW_FIELD_DELETE,
    DS_CAPABLE_LISTVIEW_WIDTH_EDIT
} DataSourceCapability;


/*
 * Data Source Definitions
 *
 * A data source is an abstraction of a PalmOS database so the
 * remainder of DB does not have to deal with the specifics of how
 * records are stored into the database. Every data source has a set
 * of function pointers implementing standard operations on the data
 * source.  */

/* Forward delcarations for the structures below. */
struct data_source_operations;
struct data_source;

/* Options that can be set on a data source. */
typedef enum data_source_options_enum {
    optionID_DisableGlobalFind = 0U,
    optionID_LFind_CaseSensitive,
    optionID_LFind_WholeWord,
    optionID_ListView_ActiveView,
    optionID_ListView_TopVisibleRecord,
} DataSourceOptionsEnum;

/* Function pointers to retrieve specific fields of a record. */
typedef struct data_source_getter
{
    char *  (*GetString) (void * data, UInt16 fieldNum);
    Int32   (*GetInteger)(void * data, UInt16 fieldNum);
    Boolean (*GetBoolean)(void * data, UInt16 fieldNum);
    void    (*GetDate)(void * data, UInt16 fieldNum,
		       UInt16* year, UInt8* mon, UInt8* day);
    void    (*GetTime)(void * data, UInt16 fieldNum,
		       UInt8* hour, UInt8* minute);
} DataSourceGetter;

/* Direction of a field sort. */
typedef enum data_source_sort_direction {
    SORT_DIRECTION_DESCENDING,
    SORT_DIRECTION_ASCENDING
} DataSourceSortDirection;

/* Instructions on how to sort a single field. */
typedef struct data_source_sort_field {
    UInt16 fieldNum;
    DataSourceSortDirection direction;
    Boolean case_sensitive;
} DataSourceSortField;

/* Instructions on how to sort a data source. */
typedef struct data_source_sort_info {
    UInt16 numFields;
    DataSourceSortField * fields;
} DataSourceSortInfo;

/* Callback function provided to the sorting function of a data source
 * to order records. -1 for less than, 0 for equal, 1 for greater than
 */
typedef int (*DataSourceSortCallback)(struct data_source *,
				      DataSourceGetter * getter1, void * data1,
				      DataSourceGetter * getter2, void * data2,
				      void * callback_data);

/* Standard sorting callback function that callers can make use of. */
extern int DS_StandardSortCallback(struct data_source *,
				   DataSourceGetter * getter1, void * data1,
				   DataSourceGetter * getter2, void * data2,
				   void * callback_data);

typedef struct data_source_operations {
    /* Close the data source. */
    void (*Close)(struct data_source *);

    /* Seek forward or backward in the data source. */
    Boolean (*Seek)(struct data_source *, UInt16 *, Int16, Int16);
    
    /* Lock a record and return a 'handle' and getter that get be used
     * to access the field data.
     */
    void (*LockRecord)(struct data_source *, UInt16 index, Boolean writable,
		       DataSourceGetter * getter, void ** data_ptr);

    /* Unlock a previously locked record. */
    void (*UnlockRecord)(struct data_source *, UInt16 index, Boolean writable,
			 void * record_data);

    /* Add a new record to the data source. */
    Err (*UpdateRecord)(struct data_source *, UInt16 index,
			Boolean doInsert, DataSourceGetter *, void *);

    /* Update a specific field of a record. */
    void (*SetField)(struct data_source *, void *, UInt16 fieldNum,
		     DataSourceGetter *, void *);

    /* Delete the specified index from the data source. */
    void (*DeleteRecord)(struct data_source *, UInt16 index);

    /* Return the unique ID for the given record. */
    UInt32 (*GetRecordID)(struct data_source *, UInt16 index);

    /* Retrieve the number of views that this data source supports. */
    UInt16 (*GetNumOfViews)(struct data_source *);

    /* Allocate and retrieve the N-th view. */
    DataSourceView* (*GetView)(struct data_source *, UInt16 viewIndex);

    /* Free a view retrieved through GetView above. */
    void (*PutView)(struct data_source *, UInt16 viewIndex, DataSourceView *);

    /* Replace (or add for invalid index) the view given. */
    void (*StoreView)(struct data_source *, UInt16 viewIndex,
		      DataSourceView *);

    /* Delete the given view. */
    void (*DeleteView)(struct data_source *, UInt16 viewIndex);

    /* Determine if a change to the schema could be made. */
    void (*CheckUpdateSchema)(struct data_source *,
			      DataSourceSchema *, UInt32* reorder,
			      Boolean* changed, Boolean* rebuild);

    /* Make a change to the schema. Assume CanUpdateSchema is true. */
    void (*UpdateSchema)(struct data_source *, DataSourceSchema *, UInt32 *);

    /* Get/Set boolean options. */
    Boolean (*GetOption_Boolean)(struct data_source *, UInt16 optionID);
    void (*SetOption_Boolean)(struct data_source *,
			      UInt16 optionID, Boolean value);

    /* Get/set options that are UInt16 types. */
    UInt16 (*GetOption_UInt16)(struct data_source *, UInt16 optionID);
    void (*SetOption_UInt16)(struct data_source *,
			     UInt16 optionID, UInt16 value);

    /* Return a key that can be passed to OpenDataSource(). */
    UInt16 (*GetKey)(struct data_source *, void * buffer);

    /* Return the Palm database (if any) backing this data source. */
    void (*GetPDB)(struct data_source *, UInt16 *, LocalID *);

    /* Get/set the title of this data source. */
    void (*GetTitle)(struct data_source *, char *, UInt32);
    void (*SetTitle)(struct data_source *, const char * title,
		     void ** key_data_ptr, UInt16 * key_size_ptr);

    /* Sort all of the records in the data source using the callback. */
    void (*Sort)(struct data_source *,
		 DataSourceSortCallback callback, void * callback_data);

    /* Return true if a particular capability is supported by the driver. */
    Boolean (*Capable)(struct data_source *, DataSourceCapability capability);

} DataSourceOperations;

/* Prototype of the function called by the enumeration method. */
typedef void (*DataSourceEnumerationFunc)(void * arg, const char* title,
					  UInt16 sourceID, void * key_data,
					  UInt32 key_size);

typedef Boolean (*DataSourceGlobalFindFunc)(FindParamsPtr,
					    struct data_source *,
					    UInt16, LocalID);

typedef struct data_source_class_operations
{
    /* Open the identified data source. */
    Err (*Open)(struct data_source *, UInt16 mode,
		void * key_data, UInt16 key_size);

    /* Open the identified data source. */
    Err (*OpenPDB)(struct data_source *, UInt16 mode,
		   UInt16 cardNo, LocalID dbID);

    /* Check to see if a title is usable. */
    Boolean (*CheckTitle)(const char* title);

    /* Create an empty database with the given title and schema. The
     * key is returned in memory that must be freed using
     * MemPtrFree().
     */
    Err (*Create)(const char* title, DataSourceSchema* schema,
		  void ** key_data_ptr, UInt16 * key_size_ptr);

    /* Returns true if the given field type is supported. */
    Boolean (*SupportsFieldType)(DataSourceFieldType);

    /* Delete the identified data source. */
    void (*Delete)(void * key_data, UInt32 key_size);

    /* True if this driver can handle the named database. */
    void (*Enumerate)(DataSourceEnumerationFunc callback, void * arg);

    /* Enumerate and open all sources willing to participate in global
     * find. If callback returns true, then enumeration function
     * should save state and also return true. Normal completion
     * returns false.
     */
    Boolean (*GlobalFind)(DataSourceGlobalFindFunc callback,
			  Boolean continuation, FindParamsPtr params);

    /* Return true if the identified data source is present. */
    Boolean (*IsPresent)(void * key_data, UInt32 key_size);

    /* Return true if both keys are identical. */
    Boolean (*CompareKey)(void * key1_data, UInt16 key1_size,
			  void * key2_data, UInt16 key2_size);

    /* Return true if this driver can handle the Palm database. */
    Boolean (*CanHandlePDB)(UInt16 cardNo, LocalID dbID,
			    UInt32 type, UInt32 creator);

    /* Beam the named data source. */
    Err (*Beam)(void * key_data, UInt16 key_size);

} DataSourceClassOperations;

typedef struct data_source {
    /* The open PalmOS database associated with this data source. */
    DmOpenRef db;

    /* The schema of this data source. */
    DataSourceSchema schema;

    /* The driver which controls this source. */
    struct data_source_driver * driver;

    /* Pointer to the operations stucture for this data source. */
    struct data_source_operations ops;

    /* Private data for use by this data source. */
    void * data;

    /* True if this database should be read-only. */
    Boolean readonly;
} DataSource;

typedef enum data_source_id_enum {
    sourceID_DB = 0U,
    sourceID_DB02x = 1U,
    sourceID_MobileDB = 2U,
    sourceID_JFile3 = 3U
} DataSourceIDEnum;

typedef struct data_source_driver {
    /* ID number of this data source. */
    UInt16 sourceID;

    /* Name of this data source driver. */
    char name[32];

    /* Class operations for this driver. */
    DataSourceClassOperations class_ops;

    /* Pointer to the next driver. (drivers shouldn't touch this) */
    struct data_source_driver * next;
} DataSourceDriver;

/* Build and return a linked list of just native data source drivers. */
extern DataSourceDriver * GetInternalDataSources(void);

/* Build and return a linked list of all data source drivers. */
extern DataSourceDriver * GetDataSources(void);

/* Free a list of data source drivers. */
extern void PutDataSources(DataSourceDriver *);

/* Copy a data source schema. */
extern void CopySchema(DataSourceSchema * dst, DataSourceSchema * src);

/* Attempt to open the database identified by the key. */
extern DataSource *
OpenDataSource(DataSourceDriver * drivers, UInt16 mode, UInt16 sourceID,
	       void * key_data, UInt16 key_size);

/* Open the database which resides in the identifed PDB. */
extern DataSource *
OpenDataSourcePDB(DataSourceDriver * drivers, UInt16 mode,
		  UInt16 cardNo, LocalID dbID);

/* Close a data source and free all data associated with it. */
extern void CloseDataSource(DataSource * source);

/* Delete the specified data source. */
extern void DeleteDataSource(DataSourceDriver * drivers, UInt16 sourceID,
			     void * key_data, UInt32 key_size);

/* Search the drivers list for a specific ID. */
extern DataSourceDriver * GetDriverByID(UInt16 sourceID);

/* Beam a data source. */
extern Err BeamDataSource(DataSourceDriver * drivers, UInt16 sourceID,
			  void * key_data, UInt16 key_size);


/*
 * Prototypes for the data source driver support functions.
 */

extern Boolean DS_PDB_CheckTitle(const char* title);

extern void DS_PDB_Delete(void * key_data, UInt32 key_size);

extern void DS_PDB_Enumerate(DataSourceEnumerationFunc callback, void * arg,
			     UInt32 type, UInt32 creator, UInt16 sourceID);

extern Boolean DS_PDB_GlobalFind(DataSourceGlobalFindFunc callback,
				 Boolean continuation, FindParamsPtr params,
				 UInt32 type, UInt32 creator,
				 Err (*OpenPDB)(DataSource *, UInt16,
						UInt16, LocalID));

extern Boolean DS_PDB_IsPresent(void * key_data, UInt32 key_size,
				UInt32 ds_type, UInt32 ds_creator);

extern Boolean DS_PDB_CompareKey(void * key1_data, UInt16 key1_size,
				 void * key2_data, UInt16 key2_size);

extern Err DS_PDB_Beam(void * key_data, UInt16 key_size);

extern UInt16 DS_PDB_GetKey(DataSource * source, void * buffer);

extern void DS_PDB_GetPDB(DataSource * source,
			  UInt16 * cardNo_ptr, LocalID * dbID_ptr);

extern void DS_PDB_GetTitle(DataSource * source, char * buf, UInt32 len);

extern void DS_PDB_SetTitle(DataSource * source, const char * title,
			    void ** key_data_ptr, UInt16 * key_size_ptr);

/* Prototypes for the driver initialization routines. */
extern DataSourceDriver * MobileDB_GetDriver(void);
extern DataSourceDriver * JFile3_GetDriver(void);

#endif
