/*
 * DB: Database Selection Form
 * Copyright (C) 1998-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "enum.h"
#include "db.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define updateCategoryChanged   0x0001
#define updateInitTable         0x0002
#define updateSortOrderChanged  0x0004

enum {
    actionID_Open, actionID_OpenReadOnly, actionID_Info,
    actionID_Delete, actionID_Rename, actionID_Duplicate, actionID_Beam,
    actionID_CreateUsing,
};

static struct {
    UInt16 nameID;
    UInt16 actionID;
    Boolean needsFeatureSet30;
} popup_menu[] = {
    { stringID_Open,         actionID_Open,         false },
    { stringID_OpenReadOnly, actionID_OpenReadOnly, false },
    { stringID_Info,         actionID_Info,         false },
    { stringID_Beam,         actionID_Beam,         true  },
    { stringID_Delete,       actionID_Delete,       false },
    { stringID_Rename,       actionID_Rename,       false },
    { stringID_Duplicate,    actionID_Duplicate,    false },
    { stringID_CreateUsing,  actionID_CreateUsing,  false },
    { 0 },
};

typedef struct
{
    UInt16 sourceID;
    Char title[64];
    /* variable-sized key data follows the above fixed portion */
} ChooserRecord;

typedef struct {
    UInt16 renamedCategories; /* bitfield of categories with a different name */
    char    categoryLabels[dmRecNumCategories][dmCategoryLength];
    UInt8    categoryUniqIDs[dmRecNumCategories];
    UInt8    lastUniqID;     /* Uniq IDs generated by the device are between */
                            /* 0 - 127.  Those from the PC are 128 - 255.   */
    UInt8    reserved1;      /* from the compiler word aligning things       */
    UInt16    reserved2;
    UInt16    reserved3;
} ChooserAppInfoType;

static UInt16 TopVisibleDatabase, CurrentCategory = dmAllCategories;
static DmOpenRef ChooserDB = 0;
static Char CategoryName[dmCategoryLength];
static Boolean ShowAllCategories = true;
static UInt16 InfoDatabase = 0;

static void FoundDatabase(void *, const char *, UInt16, void *_data,
			  UInt32) SECT_UI;
static void RescanDatabases(void) SECT_UI;
static Err AppInfoInit(DmOpenRef) SECT_UI;
static void DoSecurity(void) SECT_UI;

static void
FoundDatabase(void * arg, const char* title, UInt16 sourceID,
	      void * key_data, UInt32 key_size)
{
    DataSourceDriver * driver = (DataSourceDriver *) arg;
    MemHandle h;
    UInt16 recNum, numRecords;
    UInt32 size;
    Boolean found;
    ChooserRecord *rec, tmp;

    /* Scan the list for the provided key. */
    found = false;
    numRecords = DmNumRecords(ChooserDB);
    for (recNum = 0; recNum < numRecords; ++recNum) {
	h = DmQueryRecord(ChooserDB, recNum);
	rec = MemHandleLock(h);
	if (rec->sourceID == sourceID
	    && driver->class_ops.CompareKey(key_data, key_size,
					    (void *) (rec + 1),
					    MemHandleSize(h)
					    - sizeof(ChooserRecord))) {
	    found = true;
	    MemPtrUnlock(rec);
	    break;

	}
	MemPtrUnlock(rec);
    }

    if (!found) {
	/* Fill in a template for the fixed portion. */
	tmp.sourceID = sourceID;
	StrNCopy(tmp.title, title, sizeof(tmp.title));

	/* Calculate the total size of the record. */
	size = sizeof(ChooserRecord) + key_size;

	/* Create and store the record in the database. */
	recNum = dmMaxRecordIndex;
	h = DmNewRecord(ChooserDB, &recNum, size);
	rec = MemHandleLock(h);
	DmWrite(rec, 0, &tmp, sizeof(tmp));
	DmWrite(rec, sizeof(tmp), key_data, key_size);
	MemPtrUnlock(rec);
	DmReleaseRecord(ChooserDB, recNum, true);
    }
}

static void
RescanDatabases(void)
{
    UInt16 recNum, numRecords;
    MemHandle h;
    ChooserRecord *rec;
    DataSourceDriver * driver;
    UInt32 size;
    Boolean remove;

    /* Trim any missing data sources from the list. */
    numRecords = DmNumRecords(ChooserDB);
    for (recNum = 0; recNum < numRecords; /* inc at bottom of loop */ ) {
	/* Lock down this chooser entry. */
	h = DmQueryRecord(ChooserDB, recNum);
	rec = MemHandleLock(h);

	/* Find the data source it belongs to and ask if present. */
	driver = DriverList;
	remove = false;
	while (driver) {
	    if (driver->sourceID == rec->sourceID) {
		size = MemHandleSize(h) - sizeof(ChooserRecord);
		if (! driver->class_ops.IsPresent((void *) (rec + 1), size))
		    remove = true;

		/* We found the driver so stop searching the driver list. */
		break;
	    }

	    /* Point at the next driver in the list. */
	    driver = driver->next;
	}

	/* Unlock the chooser entry. */
	MemPtrUnlock(rec);

	/* Handle removing the record or advancing to the next record. */
	if (remove) {
	    DmRemoveRecord(ChooserDB, recNum);
	    numRecords--;
	} else {
	    ++recNum;
	}
    }

    /* Ask each data source to enumerate its data sources. */
    driver = DriverList;
    while (driver) {
	driver->class_ops.Enumerate(FoundDatabase, driver);
	driver = driver->next;
    }
}

static Err
AppInfoInit(DmOpenRef db)
{
    UInt16 cardNo;
    LocalID dbID, appInfoID;
    MemHandle h;
    ChooserAppInfoType * appInfoPtr;

    if (DmOpenDatabaseInfo(db, &dbID, NULL, NULL, &cardNo, NULL))
	return dmErrInvalidParam;

    if (DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0))
	return dmErrInvalidParam;

    if (appInfoID == 0) {
	h = DmNewHandle(db, sizeof(ChooserAppInfoType));
	if (!h)
	    return dmErrMemError;
                
	appInfoID = MemHandleToLocalID(h);
	DmSetDatabaseInfo(cardNo, dbID, 0,0,0,0,0,0,0, &appInfoID, 0,0,0);
    }
    appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);

    /* Clear the app info block. */
    DmSet(appInfoPtr, 0, sizeof(ChooserAppInfoType), 0);

    /* Initialize the categories. */
    CategoryInitialize((AppInfoPtr) appInfoPtr, 100);

    MemPtrUnlock(appInfoPtr);

    return 0;
}

static void
CreateChooserDB(void)
{
    UInt16 mode;

    if (PrivateRecordStatus == hidePrivateRecords)
	mode = dmModeReadWrite;
    else
	mode = dmModeReadWrite | dmModeShowSecret;

    /* Create the chooser database. */
    DmCreateDatabase(0, "DBDatabases", DBCreatorID, 'LIST', false);
    ChooserDB = DmOpenDatabaseByTypeCreator('LIST', DBCreatorID, mode);

    /* Initialize the app info block. */
    AppInfoInit(ChooserDB);

    /* Force all databases on the device into the chooser database. */
    RescanDatabases();
}

static void
OpenChooserDB(void)
{
    UInt16 mode;

    if (PrivateRecordStatus == hidePrivateRecords)
	mode = dmModeReadWrite;
    else
	mode = dmModeReadWrite | dmModeShowSecret;

    ChooserDB = DmOpenDatabaseByTypeCreator('LIST', DBCreatorID, mode);
    if (! ChooserDB) {
	/* Database not found. Recreate it. */
	CreateChooserDB();
    }
}

static void
CloseChooserDB(void)
{
    DmCloseDatabase(ChooserDB);
    ChooserDB = 0;
}

static Int16
ChooserCompareRecords(ChooserRecord * r1, ChooserRecord * r2, Int16 sortOrder,
		      SortRecordInfoPtr info1, 
		      SortRecordInfoPtr info2, MemHandle appInfoH)
{
    return StrCompare(r1->title, r2->title);
}

static void
SortAlphabetic(void)
{
    DmInsertionSort(ChooserDB, (DmComparF *) &ChooserCompareRecords,
		    (Int16) prefs.ChooserSortOrder);
}

static void
AddDatabase(const char* title, UInt16 sourceID,
	    void * key_data, UInt16 key_size)
{
    MemHandle h;
    UInt16 recNum, attr;
    UInt32 size;
    ChooserRecord *rec, tmp;

    /* Fill in a template for the fixed portion. */
    MemSet(&tmp, sizeof(tmp), 0);
    tmp.sourceID = sourceID;
    StrNCopy(tmp.title, title, sizeof(tmp.title));

    /* Calculate the total size of the record. */
    size = sizeof(ChooserRecord) + key_size;

    /* Create and store the record in the database. */
    recNum = dmMaxRecordIndex;
    h = DmNewRecord(ChooserDB, &recNum, size);
    rec = MemHandleLock(h);
    DmWrite(rec, 0, &tmp, sizeof(tmp));
    DmWrite(rec, sizeof(tmp), key_data, key_size);
    MemPtrUnlock(rec);
    DmReleaseRecord(ChooserDB, recNum, true);

    /* Set the category to whatever was viewed last in chooser. */
    DmRecordInfo(ChooserDB, recNum, &attr, 0, 0);
    attr &= ~dmRecAttrCategoryMask;
    if (prefs.ChooserCurrentCategory == dmAllCategories
	|| prefs.ChooserShowAllCategories)
	attr |= dmUnfiledCategory;
    else
	attr |= prefs.ChooserCurrentCategory;
    DmSetRecordInfo(ChooserDB, recNum, &attr, 0);

    /* If the sort order is alphabetic, then sort the database. We
     * assume there won't be a lot of chooser records so we do a full
     * sort.
     */
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	SortAlphabetic();
}

void
Chooser_AddDatabase(const char* title, UInt16 sourceID,
		    void * key_data, UInt16 key_size)
{
    OpenChooserDB();
    AddDatabase(title, sourceID, key_data, key_size);
    CloseChooserDB();
}

static void
RenameDatabase(UInt16 sourceID, void * old_key_data, UInt16 old_key_size,
	       void * new_key_data, UInt16 new_key_size, const char* new_title)
{
    UInt16 numRecords, recNum;
    DataSourceDriver * driver = GetDriverByID(sourceID);

    numRecords = DmNumRecords(ChooserDB);
    for (recNum = 0; recNum < numRecords; ++recNum) {
	MemHandle h;
	ChooserRecord *rec, tmp;

	h = DmGetRecord(ChooserDB, recNum);
	rec = MemHandleLock(h);
	if (rec->sourceID == sourceID
	    && driver->class_ops.CompareKey(old_key_data, old_key_size,
					    (void *) (rec + 1),
					    MemHandleSize(h) -
					    sizeof(ChooserRecord))) {
	    /* Construct the fixed portion of the new record. */
	    MemSet(&tmp, sizeof(tmp), 0);
	    tmp.sourceID = sourceID;
	    StrNCopy(tmp.title, new_title, sizeof(tmp.title));

	    /* Resize the record if needed to accomodate different key size. */
	    if (old_key_size != new_key_size) {
		MemPtrUnlock(rec);
		DmResizeRecord(ChooserDB, recNum, sizeof(tmp) + new_key_size);
		h = DmGetRecord(ChooserDB, recNum);
		rec = MemHandleLock(h);
	    }

	    /* Write the new entry into the database. */
	    DmWrite(rec, 0, &tmp, sizeof(tmp));
	    DmWrite(rec, sizeof(tmp), new_key_data, new_key_size);

	    /* Unlock the pointer and record. */
	    MemPtrUnlock(rec);
	    DmReleaseRecord(ChooserDB, recNum, true);

	    /* Stop the search here. */
	    break;
	} else {
	    /* No match. Just unlock the pointer and record. */
	    MemPtrUnlock(rec);
	    DmReleaseRecord(ChooserDB, recNum, false);
	}
    }

    /* If the sort order is alphabetic, then sort the database. We
     * assume there won't be a lot of chooser records so we do a full
     * sort.
     */
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	SortAlphabetic();
}

void
Chooser_RenameDatabase(UInt16 sourceID,
		       void * old_key_data, UInt16 old_key_size,
		       void * new_key_data, UInt16 new_key_size,
		       const char* new_title)
{
    OpenChooserDB();
    RenameDatabase(sourceID, old_key_data, old_key_size,
		   new_key_data, new_key_size, new_title);
    CloseChooserDB();
}

static Boolean
Seek(UInt16 * indexP, Int16 offset, Int16 direction)
{
    DmSeekRecordInCategory(ChooserDB, indexP, offset, direction,
			   CurrentCategory);
    if (DmGetLastErr())
	return false;

    return true;
}

static void
UpdateScrollers(void)
{
    UInt16 upIndex, downIndex, recNum;
    Boolean scrollUp, scrollDown;
    Int16 row;
    FormPtr form;
    TablePtr table;

    recNum = TopVisibleDatabase;
    scrollUp = Seek(&recNum, 1, dmSeekBackward);

    form = FrmGetActiveForm();
    table = GetObjectPtr(form, ctlID_Chooser_Table);
    row = TblGetLastUsableRow(table);
    if (row != -1)
	recNum = (UInt16) TblGetRowID(table, row);

    scrollDown = Seek(&recNum, 1, dmSeekForward);

    upIndex = FrmGetObjectIndex(form, ctlID_Chooser_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_Chooser_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);
}

static void
DrawIcon(void * table, Int16 row, Int16 col, RectangleType *b)
{
    MemHandle h;
    BitmapType * bitmap;

    h = DmGetResource(iconType, 1001);
    bitmap = (BitmapType *) MemHandleLock(h);
    WinDrawBitmap(bitmap, b->topLeft.x, b->topLeft.y + 1);
    MemPtrUnlock(bitmap);
    DmReleaseResource(h);
}

static void
DrawLabel(void * table, Int16 row, Int16 col, RectangleType *b)
{
    UInt16 recNum;
    MemHandle h;
    ChooserRecord * rec;
    FontID oldFont;

    recNum = (UInt16) TblGetRowID(table, row);
    h = DmQueryRecord(ChooserDB, recNum);
    rec = MemHandleLock(h);

    oldFont = FntSetFont(stdFont);
    WinDrawChars(rec->title, StrLen(rec->title), b->topLeft.x, b->topLeft.y);
    FntSetFont(oldFont);

    MemPtrUnlock(rec);
}

static void
LoadTable(TablePtr table)
{
    UInt16 numRows, row, lineHeight, dataHeight, tableHeight, attr;
    RectangleType r;
    UInt16 recordNum;
    FontID current_font;
    UInt32 uniqueID;
    Boolean secret, masked;

    /* Retrieve the height of the table. */
    TblGetBounds(table, &r);
    tableHeight = r.extent.y;

    /* Determine the height of the font that we draw with. */
    current_font = FntSetFont(stdFont);
    lineHeight = FntLineHeight();
    FntSetFont(current_font);

    /* Make sure that CurrentCategory is set correctly. */
    if (ShowAllCategories) {
	CurrentCategory = dmAllCategories;
    }

    numRows = TblGetNumberOfRows(table);
    recordNum = TopVisibleDatabase;
    dataHeight = 0;

    for (row = 0; row < numRows; row++) {
	/* Stop the loop if no more records are available. */
	if (! Seek(&recordNum, 0, dmSeekForward))
	    break;

	/* Stop the loop if there is no more space. */
	if (! (tableHeight >= dataHeight + lineHeight))
	    break;

	/* Extract the secret and masked (on 3.5 or higher) status. */
	DmRecordInfo(ChooserDB, recordNum, &attr, &uniqueID, 0);
	secret = (attr & dmRecAttrSecret) != 0;
	if (FeatureSet35) {
	    masked = TblRowUsable(table, row) && TblRowMasked(table, row);
	} else {
	    masked = false;
	}

	if ( ! TblRowUsable(table, row)
	     || TblGetRowData(table, row) != uniqueID
	     || (  secret && ! masked)
	     || (! secret &&   masked)) {
	    TblSetRowUsable(table, row, true);
	    TblSetRowID(table, row, recordNum);
	    TblSetRowData(table, row, uniqueID);
	    TblSetRowHeight(table, row, lineHeight);
	    TblSetRowSelectable(table, row, true);
	    TblMarkRowInvalid(table, row);

	    TblSetItemStyle(table, row, 0, customTableItem);
	    TblSetItemStyle(table, row, 1, customTableItem);

	    if (FeatureSet35) {
		if ((PrivateRecordStatus == maskPrivateRecords) && secret)
		    TblSetRowMasked(table, row, true);
		else
		    TblSetRowMasked(table, row, false);
	    }
	}

	/* Increment the record number and the data height. */
	recordNum++;
	dataHeight += lineHeight;
    }

    /* Mark the remainder of the rows as unusable. */
    while (row < numRows) {
        TblSetRowUsable(table, row, false);
        row++;
    }

    UpdateScrollers();
}

static void
InitTable(TablePtr table)
{
    UInt16 row, numRows;
    RectangleType bounds;

    TblGetBounds(table, &bounds);

    TblSetColumnUsable(table, 0, true);
    TblSetColumnWidth(table, 0, 15);
    TblSetCustomDrawProcedure(table, 0, DrawIcon);

    TblSetColumnUsable(table, 1, true);
    TblSetColumnWidth(table, 1, bounds.extent.x - 15);
    TblSetCustomDrawProcedure(table, 1, DrawLabel);

    if (FeatureSet35) {
	TblSetColumnMasked(table, 0, true);
	TblSetColumnMasked(table, 1, true);
    }

    /* Mark all of the data rows unusable so they are reloaded. */
    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++) {
        TblSetRowUsable(table, row, false);
	if (FeatureSet35) {
	    TblSetRowMasked(table, row, false);
	}
    }

    /* Load the data into the table. */
    LoadTable(table);
}

static void
Scroll(WinDirectionType direction)
{
    UInt16 newTopVisibleDatabase;

    newTopVisibleDatabase = TopVisibleDatabase;

    if (direction == winDown) {
	Seek(&newTopVisibleDatabase, 1, dmSeekForward);
    } else {
	Seek(&newTopVisibleDatabase, 1, dmSeekBackward);
    }

    if (TopVisibleDatabase != newTopVisibleDatabase) {
	TopVisibleDatabase = newTopVisibleDatabase;
	FrmUpdateForm(FrmGetActiveFormID(), 0);
    }
}

static UInt16
SelectCategory(void)
{
    FormPtr form;
    UInt16 category;
    Boolean categoryEdited;

    /* Process the category popup list. */
    category = CurrentCategory;

    form = FrmGetActiveForm();
    categoryEdited = CategorySelect(ChooserDB, form,
				    ctlID_Chooser_CategoryTrigger,
				    ctlID_Chooser_CategoryList, true,
				    &category, CategoryName, 1, 0);
        
    if (category == dmAllCategories)
	ShowAllCategories = true;
    else
	ShowAllCategories = false;

    if (categoryEdited || (category != CurrentCategory)) {
	CurrentCategory = category;
	TopVisibleDatabase = 0;
	FrmUpdateForm(FrmGetActiveFormID(), updateCategoryChanged);
    }

    return category;
}

static Boolean
UpdateForm(FormPtr form, UInt16 updateCode)
{
    TablePtr table;
    ControlPtr ctl;

    if (updateCode & frmRedrawUpdateCode)
	return false;

    table = GetObjectPtr(form, ctlID_Chooser_Table);
    ctl = GetObjectPtr(form, ctlID_Chooser_CategoryTrigger);

    if (updateCode & updateInitTable) {
	TblEraseTable(table);
	InitTable(table);
	TblDrawTable(table);
	return true;
    }

    if (updateCode & updateCategoryChanged) {
	/* Set the category trigger to the new category. */
	CategoryGetName(ChooserDB, CurrentCategory, CategoryName);
	CategorySetTriggerLabel(ctl, CategoryName);
    }

    /* Force an update of the table. */
    LoadTable(table);
    TblRedrawTable(table);

    return true;
}

static WinHandle InvertMoveIndicator(RectanglePtr itemR, WinHandle savedBits, 
				     Boolean draw)
{
    UInt16 i;
    UInt16 err;
    WinHandle winH = 0;
    RectangleType indictatorR;
    CustomPatternType pattern;
    CustomPatternType savedPattern;

    indictatorR.topLeft.x = itemR->topLeft.x;
    indictatorR.topLeft.y = itemR->topLeft.y + itemR->extent.y - 2;
    indictatorR.extent.x = itemR->extent.x;
    indictatorR.extent.y = 2;
                                
    if (draw) {
	WinGetPattern(&savedPattern);
        
	for (i = 0; i < sizeof(CustomPatternType)/2; i++) {
	    pattern[i] = grayHLinePattern;
	    pattern[i+1] = grayHLinePatternOdd;
	}
        
	WinSetPattern((const CustomPatternType *) &pattern);
        
	winH = WinSaveBits(&indictatorR, &err);
        
	WinFillRectangle(&indictatorR, 0);
        
	WinSetPattern((const CustomPatternType *) &savedPattern);
    } else {
	WinRestoreBits(savedBits,
		       indictatorR.topLeft.x, indictatorR.topLeft.y);
    }
        
    return (winH);
}

static void
SelectItem(EventPtr event)
{
    Int16 row;
    UInt16 selectedRow;
    UInt16 column;
    UInt16 recordNum;
    UInt16 selectedRecord;
    Int16 x, y;
    Boolean penDown = true;
    Boolean moving = false;
    Boolean selected = true;
    TablePtr table;
    WinHandle savedBits = 0;
    RectangleType r;

    row = event->data.tblEnter.row;
    column = event->data.tblEnter.column;
    table = event->data.tblEnter.pTable;

    // Highlight the item the pen when down on.
    selectedRecord = TblGetRowID(table, row);
    TblGetItemBounds(table, row, column, &r);
    WinInvertRectangle(&r, 0);

    while (true) {
	PenGetPoint (&x, &y, &penDown);
	if (! penDown)
	    break;
		
	if (! moving) {
	    if (prefs.ChooserSortOrder == chooserSortOrderManual) {
		if (! RctPtInRectangle(x, y, &r)) {
		    moving = true;
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    } else {
		selected = RctPtInRectangle(x, y, &r);
	    }
	}

	else if (! RctPtInRectangle(x, y, &r)) {
	    // Above the first item ?
	    if (row < 0) {
		if (y >= r.topLeft.y) {
		    row++;
		    InvertMoveIndicator(&r, savedBits, false);
		    r.topLeft.y -= r.extent.y;
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}					
	    }

	    // Move up.
	    else if (y < r.topLeft.y) {
		recordNum = TblGetRowID(table, row);
		if (Seek(&recordNum, 1, dmSeekBackward)) {
		    InvertMoveIndicator(&r, savedBits, false);
		    if (row)
			row--;
		    else {
			Scroll(winUp);
			if (TblFindRowID(table, selectedRecord, &selectedRow))
			    {
				TblGetItemBounds(table, selectedRow,
						 column, &r);
				WinInvertRectangle(&r, 0);
			    }
		    }
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
		else if (row == 0) {
		    row--;
		    InvertMoveIndicator(&r, savedBits, false);
		    r.topLeft.y -= r.extent.y;
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    }

	    // Move down
	    else {
		recordNum = TblGetRowID(table, row);
		if (Seek(&recordNum, 1, dmSeekForward)) {
		    InvertMoveIndicator(&r, savedBits, false);
		    if (row < TblGetLastUsableRow(table))
			row++;
		    else {
			Scroll(winDown);
			if (TblFindRowID(table, selectedRecord, &selectedRow))
			    {
				TblGetItemBounds(table,
						 selectedRow, column, &r);
				WinInvertRectangle (&r, 0);
			    }
		    }
		    TblGetItemBounds(table, row, column, &r);
		    savedBits = InvertMoveIndicator(&r, 0, true);
		}
	    }
	}
    }
		

    // Turn off the move indicator, if it is on.
    if (moving) {
	savedBits = InvertMoveIndicator(&r, savedBits, false);
    }
	
    // If the highlighted item is visible, unhighlight it.
    if (TblFindRowID(table, selectedRecord, &selectedRow)) {
	TblGetItemBounds(table, selectedRow, column, &r);
	WinInvertRectangle(&r, 0);
    }

    if (moving) {
	if (row >= 0) {
	    recordNum = TblGetRowID(table, row);
	    if (selectedRecord == recordNum)
		return;
			
	    recordNum++;
	} else {
	    recordNum = TblGetRowID(table, 0);;
	}

	DmMoveRecord(ChooserDB, selectedRecord, recordNum);
		
	if (selectedRecord < TopVisibleDatabase)
	    TopVisibleDatabase--;

	LoadTable(table);
	TblRedrawTable(table);
    }
		
    // If we didn't move the item, then it's been selected so we will
    // just force a tblSelectEvent to trigger the normal handler.
    else if (prefs.ChooserSortOrder == chooserSortOrderManual || selected) {
	EventType e;

	EvtCopyEvent(event, &e);
	e.eType = tblSelectEvent;
	EvtAddEventToQueue(&e);
    }
}

static void
DoSecurity(void)
{
    Boolean hiding;
 
    hiding = (PrivateRecordStatus == hidePrivateRecords);
 
    PrivateRecordStatus = SecSelectViewStatus();
 
    if (hiding ^ (PrivateRecordStatus == hidePrivateRecords)) {
	/* Close the chooser database and then reopen it again. */
	CloseChooserDB();
	OpenChooserDB();
	ErrFatalDisplayIf(!ChooserDB, "Unable to reopen database.");
    }
}

static void PopupOptionsDialog(void) SECT_UI;
static void FillDriverList(FormPtr form, ListPtr lst, UInt16 index,
			   DataSourceDriver * drivers,
			   char** *names_ptr, UInt16 *count_ptr) SECT_UI;

static void
SetupActions(FormPtr form, char *** names, UInt16 ** actions)
{
    UInt16 i, j, count, lstIndex;
    RectangleType bounds;
    ListType * lst;

    /* Count the number of actions for the popup menu. */
    count = 0;
    for (i = 0; popup_menu[i].nameID != 0; ++i) {
	if (popup_menu[i].needsFeatureSet30) {
	    if (FeatureSet30) count++;
	} else
	    count++;
    }

    /* Allocate space for the strings and actions. */
    *names = MemPtrNew(count * sizeof(char *));
    *actions = MemPtrNew(count * sizeof(UInt16));

    /* Fill in the name for each action. */
    for (i = j = 0; popup_menu[i].nameID != 0; ++i) {
	Boolean add = true;

	/* Do not add this item if a feature is missing. */
	if (popup_menu[i].needsFeatureSet30 && !FeatureSet30)
	    add = false;

	/* Add the item to the popup actions. */
	if (add) {
	    (*names)[j] = CopyStringResource(popup_menu[i].nameID);
	    (*actions)[j] = popup_menu[i].actionID;
	    j++;
	}
    }

    /* Retrieve a pointer to the popup list. */
    lstIndex = FrmGetObjectIndex(form, ctlID_Chooser_ActionList);
    lst = FrmGetObjectPtr(form, lstIndex);

    /* Setup the list choices, selection, and the trigger. */
    LstSetListChoices(lst, *names, count);
    LstSetHeight(lst, count);

    /* Determine the minimum width needed for the list. */
    FrmGetObjectBounds(form, lstIndex, &bounds);
    bounds.extent.x = 0;
    for (i = 0; i < count; ++i) {
	UInt16 width = FntCharsWidth((*names)[i], StrLen((*names)[i])) + 10;
	if (width > bounds.extent.x) bounds.extent.x = width;
    }
    FrmSetObjectBounds(form, lstIndex, &bounds);
}

static void
FreeActions(char ** names, UInt16 * actions)
{
    UInt16 i, count;

    /* Count the number of actions for the popup menu. */
    count = 0;
    for (i = 0; popup_menu[i].nameID != 0; ++i) {
	if (popup_menu[i].needsFeatureSet30) {
	    if (FeatureSet30) count++;
	} else
	    count++;
    }

    /* Free all of the allocated name strings. */
    for (i = 0; i < count; ++i) MemPtrFree(names[i]);
    MemPtrFree(names);
    MemPtrFree(actions);
}

Boolean
ChooserHandleEvent(EventPtr event)
{
    static char ** names;
    static UInt16 * actions;

    FormPtr form;
    TablePtr table;
    ControlPtr ctl;
    ListPtr lst;
    UInt16 index;
    ChooserRecord * record;
    MemHandle h;
    UInt16 mode, action;

    switch (event->eType) {
    case frmOpenEvent:
	/* If we entered this form at the start of the application, no
	 * data source will be open. If one is, then we need to close
	 * it since the user entered this form via the list view.
	 */
	if (CurrentSource) {
	    CloseDataSource(CurrentSource);
	    CurrentSource = 0;
	}

	/* Initialize our global variables. */
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_Chooser_Table);
	TopVisibleDatabase = 0;
	CurrentCategory = prefs.ChooserCurrentCategory;
	ShowAllCategories = prefs.ChooserShowAllCategories;

	/* Open the chooser database and do a rescan if need be. */
	OpenChooserDB();
	if (prefs.ChooserForceRescan) {
	    RescanDatabases();
	    prefs.ChooserForceRescan = false;
	}

	/* Initialize the table. */
	InitTable(table);

	/* Setup the action list and trigger. */
	SetupActions(form, &names, &actions);

	/* Set the category name in the trigger. */
	ctl = GetObjectPtr(form, ctlID_Chooser_CategoryTrigger);
	CategoryGetName(ChooserDB, CurrentCategory, CategoryName);
	CategorySetTriggerLabel(ctl, CategoryName);

	/* Draw the form. */
	FrmDrawForm(form);

	return true;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	return UpdateForm(form, event->data.frmUpdate.updateCode);

    case frmCloseEvent:
	CloseChooserDB();
	prefs.ChooserCurrentCategory = CurrentCategory;
	prefs.ChooserShowAllCategories = ShowAllCategories;
	FreeActions(names, actions);
	break;

    case tblEnterEvent:
	if (event->data.tblEnter.column == 1) {
	    SelectItem(event);
	    return true;
	}
	break;

    case tblSelectEvent:
	/* For PalmOS 3.5, handle private records that are masked. */
	if (FeatureSet35 && TblRowMasked(event->data.tblSelect.pTable,
					 event->data.tblSelect.row)) {
	    if (SecVerifyPW(showPrivateRecords)) {
		/* Reset the table so all private records are shown. */
		PrivateRecordStatus = showPrivateRecords;
		InitTable(event->data.tblSelect.pTable);
		TblRedrawTable(event->data.tblSelect.pTable);
		/* we fall through into the selection phase */
	    } else {
		/* If password didn't check, then bail out of selection. */
		return true;
	    }
	}

	/* If use clicked on icon, then popup the action list. */
	if (event->data.tblSelect.column == 0) {
	    RectangleType r;
	    Int16 result;

	    /* Remove the highlight on the icon. */
	    TblUnhighlightSelection(event->data.tblSelect.pTable);

	    TblGetItemBounds(event->data.tblSelect.pTable,
			     event->data.tblSelect.row,
			     event->data.tblSelect.column, &r);

	    form = FrmGetActiveForm();
	    lst = GetObjectPtr(form, ctlID_Chooser_ActionList);

	    LstSetPosition(lst, r.topLeft.x, r.topLeft.y);
	    LstSetSelection(lst, noListSelection);
	    result = LstPopupList(lst);
	    if (result >= 0)
		action = actions[(UInt16) result];
	    else
		return true;
	} else {
	    action = actionID_Open;
	}

	index = TblGetRowID(event->data.tblSelect.pTable,
			    event->data.tblSelect.row);
	h = DmQueryRecord(ChooserDB, index);
	record = MemHandleLock(h);

	switch (action) {
	case actionID_OpenReadOnly:
	    /* OPEN READ-ONLY */
	    mode = dmModeReadOnly;
	    goto do_open;

	case actionID_Open:
	    /* OPEN */
	    mode = dmModeReadWrite;
	do_open:
	    CurrentSource = OpenDataSource(DriverList, mode, record->sourceID,
					   (void *) (record + 1),
					   MemHandleSize(h) -
					   sizeof(ChooserRecord));
	    if (CurrentSource)
		FrmGotoForm(formID_ListView);
	    else
		FrmAlert(alertID_CannotOpenDB);
	    break;

	case actionID_Info:
	    /* INFO */
	    InfoDatabase = index;
	    FrmPopupForm(formID_ChooserInfoDialog);
	    break;

	case actionID_Delete:
	    /* DELETE */
	    if (FrmCustomAlert(alertID_ConfirmDatabaseDelete,
			       record->title, " ", " ") == 0) {

		DeleteDataSource(DriverList, record->sourceID,
				 (void *) (record + 1),
				 MemHandleSize(h) - sizeof(ChooserRecord));
		MemPtrUnlock(record);
		DmRemoveRecord(ChooserDB, index);
		FrmUpdateForm(FrmGetActiveFormID(), 0);
		return true;
	    }
	    break;

	case actionID_Rename:
	    /* RENAME */
	    InfoDatabase = index;
	    FrmPopupForm(formID_RenameDlg);
	    break;

	case actionID_Duplicate:
	    /* DUPLICATE */
	    InfoDatabase = index;
	    FrmPopupForm(formID_DuplicateDlg);
	    break;

	case actionID_Beam:
	    /* BEAM */
	    BeamDataSource(DriverList, record->sourceID,
			   (void *) (record + 1),
			   MemHandleSize(h) - sizeof(ChooserRecord));
	    break;

	case actionID_CreateUsing:
	    {
		DataSource * src;

		src = OpenDataSource(DriverList, dmModeReadOnly,
				     record->sourceID, (void *) (record + 1),
				     MemHandleSize(h) - sizeof(ChooserRecord));
		if (src) {
		    DesignViewInitialSchema = MemPtrNew(sizeof(DataSourceSchema));
		    CopySchema(DesignViewInitialSchema, &src->schema);
		    CloseDataSource(src);
		    FrmPopupForm(formID_CreateDlg);
		}
	    }
	    break;
	}

	MemPtrUnlock(record);
	return true;
					      
    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_Chooser_NewButton:
	    DesignViewInitialSchema = 0;
	    FrmPopupForm(formID_CreateDlg);
	    return true;

	case ctlID_Chooser_CategoryTrigger:
	    SelectCategory();
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_Chooser_UpButton:
	    Scroll(winUp);
	    break;
	case ctlID_Chooser_DownButton:
	    Scroll(winDown);
	    break;
	}
	break;

    case keyDownEvent:
        switch (event->data.keyDown.chr) {
        case pageUpChr:
            Scroll(winUp);
            return true;

        case pageDownChr:
            Scroll(winDown);
            return true;
        }
        break;

    case menuCmdBarOpenEvent:
	if (FeatureSet35) {
	    MenuCmdBarAddButton(menuCmdBarOnLeft, BarSecureBitmap,
				menuCmdBarResultMenuItem,
				menuitemID_Security, 0);
	}
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_RescanDatabases:
	    RescanDatabases();
	    FrmUpdateForm(FrmGetActiveFormID(), 0);
	    return true;

	case menuitemID_NewDatabase:
	    DesignViewInitialSchema = 0;
	    FrmPopupForm(formID_CreateDlg);
	    return true;

	case menuitemID_ChooserDisplayOpts:
	    PopupOptionsDialog();
	    return true;

	case menuitemID_Security:
	    if (FeatureSet35) {
		DoSecurity();
		FrmUpdateForm(FrmGetActiveFormID(), updateInitTable);
	    } else {
		FrmAlert(alertID_NoSupportBelowOS35);
	    }
	    return true;

	case menuitemID_Help:
	    FrmHelp(stringID_Help_Chooser);
	    return true;

	default:
	    return HandleCommonMenuEvent(event->data.menu.itemID);
	}
	break;

    default:
	break;
    }

    return false;
}

static void
PopupOptionsDialog(void)
{
    FormPtr form, oldForm;
    Char * labels[3] = {0, 0, 0};
    ControlPtr ctl;
    ListPtr lst;
    UInt8 OldChooserSortOrder = prefs.ChooserSortOrder;

    oldForm = FrmGetActiveForm();
    form = FrmInitForm(formID_ChooserOptsDialog);
    FrmSetActiveForm(form);

    labels[0] = CopyStringResource(stringID_Alphabetic);
    labels[1] = CopyStringResource(stringID_Manual);

    ctl = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderTrigger);
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	CtlSetLabel(ctl, labels[0]);
    else
	CtlSetLabel(ctl, labels[1]);

    lst = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderList);
    LstSetListChoices(lst, labels, 2);
    LstSetHeight(lst, 2);
    if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	LstSetSelection(lst, 0);
    else
	LstSetSelection(lst, 1);

    if (FrmDoDialog(form) == ctlID_ChooserOptsDialog_OkayButton) {
	lst = GetObjectPtr(form, ctlID_ChooserOptsDialog_SortOrderList);
	if (LstGetSelection(lst) == 0)
	    prefs.ChooserSortOrder = chooserSortOrderAlphabetic;
	else
	    prefs.ChooserSortOrder = chooserSortOrderManual;
    }

    FrmDeleteForm(form);
    FrmSetActiveForm(oldForm);

    MemPtrFree(labels[0]);
    MemPtrFree(labels[1]);

    if (OldChooserSortOrder != prefs.ChooserSortOrder) {
	FrmUpdateForm(formID_Chooser, updateSortOrderChanged);
	if (prefs.ChooserSortOrder == chooserSortOrderAlphabetic)
	    SortAlphabetic();
    }
}

static UInt16
ApplyChanges(UInt16 recNum, Boolean CategoryEdited,
	     UInt16 category, Boolean secret)
{
    UInt16 updateCode = 0;
    Boolean dirty = false;
    UInt16 attr;

    /* Retrieve the existing status of the record. */
    DmRecordInfo(ChooserDB, recNum, &attr, 0, 0);

    /* If the category changed, store the new category. */
    if (CategoryEdited || (attr & dmRecAttrCategoryMask) != category) {
	attr &= ~(dmRecAttrCategoryMask);
	attr |= category;

	dirty = true;
	updateCode |= updateCategoryChanged;
    }

    if (((attr & dmRecAttrSecret) == dmRecAttrSecret) != secret) {
	if (secret)
	    attr |= dmRecAttrSecret;
	else
	    attr &= ~(dmRecAttrSecret);
	dirty = true;
    }

    /* Save the new category and/or secret status. */
    if (dirty) {
	attr |= dmRecAttrDirty;
	DmSetRecordInfo(ChooserDB, recNum, &attr, 0);
    }

    return updateCode;
}

Boolean
ChooserInfoDialogHandleEvent(EventPtr event)
{
    static Char InfoName[64];
    static Boolean CategoryEdited;
    static UInt16 InfoCategory;

    FormPtr form;
    ControlPtr ctl;
    FieldPtr fld;
    ChooserRecord * record;
    MemHandle h;
    Char * label;
    UInt16 attr;
    UInt16 updateCode;
    DataSourceDriver * driver;

    form = FrmGetActiveForm();

    switch (event->eType) {
    case frmOpenEvent:
	/* Initialze the state variables. */
	CategoryEdited = false;

	/* Lock down the information on this data source. */
	h = DmQueryRecord(ChooserDB, InfoDatabase);
	record = MemHandleLock(h);

	/* Set the title to the database name. */
	StrCopy(InfoName, record->title);
	FrmSetTitle(form, InfoName);

	/* Determine the name of its data source driver. */
	driver = GetDriverByID(record->sourceID);
	fld = GetObjectPtr(form, ctlID_ChooserInfoDialog_TypeField);
	FldSetTextPtr(fld, driver->name);

	/* Unlock the information on the data source. */
	MemPtrUnlock(record);

	/* Set the category trigger label. */
	DmRecordInfo(ChooserDB, InfoDatabase, &attr, 0, 0);
	InfoCategory = attr & dmRecAttrCategoryMask;
	ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_CategoryTrigger);
	label = (char *) CtlGetLabel(ctl);
        CategoryGetName(ChooserDB, InfoCategory, label);
        CategorySetTriggerLabel(ctl, label);

	/* Set the secret checkbox to current status. */
	ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_SecretCheckbox);
	if (attr & dmRecAttrSecret)
	    CtlSetValue(ctl, 1);
	else
	    CtlSetValue(ctl, 0);

	/* Finally, draw the form. */
	FrmDrawForm(form);

	return true;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_ChooserInfoDialog_OkayButton:
	    ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_SecretCheckbox);
	    updateCode = ApplyChanges(InfoDatabase, CategoryEdited,
				      InfoCategory, CtlGetValue(ctl));
	    FrmReturnToForm(0);
	    FrmUpdateForm(formID_Chooser, updateCode);
	    return true;

	case ctlID_ChooserInfoDialog_CancelButton:
	    FrmReturnToForm(0);
	    if (CategoryEdited)
		FrmUpdateForm(formID_Chooser, updateCategoryChanged);
	    return true;

	case ctlID_ChooserInfoDialog_CategoryTrigger:
	    ctl = GetObjectPtr(form, ctlID_ChooserInfoDialog_CategoryTrigger);
	    label = (char *) CtlGetLabel(ctl);
	    CategoryEdited = CategorySelect(ChooserDB, form,
					    ctlID_ChooserInfoDialog_CategoryTrigger,
					    ctlID_ChooserInfoDialog_CategoryList,
					    false, &InfoCategory, label, 1, 0);
	    return true;

	}
	return false;

    default:
	break;
    }

    return false;
}

static void
DoDuplicate(UInt16 src_sourceID, void * src_key_data, UInt16 src_key_size,
	    DataSourceDriver * driver,
	    const char* dst_title, Boolean everything) SECT_UI;

typedef struct {
    /* Info needed to get old data out of source database. */
    DataSourceGetter getter;
    void * rec_data;

    /* Info on the source database so conversion can be determined. */
    DataSource * src;
} DuplicateData;

static char * Duplicate_GetString(void * ptr, UInt16 fieldNum) SECT_UI;
static Int32 Duplicate_GetInteger(void * ptr, UInt16 fieldNum) SECT_UI;
static Boolean Duplicate_GetBoolean(void * ptr, UInt16 fieldNum) SECT_UI;
static void Duplicate_GetDate(void * ptr, UInt16 fieldNum,
			      UInt16* year, UInt8* month, UInt8* day) SECT_UI;
static void Duplicate_GetTime(void * ptr, UInt16 fieldNum,
			      UInt8* hour, UInt8* minute) SECT_UI;

static char *
Duplicate_GetString(void * ptr, UInt16 fieldNum)
{
    static char buf[32];

    DuplicateData * data = (DuplicateData *) ptr;
    Int32 v;
    char *p;

    switch (data->src->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
        p = data->getter.GetString(data->rec_data, fieldNum);
        break;

    case FIELD_TYPE_INTEGER:
        v = data->getter.GetInteger(data->rec_data, fieldNum);
        StrPrintF(buf, "%ld", v);
        p = buf;
        break;

    case FIELD_TYPE_BOOLEAN:
        if (data->getter.GetBoolean(data->rec_data, fieldNum))
            buf[0] = '1';
        else
            buf[0] = '0';
        buf[1] = '\0';
        p = buf;
        break;

    default:
        buf[0] = '\0';
        p = buf;
	break;
    }

    return p;
}

static Int32
Duplicate_GetInteger(void * ptr, UInt16 fieldNum)
{
    DuplicateData * data = (DuplicateData *) ptr;
    Int32 v;
    char * p;

    switch (data->src->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
        p = data->getter.GetString(data->rec_data, fieldNum);
        v = String2Long(p);
        break;

    case FIELD_TYPE_INTEGER:
        v = data->getter.GetInteger(data->rec_data, fieldNum);
        break;

    case FIELD_TYPE_BOOLEAN:
        if (data->getter.GetBoolean(data->rec_data, fieldNum))
            v = 1;
        else
            v = 0;
        break;

    default:
        v = 0;
        break;
    }

    return v;
}

static Boolean
Duplicate_GetBoolean(void * ptr, UInt16 fieldNum)
{
    DuplicateData * data = (DuplicateData *) ptr;
    Boolean b;

    switch (data->src->schema.fields[fieldNum].type) {
    case FIELD_TYPE_STRING:
        b = false;
        break;

    case FIELD_TYPE_INTEGER:
        if (data->getter.GetInteger(data->rec_data, fieldNum) != 0)
            b = true;
        else
            b = false;
        break;

    case FIELD_TYPE_BOOLEAN:
        b = data->getter.GetBoolean(data->rec_data, fieldNum);
        break;

    default:
        b = false;
        break;
    }

    return b;
}

static void
Duplicate_GetDate(void * ptr, UInt16 fieldNum,
		  UInt16* year, UInt8* month, UInt8* day)
{
    DuplicateData * data = (DuplicateData *) ptr;
    DateTimeType datetime;

    switch (data->src->schema.fields[fieldNum].type) {
    case FIELD_TYPE_DATE:
        data->getter.GetDate(data->rec_data, fieldNum, year, month, day);
        break;
 
    default:
        TimSecondsToDateTime(TimGetSeconds(), &datetime);
        *year = datetime.year;
        *month = datetime.month;
        *day = datetime.day;
        break;
    }
}

static void
Duplicate_GetTime(void * ptr, UInt16 fieldNum, UInt8* hour, UInt8* minute)
{
    DuplicateData * data = (DuplicateData *) ptr;
    DateTimeType datetime;

    switch (data->src->schema.fields[fieldNum].type) {
    case FIELD_TYPE_TIME:
        data->getter.GetTime(data->rec_data, fieldNum, hour, minute);
        break;
 
    default:
        TimSecondsToDateTime(TimGetSeconds(), &datetime);
        *hour = datetime.hour;
        *minute = datetime.minute;
        break;
    }
}

static void
DoDuplicate(UInt16 src_sourceID, void * src_key_data, UInt16 src_key_size,
	    DataSourceDriver * driver,
	    const char* dst_title, Boolean everything)
{
    DataSourceGetter duplicate_getter = {
	GetString: Duplicate_GetString,
	GetInteger: Duplicate_GetInteger,
	GetBoolean: Duplicate_GetBoolean,
	GetDate: Duplicate_GetDate,
	GetTime: Duplicate_GetTime
    };
    DuplicateData data;
    DataSource *src, *dst;
    DataSourceSchema schema;
    Err err;
    void * dst_key_data;
    UInt16 dst_key_size, recNum, i;
    void * indicator_context;

    /* Display an on-screen "Working..." message. */
    indicator_context = DrawWorkingIndicator();

    /* Open the source that we are duplicating. */
    src = OpenDataSource(DriverList, dmModeReadOnly, src_sourceID,
			 src_key_data, src_key_size);
    if (!src) {
	EraseWorkingIndicator(indicator_context);
        FrmAlert(alertID_RebuildFailed);
 	return;
    }

    /* Make a local copy of the source's schema. If the destination
     * driver doesn't support all of the source's field types, then
     * change that field type to string. If the string field typeisn't
     * supported, then flag an error.
     */
    CopySchema(&schema, &src->schema);
    for (i = 0; i < schema.numFields; ++i) {
	if (! driver->class_ops.SupportsFieldType(schema.fields[i].type)) {
	    if (! driver->class_ops.SupportsFieldType(FIELD_TYPE_STRING)) {
		EraseWorkingIndicator(indicator_context);
		FrmAlert(alertID_RebuildFailed);
		MemPtrFree(schema.fields);
		CloseDataSource(src);
		return;
	    }
	    schema.fields[i].type = FIELD_TYPE_STRING;
	}
    }

    /* Create the destination data source. */
    err = driver->class_ops.Create(dst_title, &schema,
				   &dst_key_data, &dst_key_size);
    if (err != 0) {
	EraseWorkingIndicator(indicator_context);
        FrmAlert(alertID_RebuildFailed);
	MemPtrFree(schema.fields);
	CloseDataSource(src);
	return;
    }

    /* Free the copied schema now that it is outlived its usefulness. */
    MemPtrFree(schema.fields);

    /* Loop for each record and use the data source methods to have fun. */
    if (everything) {
	/* Open the destination for read/write access. */
	dst = OpenDataSource(DriverList, dmModeReadWrite, driver->sourceID,
			     dst_key_data, dst_key_size);
	if (!dst) {
	    EraseWorkingIndicator(indicator_context);
	    FrmAlert(alertID_RebuildFailed);
	    CloseDataSource(src);
	    return;
	}

	/* Fill in the information needed by our custom getter. */
	data.src = src;

	recNum = 0;
	while (1) {
	    /* Make sure that this is a valid record. */
	    if (! src->ops.Seek(src, &recNum, 0, dmSeekForward))
		break;

	    /* Copy the record to the destination. */
	    src->ops.LockRecord(src, recNum, false,
				&data.getter, &data.rec_data);
	    err = dst->ops.UpdateRecord(dst, dmMaxRecordIndex, false,
					&duplicate_getter, &data);
	    src->ops.UnlockRecord(src, recNum, false, data.rec_data);

	    /* If the copy failed, then notify the user. */
	    if (err != 0) {
		/* Remove working indicator and display the error. */
		EraseWorkingIndicator(indicator_context);
		FrmAlert(DeviceFullAlert);

		/* Close both data sources and delete the destination. */
		CloseDataSource(dst);
		CloseDataSource(src);
		DeleteDataSource(DriverList, driver->sourceID,
				 dst_key_data, dst_key_size);
		MemPtrFree(dst_key_data);
		return;
	    }

	    /* Advance the record number. */
	    recNum++;
	}

	/* Now copy each view in the source over. */
	for (i = 0; i < src->ops.GetNumOfViews(src); ++i) {
	    DataSourceView * view;

	    view = src->ops.GetView(src, i);
	    dst->ops.StoreView(dst, i, view);
	    src->ops.PutView(src, i, view);
	}

	/* Close the destination. */
	CloseDataSource(dst);
    }

    /* Close the source. */
    CloseDataSource(src);

    /* We can now add the database to the chooser list. */
    EraseWorkingIndicator(indicator_context);
    AddDatabase(dst_title, driver->sourceID, dst_key_data, dst_key_size);
    MemPtrFree(dst_key_data);
    FrmUpdateForm(formID_Chooser, 0);
}

static void
FillDriverList(FormPtr form, ListPtr lst, UInt16 index,
	       DataSourceDriver * drivers,
	       char** *names_ptr, UInt16 *count_ptr)
{
    DataSourceDriver * driver;
    UInt16 i, count;
    char ** names;
    RectangleType bounds;

    /* Count the number of drivers available. */
    count = 0;
    for (driver = drivers; driver; driver = driver->next)
	count++;

    /* Allocate space for the names. */
    names = MemPtrNew(count * sizeof(char *));

    /* Fill in the space with the driver names. */
    for (i = 0, driver = drivers; driver; ++i, driver = driver->next)
	names[i] = driver->name;

    /* Output the variables to the caller. */
    *names_ptr = names;
    *count_ptr = count;

    /* Update the list with the names. */
    LstSetListChoices(lst, names, count);
    LstSetHeight(lst, count);

    /* Adjust the width of the list to fit all the names. */
    FrmGetObjectBounds(form, index, &bounds);
    bounds.extent.x = 0;
    for (i = 0; i < count; ++i) {
	UInt16 width = FntCharsWidth(names[i], StrLen(names[i]));
	if (width + 5 > bounds.extent.x)
	    bounds.extent.x = width + 5;
    }
    FrmSetObjectBounds(form, index, &bounds);
}

Boolean
DuplicateDlgHandleEvent(EventPtr event)
{
    static char **drvnames;

    FormPtr form;
    ListPtr lst;
    ControlPtr ctl;
    FieldPtr fld;
    MemHandle h;
    char *dst_title, *p;
    DataSourceDriver * driver;
    UInt16 drvcount, i, lstIndex, fldIndex;
    Boolean everything;
    ChooserRecord * rec;

    switch (event->eType) {
    case frmOpenEvent:
	form = FrmGetActiveForm();

	/* Update the driver list and trigger in the UI. */
	lstIndex = FrmGetObjectIndex(form, ctlID_DuplicateDlg_DriverList);
	lst = FrmGetObjectPtr(form, lstIndex);
	FillDriverList(form, lst, lstIndex, DriverList, &drvnames, &drvcount);
	LstSetSelection(lst, 0);
	ctl = GetObjectPtr(form, ctlID_DuplicateDlg_DriverTrigger);
	CtlSetLabel(ctl, drvnames[0]);
	
	/* Update the extent trigger label. */
	lst = GetObjectPtr(form, ctlID_DuplicateDlg_ExtentList);
	ctl = GetObjectPtr(form, ctlID_DuplicateDlg_ExtentTrigger);
	LstSetSelection(lst, 0);
	CtlSetLabel(ctl, LstGetSelectionText(lst, 0));

	/* Give the name field a handle to use. */
        h = MemHandleNew(32);
        p = MemHandleLock(h);
        p[0] = '\0';
        MemPtrUnlock(p);
        fldIndex = FrmGetObjectIndex(form, ctlID_DuplicateDlg_NameField);
        fld = FrmGetObjectPtr(form, fldIndex);
        FldSetTextHandle(fld, h);

	/* Draw the form and put focus on name entry field. */
	FrmDrawForm(form);
	FrmSetFocus(form, fldIndex);

	return true;

    case frmCloseEvent:
	MemPtrFree(drvnames);
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_DuplicateDlg_OkayButton:
	    form = FrmGetActiveForm();

	    /* Determine if all data should be copied or just schema. */
	    lst = GetObjectPtr(form, ctlID_DuplicateDlg_ExtentList);
	    everything = (LstGetSelection(lst) == 0) ? true : false;

	    /* Figure out the driver to use. */
	    lst = GetObjectPtr(form, ctlID_DuplicateDlg_DriverList);
	    i = LstGetSelection(lst);
	    driver = DriverList;
	    while (i > 0) {
		driver = driver->next;
		--i;
	    }

	    /* Lock down the name field. */
	    fld = GetObjectPtr(form, ctlID_DuplicateDlg_NameField);
	    h = FldGetTextHandle(fld);
	    p = MemHandleLock(h);

	    /* Make sure the name isn't blank. */
	    if (StrLen(p) == 0) {
		MemPtrUnlock(p);
		FrmAlert(alertID_NameBlank);
		return true;
	    }

            /* Make sure the title doesn't already exist. */
            if (! driver->class_ops.CheckTitle(p)) {
                MemPtrUnlock(p);
		FrmAlert(alertID_NameConflict);
		return true;
            }

	    /* Now copy the name into a temporary buffer. */
	    dst_title = MemPtrNew(StrLen(p) + 1);
	    StrCopy(dst_title, p);
	    MemPtrUnlock(p);

	    /* Remove the dialog from the screen before copy operation. */
	    FrmReturnToForm(0);
	    MemPtrFree(drvnames);

	    /* Extract database key from chooser database. Duplicate. */
	    h = DmQueryRecord(ChooserDB, InfoDatabase);
	    rec = MemHandleLock(h);
	    DoDuplicate(rec->sourceID, (void *) (rec + 1),
			MemHandleSize(h) - sizeof(ChooserRecord),
			driver, dst_title, everything);
	    MemPtrUnlock(rec);
	    MemPtrFree(dst_title);

	    return true;

	case ctlID_DuplicateDlg_CancelButton:
	    FrmReturnToForm(0);
	    MemPtrFree(drvnames);
	    return true;
	}
	break;

    default:
	break;
    }

    return false;
}

Boolean
CreateDlgHandleEvent(EventPtr event)
{
    static char **drvnames;

    FormPtr form;
    ListPtr lst;
    ControlPtr ctl;
    FieldPtr fld;
    MemHandle h;
    char *name;
    DataSourceDriver * driver;
    UInt16 drvcount, lstIndex, fldIndex, i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Update the driver list and trigger in the UI. */
	form = FrmGetActiveForm();
	lstIndex = FrmGetObjectIndex(form, ctlID_CreateDlg_DriverList);
	lst = FrmGetObjectPtr(form, lstIndex);
	FillDriverList(form, lst, lstIndex, DriverList, &drvnames, &drvcount);
	LstSetSelection(lst, 0);
	ctl = GetObjectPtr(form, ctlID_CreateDlg_DriverTrigger);
	CtlSetLabel(ctl, drvnames[0]);

	/* Give the name field a handle to use. */
	h = MemHandleNew(32);
	name = MemHandleLock(h);
	name[0] = '\0';
	MemPtrUnlock(name);
	fldIndex = FrmGetObjectIndex(form, ctlID_CreateDlg_NameField);
	fld = FrmGetObjectPtr(form, fldIndex);
	FldSetTextHandle(fld, h);

	FrmDrawForm(form);
	FrmSetFocus(form, fldIndex);

	return true;

    case frmCloseEvent:
	MemPtrFree(drvnames);
	if (DesignViewInitialSchema) {
	    /* Design view will not have a chance to free initial schema. */
	    MemPtrFree(DesignViewInitialSchema->fields);
	    MemPtrFree(DesignViewInitialSchema);
	    DesignViewInitialSchema = 0;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_CreateDlg_OK:
	    form = FrmGetActiveForm();

	    /* Point at the driver the user selected. */
	    lst = GetObjectPtr(form, ctlID_CreateDlg_DriverList);
	    for (i = LstGetSelection(lst), driver = DriverList; i > 0;
		 --i, driver = driver->next)
		/* nothing */ ;

	    fld = GetObjectPtr(form, ctlID_CreateDlg_NameField);
	    h = FldGetTextHandle(fld);
	    name = MemHandleLock(h);

	    /* Make sure the field isn't blank. */
	    if (StrLen(name) == 0) {
		MemPtrUnlock(name);
		FrmAlert(alertID_NameBlank);
		return true;
	    }

	    /* Make sure the title doesn't already exist. */
	    if (! driver->class_ops.CheckTitle(name)) {
		MemPtrUnlock(name);
		FrmAlert(alertID_NameConflict);
		return true;
	    }

	    /* Set the design view variables. */
	    DesignViewDriver = driver;
	    DesignViewName = MemPtrNew(StrLen(name) + 1);
	    StrCopy(DesignViewName, name);

	    /* Unlock any variables that we locked. */
	    MemPtrUnlock(name);

	    /* Go to the design view. */
	    MemPtrFree(drvnames);
	    FrmReturnToForm(0);
	    FrmGotoForm(formID_DesignView);
	    return true;

	case ctlID_CreateDlg_Cancel:
	    FrmReturnToForm(0);
	    MemPtrFree(drvnames);
	    if (DesignViewInitialSchema) {
		/* Design view won't have a chance to free initial schema. */
		MemPtrFree(DesignViewInitialSchema->fields);
		MemPtrFree(DesignViewInitialSchema);
		DesignViewInitialSchema = 0;
	    }
	    return true;
	}
	break;

    default:
	break;
    }

    return false;
}

Boolean
RenameDlgHandleEvent(EventPtr event)
{
    static DataSource * src;

    FormPtr form;
    FieldPtr fld;
    MemHandle h;
    ChooserRecord * rec;
    char title[64], *p;

    switch (event->eType) {
    case frmOpenEvent:
	/* Open the data source we can let it rename itself. */
	h = DmQueryRecord(ChooserDB, InfoDatabase);
	rec = MemHandleLock(h);
	src = OpenDataSource(DriverList, dmModeReadOnly, rec->sourceID,
			     (void *) (rec + 1),
			     MemHandleSize(h) - sizeof(ChooserRecord));
	MemPtrUnlock(rec);

	/* Ask the data source for its title. */
	src->ops.GetTitle(src, title, sizeof(title));

	/* Update the name field with this title. */
	form = FrmGetActiveForm();
	fld = GetObjectPtr(form, ctlID_RenameDlg_NameField);
	h = MemHandleNew(StrLen(title) + 1);
	p = MemHandleLock(h);
	StrCopy(p, title);
	MemPtrUnlock(p);
	FldSetTextHandle(fld, h);

	/* Draw the form and put focus on name field. */
	FrmDrawForm(form);
	FrmSetFocus(form, FrmGetObjectIndex(form, ctlID_RenameDlg_NameField));

	return true;

    case frmCloseEvent:
	CloseDataSource(src);
	src = 0;
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_RenameDlg_OK:
	    /* Retrieve the text handle of the field. */
	    form = FrmGetActiveForm();
	    fld = GetObjectPtr(form, ctlID_RenameDlg_NameField);
	    h = FldGetTextHandle(fld);
	    p = MemHandleLock(h);

            /* Make sure the name isn't blank. */
            if (StrLen(p) == 0) {
                MemPtrUnlock(p);
                FrmAlert(alertID_NameBlank);
                return true;
            }

            /* Make sure the title doesn't already exist. */
            if (! src->driver->class_ops.CheckTitle(p)) {
                MemPtrUnlock(p);
                FrmAlert(alertID_NameConflict);
                return true;
            }

	    /* Change the title and update the chooser. */
	    {
		void * old_key_data, *new_key_data;
		UInt16 old_key_size, new_key_size;
		
		h = DmQueryRecord(ChooserDB, InfoDatabase);
		old_key_size = MemHandleSize(h) - sizeof(ChooserRecord);
		old_key_data = MemPtrNew(old_key_size);
		rec = MemHandleLock(h);
		MemMove(old_key_data, (void *) (rec + 1), old_key_size);
		MemPtrUnlock(rec);

		src->ops.SetTitle(src, p, &new_key_data, &new_key_size);
		RenameDatabase(src->driver->sourceID,
			       old_key_data, old_key_size,
			       new_key_data, new_key_size, p);
		MemPtrFree(old_key_data);
		MemPtrFree(new_key_data);
	    }

	    MemPtrUnlock(p);
	    FrmReturnToForm(0);
	    CloseDataSource(src);
	    src = 0;
	    FrmUpdateForm(formID_Chooser, updateInitTable);
	    return true;

	case ctlID_RenameDlg_Cancel:
	    FrmReturnToForm(0);
	    CloseDataSource(src);
	    src = 0;
	    return true;
	}
	break;

    default:
	break;
    }

    return false;
}
