/*
 * DB: MobileDB Interface
 * Copyright (C) 2000-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <PalmOS.h>

#include "db.h"
#include "enum.h"

#if defined(__GNUC__)
#define SECT_DRV __attribute__ ((__section__("drvsect")))
#else
#define SECT_DRV
#endif

#define CATEGORY_FIELD_LABELS          1
#define CATEGORY_DATA_RECORDS          2
#define CATEGORY_DATA_RECORDS_FILTERED 3
#define CATEGORY_PREFERENCES           4
#define CATEGORY_DATA_TYPES            5
#define CATEGORY_FIELD_LENGTHS         6

#define MAX_FIELDS 20

#define CreatorID 'Mdb1'
#define TypeID 'Mdb1'

typedef struct {
    char  field[40];   /* the filter text the user entered */
    Int8  fieldNo;     /* number of the field, -1 means any field */
    UInt8 flags;       /* reserved */
} FilterCriterion;

typedef struct {
    Int8 fieldNo;         /* field # to sort on, -1 means unused criterion */
    Boolean descending;   /* direction to sort */
    UInt8 type;           /* lowest bit: 1 means numeric, 0 means lexical */
} SortCriterion;

/* structure which comprises the application information block */
typedef struct {
    UInt16  renamedCategories; /* should be 0x0000 */
    char    categoryLabels[dmRecNumCategories][dmCategoryLength];
    UInt8   categoryUniqIDs[dmRecNumCategories]; /* 0 .. 15 */
    UInt8   lastUniqID;     /* 15 */

    UInt16  version;         /* current header version */
    UInt32  lock;            /* hash of password */
    Boolean dontSearch;   /* true, if DB should be invisible to find */
    Boolean editOnSelect; /* true, if record should be edited by default */
    UInt8   reserved[3];     /* for later use */
        
    FilterCriterion filter[3]; /* what the user typed in filters */
    SortCriterion sort[3];     /* what the user typed in sort */
} MobileDBInfoType;

typedef struct
{
    DataSourceSortCallback callback;
    void * callback_data;
    DataSource * source;
    DataSourceGetter getter;
} SortCallbackData;

/* While this pointer breaks the reentrant style of the data source
 * layer, it is necessary since PalmOS doesn't let you pass a pointer
 * to the sort callback but rather an integer instead.
 */
static SortCallbackData * sort_callback_data;

/* structure used for private data while a database is open */
struct private_data {
    UInt16 active_view;
    UInt16 top_visible_record;
    Boolean lfind_case_sensitive;
    Boolean lfind_whole_word;
};


static char * NullGetter_GetString(void *, UInt16) SECT_DRV;
static void * MobileDB_GetAppInfoPtr(DataSource *) SECT_DRV;
static UInt32 MobileDB_PackedSize(DataSourceGetter *, void *, UInt16) SECT_DRV;
static void MobileDB_PackRecord(void *, DataSourceGetter *,
				void *, UInt16) SECT_DRV;
static Boolean MobileDB_UnpackRecord(void *, UInt32, char **,
				     UInt16 *) SECT_DRV;

static void MobileDB_Close(DataSource *) SECT_DRV;
static Boolean MobileDB_Seek(DataSource *, UInt16 *, Int16, Int16) SECT_DRV;
static char * MobileDB_Getter_GetString(void *, UInt16) SECT_DRV;
static void MobileDB_LockRecord(DataSource *, UInt16, Boolean,
				DataSourceGetter *, void **) SECT_DRV;
static void MobileDB_UnlockRecord(DataSource *, UInt16, Boolean,
				  void *) SECT_DRV;
static Err MobileDB_UpdateRecord(DataSource *, UInt16, Boolean,
				 DataSourceGetter *, void *) SECT_DRV;
static void MobileDB_SetField(DataSource *, void *, UInt16,
			      DataSourceGetter *, void *) SECT_DRV;
static void MobileDB_DeleteRecord(DataSource *, UInt16) SECT_DRV;
static UInt32 MobileDB_GetRecordID(DataSource *, UInt16) SECT_DRV;
static UInt16 MobileDB_GetNumOfViews(DataSource *) SECT_DRV;
static DataSourceView * MobileDB_GetView(DataSource *, UInt16) SECT_DRV;
static void MobileDB_PutView(DataSource *, UInt16, DataSourceView *) SECT_DRV;
static void MobileDB_StoreView(DataSource *, UInt16,
			       DataSourceView *) SECT_DRV;
static void MobileDB_DeleteView(DataSource *, UInt16) SECT_DRV;
static void MobileDB_CheckUpdateSchema(DataSource *, DataSourceSchema *,
				       UInt32 *, Boolean*, Boolean*) SECT_DRV;
static void MobileDB_UpdateSchema(DataSource *,
				  DataSourceSchema *, UInt32 *) SECT_DRV;
static Boolean MobileDB_GetOption_Boolean(DataSource *, UInt16) SECT_DRV;
static void MobileDB_SetOption_Boolean(DataSource *, UInt16, Boolean) SECT_DRV;
static UInt16 MobileDB_GetOption_UInt16(DataSource *, UInt16) SECT_DRV;
static void MobileDB_SetOption_UInt16(DataSource *, UInt16, UInt16) SECT_DRV;
static Int16 MobileDB_SortCallback(void *, void *, Int16, SortRecordInfoPtr,
				   SortRecordInfoPtr, MemHandle) SECT_DRV;
static void MobileDB_Sort(DataSource *,
			  DataSourceSortCallback, void *) SECT_DRV;

static Boolean MobileDB_Capable(DataSource *, DataSourceCapability) SECT_DRV;


static Err MobileDB_OpenPDB(DataSource *, UInt16, UInt16, LocalID) SECT_DRV;
static Err MobileDB_Open(DataSource*, UInt16, void*, UInt16) SECT_DRV;
static void MakeMetaRecord(DmOpenRef, UInt8, char **, UInt16) SECT_DRV;
static void InitAppInfoBlock(DmOpenRef) SECT_DRV;
static Err MobileDB_Create(const char *, DataSourceSchema *,
			   void **, UInt16 *) SECT_DRV;
static Boolean MobileDB_SupportsFieldType(DataSourceFieldType) SECT_DRV;
static void MobileDB_Enumerate(DataSourceEnumerationFunc, void *) SECT_DRV;
static Boolean MobileDB_GlobalFind(DataSourceGlobalFindFunc, Boolean,
				   FindParamsPtr) SECT_DRV;
static Boolean MobileDB_IsPresent(void *, UInt32) SECT_DRV;
static Boolean MobileDB_CanHandlePDB(UInt16, LocalID, UInt32, UInt32) SECT_DRV;


static char *
NullGetter_GetString(void * data, UInt16 fieldNum)
{
    char * * fields = data;

    return fields[fieldNum];
}

static void *
MobileDB_GetAppInfoPtr(DataSource * source)
{
    UInt16 cardNo;
    LocalID appInfoID;

    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    return MemLocalIDToLockedPtr(appInfoID, cardNo);
}

static UInt32
MobileDB_PackedSize(DataSourceGetter * getter, void * data, UInt16 length)
{
    UInt32 size;
    UInt16 i;
    char * str;

    /* The header is the 7 byte sequence: 0xFF,0xFF,0xFF,0x01,0xFF,0x00,0x00 */
    size = 7;

    /* Each field is a 1 byte field #, the string itself, and the null byte. */
    for (i = 0; i < length; ++i) {
	str = getter->GetString(data, i);
	size += 1 + StrLen(str) + 1;
    }

    /* The trailer is the single byte 0xFF. */
    size += 1;

    return size;
}

static void
MobileDB_PackRecord(void * dest,
		    DataSourceGetter * getter, void * data, UInt16 length)
{
    UInt8 header[] = { 0xFF, 0xFF, 0xFF, 0x01, 0xFF, 0x00, 0x00 };
    UInt32 offset;
    UInt8 i;
    char * str;

    /* Pack the record header into position. */
    DmWrite(dest, 0, &header[0], sizeof(header));

    /* Pack all of the fields into the record. */
    offset = 7;
    for (i = 0; i < length; ++i) {
	/* Pack the current field number into the record. */
	DmWrite(dest, offset, &i, sizeof(i));
	offset += 1;

	/* Pack the string into the record. */
	str = getter->GetString(data, i);
	DmWrite(dest, offset, str, StrLen(str) + 1);
	offset += StrLen(str) + 1;
    }

    /* Pack the trailer into the record. */
    i = 0xFF;
    DmWrite(dest, offset, &i, sizeof(i));
}

static Boolean
MobileDB_UnpackRecord(void * record, UInt32 size,
		      char ** unpacked, UInt16* length_ptr)
{
    UInt8 header[] = { 0xFF, 0xFF, 0xFF, 0x01, 0xFF, 0x00, 0x00 };
    UInt8* p;
    UInt8 field;
    UInt16 length, i;

    /* Fill in the entire unpacked array with pointers to the empty string. */
    for (i = 0; i < MAX_FIELDS; ++i) unpacked[i] = "";

    /* Make sure that the header is valid. */
    if (size < sizeof(header)
	|| MemCmp(record, &header[0], sizeof(header)) != 0) return false;

    length = 0;
    p = ((UInt8 *) record) + sizeof(header);
    while (p < ((UInt8 *) record) + size) {
	/* Extract the field number. */
	field = *p++;
	if (field == 0xFF)
	    break;
	if (field >= MAX_FIELDS)
	    return false;
	if (field + 1 > length)
	    length = field + 1;

	/* Point at the start of this string. */
	unpacked[field] = (char *) p;
	p += StrLen(p) + 1;
    }

    /* Output the actual found length to the caller. */
    *length_ptr = length;

    return true;
}

static void
MobileDB_Close(DataSource * source)
{
    DmCloseDatabase(source->db);
    source->db = 0;
    MemPtrFree(source->schema.fields);
    if (source->data) MemPtrFree(source->data);
    source->schema.numFields = 0;
    source->schema.fields = 0;
}

static Boolean
MobileDB_Seek(DataSource * source, UInt16 * indexP,
	      Int16 offset, Int16 direction)
{
    if (DmSeekRecordInCategory(source->db, indexP, offset, direction,
			       CATEGORY_DATA_RECORDS) != 0)
        return false;

    return true;
}

static char *
MobileDB_Getter_GetString(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;

    return ptrs[1 + fieldNum];
}

static void
MobileDB_LockRecord(DataSource * source, UInt16 index, Boolean writable,
		    DataSourceGetter * getter, void ** data_ptr)
{
    void ** ptrs;
    MemHandle h;
    UInt16 count;

    ptrs = MemPtrNew((1 + MAX_FIELDS) * sizeof(void *));

    if (writable)
        h = DmGetRecord(source->db, index);
    else
        h = DmQueryRecord(source->db, index);

    ptrs[0] = MemHandleLock(h);
    MobileDB_UnpackRecord(ptrs[0], MemHandleSize(h),
			  (char **) (&ptrs[1]), &count);

    getter->GetString  = MobileDB_Getter_GetString;
    getter->GetInteger = 0;
    getter->GetBoolean = 0;
    *data_ptr = ptrs;
}

static void
MobileDB_UnlockRecord(DataSource * source, UInt16 index, Boolean writable,
		      void * record_data)
{
    void * * ptrs = record_data;

    MemPtrUnlock(ptrs[0]);

    if (writable)
        DmReleaseRecord(source->db, index, true);

    MemPtrFree(ptrs);
}

static Err
MobileDB_UpdateRecord(DataSource * source, UInt16 index, Boolean doInsert,
		      DataSourceGetter * getter, void * data)
{
    UInt32 size;
    UInt16 attr;
    MemHandle recH;
    void * packed;

    /* Determine the packed size of the new record. */
    size = MobileDB_PackedSize(getter, data, source->schema.numFields);

    if (index == dmMaxRecordIndex || doInsert) {
        /* Create a new record in the database. */
        recH = DmNewRecord(source->db, &index, size);
        if (recH == 0) {
            /* If we couldn't make the record, bail out. */
            return DmGetLastErr();
        }
    } else {
        /* Otherwise, retrieve and resize an existing record. */
        DmResizeRecord(source->db, index, size);
        recH = DmGetRecord(source->db, index);
    }

    /* Pack the new data into the record. */
    packed = MemHandleLock(recH);
    MobileDB_PackRecord(packed, getter, data, source->schema.numFields);
    MemPtrUnlock(packed);

    /* Release the write lock on the record. */
    DmReleaseRecord(source->db, index, true);

    /* Update the category so it is marked as a data record. */
    DmRecordInfo(source->db, index, &attr, 0, 0);
    attr &= ~(dmRecAttrCategoryMask);
    attr |= CATEGORY_DATA_RECORDS & dmRecAttrCategoryMask;
    DmSetRecordInfo(source->db, index, &attr, 0);

    return 0;
}

static void
MobileDB_SetField(DataSource * source, void * arg, UInt16 fieldNum,
               DataSourceGetter * getter, void * data)
{
    ErrDisplay("not supported");
}

static void
MobileDB_DeleteRecord(DataSource * source, UInt16 index)
{
    DmRemoveRecord(source->db, index);
}

static UInt32
MobileDB_GetRecordID(DataSource * source, UInt16 index)
{
    UInt32 uniqueID;

    DmRecordInfo(source->db, index, 0, &uniqueID, 0);
    return uniqueID;
}

static UInt16
MobileDB_GetNumOfViews(DataSource * source)
{
    return 1;
}

static DataSourceView *
MobileDB_GetView(DataSource * source, UInt16 viewIndex)
{
    DataSourceView * view;
    UInt16 i, recNum;
    char * p;
    MemHandle h;
    UInt16 widths[source->schema.numFields];

    /* We only support a single list view. */
    if (viewIndex != 0) return 0;

    /* Find the field widths record and extract the widths. */
    recNum = 0;
    h = DmQueryNextInCategory(source->db, &recNum, CATEGORY_FIELD_LENGTHS);
    if (h) {
	char * fields[MAX_FIELDS];
	UInt16 count;

	MobileDB_UnpackRecord(MemHandleLock(h), MemHandleSize(h),
			      fields, &count);
	for (i = 0; i < source->schema.numFields; ++i) {
	    widths[i] = (UInt16) (String2Long(fields[i]) & 0xFFFF);
	}
	MemHandleUnlock(h);
    } else {
	for (i = 0; i < source->schema.numFields; ++i) widths[i] = 80;
    }


    /* Allocate space for the list view. */
    view = MemPtrNew(sizeof(*view));

    /* Fill in the name with a localized version of "All Fields". */
    p = GetStringResource(stringID_AllFields);
    StrNCopy(view->name, p, sizeof(view->name) - 1);
    view->name[sizeof(view->name) - 1] = '\0';
    PutStringResource(p);

    /* There is a column for each field. */
    view->numCols = source->schema.numFields;
    view->cols = MemPtrNew(view->numCols * sizeof(DataSourceViewColumn));

    for (i = 0; i < source->schema.numFields; ++i) {
        view->cols[i].fieldNum = i;
        view->cols[i].width = widths[i];
    }

    return view;
}

static void
MobileDB_PutView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    MemPtrFree(view->cols);
    MemPtrFree(view);
}

static void
MobileDB_StoreView(DataSource * source, UInt16 viewIndex,
		   DataSourceView * view)
{
    DataSourceGetter getter = { GetString: NullGetter_GetString };
    char width_strings[MAX_FIELDS][16];
    char * width_ptrs[MAX_FIELDS];
    UInt16 i, recNum, attr;
    MemHandle h;
    UInt32 size;

    /* We only support a single view of the data. */
    if (viewIndex != 0 || view->numCols != source->schema.numFields)
        return;

    /* Bail out if the field numbers don't match up. */
    for (i = 0; i < source->schema.numFields; ++i) {
        if (view->cols[i].fieldNum != i) return;
    }

    /* Scan the field widths and look for any changes. */
    for (i = 0; i < source->schema.numFields; ++i) {
	StrPrintF(width_strings[i], "%u", view->cols[i].width);
	width_ptrs[i] = &width_strings[i][0];
    }

    /* Calculate the size of the final record. */
    size = MobileDB_PackedSize(&getter, width_ptrs, source->schema.numFields);

    /* Find or create the widths record. */
    recNum = 0;
    if (DmSeekRecordInCategory(source->db, &recNum, 0, dmSeekForward,
			       CATEGORY_FIELD_LENGTHS) == 0) {
	DmResizeRecord(source->db, recNum, size);
    } else {
	recNum = dmMaxRecordIndex;
	(void) DmNewRecord(source->db, &recNum, size);
    }

    /* Pack the field widths into the record. */
    h = DmGetRecord(source->db, recNum);
    MobileDB_PackRecord(MemHandleLock(h), &getter, width_ptrs,
			source->schema.numFields);
    MemHandleUnlock(h);
    DmReleaseRecord(source->db, recNum, true);

    /* Update the category so that it is marked as the width record. */
    DmRecordInfo(source->db, recNum, &attr, 0, 0);
    attr &= ~(dmRecAttrCategoryMask);
    attr |= CATEGORY_FIELD_LENGTHS & dmRecAttrCategoryMask;
    DmSetRecordInfo(source->db, recNum, &attr, 0);
}

static void
MobileDB_DeleteView(DataSource * source, UInt16 index)
{
}

static void
MobileDB_CheckUpdateSchema(DataSource * source,
			   DataSourceSchema * schema, UInt32* reorder,
			   Boolean* changed, Boolean* rebuild)
{
    UInt16 i;

    /* By default, we assume no changes happened whatsoever. */
    *changed = false;
    *rebuild = false;

    /* Adding or deleting fields requires a rebuild. */
    if (source->schema.numFields != schema->numFields) {
        *changed = true;
        *rebuild = true;
        return;
    }

    for (i = 0; i < source->schema.numFields; ++i) {
        /* Changing the order of the fields or changing the type
         * requires a full rebuild.
         */
        if (source->schema.fields[i].type != schema->fields[i].type
            || reorder[i] != i) {
            *changed = true;
            *rebuild = true;
            return;
        }

        /* Field name changes don't need a rebuild. */
        if (StrCompare(source->schema.fields[i].name,
                       schema->fields[i].name) != 0) {
            *changed = true;
        }
    }
}

static void
MobileDB_UpdateSchema(DataSource * source,
		      DataSourceSchema * schema, UInt32 * reorder)
{
    DataSourceGetter getter = { GetString: NullGetter_GetString };
    char * fields[MAX_FIELDS];
    UInt32 size;
    UInt16 i, recNum, attr;
    MemHandle h;

    /* We can only reach this point if MobileDB_CheckUpdateSchema()
     * verified that only field name changes were done. Thus, we can
     * safely just go ahead and change the field names.
     */

    /* Setup the array used to build the field names record. */
    for (i = 0; i < source->schema.numFields; ++i) {
	fields[i] = schema->fields[i].name;
    }

    /* Determine the size of the packed record. */
    size = MobileDB_PackedSize(&getter, fields, source->schema.numFields);

    /* Find or create the widths record. */
    recNum = 0;
    if (DmSeekRecordInCategory(source->db, &recNum, 0, dmSeekForward,
                               CATEGORY_FIELD_LABELS) == 0) {
        DmResizeRecord(source->db, recNum, size);
    } else {
        recNum = dmMaxRecordIndex;
        (void) DmNewRecord(source->db, &recNum, size);
    }

    /* Pack the field widths into the record. */
    h = DmGetRecord(source->db, recNum);
    MobileDB_PackRecord(MemHandleLock(h), &getter,
			fields, source->schema.numFields);
    MemHandleUnlock(h);
    DmReleaseRecord(source->db, recNum, true);

    /* Update the category so that it is marked as the width record. */
    DmRecordInfo(source->db, recNum, &attr, 0, 0);
    attr &= ~(dmRecAttrCategoryMask);
    attr |= CATEGORY_FIELD_LABELS & dmRecAttrCategoryMask;
    DmSetRecordInfo(source->db, recNum, &attr, 0);

    /* Now we need to update the current schema. */
    MemPtrFree(source->schema.fields);
    CopySchema(&source->schema, schema);
}

static Boolean
MobileDB_GetOption_Boolean(DataSource * source, UInt16 optionID)
{
    struct private_data * data = (struct private_data *) source->data;
    MobileDBInfoType* info;
    Boolean result;

    switch (optionID) {
    case optionID_DisableGlobalFind:
	info = MobileDB_GetAppInfoPtr(source);
	if (info->dontSearch)
	    result = true;
	else
	    result = false;
	MemPtrUnlock(info);
        break;

    case optionID_LFind_CaseSensitive:
        result = data->lfind_case_sensitive;
        break;

    case optionID_LFind_WholeWord:
        result = data->lfind_whole_word;
        break;

    default:
        result = false;
        break;
    }

    return result;
}

static void
MobileDB_SetOption_Boolean(DataSource * source, UInt16 optionID, Boolean value)
{
    struct private_data * data = (struct private_data *) source->data;
    MobileDBInfoType* info;

    switch (optionID) {
    case optionID_DisableGlobalFind:
	info = MobileDB_GetAppInfoPtr(source);
	DmWrite(info, OffsetOf(MobileDBInfoType, dontSearch),
		&value, sizeof(value));
	MemPtrUnlock(info);
        break;

    case optionID_LFind_CaseSensitive:
	data->lfind_case_sensitive = value;
        break;

    case optionID_LFind_WholeWord:
        data->lfind_whole_word = value;
        break;
    }
}

static UInt16
MobileDB_GetOption_UInt16(DataSource * source, UInt16 optionID)
{
    struct private_data * data = (struct private_data *) source->data;
    UInt16 result;

    switch (optionID) {
    case optionID_ListView_ActiveView:
        result = data->active_view;
        break;

    case optionID_ListView_TopVisibleRecord:
	result = data->top_visible_record;
	break;

    default:
        result = 0;
        break;
    }

    return result;
}

static void
MobileDB_SetOption_UInt16(DataSource * source, UInt16 optionID, UInt16 value)
{
    struct private_data * data = (struct private_data *) source->data;

    switch (optionID) {
    case optionID_ListView_ActiveView:
        data->active_view = value;
        break;

    case optionID_ListView_TopVisibleRecord:
	data->top_visible_record = value;
	break;

    default:
        break;
    }
}

static Int16
MobileDB_SortCallback(void * rec1, void * rec2, Int16 other,
		      SortRecordInfoPtr rec1SortInfo,
		      SortRecordInfoPtr rec2SortInfo,
		      MemHandle appInfoH)
{
    SortCallbackData * data = (SortCallbackData *) sort_callback_data;
    void * ptrs1[1 + MAX_FIELDS];
    void * ptrs2[1 + MAX_FIELDS];
    UInt32 size;
    UInt16 count;
    int result;
    MemHandle h;

#define is_data_record(cat) ( ((cat) == CATEGORY_DATA_RECORDS) \
|| ((cat) == CATEGORY_DATA_RECORDS_FILTERED))

    /* If either record is not a data record, then result is fixed. */
    if (! is_data_record(rec1SortInfo->attributes & dmRecAttrCategoryMask))
	return -1;
    if (! is_data_record(rec2SortInfo->attributes & dmRecAttrCategoryMask))
	return 1;

#undef is_data_record

    h = MemPtrRecoverHandle(rec1);
    size = MemHandleSize(h);
    MobileDB_UnpackRecord(rec1, size, (char **) (&ptrs1[1]), &count);

    h = MemPtrRecoverHandle(rec2);
    size = MemHandleSize(h);
    MobileDB_UnpackRecord(rec2, size, (char **) (&ptrs2[1]), &count);

    result = data->callback(data->source,
                            &data->getter, &ptrs1,
                            &data->getter, &ptrs2,
                            data->callback_data);

    return result;
}

static void
MobileDB_Sort(DataSource * source,
	      DataSourceSortCallback callback, void * callback_data)
{
    SortCallbackData data;

    data.callback = callback;
    data.callback_data = callback_data;
    data.source = source;
    data.getter.GetString = MobileDB_Getter_GetString;
    data.getter.GetInteger = 0;
    data.getter.GetBoolean = 0;
    sort_callback_data = &data;
    DmQuickSort(source->db, MobileDB_SortCallback, 0);
}

static Boolean
MobileDB_Capable(DataSource * source, DataSourceCapability capability)
{
    switch (capability) {
    case DS_CAPABLE_LISTVIEW_EDIT:
        return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_INSERT:
        return false;
    case DS_CAPABLE_LISTVIEW_DELETE:
        return false;
    case DS_CAPABLE_LISTVIEW_NAME_EDIT:
	return false;
    case DS_CAPABLE_LISTVIEW_FIELD_EDIT:
        return false;
    case DS_CAPABLE_LISTVIEW_FIELD_INSERT:
        return false;
    case DS_CAPABLE_LISTVIEW_FIELD_DELETE:
        return false;
    case DS_CAPABLE_LISTVIEW_WIDTH_EDIT:
        return ! source->readonly;
    default:
        return false;
    }
}

static Err
MobileDB_OpenPDB(DataSource * source, UInt16 mode, UInt16 cardNo, LocalID dbID)
{
    UInt32 dbType, size;
    UInt16 recNum, count, i;
    MemHandle h;
    char * fields[MAX_FIELDS];
    char* p;

    /* Fill in the operations structure. */
    source->ops.Close = MobileDB_Close;
    source->ops.Seek = MobileDB_Seek;
    source->ops.LockRecord = MobileDB_LockRecord;
    source->ops.UnlockRecord = MobileDB_UnlockRecord;
    source->ops.UpdateRecord = MobileDB_UpdateRecord;
    source->ops.SetField = MobileDB_SetField;
    source->ops.DeleteRecord = MobileDB_DeleteRecord;
    source->ops.GetRecordID = MobileDB_GetRecordID;
    source->ops.GetNumOfViews = MobileDB_GetNumOfViews;
    source->ops.GetView = MobileDB_GetView;
    source->ops.PutView = MobileDB_PutView;
    source->ops.StoreView = MobileDB_StoreView;
    source->ops.DeleteView = MobileDB_DeleteView;
    source->ops.CheckUpdateSchema = MobileDB_CheckUpdateSchema;
    source->ops.UpdateSchema = MobileDB_UpdateSchema;
    source->ops.GetOption_Boolean = MobileDB_GetOption_Boolean;
    source->ops.SetOption_Boolean = MobileDB_SetOption_Boolean;
    source->ops.GetOption_UInt16 = MobileDB_GetOption_UInt16;
    source->ops.SetOption_UInt16 = MobileDB_SetOption_UInt16;
    source->ops.GetKey = DS_PDB_GetKey;
    source->ops.GetPDB = DS_PDB_GetPDB;
    source->ops.GetTitle = DS_PDB_GetTitle;
    source->ops.SetTitle = DS_PDB_SetTitle;
    source->ops.Sort = MobileDB_Sort;
    source->ops.Capable = MobileDB_Capable;

    /* No private data for now in this data source. */
    source->data = MemPtrNew(sizeof(struct private_data));
    ((struct private_data *) source->data)->active_view = 0;
    ((struct private_data *) source->data)->top_visible_record = 0;
    ((struct private_data *) source->data)->lfind_case_sensitive = false;
    ((struct private_data *) source->data)->lfind_whole_word = false;

    /* Check to see if we support this type of database. */
    DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0, &dbType, 0);
    if (dbType != TypeID)
        return dmErrCantOpen;

    /* Open this database. */
    source->db = DmOpenDatabase(cardNo, dbID, mode);
    if (! source->db)
        return DmGetLastErr();

    /* Extract the field names into the data source's schema. */
    recNum = 0;
    h = DmQueryNextInCategory(source->db, &recNum, CATEGORY_FIELD_LABELS);
    if (!h) {
	DmCloseDatabase(source->db);
	return DmGetLastErr();
    }
    MobileDB_UnpackRecord(MemHandleLock(h), MemHandleSize(h),
			  fields, &count);
    source->schema.numFields = count;
    size = source->schema.numFields * sizeof(DataSourceFieldInfo);
    for (i = 0; i < count; ++i) {
	size += StrLen(fields[i]) + 1;
    }
    source->schema.fields = MemPtrNew(size);
    p = (UInt8 *) (source->schema.fields + source->schema.numFields);
    for (i = 0; i < count; ++i) {
	source->schema.fields[i].type = FIELD_TYPE_STRING;
	source->schema.fields[i].name = p;
	StrCopy(p, fields[i]);
	p += StrLen(fields[i]) + 1;
    }
    MemHandleUnlock(h);

    return 0;
}

static Err
MobileDB_Open(DataSource* source, UInt16 mode, void* key_data, UInt16 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    LocalID dbID;

    /* Make sure that the key has a valid size. */
    if (key_size != sizeof(DataSourceChooserKey))
        return dmErrCantOpen;

    /* Retrieve the location in memory of this database. */
    dbID = DmFindDatabase(key->cardNo, key->name);
    if (!dbID)
        return DmGetLastErr();

    /* Let the PDB open code handle the remainder of the open. */
    return MobileDB_OpenPDB(source, mode, key->cardNo, dbID);
}

static void
MakeMetaRecord(DmOpenRef db, UInt8 category, char ** fields, UInt16 length)
{
    UInt16 index = dmMaxRecordIndex;
    MemHandle h;
    UInt32 size;
    UInt16 attr;
    DataSourceGetter getter = { GetString: NullGetter_GetString };

    size = MobileDB_PackedSize(&getter, fields, length);
    h = DmNewRecord(db, &index, size);
    MobileDB_PackRecord(MemHandleLock(h), &getter, fields, length);
    MemHandleUnlock(h);
    DmReleaseRecord(db, index, true);

    /* Update the category so it is marked as a metadata record. */
    DmRecordInfo(db, index, &attr, 0, 0);
    attr &= ~(dmRecAttrCategoryMask);
    attr |= category;
    DmSetRecordInfo(db, index, &attr, 0);
}

static void
InitAppInfoBlock(DmOpenRef db)
{
    MemHandle appInfoHand;
    LocalID appInfoID, dbID;
    MobileDBInfoType info;
    UInt16 i, cardNo;
    void * p;

    /* Clear out the entire block. */
    MemSet(&info, sizeof(info), 0);

    /* Fill in the block. */
    info.renamedCategories = 0;
    StrCopy(info.categoryLabels[0], "Unfiled");
    StrCopy(info.categoryLabels[1], "FieldLabels");
    StrCopy(info.categoryLabels[2], "DataRecords");
    StrCopy(info.categoryLabels[3], "DataRecordsFout");
    StrCopy(info.categoryLabels[4], "Preferences");
    StrCopy(info.categoryLabels[5], "DataType");
    StrCopy(info.categoryLabels[6], "FieldLengths");
    info.lastUniqID = 15;
    info.version = 1;
    for (i = 0; i < 3; ++i) {
	info.filter[i].fieldNo = -1;
	info.sort[i].fieldNo = -1;
    }

    /* Allocate and fill a handle for app info block. */
    appInfoHand = DmNewHandle(db, sizeof(info));
    p = MemHandleLock(appInfoHand);
    DmWrite(p, 0, &info, sizeof(info));
    MemPtrUnlock(p);

    /* Set the app info block into place. */
    DmOpenDatabaseInfo(db, &dbID, 0, 0, &cardNo, 0);
    appInfoID = MemHandleToLocalID(appInfoHand);
    DmSetDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, &appInfoID, 0, 0, 0);
}

static Err
MobileDB_Create(const char* title, DataSourceSchema* schema,
             void ** key_data_ptr, UInt16 * key_size_ptr)
{
    Err err;
    LocalID dbID;
    DmOpenRef db;
    char * fields[MAX_FIELDS];
    UInt16 i;
    DataSourceChooserKey * key;

    err = DmCreateDatabase(0, title, CreatorID, TypeID, false);
    if (err)
        return err;

    dbID = DmFindDatabase(0, title);
    if (!dbID)
        return DmGetLastErr();

    db = DmOpenDatabase(0, dbID, dmModeReadWrite);
    if (!db)
        return DmGetLastErr();

    /* Build the application information block. */
    InitAppInfoBlock(db);

    /* Create the field names record. */
    for (i = 0; i < schema->numFields; ++i) fields[i] = schema->fields[i].name;
    MakeMetaRecord(db, CATEGORY_FIELD_LABELS, fields, schema->numFields);

    /* Create the field types record. */
    for (i = 0; i < MAX_FIELDS; ++i) fields[i] = "str";
    MakeMetaRecord(db, CATEGORY_DATA_TYPES, fields, MAX_FIELDS);

    /* Create the widths record. */
    for (i = 0; i < MAX_FIELDS; ++i) fields[i] = "80";
    MakeMetaRecord(db, CATEGORY_FIELD_LENGTHS, fields, MAX_FIELDS);

    /* Create the preferences record. */
    for (i = 0; i < MAX_FIELDS; ++i) fields[i] = "\0x01";
    MakeMetaRecord(db, CATEGORY_PREFERENCES, fields, MAX_FIELDS);

    /* Close the database. */
    DmCloseDatabase(db);

    /* Add this database into the chooser screen. */
    key = (DataSourceChooserKey *) MemPtrNew(sizeof(DataSourceChooserKey));
    MemSet(key, sizeof(*key), 0);
    key->cardNo = 0;
    StrNCopy(key->name, title, sizeof(key->name) - 1);
    key->name[sizeof(key->name)-1] = '\0';
    *key_data_ptr = key;
    *key_size_ptr = sizeof(*key);

    return 0;
}

static Boolean
MobileDB_SupportsFieldType(DataSourceFieldType type)
{
    switch (type) {
    case FIELD_TYPE_STRING:
	return true;
    default:
	return false;
    }
}

static void
MobileDB_Enumerate(DataSourceEnumerationFunc callback, void * arg)
{
    return DS_PDB_Enumerate(callback, arg,
			    TypeID, CreatorID, sourceID_MobileDB);
}

static Boolean
MobileDB_GlobalFind(DataSourceGlobalFindFunc callback, Boolean continuation,
		    FindParamsPtr params)
{
    return DS_PDB_GlobalFind(callback, continuation, params,
			     TypeID, CreatorID, MobileDB_OpenPDB);
}

static Boolean
MobileDB_IsPresent(void * key_data, UInt32 key_size)
{
    return DS_PDB_IsPresent(key_data, key_size, TypeID, CreatorID);
}

static Boolean
MobileDB_CanHandlePDB(UInt16 cardNo, LocalID dbID, UInt32 type, UInt32 creator)
{
    return (creator == CreatorID) && (type == TypeID);
}

DataSourceDriver *
MobileDB_GetDriver(void)
{
    DataSourceDriver * driver = MemPtrNew(sizeof(DataSourceDriver));

    StrCopy(driver->name, "MobileDB");
    driver->sourceID = sourceID_MobileDB;
    driver->class_ops.Open = MobileDB_Open;
    driver->class_ops.OpenPDB = MobileDB_OpenPDB;
    driver->class_ops.CheckTitle = DS_PDB_CheckTitle;
    driver->class_ops.Create = MobileDB_Create;
    driver->class_ops.SupportsFieldType = MobileDB_SupportsFieldType;
    driver->class_ops.Delete = DS_PDB_Delete;
    driver->class_ops.Enumerate = MobileDB_Enumerate;
    driver->class_ops.GlobalFind = MobileDB_GlobalFind;
    driver->class_ops.IsPresent = MobileDB_IsPresent;
    driver->class_ops.CompareKey = DS_PDB_CompareKey;
    driver->class_ops.CanHandlePDB = MobileDB_CanHandlePDB;
    driver->class_ops.Beam = DS_PDB_Beam;

    return driver;
}
