/*
 * DB: JFile v3.x Interface
 * Copyright (C) 2000-2001 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>

#include "db.h"
#include "enum.h"

#if defined(__GNUC__)
#define SECT_DRV __attribute__ ((__section__("drvsect")))
#else
#define SECT_DRV
#endif

#define MAX_FIELDS 20

#define CreatorID 'JBas'
#define TypeID 'JbDb'

typedef struct jfile3_appinfo_type
{
    char fieldNames[MAX_FIELDS][21];
    UInt16 fieldTypes[MAX_FIELDS];
    UInt16 numFields;
    UInt16 version;
    UInt16 fieldWidths[MAX_FIELDS];
    UInt16 editViewLabelWidth;
    UInt16 sortFields[3];
    UInt16 findField;
    UInt16 filterField;
    char findString[16];
    char filterString[16];
    UInt16 flags;
    UInt16 firstColumn;
    char password[12];
} JFile3AppInfoType;

/* structure used for private data while a database is open */
struct private_data {
    UInt16 active_view;
    UInt16 top_visible_record;
    Boolean lfind_case_sensitive;
    Boolean lfind_whole_word;
};

typedef struct
{
    DataSourceSortCallback callback;
    void * callback_data;
    DataSource * source;
    DataSourceGetter getter;
} SortCallbackData;
 
/* While this pointer breaks the reentrant style of the data source
 * layer, it is necessary since PalmOS doesn't let you pass a pointer
 * to the sort callback but rather an integer instead.
 */
static SortCallbackData * sort_callback_data;

static JFile3AppInfoType * JFile3_GetAppInfoPtr(DataSource *) SECT_DRV;
static UInt32 JFile3_PackedRecordSize(DataSource *,
				      DataSourceGetter *, void *) SECT_DRV;
static void JFile3_PackRecord(DataSource *, void *, DataSourceGetter *,
			      void *) SECT_DRV;
static void JFile3_UnpackRecord(DataSource *, void *, void * arg[]) SECT_DRV;
static void JFile3_Close(DataSource *) SECT_DRV;
static Boolean JFile3_Seek(DataSource *, UInt16 *, Int16, Int16) SECT_DRV;
static char * JFile3_Getter_GetString(void *, UInt16) SECT_DRV;
static void JFile3_LockRecord(DataSource *, UInt16, Boolean,
			      DataSourceGetter *, void **) SECT_DRV;
static void JFile3_UnlockRecord(DataSource *,UInt16,Boolean,void *) SECT_DRV;
static Err JFile3_UpdateRecord(DataSource *, UInt16, Boolean,
			       DataSourceGetter *, void *) SECT_DRV;
static void JFile3_SetField(DataSource *, void *, UInt16,
			    DataSourceGetter *, void *) SECT_DRV;
static void JFile3_DeleteRecord(DataSource *, UInt16) SECT_DRV;
static UInt32 JFile3_GetRecordID(DataSource *, UInt16) SECT_DRV;
static UInt16 JFile3_GetNumOfViews(DataSource *) SECT_DRV;
static DataSourceView * JFile3_GetView(DataSource *, UInt16) SECT_DRV;
static void JFile3_PutView(DataSource *, UInt16, DataSourceView *) SECT_DRV;
static void JFile3_StoreView(DataSource *, UInt16,
			     DataSourceView *) SECT_DRV;
static void JFile3_DeleteView(DataSource *, UInt16) SECT_DRV;
static void JFile3_CheckUpdateSchema(DataSource *, DataSourceSchema *,
				     UInt32 *, Boolean*, Boolean*) SECT_DRV;
static void JFile3_UpdateSchema(DataSource *,
				DataSourceSchema *, UInt32 *) SECT_DRV;
static Boolean JFile3_GetOption_Boolean(DataSource *, UInt16) SECT_DRV;
static void JFile3_SetOption_Boolean(DataSource *, UInt16, Boolean) SECT_DRV;
static UInt16 JFile3_GetOption_UInt16(DataSource *, UInt16) SECT_DRV;
static void JFile3_SetOption_UInt16(DataSource *, UInt16, UInt16) SECT_DRV;
static Int16 JFile3_SortCallback(void *, void *, Int16,
				 SortRecordInfoPtr, SortRecordInfoPtr,
				 MemHandle appInfoH) SECT_DRV;
static void JFile3_Sort(DataSource *, DataSourceSortCallback, void *) SECT_DRV;
static Boolean JFile3_Capable(DataSource *, DataSourceCapability) SECT_DRV;
 
static Err JFile3_OpenPDB(DataSource *, UInt16, UInt16, LocalID) SECT_DRV;
static Err JFile3_Open(DataSource*, UInt16, void*, UInt16) SECT_DRV;
static Err JFile3_Create(const char *, DataSourceSchema *,
                           void **, UInt16 *) SECT_DRV;
static Boolean JFile3_SupportsFieldType(DataSourceFieldType) SECT_DRV;
static void JFile3_Enumerate(DataSourceEnumerationFunc, void *) SECT_DRV;
static Boolean JFile3_GlobalFind(DataSourceGlobalFindFunc, Boolean,
                                   FindParamsPtr) SECT_DRV;
static Boolean JFile3_IsPresent(void *, UInt32) SECT_DRV;
static Boolean JFile3_CanHandlePDB(UInt16, LocalID, UInt32, UInt32) SECT_DRV;


static JFile3AppInfoType *
JFile3_GetAppInfoPtr(DataSource * source)
{
    UInt16 cardNo;
    LocalID appInfoID;

    DmOpenDatabaseInfo(source->db, 0, 0, 0, &cardNo, 0);
    appInfoID = DmGetAppInfoID(source->db);
    return MemLocalIDToLockedPtr(appInfoID, cardNo);
}

static void
JFile3_Close(DataSource * source)
{
    DmCloseDatabase(source->db);
    source->db = 0;
    MemPtrFree(source->schema.fields);
    if (source->data) MemPtrFree(source->data);
    source->schema.numFields = 0;
    source->schema.fields = 0;
}

static Boolean
JFile3_Seek(DataSource * source, UInt16 * indexP, Int16 offset,
	    Int16 direction)
{
    DmSeekRecordInCategory(source->db, indexP, offset, direction,
                           dmAllCategories);
    if (DmGetLastErr())
        return false;
 
    return true;
}

static UInt32
JFile3_PackedRecordSize(DataSource * source,
			DataSourceGetter * getter, void * data)
{
    UInt32 size;
    UInt16 i;
    char * str;
    char buf[32];

    size = source->schema.numFields * sizeof(UInt16);
    for (i = 0; i < source->schema.numFields; i++) {
        switch (source->schema.fields[i].type) {
        case FIELD_TYPE_STRING:
            str = getter->GetString(data, i);
            size += StrLen(str) + 1;
            break;
 
        case FIELD_TYPE_BOOLEAN:
            size += 2;
            break;
 
        case FIELD_TYPE_INTEGER:
	    StrIToA(buf, getter->GetInteger(data, i));
            size += StrLen(buf) + 1;
            break;

        default:
	    /* For the unsupported field types, just null out the field. */
            size += 2;
            break;
	}
    }

    return size;
}

static void
JFile3_PackRecord(DataSource * source, void * dest,
		  DataSourceGetter * getter, void * data)
{
    UInt32 offset = 0;
    UInt16 i, year;
    UInt8 b, hour, minute, month, day;
    char * str;
    char buf[32];

    for (i = 0; i < source->schema.numFields; i++) {
        switch (source->schema.fields[i].type) {
        case FIELD_TYPE_STRING:
            str = getter->GetString(data, i);
            DmWrite(dest, offset, str, StrLen(str) + 1);
            offset += StrLen(str) + 1;
            break;

        case FIELD_TYPE_BOOLEAN:
	    /* Write the boolean value as a numeric character. */
            b = (getter->GetBoolean(data, i)) ? '1' : '0';
            DmWrite(dest, offset, &b, sizeof(UInt8));
            offset += sizeof(UInt8);

	    /* Write the teriminating null byte. */
	    b = 0;
	    DmWrite(dest, offset, &b, sizeof(UInt8));
	    offset += sizeof(UInt8);
	    
            break;
 
        case FIELD_TYPE_INTEGER:
	    StrIToA(buf, getter->GetInteger(data, i));
            DmWrite(dest, offset, buf, StrLen(buf) + 1);
            offset += StrLen(buf) + 1;
            break;

	case FIELD_TYPE_DATE:
	    getter->GetDate(data, i, &year, &month, &day);
	    StrPrintF(buf, "%u/%u/%u", (UInt16) day, (UInt16) month, year);
            DmWrite(dest, offset, buf, StrLen(buf) + 1);
            offset += StrLen(buf) + 1;
	    break;

	case FIELD_TYPE_TIME:
	    getter->GetTime(data, i, &hour, &minute);
	    StrPrintF(buf, "%u:%u", (UInt16) hour, (UInt16) minute);
            DmWrite(dest, offset, buf, StrLen(buf) + 1);
            offset += StrLen(buf) + 1;
	    break;

        default:
	    /* Write two null bytes for unknown field types. */
	    b = 0;
	    DmWrite(dest, offset, &b, sizeof(UInt8));
	    offset += sizeof(UInt8);
	    b = 0;
	    DmWrite(dest, offset, &b, sizeof(UInt8));
	    offset += sizeof(UInt8);
            break;
        }
    }
}

static void
JFile3_UnpackRecord(DataSource * source, void * packed, void * unpacked[])
{
    UInt8 * rec = packed;
    UInt16 i;
 
    for (i = 0; i < source->schema.numFields; i++) {
	unpacked[i] = rec;
	rec += StrLen(rec) + 1;
    }
}

static char *
JFile3_Getter_GetString(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;
 
    return ptrs[fieldNum];
}
 
static Int32
JFile3_Getter_GetInteger(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;
    Int32 value;
 
    value = String2Long(ptrs[fieldNum]);
    return value;
}

static Boolean
JFile3_Getter_GetBoolean(void * data, UInt16 fieldNum)
{
    void * * ptrs = data;
 
    if ( *((UInt8 *) ptrs[fieldNum]) == '1')
        return true;
    else
        return false;
}

static void
JFile3_Getter_GetDate(void * data, UInt16 fieldNum,
		      UInt16* year, UInt8* mon, UInt8* day)
{
    void * * ptrs = data;
    char buf[StrLen(ptrs[fieldNum]) + 1], *p, *str;
    Int32 value;

    str = ptrs[fieldNum];
    p = StrChr(str, '/');
    StrNCopy(buf, str, p - str);
    value = String2Long(str);
    *day = (UInt8) (value & 0xFF);

    str = p + 1;
    p = StrChr(str, '/');
    StrNCopy(buf, str, p - str);
    value = String2Long(str);
    *mon = (UInt8) (value & 0xFF);

    str = p + 1;
    p = StrChr(str, '/');
    StrNCopy(buf, str, p - str);
    value = String2Long(str);
    *year = (UInt16) (value & 0xFFFF);
}

static void
JFile3_Getter_GetTime(void * data, UInt16 fieldNum, UInt8* hour, UInt8* minute)
{
    *hour = 13;
    *minute = 30;
}

static void
JFile3_LockRecord(DataSource * source, UInt16 index, Boolean writable,
                 DataSourceGetter * getter, void ** data_ptr)
{
    void ** ptrs, *packed;
    MemHandle h;
 
    ptrs = MemPtrNew(source->schema.numFields * sizeof(void *));
 
    if (writable)
        h = DmGetRecord(source->db, index);
    else
        h = DmQueryRecord(source->db, index);
 
    packed = MemHandleLock(h);
    JFile3_UnpackRecord(source, packed, ptrs);
 
    getter->GetString  = JFile3_Getter_GetString;
    getter->GetInteger = JFile3_Getter_GetInteger;
    getter->GetBoolean = JFile3_Getter_GetBoolean;
    getter->GetDate    = JFile3_Getter_GetDate;
    getter->GetTime    = JFile3_Getter_GetTime;
    *data_ptr = ptrs;
}

static void
JFile3_UnlockRecord(DataSource * source, UInt16 index, Boolean writable,
		    void * record_data)
{
    void * * ptrs = record_data;
 
    MemPtrUnlock(ptrs[0]);
 
    if (writable)
        DmReleaseRecord(source->db, index, true);
 
    MemPtrFree(ptrs);
}

static Err
JFile3_UpdateRecord(DataSource * source, UInt16 index, Boolean doInsert,
		    DataSourceGetter * getter, void * data)
{
    UInt32 size;
    MemHandle recH;
    void * packed;
 
    /* Determine the packed size of the new record. */
    size = JFile3_PackedRecordSize(source, getter, data);
 
    if (index == dmMaxRecordIndex || doInsert) {
        /* Create a new record in the database. */
        recH = DmNewRecord(source->db, &index, size);
        if (recH == 0) {
            /* If we couldn't make the record, bail out. */
            return DmGetLastErr();
        }
    } else {
        /* Otherwise, retrieve and resize an existing record. */
        DmResizeRecord(source->db, index, size);
        recH = DmGetRecord(source->db, index);
    }

    /* Pack the new data into the record. */
    packed = MemHandleLock(recH);
    JFile3_PackRecord(source, packed, getter, data);
    MemPtrUnlock(packed);
 
    /* Release the write lock on the record. */
    DmReleaseRecord(source->db, index, true);
    
    return 0;
}

static void
JFile3_SetField(DataSource * source, void * data, UInt16 fieldNum,
		DataSourceGetter * getter, void * getter_arg)
{
    void * * ptrs = data;
    char b;

    switch (source->schema.fields[fieldNum].type) {
    case FIELD_TYPE_BOOLEAN:
        b = getter->GetBoolean(getter_arg, fieldNum) ? '1' : '0';
	DmWrite(ptrs[fieldNum], 0, &b, sizeof(char));
	break;

    default:
	ErrDisplay("JFile3_SetField: only boolean supported");
	break;
    }
}
 
static void
JFile3_DeleteRecord(DataSource * source, UInt16 index)
{
    DmRemoveRecord(source->db, index);
}
 
static UInt32
JFile3_GetRecordID(DataSource * source, UInt16 index)
{
    UInt32 uniqueID;
 
    DmRecordInfo(source->db, index, 0, &uniqueID, 0);
    return uniqueID;
}

static UInt16
JFile3_GetNumOfViews(DataSource * source)
{
    return 1;
}

static DataSourceView *
JFile3_GetView(DataSource * source, UInt16 viewIndex)
{
    DataSourceView * view;
    UInt16 i;
    char * p;
    JFile3AppInfoType* hdr;

    /* We only support a single list view. */
    if (viewIndex != 0) return 0;

    /* Allocate space for the list view. */
    view = MemPtrNew(sizeof(*view));

    /* Fill in the name with a localized version of "All Fields". */
    p = GetStringResource(stringID_AllFields);
    StrNCopy(view->name, p, sizeof(view->name) - 1);
    view->name[sizeof(view->name) - 1] = '\0';
    PutStringResource(p);

    /* There is a column for each field. */
    view->numCols = source->schema.numFields;
    view->cols = MemPtrNew(view->numCols * sizeof(DataSourceViewColumn));

    /* Fill in the column information. */
    hdr = JFile3_GetAppInfoPtr(source);
    for (i = 0; i < source->schema.numFields; ++i) {
        view->cols[i].fieldNum = i;
        view->cols[i].width = hdr->fieldWidths[i];
    }
    MemPtrUnlock(hdr);

    return view;
}

static void
JFile3_PutView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    MemPtrFree(view->cols);
    MemPtrFree(view);
}

static void
JFile3_StoreView(DataSource * source, UInt16 viewIndex, DataSourceView * view)
{
    JFile3AppInfoType* hdr;
    UInt16 i;

    /* We only support a single list view. */
    if (viewIndex != 0 || view->numCols != source->schema.numFields)
        return;

    /* Bail out if the field numbers don't match up. */
    for (i = 0; i < source->schema.numFields; ++i) {
        if (view->cols[i].fieldNum != i) return;
    }

    /* Store the new field widths into the header. */
    hdr = JFile3_GetAppInfoPtr(source);
    for (i = 0; i < view->numCols; ++i) {
	UInt16 width = view->cols[i].width;

	DmWrite(hdr, OffsetOf(JFile3AppInfoType, fieldWidths[i]),
		&width, sizeof(width));
    }
    MemPtrUnlock(hdr);
}

static void
JFile3_DeleteView(DataSource * source, UInt16 index)
{
}

static void
JFile3_CheckUpdateSchema(DataSource * source,
			 DataSourceSchema * schema, UInt32* reorder,
			 Boolean* changed, Boolean* rebuild)
{
    UInt16 i;

    /* By default, we assume no changes happened whatsoever. */
    *changed = false;
    *rebuild = false;

    /* Adding or deleting fields requires a rebuild. */
    if (source->schema.numFields != schema->numFields) {
        *changed = true;
        *rebuild = true;
        return;
    }

    for (i = 0; i < source->schema.numFields; ++i) {
        /* Changing the order of the fields or changing the type
         * requires a full rebuild.
         */
        if (source->schema.fields[i].type != schema->fields[i].type
            || reorder[i] != i) {
            *changed = true;
            *rebuild = true;
            return;
        }

        /* Field name changes don't need a rebuild. */
        if (StrCompare(source->schema.fields[i].name,
                       schema->fields[i].name) != 0) {
            *changed = true;
        }
    }
}

static void
JFile3_UpdateSchema(DataSource * source,
		    DataSourceSchema * schema, UInt32 * reorder)
{
    JFile3AppInfoType* hdr;
    UInt32 offset;
    UInt16 i;

    /* We can only reach this point if JFile3_CheckUpdateSchema()
     * verified that only field name changes were done. Thus, we can
     * safely just go ahead and change the field names.
     */

    /* Retrieve the app info block which contains all the metadata. */
    hdr = JFile3_GetAppInfoPtr(source);

    /* Copy the new field names into the header. */
    offset = OffsetOf(JFile3AppInfoType, fieldNames);
    for (i = 0; i < schema->numFields; ++i) {
        DmSet(hdr, offset, sizeof(hdr->fieldNames[i]), 0);
	DmWrite(hdr, offset, schema->fields[i].name,
		StrLen(schema->fields[i].name) + 1);
	offset += sizeof(hdr->fieldNames[i]);
    }

    /* Zero out the remaining field name positions. */
    for ( /* i is set above */ ; i < MAX_FIELDS; ++i) {
        MemSet(&(hdr->fieldNames[i]), sizeof(hdr->fieldNames[i]), 0);
    }

    /* Put away our reference to the app info block. */
    MemPtrUnlock(hdr);

    /* Now we need to update the current schema. */
    MemPtrFree(source->schema.fields);
    CopySchema(&source->schema, schema);
}

static Boolean
JFile3_GetOption_Boolean(DataSource * source, UInt16 optionID)
{
    struct private_data * data = (struct private_data *) source->data;
    Boolean result;

    switch (optionID) {
    case optionID_DisableGlobalFind:
	result = false;
	break;

    case optionID_LFind_CaseSensitive:
        result = data->lfind_case_sensitive;
        break;

    case optionID_LFind_WholeWord:
        result = data->lfind_whole_word;
        break;

    default:
        result = false;
        break;
    }

    return result;
}

static void
JFile3_SetOption_Boolean(DataSource * source, UInt16 optionID, Boolean value)
{
    struct private_data * data = (struct private_data *) source->data;

    switch (optionID) {
    case optionID_DisableGlobalFind:
	break;

    case optionID_LFind_CaseSensitive:
        data->lfind_case_sensitive = value;
        break;

    case optionID_LFind_WholeWord:
        data->lfind_whole_word = value;
        break;
    }
}

static UInt16
JFile3_GetOption_UInt16(DataSource * source, UInt16 optionID)
{
    struct private_data * data = (struct private_data *) source->data;
    UInt16 result;

    switch (optionID) {
    case optionID_ListView_ActiveView:
        result = data->active_view;
        break;

    case optionID_ListView_TopVisibleRecord:
        result = data->top_visible_record;
        break;

    default:
        result = 0;
        break;
    }

    return result;
}

static void
JFile3_SetOption_UInt16(DataSource * source, UInt16 optionID, UInt16 value)
{
    struct private_data * data = (struct private_data *) source->data;

    switch (optionID) {
    case optionID_ListView_ActiveView:
        data->active_view = value;
        break;

    case optionID_ListView_TopVisibleRecord:
        data->top_visible_record = value;
        break;

    default:
        break;
    }
}

static Int16
JFile3_SortCallback(void * rec1, void * rec2, Int16 other,
		    SortRecordInfoPtr rec1SortInfo,
		    SortRecordInfoPtr rec2SortInfo, MemHandle appInfoH)
{
    SortCallbackData * data = (SortCallbackData *) sort_callback_data;
    void * ptrs1[data->source->schema.numFields];
    void * ptrs2[data->source->schema.numFields];
    int result;
 
    JFile3_UnpackRecord(data->source, rec1, ptrs1);
    JFile3_UnpackRecord(data->source, rec2, ptrs2);
 
    result = data->callback(data->source,
                            &data->getter, &ptrs1,
                            &data->getter, &ptrs2,
                            data->callback_data);

    return result;
}

static void
JFile3_Sort(DataSource * source,
	    DataSourceSortCallback callback, void * callback_data)
{
    SortCallbackData data;
 
    data.callback = callback;
    data.callback_data = callback_data;
    data.source = source;
    data.getter.GetString = JFile3_Getter_GetString;
    data.getter.GetInteger = JFile3_Getter_GetInteger;
    data.getter.GetBoolean = JFile3_Getter_GetBoolean;
    data.getter.GetDate = JFile3_Getter_GetDate;
    data.getter.GetTime = JFile3_Getter_GetTime;
    sort_callback_data = &data;
    DmQuickSort(source->db, JFile3_SortCallback, 0);
}

static Boolean
JFile3_Capable(DataSource * source, DataSourceCapability capability)
{
    switch (capability) {
    case DS_CAPABLE_LISTVIEW_EDIT:
        return ! source->readonly;
    case DS_CAPABLE_LISTVIEW_INSERT:
        return false;
    case DS_CAPABLE_LISTVIEW_DELETE:
        return false;
    case DS_CAPABLE_LISTVIEW_NAME_EDIT:
        return false;
    case DS_CAPABLE_LISTVIEW_FIELD_EDIT:
        return false;
    case DS_CAPABLE_LISTVIEW_FIELD_INSERT:
        return false;
    case DS_CAPABLE_LISTVIEW_FIELD_DELETE:
        return false;
    case DS_CAPABLE_LISTVIEW_WIDTH_EDIT:
        return ! source->readonly;
    default:
        return false;
    }
}

static Err
JFile3_OpenPDB(DataSource * source, UInt16 mode, UInt16 cardNo, LocalID dbID)
{
    UInt32 dbType, size;
    JFile3AppInfoType* hdr;
    UInt16 i;
    UInt8* p;

    /* Fill in the operations structure. */
    source->ops.Close = JFile3_Close;
    source->ops.Seek = JFile3_Seek;
    source->ops.LockRecord = JFile3_LockRecord;
    source->ops.UnlockRecord = JFile3_UnlockRecord;
    source->ops.UpdateRecord = JFile3_UpdateRecord;
    source->ops.SetField = JFile3_SetField;
    source->ops.DeleteRecord = JFile3_DeleteRecord;
    source->ops.GetRecordID = JFile3_GetRecordID;
    source->ops.GetNumOfViews = JFile3_GetNumOfViews;
    source->ops.GetView = JFile3_GetView;
    source->ops.PutView = JFile3_PutView;
    source->ops.StoreView = JFile3_StoreView;
    source->ops.DeleteView = JFile3_DeleteView;
    source->ops.CheckUpdateSchema = JFile3_CheckUpdateSchema;
    source->ops.UpdateSchema = JFile3_UpdateSchema;
    source->ops.GetOption_Boolean = JFile3_GetOption_Boolean;
    source->ops.SetOption_Boolean = JFile3_SetOption_Boolean;
    source->ops.GetOption_UInt16 = JFile3_GetOption_UInt16;
    source->ops.SetOption_UInt16 = JFile3_SetOption_UInt16;
    source->ops.GetKey = DS_PDB_GetKey;
    source->ops.GetPDB = DS_PDB_GetPDB;
    source->ops.GetTitle = DS_PDB_GetTitle;
    source->ops.SetTitle = DS_PDB_SetTitle;
    source->ops.Sort = JFile3_Sort;
    source->ops.Capable = JFile3_Capable;

    /* Private data to emulate some options supported other drivers. */
    source->data = MemPtrNew(sizeof(struct private_data));
    ((struct private_data *) source->data)->active_view = 0;
    ((struct private_data *) source->data)->top_visible_record = 0;
    ((struct private_data *) source->data)->lfind_case_sensitive = false;
    ((struct private_data *) source->data)->lfind_whole_word = false;

    /* Check to see if we support this type of database. */
    DmDatabaseInfo(cardNo, dbID, 0, 0, 0, 0, 0, 0, 0, 0, 0, &dbType, 0);
    if (dbType != TypeID)
        return dmErrCantOpen;

    /* Open this database. */
    source->db = DmOpenDatabase(cardNo, dbID, mode);
    if (! source->db)
        return DmGetLastErr();

    /* Retrieve the app info block which contains all the metadata. */
    hdr = JFile3_GetAppInfoPtr(source);

    /* Check the size of the app info block. */
    if (!hdr || MemPtrSize(hdr) < sizeof(JFile3AppInfoType)) {
	MemPtrUnlock(hdr);
	DmCloseDatabase(source->db);
	return dmErrCantOpen;
    }

    /* Verify that the version is what we expect. */
    if (hdr->version != 452) {
	MemPtrUnlock(hdr);
	DmCloseDatabase(source->db);
	return dmErrCantOpen;
    }

    /* Extract the number of fields from the header. */
    source->schema.numFields = hdr->numFields;

    /* Determine how much space needs to be allocated for schema fields. */
    size = source->schema.numFields * sizeof(DataSourceFieldInfo);
    for (i = 0; i < source->schema.numFields; ++i) {
        size += StrLen(hdr->fieldNames[i]) + 1;
    }
    source->schema.fields = MemPtrNew(size);

    /* Extract the entire schema from the database header. */
    p = (UInt8 *) (source->schema.fields + source->schema.numFields);
    for (i = 0; i < source->schema.numFields; ++i) {
	/* Extract the field type from the header. */
        switch (hdr->fieldTypes[i]) {
        case 0x0001:
            source->schema.fields[i].type = FIELD_TYPE_STRING;
            break;

        case 0x0002:
            source->schema.fields[i].type = FIELD_TYPE_BOOLEAN;
            break;

        case 0x0004:
            source->schema.fields[i].type = FIELD_TYPE_DATE;
            break;

        case 0x0008:
	    source->schema.fields[i].type = FIELD_TYPE_INTEGER;
            break;

        case 0x0010: /* FLOAT */
            source->schema.fields[i].type = FIELD_TYPE_STRING;
            break;

        case 0x0020:
            source->schema.fields[i].type = FIELD_TYPE_TIME;
            break;

	case 0x0040: /* LIST */
            source->schema.fields[i].type = FIELD_TYPE_STRING;
            break;

        default:
            source->schema.fields[i].type = FIELD_TYPE_STRING;
	    break;
        }

	/* Copy the field name into the schema and point schema at it. */
        StrCopy(p, hdr->fieldNames[i]);
	source->schema.fields[i].name = p;
        p += StrLen(hdr->fieldNames[i]) + 1;
    }

    /* Put away our reference to the app info block. */
    MemPtrUnlock(hdr);

    return 0;
}

static Err
JFile3_Open(DataSource* source, UInt16 mode, void* key_data, UInt16 key_size)
{
    DataSourceChooserKey * key = (DataSourceChooserKey *) key_data;
    LocalID dbID;

    /* Make sure that the key has a valid size. */
    if (key_size != sizeof(DataSourceChooserKey))
        return dmErrCantOpen;

    /* Retrieve the location in memory of this database. */
    dbID = DmFindDatabase(key->cardNo, key->name);
    if (!dbID)
        return DmGetLastErr();

    /* Let the PDB open code handle the remainder of the open. */
    return JFile3_OpenPDB(source, mode, key->cardNo, dbID);
}

static Err
JFile3_Create(const char* title, DataSourceSchema* schema,
	      void ** key_data_ptr, UInt16 * key_size_ptr)
{
    return 0;
}

static Boolean
JFile3_SupportsFieldType(DataSourceFieldType type)
{
    switch (type) {
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_BOOLEAN:
    case FIELD_TYPE_INTEGER:
    case FIELD_TYPE_DATE:
    case FIELD_TYPE_TIME:
        return true;
    default:
        return false;
    }
}

static void
JFile3_Enumerate(DataSourceEnumerationFunc callback, void * arg)
{
    return DS_PDB_Enumerate(callback, arg, TypeID, CreatorID, sourceID_JFile3);
}

static Boolean
JFile3_GlobalFind(DataSourceGlobalFindFunc callback, Boolean continuation,
                    FindParamsPtr params)
{
    return DS_PDB_GlobalFind(callback, continuation, params,
                             TypeID, CreatorID, JFile3_OpenPDB);
}

static Boolean
JFile3_IsPresent(void * key_data, UInt32 key_size)
{
    return DS_PDB_IsPresent(key_data, key_size, TypeID, CreatorID);
}

static Boolean
JFile3_CanHandlePDB(UInt16 cardNo, LocalID dbID, UInt32 type, UInt32 creator)
{
    return (creator == CreatorID) && (type == TypeID);
}

DataSourceDriver *
JFile3_GetDriver(void)
{
    DataSourceDriver * driver = MemPtrNew(sizeof(DataSourceDriver));

    StrCopy(driver->name, "JFile v3.x");
    driver->sourceID = sourceID_JFile3;
    driver->class_ops.Open = JFile3_Open;
    driver->class_ops.OpenPDB = JFile3_OpenPDB;
    driver->class_ops.CheckTitle = DS_PDB_CheckTitle;
    driver->class_ops.Create = JFile3_Create;
    driver->class_ops.SupportsFieldType = JFile3_SupportsFieldType;
    driver->class_ops.Delete = DS_PDB_Delete;
    driver->class_ops.Enumerate = JFile3_Enumerate;
    driver->class_ops.GlobalFind = JFile3_GlobalFind;
    driver->class_ops.IsPresent = JFile3_IsPresent;
    driver->class_ops.CompareKey = DS_PDB_CompareKey;
    driver->class_ops.CanHandlePDB = JFile3_CanHandlePDB;
    driver->class_ops.Beam = DS_PDB_Beam;

    return driver;
}
