/******************************************************************************

	sessions.c

	Handle multiple sessions. Create a new thread for each session.
	Multiple sessions may be through FIFOs or sockets.

******************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include "kernel/config.h"
#ifdef MULTITHREADED
#include <pthread.h>
#endif
#include "client.h"
#include "sync.h"
#include "auth/auth.h"

#define MAX_LINE 120

extern void *client (void*);
extern unsigned int clients, tot_clients;

void (*terminator) (void);
void (*closedown) (FILE*, FILE*);

/*
 *		Shutdown client functions
 */

#ifdef MULTITHREADED

static int sockfd;
static FILE *serv;
static char *m_fifo;

void term_sockets ()
{ shutdown (sockfd, 0); }

void term_fifo ()
{ fclose (serv); unlink (m_fifo); }

void close_sockets (FILE *a, FILE *b)
{ shutdown (fileno (a), 2); }

void close_fifos (FILE *a, FILE *b)
{ fclose (a); fclose (b); }

#else

void term_sockets () { }
void term_fifo () { }

#endif

void term_dummy () { }
void close_dummy (FILE*x, FILE*y) { }

#ifdef MULTITHREADED
/*
 * Clients are connected on TCP/IP with sockets.
 * A telnet (netcat) to the desired port shall do the job.
 */
void socket_tcp (short int port)
{
	struct sockaddr_in s_sock;

	if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) == -1)
	{
		perror ("Main socket");
		exit (1);
	}

	s_sock.sin_family = AF_INET;
	s_sock.sin_addr.s_addr = htonl (INADDR_ANY);
	s_sock.sin_port = htons (port);

	if (bind (sockfd, (struct sockaddr*)&s_sock, sizeof s_sock) == -1)
	{
		perror ("Bind");
		exit (1);
	}
}

void socket_unix (char *path)
{
	struct sockaddr_un s_sock;

	if ((sockfd = socket (AF_UNIX, SOCK_STREAM, 0)) == -1)
	{
		perror ("Main socket");
		exit (1);
	}

	s_sock.sun_family = AF_UNIX;
	unlink (path);
	strcpy (s_sock.sun_path, path);

	if (bind (sockfd, (struct sockaddr*)&s_sock, sizeof s_sock) == -1)
	{
		perror ("Bind");
		exit (1);
	}
}

void do_socket ()
{
	int clifd, ret;
	struct thread_params thp;
	pthread_t threadd;
	struct sockaddr_in c_sock;
	struct sembuf sem_operation = { 0, 0, SEM_UNDO };

	terminator = term_sockets;
	closedown = close_sockets;
	listen (sockfd, 5);

	while (1)
	{
		clifd = accept (sockfd, (struct sockaddr*)&c_sock, &ret);
		if (clifd == -1)
		{
			perror ("accept");
			continue;
		}

		thp.readf = fdopen (clifd, "r");
		thp.reply = fdopen (clifd, "w");
		thp.sue = 0;
		if (!thp.readf || !thp.reply)
			continue;

		auth (0, thp.readf, thp.reply);

		SYNC_CLI

		ret = pthread_create (&threadd, NULL, client, &thp);

		if (ret != 0)
			printf ("Can't create thread\n");
		else
		{
			pthread_detach (threadd);
			++clients;
			++tot_clients;
		}
	}
}

/*
 * Clients connect through FIFO specials.
 * Each client must send a string ":<read FIFO> <reply FIFO>" to 
 * establish com. Each client has its own FIFO pair.
 */
void fifo_sessions (char *main_fifo)
{
	char *c;
	struct thread_params thp;
	pthread_t threadd;
	int ret;
	char line_buf [MAX_LINE + 1];
	struct sembuf sem_operation = { 0, 0, SEM_UNDO };

	m_fifo = main_fifo;
	terminator = term_fifo;
	closedown = close_fifos;

	if (mkfifo (main_fifo, 0666) == -1)
	{
		perror ("Can't create master fifo");
		exit (1);
	}

	serv = fopen (main_fifo, "r");

	if (!serv)
	{
		perror ("Can't open master fifo");
		exit (1);
	}

	while (1)
	{
		line_buf [0] = 0;
		if (!fgets (line_buf, MAX_LINE, serv))
		{
			/* A client leaving the master FIFO will
			 * result in a continouus NULL. Close and reopen
			 * the FIFO to block again...
			 */
			fclose (serv);
			serv = fopen (main_fifo, "r");
			if (!serv)
			{
				perror ("Can't open master fifo");
				exit (1);
			}
			continue;
		}


		if (line_buf [0] != ':'  || line_buf [0] == 0
		|| line_buf [strlen (line_buf) - 1] != '\n')
			continue;

		line_buf [strlen (line_buf) - 1] = 0;
		if (!(c = strchr (line_buf, ' ')))
			continue;
		*c = 0;

		thp.readf = fopen (line_buf + 1, "r");
		thp.reply = fopen (c + 1, "w");
		thp.sue = 1;

		if (!thp.readf || !thp.reply)
			continue;

		SYNC_CLI
		ret = pthread_create (&threadd, NULL, client, &thp);

		if (ret != 0)
			printf ("Can't create thread\n");
		else
		{
			pthread_detach (threadd);
			++clients;
			++tot_clients;
		}
	}
}

#else

void do_socket ()
 { fputs ("Available only in multithreaded [mc]ode\n", stderr); exit (1); }
void socket_tcp (short int port) { do_socket (); }
void socket_unix (char *path) { do_socket (); }
void fifo_sessions (char *main_fifo) { do_socket (); }

#endif

/*
 * Single user from stdin/stdout.
 */
void single_std ()
{
	struct thread_params thp = { 1, stdin, stdout };

	clients = tot_clients = 1;
	terminator = term_dummy;
	closedown = close_dummy;

	client (&thp);
}
