/*****************************************************************************

	elementary.c 
 
 	All the elementary actions of Lndbase performed here.
	Thus providing the interface to lndb_calls

*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "global.h"
#include "polytypes.h"

#include "Polytypes/cistr.h"

PolymorphicType *polytype[] = {
	&cistr
};

const int NPTYPES = sizeof polytype / sizeof polytype [0];
const char MEM_FAIL [] = "Memory allocation Failure. sleeping for retry..\n";

Element **element;
Group root = { 0, "", NULL, NULL, NULL, NULL, NULL };

unsigned int Elements;
unsigned int ElTop, ElArrTop;

static Element *e_hash [ELEMENT_HASH];

#ifdef USE_SFMALLOC
 #include "sfmalloc.h"
 static SFMPOOL *esm, *gsm, *lsm;
 #define allocelem(x) x = (Element*) sfmalloc (esm)
 #define allocgrp(x) x = (Group*) sfmalloc (gsm)
 #define alloclink(x) x = (Link*) sfmalloc (lsm)
 #define freeelem(x) sfmalloc_free (esm, x)
 #define freegrp(x) sfmalloc_free (gsm, x)
 #define freelink(x) sfmalloc_free (lsm, x)
 void init_sfmalloc ()
 {
 	esm = sfmalloc_creat (sizeof (Element), 8);
	lsm = sfmalloc_creat (sizeof (Link), 9);
	gsm = sfmalloc_creat (sizeof (Group), 7);
 }
#else
 #define allocelem(x) x = (Element*) malloc (sizeof (Element))
 #define allocgrp(x) x = (Group*) malloc (sizeof (Group))
 #define alloclink(x) x = (Link*) malloc (sizeof (Link))
 #define freeelem(x) free (x)
 #define freegrp(x) free (x)
 #define freelink(x) free (x)
#endif

extern void freedataof (Element*);
extern int comparedataof (Element*, const char*);

/******************************************************************************
 *				Group routines
 * From the Dynamic Binary Search Tree Lndbase specific implementation.
 *****************************************************************************/

extern Group*		dbst_find (Group*, const char*);
extern unsigned int	dbst_insert (Group*);
extern unsigned int	dbst_init_insert (Group*);
extern void		dbst_rmv (Group*);

/*
 * Create a new group. No node_name duplicate must exist.
 */
Group *add_group (Group *p, const char *node_name)
{
	Group *g;

#ifdef BUG_TRAP
	assert (p);
	assert (dbst_find (p->child, node_name) == NULL);
#endif
	GUARD (allocgrp (g))
	GUARD (g->node_name = (char*) malloc (strlen (node_name) + 1))
	g->cnt = 0;
	g->child = NULL;
	g->e = NULL;
	strcpy (g->node_name, node_name);
	g->parent = p;
	dbst_insert (g);

	return g;
}

/*
 * This to be called once at initialization. A different approach to
 * make add_group and init_add_group one function could be done.
 */
Group *init_add_group (Group *p, const char *node_name)
{
	Group *g;

#ifdef BUG_TRAP
	assert (p);
#endif
	GUARD (allocgrp (g))
	GUARD (g->node_name = (char*) malloc (strlen (node_name) + 1))
	strcpy (g->node_name, node_name);
	g->cnt = 0;
	g->parent = p;
	g->child = NULL;
	dbst_init_insert (g);
	g->e = NULL;

	return g;
}

/*
 * The group should NOT have children, nor elements.
 */
void del_group (Group *g)
{
#ifdef BUG_TRAP
	assert (!g->e);
	assert (!g->child);
	assert (g != &root);
#endif
	dbst_rmv (g);
	freegrp (g);
}

/*
 * Find a group. The argument is an array of pointers to the subdirectories.
 */
Group *find_group (Group *g, char *arrv [])
{
#ifdef BUG_TRAP
	assert (g);
#endif
	while (*arrv)
		if (!(g = dbst_find (g->child, *arrv++)))
			return NULL;

	return g;
}

/*
 * Rename/move a group.
 * The group does not need to get a new name, thus c may be NULL.
 * Also, we can't rename a.b.c to a.b.c.d because we will have then
 *  broken the first rule of our multidimensional tree definition.
 */
int ch_parent (Group *g, Group *parent, const char *c)
{
	Group *tst;

	for (tst = parent; tst; tst = tst->parent)
		if (tst == g) return 0;

	dbst_rmv (g);
	if (c)
	{
		free (g->node_name);
		GUARD (g->node_name = (char*) malloc (strlen (c) + 1))
		strcpy (g->node_name, c);
	}
	g->parent = parent;
	dbst_insert (g);

	return 1;
}

/*****************************************************************************
 *				Elements & Groups
 ****************************************************************************/
static void elem_off_grp (Element *e)
{
	// Removal
	if (!e->pgrp)
		e->group->e = e->ngrp;
	else
		e->pgrp->ngrp = e->ngrp;

	if (e->ngrp)
		e->ngrp->pgrp = e->pgrp;

	e->group->cnt--;
}

static void elem_in_grp (Element *e, Group *g)
{
	// Insertal
	if ((e->ngrp = g->e))
		e->ngrp->pgrp = e;
	g->e = e;
	g->cnt++;
	e->pgrp = NULL;
	e->group = g;
}

/*
 * Change the group of an element.
 */
void chgrp (Element *e, Group *g)
{
	elem_off_grp (e);
	elem_in_grp (e, g);
}


/******************************************************************************
 *				Polymorphic Types
 *****************************************************************************/
void init_polytypes ()
{
	int i;

	for (i = 0; i < NPTYPES; i++)
		polytype [i]->init ();
}

/*
 * Special. When the element's data is changed, the element is removed
 * and reentered from its PMTs. That's because of hashing etc....
 */
static void global_reg (Element *e)
{
	short int i;

	for (i = 0; i < NPTYPES; i++)
		if ((TSTBIT(e, i)))
			if (!polytype [i]->reg (e))
				CLRBIT(e, i);
}
static void global_unreg (Element *e)
{
	short int i;

	for (i = 0; i < NPTYPES; i++)
		if ((TSTBIT(e, i)))
		{
			polytype [i]->unreg (e);
			CLRBIT(e, i);
		}
}

int register_type (Element *e, short int t)
{
	if ((TSTBIT(e, t)))
		return 0;
	if (polytype [t]->reg (e))
	{
		SETBIT(e, t);
		return 1;
	}
	return 0;
}

int unregister_type (Element *e, short int t)
{
	if (!(TSTBIT(e, t)))
		return 0;
	polytype [t]->unreg (e);
	CLRBIT(e, t);
	return 1;
}

short int pmt_byname (const char *c)
{
	int i;

	for (i = 0; i < NPTYPES; i++)
		if (strcmp (c, polytype [i]->code) == 0)
			return i;

	return -1;
}

/*
 * The cistr is builtin and thus give it a special interface
 */
unsigned int *find_cistr (char *name)
{
	return polytype [0]->locate (name, strlen (name), NULL);
}

unsigned int *locate (char *opts[], char *ptr, unsigned int len, char *pmt)
{
	short int i;

	if ((i = pmt_byname (pmt)) == -1)
		return BAD_PMT;

	return polytype [i]->locate (ptr, len, opts);
}

/*
 * Can be sequentially called and it will return the names of all
 * the pmts of an element. Provided for lndb_calls.
 * e == NULL will return all the pmts.
 */
char *show_pmts (Element *e, short int *ip)
{
	while (*ip < NPTYPES)
	{
		if (e == NULL || (TSTBIT(e, *ip)))
			return polytype [(*ip)++]->code;
		++*ip;
	}
	return NULL;
}

unsigned int pmt_rusage (short int t)
{
	return polytype [t]->rusage ();
}
/******************************************************************************
 *				    Links
 *****************************************************************************/
/*
 * Verify the existance of a link
 * One of the links is returned - to get link id.
 */
Link *find_link (Element *e1, Element *e2)
{
	Link *l;

	for (l = e1->link; l; l = l->next)
		if (l->e == e2) return l;

	return NULL;
}

/*
 * Create a new link. It must be called twice to really create a link
 * add_link (e1, e2) and add_link (e2, e1) since links are bidirectional.
 */
static Link *add_link (Element *e1, Element *e2)
{
	Link *l;

	GUARD (alloclink (l))
	l->next = e1->link;
	l->e = e2;
	e1->link = l;

	return l;
}
/*
 * Like add_link it must be called twice to do the job.
 * This one removes the link thats ``hanging'' from e1 and points to e2.
 */
static void del_link (Element *e1, Element *e2)
{
	Link *l, *i;

	l = e1->link;
	if (l->e == e2)
	{
		e1->link = l->next;
		freelink (l);
		return;
	}

	while (l)
	{
		i = l->next;
		if (i->e == e2)
		{
			l->next = i->next;
			freelink (i);
			break;
		}
		l = i;
	}
}

void link_elems (unsigned int i1, unsigned int i2, short int lid)
{
#ifdef BUG_TRAP
	assert (i1 < Elements && i2 < Elements);
#endif
	add_link (element [i1], element [i2])->lid = lid;
	add_link (element [i2], element [i1])->lid = lid;
}
void break_link (unsigned int i1, unsigned int i2)
{
#ifdef BUG_TRAP
	assert (i1 < Elements && i2 < Elements);
#endif
	del_link (element [i1], element [i2]);
	del_link (element [i2], element [i1]);
}

unsigned int count_links (unsigned int e)
{
	Link *l = element [e]->link;

	for (e = 0; l; e++)
		l = l->next;

	return e;
}

/******************************************************************************
 *				Elements
 *****************************************************************************/
/*
 * Hash signature. Xor'd samples from the data.
 * All we should want is fair distribution of elements/hash.
 */
static inline unsigned short int get_hash (const char *ptr, unsigned int len)
{
	char ca [2];

	ca [0] = *ptr ^ *(ptr + 3 * len / 4);
	ca [1] = *(ptr + len / 2) ^ *(ptr + len / 4);
	return (*(unsigned short int*)ca) % ELEMENT_HASH;
}

/*
 * Allocate a new Element, reallocate the element array if needed.
 * Nothing but the id of the Element is initialized.
 * Rationale.
 *	The element[] array is always expanding. Even if all the elements are
 *	removed and then a new element is created it will still expand the
 *	array. It follows the logic of `keeping IDs static'.
 *	After a major database cleanup, it's good to defrag & save & restart.
 */
static Element *creat_element ()
{
#define ELEM_INCR 100
	Element *e;
	Element **ee;

	if (ElTop == ElArrTop)
	{
		size_t i = sizeof (Element*) * (ElArrTop += ELEM_INCR);
		GUARD (ee = realloc (element, i))
		element = ee;
	}

	GUARD (allocelem(e))
	++Elements;

	return element [e->id = ElTop++] = e;
}

/*
 * Add an element to the search hash lists.
 * The hash signature is passed calculated because the element
 * may not be in memory. This is generic.
 */
static void to_hash (Element *e, unsigned short int sig)
{
	unsigned int i = e->len;
	Element *ep;

	ep = e_hash [sig];
	if (!ep || ep->len >= i)
	{
		if ((e->next = ep))
			e->next->prev = e;
		e_hash [sig] = e;
		e->prev = (Element*) &e_hash [sig];
		return;
	}
	for (;;ep = ep->next)
		if (!ep->next || ep->next->len >= i)
			break;
	e->prev = ep;
	if ((e->next = ep->next)) e->next->prev = e;
	ep->next = e;
}

/*
 * Remove an element from the hash tree/table/lists.
 * Normally, the ->prev member of the first element of the llist should be
 * NULL. But the hack is that its the (Element**) pointer of the location in
 * the e_hash table. This way we can remove the first element of the llist in
 * one step.
 */
static void off_hash (Element *e)
{
	int i;

	i = (Element**) e->prev - &e_hash [0];

	if (i >= 0 && i < ELEMENT_HASH)
	{
#ifdef BUG_TRAP
		assert (e_hash [i] == e);
#endif
		e_hash [i] = e->next;
	}
	else
		e->prev->next = e->next;

	if (e->next)
		e->next->prev = e->prev;
}

/*
 * New Element. It must not already exist in the database.
 * This one does nothing about the data, used when loading elements.
 */
Element *newelement (Group *g, const char *data, unsigned int len)
{
	Element *e;

	e = creat_element ();
	e->link = NULL;
	e->len = len;
	e->group = g;
	e->pgrp = NULL;
	if ((e->ngrp = g->e))
		e->ngrp->pgrp = e;
	g->e = e;
	++g->cnt;
	memset (&e->pmt_bool, 0, sizeof (polymorphic_bits));
	to_hash (e, get_hash (data, len));

	return e;
}

Element *new_element (Group *g, char *data, unsigned int len)
{
	Element *e;

	datato (e = newelement (g, data, len), data, len);
	return e;
}

/*
 * Rename reserved elements. Avoid freeing the previous data.
 */
void true_data (unsigned int i, char *data, unsigned int len)
{
	Element *e;
	off_hash (e = element [i]);
	datato (e, data, len);
	to_hash (e, get_hash (data, len));
}

/*
 * Remove an element
 */
void del_elem (unsigned int id)
{
	Element *e;
	Link *l;

#ifdef BUG_TRAP
	assert (id < ElTop);
	assert (element [id]);
#endif

	/*
	 * We have to:
	 *	free our links.
	 *	free the links from other elements to us.
	 *	Remove from the group.
	 *	remove from the hash table/llists.
	 *	unregister the element from its polymorphic types.
	 *	void the position in `element[]'
	 *	free the element (and it's data-->remove tmp files...)
	 */
	e = element [id];
	while ((l = e->link))
		del_link (l->e, e), del_link (e, l->e);
	elem_off_grp (e);
	off_hash (e);
	global_unreg (e);
	element [id] = NULL;
	freedataof (e);
	--Elements;
}
	
/*
 * Locate an element by content, if exists.
 * Returns id or ElTop.
 */
unsigned int eexists (char *data, unsigned int len)
{
	Element *e;

	e = e_hash [get_hash (data, len)];
	while (e && e->len < len)
		e = e->next;
	if (len < BIG_ELEMENT)
	{
		// No need to jump to vdata.c:comparedataof
		// The element is definately in some kind
		// of memory. It's faster this way
		for (; e && e->len == len; e = e->next)
			if (memcmp (e->data, data, len) == 0)
				return e->id;
	}
	else
		for (; e && e->len == len; e = e->next)
			if (comparedataof (e, data) == 0)
				return e->id;

	return FREE_ID;
}

/*
 * Change the data of an element.
 * We have to leave and reenter in the PMTs
 */
void change_dataof (Element *e, char *ptr, unsigned int len)
{
	char pmt_tmp [PMT_BYTES];

	memcpy (pmt_tmp, e->pmt_bool.ch, PMT_BYTES);
	global_unreg (e);
	off_hash (e);
	freedataof (e);
	memcpy (e->pmt_bool.ch, pmt_tmp, PMT_BYTES);
	datato (e, ptr, len);
	to_hash (e, get_hash (ptr, len));
	global_reg (e);
}

/********************************* Defrag *************************************/

/*
 * All sessions MUST be closed or alternatively this should be
 * called after a fork () so it won't change the actual data.
 */
void defrag ()
{
	unsigned int v, e;

	for (v = 0, e = ElTop - 1;;)
	{
		while ( element [v]) v++;
		if (v >= Elements) break;
		while (!element [e]) e--;
		element [v] = element [e];
		element [e] = NULL;
		element [v]->id = v;
	}
	ElTop = Elements;
}
