/******************************************************************************
	dbstree.C 

	Dynamic Binary Search Tree used throughout the entire project.

	A dbsNode applied as a parent class to a specific class will provide
	transparent storage in a dbstree. As a result no duplicates may be
	stored and locating stored data is performed in less than 32 steps.

	Check dbstree.tex for technical information on this tree.

*****************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

//
// int compare (dbsNode*)	: compare two nodes
// int compare ()		: compare a node with external
//				: node-candidate data
//
// dbsNode::dbsNode must be called after a dbsFind.
// This is the normal procedure, search if its in there and if not insert it.
// But except from the normal procedure this is a MUST. dbsFind if the
// requested data is not found will set the FoundSlot which the dbsNode::dbsNode
// uses to insert into the tree. We avoid two searches - one to check if in
// there and one to find where to insert the new node.
// dbsFind also sets a few other things, depth of the FoundSlot and the
// root of the tree.
//
// So, we call dbsFind and "new node" in the SAME code block without other
// dbsActions between them.
//
// dbstree is not so big. Much of the code is for random access by in-order
// in N/16 steps average case.
//
// Q: Why do we always have to do:  Tree.dbsRemove (this)
//    in the dbsNodes? Couldn't dbsRemove() be a member of dbsNode
//    instead of dbsTree?
// A: In order to remove we must know the tree and the node,
//    nodes do not know their tree because that would be extra
//    sizeof (pointer) to each node.
//    The other case would be:  this.dbsRemove (&Tree)
//    at dbsNode's destructor. But destructors do not take arguments
//    and there should must be a way for the destructor to know
//    the node's tree. A caller function knowing both could not
//    pass the tree to the destructor, unless with static hacks.
//
#include <stdio.h>
#include <assert.h>
#include "dbstree.h"

dbsTree::dbsTree ()
{
	nnodes = 0;
	root = NULL;
	FoundSlot = NULL;
	nrt = nrl = nrm = 0;
	iAccess = true;
	mod = false;
}

/*
 * Guide nodes are root, root->less, root->mode
 * those know their Inorder numeric index in the tree.
 * May need to recalculate after a dbsBalance ()
 * -- this applies if iAccess==true
 */
void dbsTree::count_nodes (dbsNode *n)
{
	++FoundDepth;
if (n->more == n) fprintf (stderr, "Rare Bug: F*&KED UP!!!\n");
	if (n->more) count_nodes (n->more);
	if (n->less) count_nodes (n->less);
}
void dbsTree::count_guides ()		// O(n)
{
	nrl = -1, nrm = nrt = 0;
	if (!root) return;
	if (root->less) {
		FoundDepth = 0;
		if (root->less->less)
			count_nodes (root->less->less);
		nrl = FoundDepth;
		FoundDepth = 0;
		if (root->less->more)
			count_nodes (root->less->more);
		nrt = nrl + 1 + FoundDepth;
	}
	if (root->more) {
		FoundDepth = 0;
		if (root->more->less)
			count_nodes (root->more->less);
		nrm = nrt + 1 + FoundDepth;
	}
	FoundDepth = 0;
}

/*
 * Balancing a binary tree. This happens whenever the height reaches 32.
 */
void dbsTree::tree_to_array (dbsNode *n)
{
	if (n->less) tree_to_array (n->less);
	*FoundSlot++ = n;
	if (n->more) tree_to_array (n->more);
}
void dbsTree::dbsBalance ()		// O(n)
{
	dbsNode **npp;
	unsigned long long i, j, k, D, Y, k2, k3;

	if (!root) return;

	npp = FoundSlot = (dbsNode**) alloca (sizeof (dbsNode*) * nnodes);
	tree_to_array (root);

	root = npp [nnodes / 2];
	for (D = nnodes + 1, i = 4; i <= D; i *= 2)
		for (j = 2; j < i; j += 4)
		{
			k3 = nnodes * j / i;
			npp [k3]->less = npp [nnodes * (j - 1) / i],
			npp [k3]->more = npp [nnodes * (j + 1) / i];
		}
	k = nnodes + 1 - (Y = i / 2);
	if (k == 0)
	{
		for (i /=2, j = 1; j < i; j += 2)
			k3 = nnodes * j / i,
			npp [k3]->less = npp [k3]->more = NULL;
		return;
	}

	for (j = 2; j < i; j += 4)
	{
		k3 = nnodes * j / i;
		D = (k2 = (j - 1) * nnodes / i) * Y % nnodes;
		if (D >= k || D == 0)
			npp [k3]->less = NULL;
		else
		{
			npp [k3]->less = npp [k2];
			npp [k2]->less = npp [k2]->more = NULL;
		}
		D = (k2 = (j + 1) * nnodes / i) * Y % nnodes;
		if (D >= k || D == 0)
			npp [k3]->more = NULL;
		else
		{
			npp [k3]->more = npp [k2];
			npp [k2]->less = npp [k2]->more = NULL;
		}
	}

	dbsNode *np;
	for (np = root; np->less; np = np->less);

	np->less = npp [0];
	npp [0]->less = npp [0]->more = NULL;

	if (iAccess) count_guides ();
}

void dbsNode::addself (dbsTree *t)
{
	*t->FoundSlot = this;
	++t->nnodes;

	t->mod = true;

	/* Indexer */
	if (t->iAccess) if (t->FoundDepth == 0)
		{ t->nrl = -1, t->nrm = t->nrt = 0; }
	else switch (t->dflag) {
		case 0: t->nrl++;
		case 2: t->nrt++;
		case 1: t->nrm++;
	}

	if (t->FoundDepth >= DBS_MAGIC)
		t->dbsBalance ();
	t->FoundSlot = NULL;	// Bug traper
}

dbsNode::dbsNode (dbsTree *t)
{
	if (t->FoundSlot) addself (t);

	less = more = NULL;
}

dbsNode::~dbsNode ()
{ }

/*
 */
void dbsTree::dbsRemove (dbsNode *t)		// O(log n)
{
	dbsNode *np, *nl, *nr, *nlp, *nrp;
	int isroot;
	unsigned int i, j;
	int help = 0;

	isroot = (np = t->myParent (this)) == NULL;

	--nnodes;
	mod = true;

	/* Indexer */
	if (iAccess) {
		// reevaluate guide node indexes
		// hard one is when root, root->less or root->more removed
		if (isroot) {
			nrm--;
			help = 3;
		} else if (t->compare (root) > 0) {
			int cm;
			if ((cm = t->compare (root->more)) < 0)
				nrm--;
			else if (cm == 0) help = 1;
		} else {
			int cm;
			nrm--;
			nrt--;
			if ((cm = t->compare (root->less)) < 0)
				nrl--;
			else if (cm == 0) help = 2;
		}
	}


	if (!(t->less && t->more))
	{
		if (isroot)
			root = (t->less) ? t->less : t->more;
		else
			if (np->less == t)
				np->less = (t->less) ? t->less : t->more;
			else
				np->more = (t->less) ? t->less : t->more;
		if (help) count_guides ();
		return;
	}

	for (i = 0, nlp = NULL, nl = t->less; nl->more; i++)
		nlp = nl, nl = nl->more;
	for (j = 0, nrp = NULL, nr = t->more; nr->less; j++)
		nrp = nr, nr = nr->less;

	if (i >= j)		// the smallest from bigger ones
	{
		if (isroot) root = nl;
		else
			if (np->less == t) np->less = nl;
			else np->more = nl;
		if (nlp)
		{
			nlp->more = nl->less;
			nl->less = t->less;
		}
		nl->more = t->more;
	}
	else	// Mirror situation
	{
		if (isroot) root = nr;
		else
			if (np->less == t) np->less = nr;
			else np->more = nr;
		if (nrp)
		{
			nrp->less = nr->more;
			nr->more = t->more;
		}
		nr->less = t->less;
	}

	/* Indexer is here and needs help */
	if (help && i >= j)
		switch (help) {
		case 2:
			nrl--;
			break;
		case 1:
			nrm--;
			break;
		case 3:
			nrt--;
		}
}

dbsNode *dbsNode::myParent (dbsTree *t)		// O(log n)
{
	dbsNode *np;

	if ((np = t->root) == this)
		return NULL;

	while (np)
		if (compare (np) < 0)
			if (np->less == this) break;
			else np = np->less;
		else
			if (np->more == this) break;
			else np = np->more;

	assert (np);

	return np;
}

dbsNode *dbsTree::dbsFind ()		// O (log n)
{
	dbsNode *d;
	int i;

	dflag = FoundDepth = 0;

	if (!(d = root)) {
		FoundSlot = &root;
		return NULL;
	}

	++FoundDepth;
	/* Indexer */
	if (iAccess) {		// first two comparisons saved in dflag
		for (;FoundDepth <= 2; ++FoundDepth)
			if ((i = d->compare ()) == 0) {
				FoundSlot = NULL;
				return d;
			} else if (i < 0) {
				dflag |= FoundDepth;
				if (d->more) d = d->more;
				else {
					FoundSlot = &d->more;
					return NULL;
				}
			} else {
				dflag &= ~FoundDepth;
				if (d->less) d = d->less;
				else {
					FoundSlot = &d->less;
					return NULL;
				}
			}
	}

	for (;; ++FoundDepth) {
		if ((i = d->compare ()) == 0) {
			FoundSlot = NULL;
			return d;
		}
		if (i < 0)
			if (d->more) d = d->more;
			else {
				FoundSlot = &d->more;
				return NULL;
			}
		else
			if (d->less) d = d->less;
			else {
				FoundSlot = &d->less;
				return NULL;
			}
	}
}

/*
 * Moving in the tree.
 *  If we want for every node of the tree: foreach ()	O(n)
 *  To copy the tree - preorder: copytree ()		O(n)
 *  Safe to node deletion (but no dbsRemove, just
 *	root=NULL, cnt=0) - postorder: deltree ()	O(n) 
 *  To a specific index: operator [i]			O(n)  ...(n/16)
 *     But for every node with operator[] total is ... n^2  CAREFUL!
 *  Inorder next/prev dbsNext, dbsPrev:			O(log n)
 *  For a scroller area Use dbsNext, dbsPrev and keep a sample node
 *  if the tree is modified, reget the sample node from operator [].
 *
 */

void dbsTree::walk_tree (dbsNode *n, void (*foo)(dbsNode*))
{
	foo (n);
	if (n->less) walk_tree (n->less, foo);
	if (n->more) walk_tree (n->more, foo);
}
void dbsTree::walk_tree_io (dbsNode *n, void (*foo)(dbsNode*))
{
	if (n->less) walk_tree_io (n->less, foo);
	foo (n);
	if (n->more) walk_tree_io (n->more, foo);
}
void dbsTree::walk_tree_d (dbsNode *n, void (*foo)(dbsNode*))
{
	if (n->less) walk_tree_d (n->less, foo);
	if (n->more) walk_tree_d (n->more, foo);
	foo (n);
}


void dbsTree::copytree (void (*f)(dbsNode*))
{
	if (root) walk_tree (root, f);
}

void dbsTree::foreach (void (*f)(dbsNode*))
{
	if (root) walk_tree_io (root, f);
}

void dbsTree::deltree (void (*f)(dbsNode*))
{
	if (root) walk_tree_d (root, f);
}
/*
 * Indexed by inorder access
 */

void dbsTree::io_nth (dbsNode *n)
{
	if (n->less) io_nth (n->less);
	if (tb) return;
	if (FoundDepth == tmp) {
		tb = true;
		gNode = n;
		return;
	}
	++FoundDepth;
	if (n->more) io_nth (n->more);
}

void dbsTree::rio_nth (dbsNode *n)
{
	if (n->more) rio_nth (n->more);
	if (tb) return;
	if (FoundDepth == tmp) {
		tb = true;
		gNode = n;
		return;
	}
	++FoundDepth;
	if (n->less) rio_nth (n->less);
}

dbsNode *dbsTree::operator [] (int i)
{
	// Return the Nth by Inorder node in N/8 steps worst case
	if (i >= nnodes) return NULL;

	tb = false;
	FoundDepth = 0;
	if (i == nrt) return root;

	if (i < nrt) {
		if (i == nrl) return root->less;
		if (i < nrl)
			if (i < nrl / 2) {
				tmp = i;
				io_nth (root->less);
			} else {
				tmp = nrl - i - 1;
				rio_nth (root->less->less);
			}
		else {
			if (i < nrl + (nrt - nrl) / 2) {
				tmp = i - nrl - 1;
				io_nth (root->less->more);
			} else {
				tmp = nrt - i - 1;
				rio_nth (root->less);
			}
		}
		return gNode;
	}

	if (i == nrm) return root->more;

	i -= nrt + 1;
	int sv = nrm;
	nrm -= nrt + 1;
	int N = nnodes - (nrt + 1);

	if (i < nrm)
		if (i < nrm / 2) {
			tmp = i;
			io_nth (root->more);
		} else {
			tmp = nrm - i - 1;
			rio_nth (root->more->less);
		}
	else {
		if (i < nrm + (N - nrm) / 2) {
			tmp = i - nrm - 1;
			io_nth (root->more->more);
		} else {
			tmp = N - i - 1;
			rio_nth (root->more);
		}
	}

	nrm = sv;
	return gNode;
}

dbsNode *dbsTree::dbsNext (dbsNode *d)
{
	if (root == NULL) return NULL;

	if (d->more) {
		d = d->more;
		while (d->less) d = d->less;
		return d;
	}

	while (1) {
		dbsNode *p = d->myParent (this);
		if (!p) return NULL;
		if (p->more != d)
			return p;
		d = p;
	}
}

dbsNode *dbsTree::dbsPrev (dbsNode *d)
{
	if (root == NULL) return NULL;

	if (d->less) {
		d = d->less;
		while (d->more) d = d->more;
		return d;
	}

	while (1) {
		dbsNode *p = d->myParent (this);
		if (!p) return NULL;
		if (p->less != d)
			return p;
		d = p;
	}
}

//***************************************************************
// A tree of case sensitive strings -- char *Name
//***************************************************************

char *dbsNodeStr::Query;

int dbsNodeStr::compare (dbsNode *n)
{
	return strcmp (Name, ((dbsNodeStr*) n)->Name);
}

int dbsNodeStr::compare ()
{
	return strcmp (Name, Query);
}

dbsNodeStr::dbsNodeStr (dbsTree *t) : dbsNode (t)
{ }

dbsNodeStr::~dbsNodeStr ()
{ }
