/*****************************************************************************

	interface to server

	All command instances are global initialized at runtime.

	Failure to get something (not connected, timeout..) is
	transformed to a fake server reply.

*****************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <sys/stat.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

#include "iServ.h"
#include "epil.h"
#include "sfFetch.h"

extern char *StrDup (char*);

#ifndef __GLIBC__
 typedef int socklen_t;
#endif

#define MAX_LINE 200
#define CLIENT_TIMEOUT 100000

char iCommand::query [QUERYLEN];

bool connected = false;
bool issue = false;

//*********************** Command Interface ************************

iCommand Ln_New		("new", 1);
iCommand Ln_Alt		("alt", 1);
iCommand Ln_Quit	("quit");
iCommand Ln_Chgrp	("chgrp", 1);
iCommand Ln_Del		("del", 1);
iCommand Ln_Link	("link", 1);
iCommand Ln_Mkgrp	("mkgrp", 1);
iCommand Ln_Rengrp	("rengrp", 1);
iCommand Ln_Rmgrp	("rmgrp", 1);
iCommand Ln_Unlink	("unlink", 1);
iCommand Ln_Unset	("unset", 1);
iCommand Ln_Set		("set", 1);
iCommand Ln_Cd		("cd");
iCommand Ln_Sue		("sue");
iCommand Ln_Save	("save", 1);
iCommand Ln_Shutdown	("shutdown", 1);

// LR expect
iCommand Ln_Groupof	("groupof");
iCommand Ln_Idof	("idof");
iCommand Ln_Lenof	("lenof");
iCommand Ln_Pwd		("pwd");
iCommand Ln_Nlinks	("nlinks");	
iCommand Ln_Vlink	("vlink");
iCommand Ln_Nls		("nls");

// MLR expect
iCommand Ln_Dir		("dir");
iCommand Ln_Find	("find");
iCommand Ln_Ls		("ls");
iCommand Ln_Stats	("stats");
iCommand Ln_Typesof	("typesof");
iCommand Ln_Links	("links");
iCommand Ln_Glinks	("glinks");

// shR expect
iCommand Ln_Sh		("sh");

// Content Supplied
iCommandCL Ln_Newcl	("newcl", 1);
iCommandCL Ln_Altcl	("altcl", 1);

// LR expect
iCommandCL Ln_Idofcl	("idofcl");

// MLR expect
iCommandCL Ln_Locate	("locate");

//************************* Implementation ************************

static FILE *lndb_send, *lndb_get;

static int fread_timeout (char *t, unsigned int cl, FILE *inf, time_t msec)
{
	unsigned int offset = 0, i;
	pollfd pfd = { fileno (inf), POLLIN + POLLHUP, 0 };

	fcntl (fileno (inf), F_SETFL, O_NONBLOCK);
	do {
		i = fread (t + offset, 1, cl - offset, inf);
		if ((offset += i) == cl)
		{
			fcntl (fileno (inf), F_SETFL,
				fcntl (fileno (inf), F_GETFD) & ~O_NONBLOCK);
			return 0;
		}
	} while (poll (&pfd, 1, msec) > 0
		 && !(pfd.revents & POLLHUP));

	return 1;
}

static int fgets_timeout (char *ptr, unsigned int len, FILE *inf, time_t msec)
{
	pollfd pfd = { fileno (inf), POLLIN + POLLHUP, 0 };

	fcntl (fileno (inf), F_SETFL, O_NONBLOCK);

	if (!fgets (ptr, len, inf))
	{
		if (poll (&pfd, 1, msec) <= 0 || (pfd.revents & POLLHUP))
			return 1;

		fgets (ptr, len, inf);
	}
	fcntl (fileno (inf), F_SETFL,
		fcntl (fileno (inf), F_GETFD) & ~O_NONBLOCK);

	return 0;
}

// ******************************* Reply ************************************

Reply::Reply ()
{
	StatusLine = NULL;
	iCommand::setReply (this);
	ds = 1;
}

Reply::Reply (Reply &R)
{
	StatusLine = R.StatusLine;
	StatusCode = R.StatusCode;
	ds = R.ds;
}

static char badreply [] =
 "- 100 Bad reply from Lndbase server! Protocol error or bug";

int Reply::CheckReplyStatus (char *c)
{
	if (*c != '-' && *c != '*')
		return 0;

	if (c [1] != ' ' || !isdigit (c [2]))
		return 0;

	StatusLine = StrDup (c);

	if (!(c = strchr (StatusLine + 2, ' ')))
		StatusLine = StrDup (badreply);

	*c = 0;
	StatusCode = atoi (StatusLine + 2);
	*c = ' ';

	return 1;
}

int Reply::Status ()
{
	return StatusLine [0] == '*';
}

int Reply::CheckLine (char *c)
{
	int i = strlen (c);

	if (i == 0 || c [i - 1] != '\n')
		StatusLine = StrDup (badreply);

	c [i - 1] = 0;

	return i;
}

static char servereof [] =
 "- 100 Timeout waiting for reply from server";

void Reply::Get ()
{
	char tmp [MAX_LINE];

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT)) {
		CheckReplyStatus (servereof);
		return;
	}

	CheckLine (tmp);
	if (!CheckReplyStatus (tmp))
		CheckReplyStatus (badreply);
}

void Reply::Release ()
{
	if (StatusLine) delete StatusLine;
	ds = 0;
}

Reply::~Reply ()
{
	if (ds) Release ();
}

// **************************** LineReply **********************************

LineReply::LineReply () : Reply ()
{
	DataLine = NULL;
}

LineReply::LineReply (LineReply &R) : Reply (R)
{
	DataLine = R.DataLine;
}

void LineReply::Get ()
{
	char tmp [MAX_LINE];
	int i;

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT)) {
		CheckReplyStatus (servereof);
		return;
	}

	i = CheckLine (tmp);
	if (CheckReplyStatus (tmp))
		return;

	DataLine = StrDup (tmp);
	DataLine [i - 1] = 0;

	Reply::Get ();
}

void LineReply::Release ()
{
	if (DataLine) delete DataLine;
	Reply::Release ();
}

// ***************************** MultiLineReply *****************************

MultiLineReply::MultiLineReply () : Reply ()
{
	DataLine = NULL;
	Lines = 0;
}

MultiLineReply::MultiLineReply (MultiLineReply &R) : Reply (R)
{
	DataLine = R.DataLine;
	Lines = R.Lines;
}

void MultiLineReply::Get ()
{
	char tmp [MAX_LINE];
	int i;
	struct llist {
		char *str;
		llist *next;
	} *first = NULL, *curr = NULL /*avoid waring*/;

	for (Lines = 0;; Lines++)
	{
		if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10*CLIENT_TIMEOUT))
			strcpy (tmp, servereof);

		i = CheckLine (tmp);
		if (CheckReplyStatus (tmp))
			break;

		llist *nanuk = (llist*) alloca (sizeof (llist));
		if (!first)
			first = nanuk;
		else
			curr->next = nanuk;

		curr = nanuk;
		nanuk->str = new char [i];
		strcpy (nanuk->str, tmp);
	}

	DataLine = new char* [Lines + 1];
	for (i = 0, curr = first; i < Lines; i++, curr = curr->next)
		DataLine [i] = curr->str;
	DataLine [i] = NULL;
}

void MultiLineReply::Release ()
{
	int i;

	for (i = 0; i < Lines; i++)
		delete DataLine [i];

	if (DataLine) delete DataLine;
	Reply::Release ();
}

// ****************************** shReply **********************************

shReply::shReply () : Reply ()
{
	Data = NULL;
	ContentLength = 0;
}

shReply::shReply (shReply &R) : Reply (R)
{
	Data = R.Data;
	ContentLength = R.ContentLength;
}

void shReply::Get ()
{
	char tmp [MAX_LINE];

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT))
		strcpy (tmp, servereof);

	CheckLine (tmp);
	if (CheckReplyStatus (tmp))
		return;

	if (strncmp (tmp, "CL: ", 4)) {
		CheckReplyStatus (badreply);
		return;
	}

	Data = new char [1 + (ContentLength = atoi (tmp + 4))];
	Data [ContentLength] = 0;

	if (fread_timeout (Data, ContentLength, lndb_get, 10 * CLIENT_TIMEOUT))
	{
		delete Data;
		CheckReplyStatus (servereof);
		return;
	}

	fgetc (lndb_get);

	Reply::Get ();
}

void shReply::Release ()
{
	if (Data) delete Data;
	Reply::Release ();
}

//**************************************************************************//

static int connect_tcp (char *host)
{
	int sock;
	sockaddr_in addr;
	hostent *ho;
	char *p;

	if ((p = strchr (host, ':')) == NULL)
		return 0;

	*p = 0;
	if (!(ho = gethostbyname (host)))
		return 0;

	sock = socket (AF_INET, SOCK_STREAM, 0);
	addr.sin_family = AF_INET;
	addr.sin_port = htons (strtoul (p + 1, NULL, 10));
	addr.sin_addr.s_addr = (*(struct in_addr*)*ho->h_addr_list).s_addr;

	if (connect (sock, (sockaddr*)&addr, sizeof addr) == -1)
		return 0;

	lndb_get = fdopen (sock, "r");
	lndb_send = fdopen (sock, "w");

	return 1;
}

static int connect_unix (char *c)
{
	int sock;
	sockaddr_un addr;

	sock = socket (AF_UNIX, SOCK_STREAM, 0);
	addr.sun_family = AF_UNIX;
	sprintf (addr.sun_path, c);

	if (connect (sock, (struct sockaddr*)&addr, sizeof addr) == -1)
		return 0;

	lndb_get = fdopen (sock, "r");
	lndb_send = fdopen (sock, "w");

	return 1;
}

// ************************* iCommands *************************************

Reply *iCommand::CurrentReply;

iCommand::iCommand (char *c, bool su)
{ cmd_name = c; suecmd = su; }

void iCommand::setReply (Reply *R)
{ CurrentReply = R; }

static char notsue [] = "- 1000 No permission to modify the database:";
static char notconn [] = "- 1000 Not connected to Lndbase server";

int iCommand::SendQuery ()
{
	if (!connected) {
		CurrentReply->CheckReplyStatus (notconn);
		return CurrentReply->Status ();
	}
	if (!issue && suecmd)
		throw epilException (notsue, cmd_name);
	fputs (query, lndb_send);
	fflush (lndb_send);
	CurrentReply->Get ();
	return CurrentReply->Status ();
}
void iCommandCL::SendContent (void *content)
{
	if (!connected) {
		CurrentReply->CheckReplyStatus (notconn);
		return;
	}
	if (!issue && suecmd)
		throw epilException (notsue, cmd_name);
	fwrite (content, 1, cl, lndb_send);
	fputc ('\n', lndb_send);
	fflush (lndb_send);
	CurrentReply->Get ();
}

void iCommand::operator () (void)
{
	sprintf (query, "%s\n", cmd_name);
	SendQuery ();
}
void iCommand::operator () (unsigned int x)
{
	sprintf (query, "%s %u\n", cmd_name, x);
	SendQuery ();
}
void iCommand::operator () (char *x)
{
	sprintf (query, "%s %s\n", cmd_name, x);
	SendQuery ();
}
void iCommand::operator () (unsigned int x, char *c)
{
	sprintf (query, "%s %u %s\n", cmd_name, x, c);
	SendQuery ();
}
void iCommand::operator () (unsigned int x, unsigned int c)
{
	sprintf (query, "%s %u %u\n", cmd_name, x, c);
	SendQuery ();
}
void iCommand::operator () (unsigned int x, unsigned int c, int y)
{
	sprintf (query, "%s %u %u %i\n", cmd_name, x, c, y);
	SendQuery ();
}
void iCommand::operator () (char *x, char *c)
{
	sprintf (query, "%s %s %s\n", cmd_name, x, c);
	SendQuery ();
}
void iCommandCL::operator () (unsigned int i, void *v)
{
	iCommand::operator () (cl = i);
	if (CurrentReply->Status ())
		SendContent (v);
}
void iCommandCL::operator () (unsigned int i, unsigned int j, void *v)
{
	iCommand::operator () (i, cl = j);
	if (CurrentReply->Status ())
		SendContent (v);
}
void iCommandCL::operator () (unsigned int ol, unsigned int c, char *pmt, char **l, void *v)
{
	sprintf (query, "%s %i %i %s\n", cmd_name, ol, c, pmt);

	if (!SendQuery ()) return;

	for (unsigned int i = 0; i < ol; i++) {
		fputs (l [i], lndb_send);
		fputc ('\n', lndb_send);
	}
	fwrite (v, 1, c, lndb_send);
	fputc ('\n', lndb_send);
	fflush (lndb_send);
	CurrentReply->Get ();
}

//*****************************************************************************
// epil macros from this code
//*****************************************************************************

static void pushMLR (MultiLineReply *MLR)
{
	int i, j = 1;
	char *out;

	for (i = 0; i < MLR->Lines; i++)
		j += strlen (MLR->DataLine [i]) + 1;

	out = (char*) alloca (j + 1);
	out [0] = 0;
	for (i = 0; i < MLR->Lines; i++)
		strcat (strcat (out, MLR->DataLine [i]), "\n");

	ep_push (out, j);
}

static void get_welcome ()
{
	MultiLineReply MLR;
	MLR.Get ();

	FETCH_MLREPLY (&MLR);
	pushMLR (&MLR);
}

static void coded_connect_unix ()
{
	char *c;
	TMP_POP_ALLOC(c)

	if (!connected)
		if ((ep_boolean = connected = connect_unix (c)))
			get_welcome ();
}

static void coded_connect_tcp ()
{
	char *c;
	TMP_POP_ALLOC(c)

	if (!connected)
		if ((ep_boolean = connected = connect_tcp (c)))
			get_welcome ();
}

static void coded_login ()
{
	char *c;
	TMP_POP_ALLOC(c)

	if (!issue) {
		Reply R;
		Ln_Sue (c);
		issue = ep_boolean = R.Status ();
	}
}

static void coded_lnstats ()
{
	MultiLineReply MLR;

	Ln_Stats ();
	FETCH_MLREPLY (&MLR);
	pushMLR (&MLR);
}

void iServ ()
{
	CODED(connect_unix);
	CODED(connect_tcp);
	CODED(login);
	CODED(lnstats);
}
