//******************************************************************************
//
//	Making an editor out of a readline comes true.
//	The good thing is that all the control characters an editor might
//	need ARE virtuals in breadline. So we can proceed and dump these 
//	inside the editor object.
//	The smallest editor in the history of readlines...
//
//******************************************************************************
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "dLIST.h"
#include "editor.h"
#include "rows.h"

extern int c_scr_bg, c_scr_fg, c_scr_rv;

//**************************************************
// internal readlines
//**************************************************
editor::intrl::intrl (char *c, int w) :
	breadline (), normtab (), novarrow ()
{
	Width = w;
	prompt = NULL;
	line.reset (c);
	Vcur = 0;
}

//**************************************************
// show cursor or vanish cursor
//**************************************************
void editor::scursor ()
{
	if (JCursor::Readline == this)
		pcursor (VLINE->line.cursor - VLINE->Vcur, nc - nt,
		      (VLINE->line.line [VLINE->line.cursor]) ?: ' ');
}

//**************************************************
// draw a readline at row r
//**************************************************
void editor::drawline (breadline *b, int r)
{
	if (b->line.cursor > b->Vcur + b->Width)
		b->Vcur = 1 + b->line.cursor - b->Width;
	puttext (0, r, b->line.line + b->Vcur, c_scr_fg);
	if (b->Vcur + b->Width > b->line.linelen)
		puttext (b->line.linelen - b->Vcur, r, "  ", c_scr_fg);
}

//**************************************************
// draw all the lines
//**************************************************
void editor::drawalllines ()
{
	dlistNode *b = t;
	char spaces [tX];

	memset (spaces, ' ', tX - 1);
	spaces [tX - 1] = 0;

	vscreen::draw ();
	for (int i = 0; i < (int)tY; i++) {
		if (b) drawline (LINE(b), i);
		else puttext (0, i, spaces, 1);
		if (b) b = rlList.Next (b);
	}
	scursor ();
}

//**************************************************
// vertical cursor movement
//**************************************************
int editor::vmove (bool up)
{
	dlistNode *e;

	e = ((up) ? rlList.Prev (vcursor) : rlList.Next (vcursor));

	if (!e) return 0;

	if (up) {
		if (--nc < nt) {
			t = e;
			nt = nc;
			drawalllines ();
		}
	} else if (++nc >= nt + (int)tY) {
			nt++;
			t = rlList.Next (t);
			drawalllines ();
	}
	vcursor = e;

	return 1;
}

void editor::do_varrow ()
{
	dlistNode *tmp = vcursor;
	if (vmove (cc == K_UP)) {
		VLINE->line.cursor =
			(LINE(tmp)->line.cursor > VLINE->line.linelen) ?
			(VLINE->line.linelen) : LINE(tmp)->line.cursor;
		if (VLINE->line.cursor > VLINE->Width)
			drawline (VLINE, nc - nt);
		if (LINE(tmp)->Vcur) {
			LINE(tmp)->do_char (K_HOME);
			drawline (LINE(tmp), nc - nt + ((cc == K_UP) ? 1 : -1));
		}
	}
	scursor ();
}

void editor::do_harrow ()
{
	if (cc == K_LEFT) {
		if (VLINE->line.cursor > 0)
			VLINE->do_char (cc);
		else if (vmove (true)) {
			VLINE->line.cursor = VLINE->line.linelen;
			drawline (VLINE, nc - nt);
		}
	} else
		if (VLINE->line.cursor < VLINE->line.linelen)
			VLINE->do_char (cc);
		else {
			dlistNode *tmp = vcursor;
			if (vmove (false)) {
				LINE(tmp)->do_char (K_HOME);
				drawline (LINE(tmp), nc - 1 - nt);
				VLINE->line.cursor = 0;
			}
		}
	drawline (VLINE, nc - nt);
	scursor ();
}

//**************************************************
// enter is the line creator
//**************************************************
void editor::do_enter ()
{
	intrl *i = new intrl (VLINE->line.line + VLINE->line.cursor, tX);
	i->Width = tX;
	VLINE->line.deltoend ();
	VLINE->do_char (K_HOME);
	rlList.addafter (vcursor, i);
	vmove (false);
	drawalllines ();
	ntot++;
}

//**************************************************
// bs & del may be the line destroyers
//**************************************************
void editor::do_bsdel ()
{
	dlistNode *b;

	if (cc == '\b') {
		if (VLINE->line.cursor > 0) {
			VLINE->do_char (cc);
			drawline (VLINE, nc - nt);
			scursor ();
		} else if ((b = rlList.Prev (vcursor))) {
			LINE(b)->line.cursor = LINE(b)->line.linelen;
			LINE(b)->line.Strcat (VLINE->line.line);
			delete VLINE;
			rlList.dremove (vcursor);
			vcursor = b;
			if (--nc < nt) {
				--nt;
				t = b;
			}
			drawalllines ();
			--ntot;
		} else scursor ();
	} else
		if (VLINE->line.cursor < VLINE->line.linelen) {
			VLINE->do_char (cc);
			drawline (VLINE, nc - nt);
			scursor ();
		} else if ((b = rlList.Next (vcursor))) {
			VLINE->line.Strcat (LINE(b)->line.line);
			delete rlList.XFor (b);
			rlList.dremove (b);
			drawalllines ();
			--ntot;
		} else scursor ();
}

void editor::do_pg ()
{
	for (int i = 0; i < 20; i++)
		vmove (cc == K_PGUP);
	scursor ();
}

void editor::do_hoend ()
{
	VLINE->do_char(cc);
	drawline (VLINE, nc - nt);
	scursor ();
}

void editor::do_cchar ()
{
	VLINE->do_char (cc);
	drawline (VLINE, nc - nt);
	scursor ();
}

//**************************************************
// drawable->vscreen-> ancestors
//**************************************************

void editor::draw ()
{
	vscreen::draw ();
	drawalllines ();
	scursor ();
}

void editor::present ()
{ }

bool editor::onclick ()
{
	if (!checkclick ()) return false;

	// FIXME: make the cursor appear at the click X, Y
	JCursor::changerl (this, font, fg, bg, true);
	scursor ();

	return true;
}

//**************************************************
//:::::::::::: Readline inherits Ok
//**************************************************

editor::editor (int w, int h)
: breadline (), normtab (), vscreen (0, w, h, c_scr_bg, c_scr_fg)
{
	nc = nt = 0;
	ntot = 1;
	rlList.add (new intrl ("", tX));
	t = vcursor = rlList.Start;
	prompt = NULL;
}

editor::~editor ()
{ }

//************************* Export/Import the text ************************

int editor::textsize ()
{
	int s = 0;
	dlistNode *d;

	for (d = rlList.Start; d; d = rlList.Next (d))
		s += LINE(d)->line.linelen + 1;

	return s;
}

char *editor::textcpy (char *dest)
{
	int p;
	dlistNode *d;

	for (p = 0, d = rlList.Start; d; d = rlList.Next (d)) {
		dest [p] = 0;
		strcpy (dest + p, LINE(d)->line.line);
		p += LINE(d)->line.linelen;
		dest [p++] = '\n';
	}

	dest [p] = 0;

	return dest;
}

void editor::load_text (char *c)
{
	char *p;

	while (rlList.Start) {
		delete rlList.XFor (rlList.Start);
		rlList.dremove (rlList.Start);
	}
	nc = nt = 0;
	ntot = 1;

	if (*c) {
		while ((p = strchr (c, '\n'))) {
			*p = 0;
			rlList.add (new intrl (c, tX));
			*p = '\n';
			c = p + 1;
			++ntot;
		}

		if (*c) rlList.add (new intrl (c, tX));
	} else rlList.add (new intrl ("", tX));

	t = vcursor = rlList.Start;
}
