#!/usr/bin/env python
#
# Mapil - Mailer And Pgp Interface Layer
#
# Send questions, bug reports, etc to me, Ben Escoto <bescoto@stanford.edu>
#
# Don't charge anyone for this program, don't remove my name, if you
# copy it give me credit, etc.
#
# This is the Mapil python source code.  Mapil is a program that can
# be used with PGP to encrypt, sign, or decode email messages.  The
# code is organized into a few main sections: IO, mime, policies, pgp,
# and misc.  Each is preceded by a section header, so look for lots of
# '#' to move to another section.

import os, sys, string, getopt, rfc822, tempfile, regex, \
       rand, termios, TERMIOS, types, StringIO, base64


# First, the global dictionary and variable definitions:

Version = '0.2.1'

# All the default settings follow.  None means the user should be
# prompted to choose.  For Signkey, if 'D', use PGP's default signkey.

Settings = {'Action':'Prompt',
	    'Verbose':'5',
	    'DisablePolicies':'N',
	    'Policiesfilename':os.environ['HOME']+'/.mapilpolicies',
	    'UpdateRecord':'Prompt',
	    'Inputfilename':None,
	    'PGPDirectory':None,
	    'PGPVersion':'PGP-2.6',
	    'WriteBackup':'Y',
	    'Backupfilename':os.environ['HOME']+'/.mapilpolicies~',
	    'TempDir':None,
	    'Interactive':'N',
	    'Batchmode':'N',
	    'CheckQP':'Loose',
	    'AsymCipher':'RSA',
	    'BlockCipher':'IDEA',
	    'Hash':'MD5',
	    'CheckQP':'Loose',
	    'Signkey':'D',
	    'Protocol':'P/MIME',
	    'PassPhrase':None,
	    'SaveProfile':'Prompt',
	    'Confirm':'Y',
	    'Recipients':[],
	    'RecFix':None,
	    'RecKeys':[],
	    'X-MapilReport':'Y',
	    'TransferEncoding':'AsciiArmor'}

Choices = {'Action':['Decrypt','Encrypt','Sign','EncryptSign','None'],
	   'PGPVersion':['PGP-2.6', 'PGP-5.0'],
	   'BlockCipher':['IDEA', 'CAST', '3DES'],
	   'AsymCipher':['RSA', 'DH'],
	   'Hash':['MD5', 'SHA'],
	   'Protocol':['dump','P/MIME'],
	   'CheckQP':['Loose', 'Strict', 'None'],
	   'TransferEncoding':['AsciiArmor','Base64']}



######################################################################
########## Class UserIO - contains the user input/output code ########
########## Class TempFile - contains code relating to file IO ########
######################################################################

class UserIO:

    # Handles input and feedback to the user.  Also contains error
    # handling code... Perhaps this should have been done with
    # exceptions?

    def Fatal(self, errorstring, exitval):
	if 0 < Settings['Verbose']:
	    print errorstring
	TF.DelTemps()
	sys.exit(exitval)

	# Uncomment below for exception tracing from interpreter
	# raise 'Exiting '+str(exitval)


    def Warn(self, errorstring, vlevel):
	if vlevel <= Settings['Verbose']:
	    print errorstring


    def Say(self, message, vlevel):
	# Tell the user something if verbosity level is high enough

	if vlevel <= Settings['Verbose']:
	    print message


    def Menu(self, introtext, choicelist):
	# choice list is a list of strings, Menu returns integer

	l = len(choicelist)
	sys.stdout.write(introtext+'\n')

	for i in range(0, l):
	    sys.stdout.write('     '+str(i+1)+') '+choicelist[i]+'\n')

	while 1:
	    sys.stdout.write('\nEnter Choice (1-'+str(l)+') or Q to quit: ')
	    response = sys.stdin.readline()

	    if response[0] == 'Q':
		IO.Fatal('User aborted', 2)
	    try: n = string.atoi(response)
	    except ValueError:
		pass
	    else:
		if n <= l and n >= 1:
		    return n

	    sys.stdout.write('Invalid response\n')


    def GetAction(self):
	# Ask the user what to do

	i = self.Menu('Which action would you like to perform on ' + \
		      'this message?\n', Choices['Action'])
	Settings['Action'] = Choices['Action'][i-1]


    def GetAsymCipher(self):

	i = self.Menu('Which public key encryption algorithm should ' + \
		      'be used?\n', Choices['AsymCipher'])
	Settings['AsymCipher'] = Choices['AsymCipher'][i-1]


    def GetProtocol(self):
	i = self.Menu('Which protocol should this message follow?\n',
		      Choices['Protocol'])
	Settings['Protocol'] = Choices['Protocol'][i-1]


    def GetQPChecking(self):
	i = self.Menu('Select a degree of QP checking:\n',
		      Choices['CheckQP'])
	Settings['CheckQP'] = Choices['CheckQP'][i-1]


    def GetRecKeys(self):
	# Ask user for recipients to encrypt to.

	sys.stdout.write("""
Please enter the UserIDs or KeyIDs to use when encrypting your message.
KeyIDs must start with 0x.  For other quoting rules, please see
the documentation.\n\n""")

	sys.stdout.write('Recipient keys or Q to quit: ')
	r = sys.stdin.readline()

	if string.strip(r) == 'Q':
	    IO.Fatal('User aborted',2)
	rlist = ParseLine(r)

	if len(rlist) == 0:
	    sys.stdout.write('Empty recipient list, try again.\n')
	    self.GetRecKeys()
	else:
	    Settings['RecKeys'] = rlist


    def GetSigner(self):
	# Ask for the sign key to use

	sys.stdout.write("""
Enter the UserID or KeyID of the key you want to sign the message with.
Leaving this empty will choose the default PGP key.  KeyID's must start
with 0x.  Since there is only one field, there is no need for quoting.
\n""")

	sys.stdout.write('Signing ID or Q to quit: ')
	input = string.strip(sys.stdin.readline())

	if input:
	    if input == 'Q':
		IO.Fatal('User aborted', 2)
	    else: Settings['Signkey'] = input
	else:
	    Settings['Signkey'] = 'D'


    def GetPass(self, key = None):
	# Get the Pass Phrase.  This is copied from an example in the
	# python library documentation.

	if key:
	    sys.stdout.write('\nEnter the passphrase for key ' + key + \
			     'or Q to quit.\nOn most systems it will ' + \
			     'not be echoed.\n')
	else: sys.stdout.write('\nEnter passphrase or Q to quit.  It ' + \
			       'will not be echoed on most systems.\n')
	
	fd = sys.stdin.fileno()
	old = termios.tcgetattr(fd)
	new = termios.tcgetattr(fd)
	new[3] = new[3] & ~TERMIOS.ECHO          # lflags

	try:
	    termios.tcsetattr(fd, TERMIOS.TCSADRAIN, new)
	    passphrase = raw_input('Passphrase: ')
	finally:
	    termios.tcsetattr(fd, TERMIOS.TCSADRAIN, old)

	sys.stdout.write('\n')
	if len(passphrase) == 0:
	    Settings['PassPhrase'] = None
	else:
	    if passphrase == 'Q':
		IO.Fatal('User aborted', 2)
	    else: Settings['PassPhrase'] = passphrase


    def GetUpdate(self, record):
	# Ask the user whether or not to write the profile.

	sys.stdout.write('\nUpdate the policies file with the following line?\n')
	sys.stdout.write(record.ToString() + '\n')
	sys.stdout.write('(Y/N)? ')
	input = string.strip(sys.stdin.readline())

	if input == 'Y' or input == 'y':
	    Settings['UpdateRecord'] = 'Y'
	elif input == 'N' or input == 'n':
	    Settings['UpdateRecord'] = 'N'
	else:
	    sys.stdout.write('Invalid response.')
	    self.GetUpdate(record)


    def ReviewSettings(self):
	# Print some of settings in numbered list, then give menu and
	# allow user to change some of them.

	choicelist = ['Action: '+Settings['Action'],
		      'Message format: '+Settings['Protocol'],
		      'Recipient Keys: ' + QuoteList(Settings['RecKeys']),
		      'Signing Key: '+Settings['Signkey'],
		      'CheckQP: '+Settings['CheckQP']]

	if Settings['Confirm'] == 'N' or Settings['Batchmode'] == 'Y':
	    print '\nSetting summary:'
	    print string.join(choicelist, '\n')
	    print
	else:
	    choicelist = choicelist + ['Keep above settings']
	    choice = self.Menu('\nYou may change some of the settings here:',
		 choicelist)

	    if choice == 1:
		self.GetAction()
	    elif choice == 2:
		self.GetProtocol()
	    elif choice == 3:
		self.GetRecKeys()
	    elif choice == 4:
		self.GetSigner()
	    elif choice == 5:
		self.GetQPChecking()
	    elif choice == 6:
		return

	    self.ReviewSettings()


    def Report(self, protocol, outsum):
	# outsum is the dictionary returned by ParseOutput, protocol
	# is dump, or P/MIME, etc.
	# Return a list of report lines.

	if protocol == 'P/MIME':
	    report = ['Message was in P/MIME format.\n']
	elif protocol == 'dump':
	    report = ['Message was a PGP dump.\n']
	else: 
	    report = ['PGP encoding not found']
	    return

	if outsum['signature'] == -1:
	    report.append('PGP section just seemed to be a signature!\n')
	    return report
	elif outsum['signature'] == 1:
	    if outsum['signer'][0:2] == '0x':
		report.append('Signature by unknown key ' + \
			      outsum['signer'] + '.\n')
	    else:
		if outsum['goodsignature']:
		    str = 'Good '
		else: str = 'Bad '
		
		if outsum['certified']:
		    str = str + 'certified '
		else: str = str + 'uncertified '

		report.append(str + 'signature by ' + \
			      outsum['signer'] + '.\n')

	if outsum['encrypted'] == -1:
	    report.append('Encrypted to recipients: ' + \
		  string.join(outsum['recipients']) + '.\n')
	elif outsum['encrypted'] == 1:
	    report.append('Message was decrypted with key ' + \
		  outsum['recipients'][0] + '.\n')
	else: report.append('Message was not encrypted.\n')

	return report



class TempFile:

    # Class for keeping track of temporary files and methods for
    # moving or deleting files, etc.

    def __init__(self):
	# Set the list of tempfiles to []

	self.tflist = []


    def OpenFile(self, filename, mode = 'r'):
	
	try: f = open(filename, mode)
	except IOError:
	    IO.Fatal('Unable to open file ' + filename + \
		     ' in mode ' + mode + '.',6)
	return f


    def Make(self):
	# returns a temp filename.

	filename = tempfile.mktemp()
	self.tflist.append(filename)
	return filename


    def Delete(self, filename):
	if filename in self.tflist:
	    self.tflist.remove(filename)
	
	try: os.remove(filename)
	except os.error:
	    IO.Fatal('Unable to remove file '+filename+'.', 28)


    def Move(self, source, destination):
	# Use os.rename if possible, otherwise read and write
	# the file, and delete source.

	try:
	    os.rename(source, destination)
	    if source in self.tflist:
		self.tflist.remove(source)
	except os.error:
	    fdest = self.OpenFile(destination, 'w')
	    fsrc = self.OpenFile(source, 'r')
	    fdest.write(fsrc.read())
	    fdest.close()
	    fsrc.close()
	    self.Delete(source)


    def DelTemps(self):
	try: map(os.remove, self.tflist)
	except os.error:
	    pass


######################################################################
################# MimePart Class - Mime related code #################
######################################################################

class MimePart(rfc822.Message):

    # This class builts on the standard rfc822.Message type like the
    # mimetools suite does, but contains more functions.  Much of it
    # is copied directly from mimetools - maybe I will submit this to
    # become part of the standard package.

    # The semi-public variables:
    # type - by constructor as in multipart/mixed
    # maintype, subtype - again by constructor

    # WriteParts assigns the following:
    # partfilenames - list of filenames of each part, for multipart
    # mimeparts - a list of MimeParts corresponding to the subparts.
    # numparts - 0 if simple, >0 for multipart.

    def __init__(self, filename):
	# Open file for reading, and initialize the rfc.Message from
	# the resulting file pointer.  File is required to be
	# seekable.

	# Let '--------' count as a message separator

	self.filename = filename
	self.fp = TF.OpenFile(filename)

	rfc822.Message.__init__(self, self.fp, 1)

	self.encodingheader = self.getheader('content-transfer-encoding')
	self.typeheader = self.getheader('content-type')
	self.parsetype()
	self.parseplist()

	self.mimeparts = []
	self.partfilenames = []
	self.numparts = 0
	self.bodyfilename = None


    # Allow the MH separator '--------' between body and headers
    def islast(self, line):
	return line in ('\r\n', '\n', '--------\n')


    def UnwriteParts(self, recurse = 1):

	# Remove all the files that WriteParts() and may have written.
	# This seemed like a less bug-prone solution than writing an
	# equivalent destructor.

	for fn in self.partfilenames:
	    TF.Delete(fn)
	self.partfilenames = []

	if self.bodyfilename:
	    TF.Delete(self.bodyfilename)
	    self.bodyfilename = None

	if recurse:
	    for m in self.mimeparts:
		m.UnwriteParts()


    def WriteParts(self, recurse = 1):
	# This can be used to save all the parts of a multipart
	# message into a tree structure.  Does nothing for simple data
	# type.  Looks like fp must be at beginning of body.

	if self.maintype == 'multipart':

	    # Create the necessary boundaries for checking.  Is it
	    # necessary to support \r\n messages on a unix system?

	    boundary = self.getparam('boundary')
	    midboundary = '--'+boundary+'\n'
	    finalboundary = '--'+boundary+'--\n'

	    curline = self.fp.readline()
	    while curline and curline != midboundary:
		curline = self.fp.readline()
	    if not curline:
		IO.Warn('Invalid multipart message, continuing.',3)

	    while curline and curline != finalboundary:
		
		# Create tempfile, increment lists
		self.partfilenames.append(TF.Make())
		curfp = TF.OpenFile(self.partfilenames[self.numparts], 'w')
		self.numparts = self.numparts + 1

		# The following is somewhat awkward, but it tries to write
		# the exact part, without including the trailing linefeed,
		# and without removing a linefeed from every line read.

		prevline = self.fp.readline()
		curline = self.fp.readline()
		while curline and curline != midboundary and \
		      curline != finalboundary:
		    curfp.write(prevline)
		    prevline = curline
		    curline = self.fp.readline()
		curfp.write(prevline[:-1])
		curfp.close()
	    if not curline:
		IO.Warn('No closing boundary in message',3)

	    # recurse down the part tree if recurse set
	    if recurse:
		self.mimeparts = map(MimePart, self.partfilenames)
		for m in self.mimeparts:
		    m.WriteParts(1)


    def WriteBody(self):
	# Writes the body of the part into a file, set self.bodyfilename

	self.bodyfilename = TF.Make()
	bodyfp = TF.OpenFile(self.bodyfilename, 'w')
		
	if self.encodingheader and \
	   string.lower(self.encodingheader) == 'base64':
	    base64.decode(self.fp, bodyfp)
	else: bodyfp.write(self.fp.read())

	bodyfp.close()


    # The following four functions, parsetype, parseplist, getparam,
    # and getparamnames were taken verbatim from the standard library
    # mimetools.
    
    def parsetype(self):
	str = self.typeheader
	if str == None:
	    str = 'text/plain'
	if ';' in str:
	    i = string.index(str, ';')
	    self.plisttext = str[i:]
	    str = str[:i]
	else:
	    self.plisttext = ''
	fields = string.splitfields(str, '/')
	for i in range(len(fields)):
	    fields[i] = string.lower(string.strip(fields[i]))
	self.type = string.joinfields(fields, '/')
	self.maintype = fields[0]
	self.subtype = string.joinfields(fields[1:], '/')


    def parseplist(self):
	str = self.plisttext
	self.plist = []
	while str[:1] == ';':
	    str = str[1:]
	    if ';' in str:
		# XXX Should parse quotes!
		end = string.index(str, ';')
	    else:
		end = len(str)
	    f = str[:end]
	    if '=' in f:
		i = string.index(f, '=')
		f = string.lower(string.strip(f[:i])) + \
		    '=' + string.strip(f[i+1:])
	    self.plist.append(string.strip(f))
	    str = str[end:]


    def getparam(self, name):
	name = string.lower(name) + '='
	n = len(name)
	for p in self.plist:
	    if p[:n] == name:
		return rfc822.unquote(p[n:])
	return None


    def getparamnames(self):
	result = []
	for p in self.plist:
	    i = string.find(p, '=')
	    if i >= 0:
		result.append(string.lower(p[:i]))
	return result


    def GetAddresses(self):
	# Returns a list of the email address occuring in the To:,
	# Cc:, Bcc:, and Dcc: headers of a message, only in lower case.
	# My attempt at a obfuscated one-liner

	return map(string.lower, map(lambda t: t[1], \
		reduce(lambda x,y: x+y, map(self.getaddrlist, \
					    ['To','Cc','Bcc','Dcc']))))


    def GetPrimaryAddress(self):
	# Returns lower case version of the first email address in
	# the To: header

	try: str=string.lower(self.getaddrlist('To')[0][1])
	except (IndexError, TypeError):
	    return None
	else: return str


    def MakeBoundary(self):
	return '=_Mapil-'+Version+'_'+str(rand.rand())+str(rand.rand())


    def CheckQP(self, strict = 0):
	# Assumes unix line ending conventions.  Checks for line
	# length, invalid characters, and From.  if strict, also check
	# for tabs, and characters !"#$@[\]^`{|}~
	# Body must already be written
	# Returns 1 if passes, None if fails

	fp = TF.OpenFile(self.bodyfilename)
	if strict:
	    re =  regex.compile('^\\([%-?A-Za-z _]*[%-?A-Za-z_]\\|\\b\\)\n$')
	else:
	    re = regex.compile('^\\([!-~ \t]*[!-~]\\|\\b\\)\n$')

	line = fp.readline()
	while line:
	    if len(line) > 77:
		break
	    if re.search(line) < 0:
		break
	    if line[:5] == 'From ':
		break
	    line = fp.readline()

	else:
	    fp.close()
	    return 1
	fp.close()


    def QPEncode(self):
	# strictly QP encodes the body, replacing the body.
	# Body must be written already.

	self.ChangeHeader('Content-Transfer-Encoding','quoted-printable')

	fp = TF.OpenFile(self.bodyfilename)
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile, 'w')

	line = fp.readline()
	while line:
	    tlist = []
	    haslf = 0

	    # First encode the line, without worring about length
	    for i in range(len(line)):
		j = ord(line[i])
		if j == 10:
		    haslf = 1
		elif (j >= 37 and j <= 60) or j == 62 or j == 63 or \
		   (j >= 65 and j <= 90) or (j >= 97 and j <= 122) or \
		   j == 32 or j == 95:
		    tlist.append(line[i])
		else:
		    # If bad char, convert to hex and quote
		    h = string.upper(hex(j)[2:])
		    if len(h) == 2:
			tlist.append('='+h)
		    else: tlist.append('=0'+h)

	    tstart = 0
	    tlength = 0

	    if tlist:
		# Now check for From.  This seems inefficient, oh well
		if string.join(tlist[:5],'') == 'From ':
		    tlist[0] = '=46'

		# Fix trailing spaces
		if tlist[-1] == ' ':
		    tlist[-1] = '=20'

		# Now built line of correct length
		for i in range(len(tlist)):
		    tlength = tlength + len(tlist[i])
		    if tlength > 75:
			outfp.write(string.join(tlist[tstart:i], '')+'=\n')
			tstart = i
			tlength = len(tlist[i])

	    if haslf:
		outfp.write(string.join(tlist[tstart:], '')+'\n')
	    else: outfp.write(string.join(tlist[tstart:],''))

	    line = fp.readline()

	fp.close()
	outfp.close()
	TF.Delete(self.bodyfilename)
	self.bodyfilename = outfile


    def GetHeaderPos(self, name):
	# Return index of first header line in self.headers starting with
	# name:.  If not found, return -1.

	length = len(name)+1
	name = string.lower(name)

	for i in range(len(self.headers)):
	    if string.lower(self.headers[i][:length]) == name+':':
		return i

	return -1


    def ChangeHeader(self, name, newvalue):
	# Changes first existing header, or adds one if not already
	# present

	header = name+': '+newvalue+'\n'
	i = self.GetHeaderPos(name)

	if i == -1:
	    self.headers.append(header)
	else:
	    length = len(self.getfirstmatchingheader(name))
	    self.headers[i] = header

	    # Get rid of any continuation lines
	    del self.headers[i+1:i+length]


    def RemoveHeader(self, name):
	# Removes header matching name

	i = self.GetHeaderPos(name)
	if i != -1:
	    length = len(self.getfirstmatchingheader(name))
	    del self.headers[i:i+length]


    def CopyHeaders(self, M):
	# set the headers Content-Type and Content-Transfer-Encodings
	# to those of MimePart M.

	if M.typeheader:
	    self.ChangeHeader('Content-Type', M.typeheader)
	else: self.RemoveHeader('Content-Type')

	if M.encodingheader:
	    self.ChangeHeader('Content-Transfer-Encoding', M.encodingheader)
	else: self.RemoveHeader('Content-Transfer-Encoding')


    def BeforeBody(self, string):
	# Writes a new file that contains string + the body of the
	# current message.  Assumes that self.fp is at the beginning
	# of the body if not self.bodyfilename.  Returns filename.

	fn = TF.Make()
	fp = TF.OpenFile(fn, 'w')
	fp.write(string)

	if self.bodyfilename:
	    bodyfp = TF.OpenFile(self.bodyfilename)
	    fp.write(bodyfp.read())
	    bodyfp.close()
	else:
	    fp.write(self.fp.read())
	    self.fp.close()
	fp.close()

	return fn


    def Micalg(self, hash):
	# Given the hash setting in Settings['Hash'], returns the P/MIME
	# micalg designation

	if hash == 'MD5':
	    return 'pgp-md5'
	elif hash == 'SHA':
	    return 'pgp-sha1'


    def XMapilCheck(self):
	# Replaces all the X-Mapil headers in the message with
	# X-Mapil-Forged lines.

	while 1:
	    i = self.GetHeaderPos('X-Mapil')
	    if i == -1: break

	    self.headers[i] = 'X-Mapil-Forged' + \
			      self.headers[i][len('X-Mapil'):]


    def Report(self, protocol, outsum):
	# Get report from IO.report, add to headers if necessary.
	# If existing X-Mapil header, change to X-Mapil-Forged.
	# Then report to user.

	report = IO.Report(protocol, outsum)

	# Write headers if necessary
	if Settings['X-MapilReport'] == 'Y':
	    self.headers.append('X-Mapil: ' + report[0])

	    for i in range(1, len(report)):
		self.headers.append('         ' + report[i])

	# Tell user
	IO.Say('\n' + string.join(report, ''), 2)


    def GetProtocol(self):
	# Sets Settings['Protocol'] and Settings['Action'].  First
	# check Content-Type.  Failing that, look at first line.
	# Action is either 'Sign' or 'Encrypt'.

	if self.type == 'multipart/signed':
	    Settings['Protocol'] = 'P/MIME'
	    Settings['Action'] = 'Sign'
	elif self.type == 'multipart/encrypted':
	    Settings['Protocol'] = 'P/MIME'
	    Settings['Action'] = 'Encrypt'
	elif self.type != 'text/plain' and self.type != 'application/pgp':
	    Settings['Protocol'] = 'None'
	    Settings['Action'] = 'None'
	else:
	    line = string.strip(self.fp.readline())
	    self.rewindbody()

	    if line == '-----BEGIN PGP SIGNED MESSAGE-----':
		Settings['Protocol'] = 'dump'
		Settings['Action'] = 'Sign'
	    elif line == '-----BEGIN PGP MESSAGE-----':
		Settings['Protocol'] = 'dump'
		Settings['Action'] = 'Encrypt'
	    else:
		Settings['Protocol'] = 'None'
		Settings['Action'] = 'None'


    def PMIMEsign(self):
	# PGP/MIME clearsign the current message, changing the headers
	# and QP Encoding if necessary.  Returns the filename of the
	# new message.

	# Check to see if QP should be checked, write body if so.
	if Settings['CheckQP'] != 'None' and \
	   self.maintype != 'message' and \
	   self.maintype != 'multipart':
	    if not self.bodyfilename:
		self.WriteBody()
	    if not self.CheckQP(Settings['CheckQP'] == 'Strict'):
		self.QPEncode()
		
	oldheaders = self.getfirstmatchingheader('Content-Type') + \
		     self.getfirstmatchingheader('Content-Transfer-Encoding')

	# Write file to be signed, then make signature
	infile = self.BeforeBody(string.join(oldheaders,'') + '\n')
	sigfile = EI.Encode('MakeSig', infile)

	boundary = self.MakeBoundary()
	newtype = 'multipart/signed; boundary="' + boundary + '";\n' + \
		  '    micalg="' + self.Micalg(Settings['Hash']) + \
		  '"; protocol="application/pgp-signature"'
	self.ChangeHeader('Content-Type', newtype)
	self.ChangeHeader('Content-Transfer-Encoding','7bit')
	self.ChangeHeader('Mime-Version','1.0')

	# Write the output file
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile,'w')

	if self.unixfrom:
	    outfp.write(self.unixfrom)
	outfp.writelines(self.headers)
	outfp.write('\n--'+boundary+'\n')

	# Write data that was signed into outfile, then delete infile
	infp = TF.OpenFile(infile)
	outfp.write(infp.read())
	infp.close()
	TF.Delete(infile)

	outfp.write('\n--'+boundary+'\n')
	outfp.write('Content-Type: application/pgp-signature\n')
	sigfp = TF.OpenFile(sigfile)

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    outfp.write('\n')
	    # Write signature into output file, changing BEGIN/END PGP
	    # MESSAGE to BEGIN/END PGP SIGNATURE if necessary.  Delete
	    # signature file afterwards.

	    line = sigfp.readline()
	    while line:
		if line == '-----BEGIN PGP MESSAGE-----\n':
		    outfp.write('-----BEGIN PGP SIGNATURE-----\n')
		elif line == '-----END PGP MESSAGE-----\n':
		    outfp.write('-----END PGP SIGNATURE-----\n')
		else: outfp.write(line)
		line = sigfp.readline()

	else:
	    outfp.write('Content-Transfer-Encoding: base64\n\n')
	    base64.encode(sigfp, outfp)

	sigfp.close()
	TF.Delete(sigfile)

	# Write last boundary
	outfp.write('\n--'+boundary+'--\n')
	outfp.close()

	return outfile


    def PMIMEencrypt(self, sign = 1):
	# PGP/MIME encrypt or encryptsign the current message.
	# Changes headers like PMIMEsign, but doesn't QPencode or
	# check.  Returns filename of new message.  The seek point
	# on the message is required to be right before the body.

	oldheaders = self.getfirstmatchingheader('Content-Type') + \
		     self.getfirstmatchingheader('Content-Transfer-Encoding')

	boundary = self.MakeBoundary()
	newtype = 'multipart/encrypted; ' + \
		  'protocol="application/pgp-encrypted";\n' + \
		  '    boundary="' + boundary + '"'
	self.ChangeHeader('Content-Type', newtype)
	self.ChangeHeader('Content-Transfer-Encoding','7bit')
	self.ChangeHeader('Mime-Version','1.0')

	# Write file to be encrypted, encrypt, and then delete it
	infile = self.BeforeBody(string.join(oldheaders,'') + '\n')
	if sign:
	    encfile = EI.Encode('EncryptSign',infile)
	else: encfile = EI.Encode('Encrypt',infile)
	TF.Delete(infile)

	# Write output file
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile,'w')

	if self.unixfrom:
	    outfp.write(self.unixfrom)
	outfp.writelines(self.headers)
	outfp.write('\n--'+boundary+'\n')
	outfp.write('Content-Type: application/pgp-encrypted\n\n')
	outfp.write('Version: 1\n\n')
	outfp.write('--'+boundary+'\n')
	outfp.write('Content-Type: application/octet-stream\n')

	# Write encrypted message into output file, then delete
	encfp = TF.OpenFile(encfile)

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    outfp.write('\n')
	    outfp.write(encfp.read())
	else:
	    outfp.write('Content-Transfer-Encoding: base64\n\n')
	    base64.encode(encfp, outfp)

	encfp.close()
	TF.Delete(encfile)

	# Write final boundary
	outfp.write('\n--'+boundary+'--\n')
	outfp.close()

	return outfile


    def Dump(self, action):
	# encrypts, signs, or encrypt and signs the current message by
	# just replacing the current body with the PGP output.  Valid
	# actions are 'ClearSign', 'Encrypt', and 'EncryptSign'.

	self.WriteBody()

	encfile = EI.Encode(action, self.bodyfilename)
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile, 'w')
	encfp = TF.OpenFile(encfile)

	if self.unixfrom:
	    outfp.write(self.unixfrom)

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    outfp.writelines(self.headers)
	    outfp.write('\n')
	    outfp.write(encfp.read())
	else:
	    self.ChangeHeader('Content-Transfer-Encoding', 'base64')
	    outfp.writelines(self.headers)
	    outfp.write('\n')
	    base64.encode(encfp, outfp)

	encfp.close()
	outfp.close()
	TF.Delete(encfile)
	return outfile


    def PMIMEdecrypt(self):
	# Assumes message is in P/MIME encrypted or encrypt signed form.

	self.WriteParts()
	if self.numparts != 2:
	    IO.Fatal('Invalid or unrecognized message format.',29)

	pgppart = MimePart(self.partfilenames[1])
	pgppart.WriteBody()
	cleartext, outsum = EI.Decode('Decrypt', pgppart.bodyfilename)
	pgppart.UnwriteParts()

	# Read headers from previously encrypted file
	M = MimePart(cleartext)

	# Change headers of message to those of part.
	self.CopyHeaders(M)

	# Give report, add report headers
	self.Report('P/MIME', outsum)

	# Start to write outfile
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile, 'w')

	# Write headers, then body of cleartext file
	if self.unixfrom:
	    outfp.write(self.unixfrom)
	outfp.writelines(self.headers)
	outfp.write('\n')
	outfp.write(M.fp.read())

	M.fp.close()
	outfp.close()
	TF.Delete(cleartext)

	return outfile


    def PMIMEchecksig(self):
	# Check the signature of a message in P/MIME signed format

	self.WriteParts()
	if self.numparts != 2:
	    IO.Fatal('Invalid or unrecognized message format.',29)

	pgppart = MimePart(self.partfilenames[1])
	pgppart.WriteBody()
	None, outsum = EI.Decode('CheckSig', pgppart.bodyfilename, \
		  self.partfilenames[0])
	pgppart.UnwriteParts()

	# Change headers of message to that of subpart
	M = MimePart(self.partfilenames[0])
	self.CopyHeaders(M)

	# Report to user, write report headers
	self.Report('P/MIME', outsum)

	# Write headers, then body of cleartext file to outfile
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile, 'w')

	if self.unixfrom:
	    outfp.write(self.unixfrom)
	outfp.writelines(self.headers)
	outfp.write('\n')
	outfp.write(M.fp.read())
	
	M.fp.close()
	outfp.close()

	return outfile


    def DumpDecrypt(self):
	# Decrypts or checks the signature of a message, whose body is
	# a PGP output

	self.WriteBody()

	cleartext, outsum = EI.Decode('Decrypt', self.bodyfilename)

	# Write report headers, report to user
	self.Report('dump', outsum)

	cleartextfp = TF.OpenFile(cleartext)
	outfile = TF.Make()
	outfp = TF.OpenFile(outfile, 'w')

	if self.unixfrom:
	    outfp.write(self.unixfrom)
	outfp.writelines(self.headers)
	outfp.write('\n')
	outfp.write(cleartextfp.read())

	cleartextfp.close()
	outfp.close()
	TF.Delete(cleartext)

	return outfile



######################################################################
#### Classes Policies and Record - code relating to policies file ####
######################################################################

class Policies:

    # Interface to the policies file holding user and key information
    # See documentation for format

    # variables: mode can be 'r', 'r+', or 'a'
    # fp is the file object if open

    def __init__(self):

	self.mode = None
	self.fp = None
	self.CheckFile()


    def AddRecord(self, rec):
	# Add a line at the end

	self.Open('a')
	self.fp.write(rec.ToString()+'\n')
	self.Close()


    def GetRecord(self, address, regexp = 1, exact = 0):
	# Get the first record matching address.  If regexp, include
	# regexp lines.  If not regexp, through out those lines, and
	# look for substring match.  Return None if no match

	self.Open()
	
	line = self.fp.readline()
	linenum = 1

	while line:
	    t = self.TestLine(line, address, regexp, exact)
	    if t == -1:
		IO.Fatal('Invalid line '+str(linenum)+' in policies file.',13)
	    elif t:
		self.Close()
		return t

	    line = self.fp.readline()
	    linenum = linenum+1

	self.Close()
	return None


    def ChangeRecord(self, address, replacement = None):
	# Write each line to temp file, skipping over matched line.
	# Then move policies file to backup, tempfile to policies file.
	# Doesn't do anything if no match.  Returns 1 if address found,
	# 0 otherwise.

	self.Open()
	matched = 0

	tempfn = TF.Make()
	tempfp = TF.OpenFile(tempfn,'w')

	line = self.fp.readline()
	linenum = 1

	while line:
	    t = self.TestLine(line, address, 0, 1)
	    if t == -1:
		IO.Fatal('Invalid line '+str(linenum)+' in policies file.',13)

	    # Only write if it doesn't match
	    if t:
		matched = 1
		if replacement:
		    tempfp.write(replacement.ToString()+'\n')
		break
	    else: tempfp.write(line)

	    line = self.fp.readline()
	    linenum = linenum+1

	# Now just copy the rest
	tempfp.write(self.fp.read())

	tempfp.close()
	self.Close()

	# Now switch the files around.
	if Settings['WriteBackup'] == 'Y':
	    TF.Move(Settings['Policiesfilename'], \
		    Settings['Backupfilename'])

	TF.Move(tempfn,Settings['Policiesfilename'])

	return matched


    def Open(self, mode = 'r'):
	self.fp = TF.OpenFile(Settings['Policiesfilename'], mode)


    def CheckFile(self):
	# Create the policies file if it doesn't exist already

	try: os.stat(Settings['Policiesfilename'])
	except os.error:
	    f = TF.OpenFile(Settings['Policiesfilename'],'w')
	    f.close()
	    IO.Warn(Settings['Policiesfilename'] + ' not found, Creating..', 4)


    def TestLine(self, line, address, regexp, exact):
	# Is this too slow?
	# record if match, None if no match, -1 if invalid

	record = Record(line)
	if not record.address:
	    return -1

	if exact or record.regexp != 'Y':
	    if record.address == address:
		return record
	else:
	    if regexp:
		re = regex.compile(record.address)
		if re.search(address) >= 0:
		    return record
	

    def Close(self):
	if self.fp:
	    self.fp.close()


class Record:
    # Represents an individual line in the policies file

    # variables are valid address, regexp, reckeys,
    # action, signer, asym, block, hash

    def __init__(self, r, address = None):
	# If r is None, return Record based on Settings.  Otherwise
	# make Record from a list or string.  Also checks formatting.

	if r == None:
	    r = [address, 'N', QuoteList(Settings['RecKeys']),
		 self.GetActionLetter(Settings['Action']),
		 Settings['Signkey'], Settings['AsymCipher'],
		 'D', 'D', Settings['Protocol']]

	elif type(r) is types.StringType:
	    r = ParseLine(r)

	if len(r) < 9:
	    self.address = None
	    return

	# The following could be written (...) = tuple() but it is
	# easier to refer to this way.
	self.address = r[0]
	self.regexp = r[1]
	self.reckeys = ParseLine(r[2])
	self.action = r[3]
	self.signer = r[4]
	self.asym = r[5]
	self.block = r[6]
	self.hash = r[7]
	self.protocol = r[8]
	
	if self.regexp not in ['Y','N']:
	    self.address = None
	    return

	if self.action not in ['E', 'S', 'ES', 'C', 'D', 'N']:
	    self.address = None
	    return

	if self.asym not in ['RSA', 'C', 'D']:
	    self.address = None
	    return

	if self.block not in ['IDEA', 'C', 'D']:
	    self.address = None
	    return

	if self.hash not in ['MD5', 'C', 'D']:
	    self.address = None

	if self.protocol not in ['P/MIME','dump','C','D']:
	    self.address = None


    def ToString(self):
	# Returns a string representation of the record

	if not self.address:
	    return None

	return QuoteList([self.address, self.regexp,
			  QuoteList(self.reckeys), self.action,
			  self.signer, self.asym, self.block,
			  self.hash, self.protocol])


    def GetAction(self, letter):
	#  Returns the action, where letter is 'E', 'S', 'C', or 'ES'

	if letter == 'E':
	    return 'Encrypt'
	elif letter == 'S':
	    return 'Sign'
	elif letter == 'ES':
	    return 'EncryptSign'
	elif letter == 'N':
	    return 'None'
	elif letter == 'D' or letter == 'C':
	    return letter


    def GetActionLetter(self, action):
	# Returns the letter for a given action, the opposite of
	# Record.GetAction

	if action == 'Encrypt':
	    return 'E'
	elif action == 'EncryptSign':
	    return 'ES'
	elif action == 'Sign':
	    return 'S'
	elif action == 'None':
	    return 'N'


    def CompareRecord(self, record):
	# Returns 1 if records are equal, 0 if otherwise.  Doesn't
	# check address, but do check all others

	return [self.regexp, self.reckeys, self.action, self.signer,
		self.asym, self.block, self.hash, self.protocol] == \
		[record.regexp, record.reckeys, record.action,
		 record.signer, record.asym, record.block, record.hash,
		 record.protocol]


    def SetSettingsField(self, field, value, dict):
	# Set Settings[field] = newvalue unless C or D.  if
	# dict[field], clear Settings[field] if conflict.  If value is
	# not D, set dict[field].

	if value == 'C' or \
	   (value != 'D' and dict[field] and Settings[field] != value):
	    Settings[field] = None
	elif value != 'D':
	    Settings[field] = value
	    dict[field] = 1


    def SetSettings(self, address, dict):
	# changes the Settings based on the current record.  If a
	# specified recipient is 'D', then add address to RecKeys.  If
	# any field is a 'C', the Settings field is made None if any
	# other field is a 'D', Settings not changed.

	if not self.address: return
	fields = ['Protocol', 'Signkey', 'Action', 'AsymCipher']
	values = [self.protocol, self.signer, \
		  self.GetAction(self.action), self.asym]

	for i in range(len(fields)):
	    self.SetSettingsField(fields[i], values[i], dict)

	# Add the list of reckeys to RecKeys
	for i in range(len(self.reckeys)):
	    if self.reckeys[i] == 'C':
		AddRecKeys(None)
		break
	    elif self.reckeys[i] == 'D':
		Settings['RecKeys'].append(address)
	    else: Settings['RecKeys'].append(self.reckeys[i])



######################################################################
############ Classes ExecBase, PGP2Interface, ExecInter ##############
############   contains code interfacing to PGP         ##############
######################################################################

class ExecBase:

    # This is the base class for the specific interfaces

    # Important variables:
    # filename - filename of input file, usually necessary
    # outfile - Where to put PGP outputfile.  Some of these have
    #     to end in .asc or .pgp
    # output - Output of process, if in batch mode
    # outsum - Output summary dictionary set by ParseOutput

    def __init__(self, filename):

	self.filename = filename
	self.output = ''
	self.outsum = {}


    def Exec(self, commandline):
	# if not interactive, run commandline, with stderr output
	# discarded, and puts the stdout into self.output.
	# if inactive, just system()

	IO.Say('Running '+commandline,6)

	if Settings['Interactive'] == 'Y':
	    os.system(commandline + ' 2>&1')
	else:
	    f = os.popen(commandline + ' 2>&1','r')
	    self.output = f.read()
	    f.close()

	IO.Say(self.output,7)


    def UnknownOutput(self):

	msg = """
	The encryption program has generated unexpected
	output.  This could be caused by a bug in mapil or by using a
	newer and unsupported version of the encryption program."""
	IO.Fatal(msg, 9)


class PGP2Interface(ExecBase):

    # This contains basic functions to do PGP 2.x tasks

    def GetPGPname(self):

	if Settings['PGPDirectory']:
	    return Settings['PGPDirectory']+'/pgp'
	else: return 'pgp'


    def CheckOutputfile(self, filename = None):
	# Unfortunately, PGP seems to force its output files to be named
	# xxxx.asc

	if not filename:
	    filename = self.filename

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    self.outfile = filename+'.asc'
	else: self.outfile = filename+'.pgp'

	try: os.stat(self.outfile)
	except os.error: return

	IO.Fatal('Output file '+self.outfile+' already exists.  Sorry!',10)


    def Exec(self, options, args = None, checkoutput = 1, setpass = 1):

	if not args:
	    args = self.filename

	if checkoutput:
	    self.CheckOutputfile()

	commandline = self.GetPGPname() + ' ' + options

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    commandline = commandline + '+ARMOR = ON'
	else:
	    commandline = commandline + '+ARMOR = OFF'

	if Settings['Interactive'] == 'Y':
	    commandline = commandline + ' ' + args
	else:
	    if setpass and Settings['PassPhrase']:
		os.environ['PGPPASS'] = Settings['PassPhrase']
	    commandline = commandline + ' +batchmode +verbose=1 ' + args

	ExecBase.Exec(self, commandline)


    def MakeSig(self):
	# Make a detached signature
	
	if Settings['Signkey'] != 'D':
	    self.Exec('-stb', self.filename + ' -u ' + Settings['Signkey'])
	else: self.Exec('-stb')


    def ClearSign(self):
	# Clearsign file

	if Settings['Signkey'] != 'D':
	    self.Exec('-st', self.filename + ' -u ' + Settings['Signkey'])
	else: self.Exec('-st')


    def Encrypt(self):
	# Encrypt a file

	self.Exec('-te', self.filename + ' ' + \
		  QuoteList(Settings['RecKeys']), 1, 0)


    def EncryptSign(self):
	# Encrypt and sign a file

	l = self.filename + ' ' + QuoteList(Settings['RecKeys'])
	if Settings['Signkey'] != 'D':
	    l = l + ' -u ' + Settings['Signkey']

	self.Exec('-set', l)


    def Decrypt(self):
	# Decrypt a file.  This can also be used to check a file, if
	# it not clear whether or not the file is encrypted or just
	# signed.

	self.outfile = TF.Make()
	self.Exec('', self.filename + ' -o ' + self.outfile, 0, 1)


    def CheckSig(self, cleartext):
	# Check a detached signature against filename cleartext

	self.Exec('', self.filename + ' ' + cleartext, 0, 0)


    def GetKeyIDs(self, userid): 
	# Takes a user ID, runs PGP to find the key, and returns list
	# of tuples (x,y) where x is the KeyID and y is the assorted
	# other information.  The list can be empty.  Returns None
	# if error.

	# Quote the userid if necessary before passing to Exec
	quote = GetQuote(userid)
	userid = quote + userid + quote

	self.Exec('-kv', userid, 0, 0)

	if regex.search('0 matching keys found.', self.output) >= 0:
	    return []
	else:
	    keytext = regex.compile('^Type bits.*\n\\(\\(.\\|\n\\)*\\)\n[0-9]+ matching keys? found.')
	    if keytext.search(self.output) < 0:
		return

	    templist = string.split(keytext.group(1), '\n')
	    keyline = regex.compile('^\\(pub\\|sec\\)\W*[0-9]+/\\([0-9A-F]+\\)')
	    # Build outlist from templist
	    outlist = []
	    keyid, totalkey = None, None
	    for i in range(len(templist)):
		if keyline.search(templist[i]) >= 0:
		    if keyid:
			outlist.append((keyid, totalkey))
		    keyid = keyline.group(2)
		    totalkey = templist[i]
		else:
		    totalkey = totalkey+'\n'+templist[i]
	    outlist.append((keyid, totalkey))
	    return outlist


    def CheckError(self):
	# Scan the output for errors 

	# Possible errors: ('baduser',None), ('badpass', None),
	# ('badrecipient',xxx), ('other', None), None

	re = regex.compile('^\007Cannot find the public key matching' + \
			   ' userid \'\\(.*\\)\'$')

	if regex.search('^\007Error:  Bad pass phrase\\.',self.output) >= 0:
	    return 'badpass', None
	elif re.search(self.output) >= 0:
	    return ('badrecipient', re.group(1))
	elif regex.search('^\007Signature error',self.output) >= 0 and \
	     regex.search('^\007Key matching userid .* not found', 
			  self.output) >= 0:
	    return 'baduser', None
	else:
	    r1 = regex.compile('\007\W?Error: \\(.*\\)$')
	    if r1.search(self.output) >= 0:
		return 'other', r1.group(1)


    def ParseCodingOutput(self):
	# only set outsum['error'].  If file exists, must have worked.
	# Otherwise, set error.  This can be run in interactive mode.
	# Removes output if error.

	self.outsum['error'] = None

	if Settings['Interactive'] == 'N':
	    self.outsum['error'] = self.CheckError()

	try: os.stat(self.outfile)
	except:
	    if not self.outsum['error']:
		self.outsum['error'] = ('other', None)
	else:
	    if self.outsum['error']:
		TF.Delete(self.outfile)

	if not self.outsum['error']:
	    IO.Say('PGP seemed successful.',6)
	elif self.outsum['error'][0] == 'badrecipient':
	    IO.Warn('PGP Error: Cannot find key for user ' + \
		    self.outsum['error'][1] + '.',4)
	else:
	    IO.Warn('PGP Error: '+self.outsum['error'][0],4)


    def ParseDecodingOutput(self):
	# sets outsum, the output summary dictionary

	# Dictionary keys:  Should be checked in following order.
	# error - None if valid message, (error, info) if not
	# signature - 1 if signed, 0 if not, -1 if file is only a signature
	# goodsignature - 1 if signature was good, 0 if not, -1 if unknown
	# certified - 1 if signature was trusted, 0 if not
	# signer - userID of signer
	# encrypted - 1 if message was encrypted, 0 if not, -1 if no sec key
	# recipients - list of userIDs message was encrypted to

	if regex.search('File has signature\\.', self.output) >= 0:

	    self.outsum['signature'] = 1
	    r1 = regex.compile('^\\(Good\\|Bad\\) signature' + \
			       ' from user "\\(.*\\)"\\.$')
	    r2 = regex.compile('^?.Key matching expected Key ID ' + \
			       '\\([0-9A-F]+\\) not found')
	    r3 = regex.compile('^For a usage summary, type:')
	
	    if r1.search(self.output) >= 0:
		self.outsum['signer'] = r1.group(2)
		if r1.group(1) == 'Good':
		    self.outsum['goodsignature'] = 1
		else:
		    self.outsum['goodsignature'] = 0
		if regex.search('^WARNING:  Because this public key is' + \
				' not certified',self.output) >= 0:
		    self.outsum['certified'] = 0
		else: self.outsum['certified'] = 1
	    elif r2.search(self.output) >= 0:
		self.outsum['signer'] = '0x' + r2.group(1)
		self.outsum['goodsignature'] = -1
	    elif r3.search(self.output) >= 0:
		self.outsum['signature'] = -1
	    else: self.UnknownOutput()

	else: self.outsum['signature'] = 0

	if regex.search('^File is encrypted\\.', self.output) >= 0:
	    r2 = regex.compile('Key for user ID: \\(.*\\)$')
	    r1 = regex.compile('^This message can only be read by:\\W' +
			       '\\(\\(.+\\W\\)*\\)')

	    if r2.search(self.output) >= 0:
		self.outsum['encrypted'] = 1
		self.outsum['recipients'] = [r2.group(1)]
	    elif r1.search(self.output) >= 0:
		self.outsum['encrypted'] = -1
		self.outsum['recipients'] = r1.group(1)
	    else: self.UnknownOutput()

	else: self.outsum['encrypted'] = 0

	self.outsum['error'] = self.CheckError()
	if self.outsum['error']: 
	    if self.outsum['error'][0] == 'other':
		IO.Warn('PGP Error: '+self.outsum['error'][1], 4)

	    elif not self.outsum['error'][0] == 'badpass':
		# Do not warn about a bad pass phrase since this is
		# handled later.
		IO.Warn('PGP Error: ' + self.outsum['error'][0] + \
			' ' + str(self.outsum['error'][1]), 4)
	    return
	
	IO.Say('PGP Parse Output results:\n'+str(self.outsum),8)


    def GetRecipients(self):
	# Subroutine of ParseDecodingOutput, returns list of recipients

	recipients = []
	list = string.split(self.output,'\n')
	i = list.index('This message can only be read by:')+1

	while list[i]:
	    if list[i][:8] == '  keyID:':
		recipients.append('0x'+list[i][9:])
	    else: recipients.append(list[i][2:])
	    i = i+1

	return recipients



class PGP5Interface(ExecBase):

    # This contains basic functions for PGP 5.0 tasks

    def GetPGPname(self, alias):

	if Settings['PGPDirectory']:
	    return Settings['PGPDirectory']+'/'+alias
	else: return alias


    def Exec(self, alias, options, setpass = 1, addout = 1, setfile = 1):
	self.outfile = TF.Make()

	if addout: out = ' -o ' + self.outfile
	else: out = ''

	if setfile: file = ' ' + self.filename
	else: file = ''

	if Settings['TransferEncoding'] == 'AsciiArmor':
	    commandline = 'echo | ' + self.GetPGPname(alias) + \
			  ' --batchmode +ARMOR=ON ' + options + file + out
	else:
	    commandline = 'echo | ' + self.GetPGPname(alias) + \
			  ' --batchmode +ARMOR=OFF ' + options + file + out

	if setpass and Settings['Interactive'] != 'Y' and \
	   Settings['PassPhrase']:
	    os.environ['PGPPASS'] = Settings['PassPhrase']

	ExecBase.Exec(self, commandline)


    def MakeSig(self):
	# Make detached signature

	if Settings['Signkey'] == 'D':
	    self.Exec('pgps', '-bt')
	else: self.Exec('pgps', '-bt -u ' + Settings['Signkey'])

	# This is a horrible hack, but look at the file size of the
	# produced signature to set the Hash value.  It seems that MD5
	# hashs are all 355 bytes, while SHA1 hashes are all 237
	# bytes.  If the size of the output file is less than 300
	# bytes, set Settings['Hash'] to 'SHA'.

	try:
	    filestat = os.stat(self.outfile)
	    if filestat[6] <= 300:
		Settings['Hash'] = 'SHA'
	    else: Settings['Hash'] = 'MD5'
	except os.error:
	    pass


    def ClearSign(self):
	# Clearsign a file

	if Settings['Signkey'] == 'D':
	    self.Exec('pgps', '-t')
	else: self.Exec('pgps', '-t -u ' + Settings['Signkey'])


    def MakeRecipientArgs(self):
	# Returns Settings['RecKeys'] quoted, with -r before
	# each recipient.

	templist = []
	for i in range(len(Settings['RecKeys'])):
	    templist.append('-r')
	    templist.append(Settings['RecKeys'][i])
	return QuoteList(templist)


    def Encrypt(self):
	# Encrypt a file

	self.Exec('pgpe', '-t ' + self.MakeRecipientArgs(), 0)


    def EncryptSign(self):
	# Encrypt and sign a file

	if Settings['Signkey'] == 'D':
	    self.Exec('pgpe', '-st ' + self.MakeRecipientArgs())
	else:
	    self.Exec('pgpe', '-st ' + self.MakeRecipientArgs() + \
		      ' -u ' + Settings['Signkey'])

    def Decrypt(self):
	# Decrypt or check a file.

	self.Exec('pgpv', '', 1)


    def CheckSig(self, cleartext):
	# The filename of the signature must, plus or minus a
	# signature, be the cleartext filename
	
	self.Exec('pgpv', '-o '+cleartext, 0, 0)


    def GetKeyIDs(self, userid):
	# Takes a userID, runs PGP to find the key, and returns list
	# of tuples (x,y) where x is the KeyID and y is the assorted
	# other information.  The list can be empty.  Returns None
	# if error

	# Quote the userid if necessary before passing to Exec
	quote = GetQuote(userid)
	userid = quote + userid + quote

	self.Exec('pgpk', '-l ' + userid, 0, 0, 0)

	outIO = StringIO.StringIO(self.output)
	keyline = regex.compile('^\\(pub\\|sec\\)...?[0-9]+ \\(0x[0-9A-F]+\\)')
	matchkeys = regex.compile('^[0-9]+ matching')
	outlist = []

	newkey = 1
	keytext, keyid = None, None
	curline = outIO.readline()
	while curline:
	    if newkey and keyline.search(curline) >= 0:
		if keytext:
		    outlist.append((keyid, keytext))
		keyid = keyline.group(2)
		keytext = curline
		newkey = 0
	    elif matchkeys.search(curline) >= 0:
		break
	    elif curline != '\n' and keytext:
		keytext = keytext + curline
	    else:
		newkey = 1
	    curline = outIO.readline()

	if keytext:
	    outlist.append((keyid, keytext))

	return outlist


    def CheckError(self):
	# Scan output for errors

	# Possible errors: ('baduser', user), ('badpass', None),
	# ('badrecipient', recipient) All others should be caught when
	# looking for the output file.

	re1 = regex.compile('^Cannot find a private key for '+
			   'signing: \\(.*\\)$')
	re2 = regex.compile('^No encryption keys found for: \\(.*\\)$')

	if regex.search('^Enter pass phrase:', self.output) >= 0:
	    return 'badpass', None
	elif re1.search(self.output) >= 0:
	    return ('baduser', re1.group(1))
	elif re2.search(self.output) >= 0:
	    return ('badrecipient', re2.group(1))


    def ParseCodingOutput(self):
	# This is copied from PGP2Interface

	self.outsum['error'] = None

	if Settings['Interactive'] == 'N':
	    self.outsum['error'] = self.CheckError()

	try: os.stat(self.outfile)
	except:
	    if not self.outsum['error']:
		self.outsum['error'] = ('other', None)
	else:
	    if self.outsum['error']:
		TF.Delete(self.outfile)

	if not self.outsum['error']:
	    IO.Say('PGP seemed successful.',6)
	elif self.outsum['error'][0] == 'badrecipient':
	    IO.Warn('PGP Error: Cannot find key for user ' + \
		    self.outsum['error'][1] + '.',4)
	elif self.outsum['error'][0] == 'baduser':
	    IO.Warn('PGP Error: Bad key ' + self.outsum['error'][1] + \
		    ' specified for signing.', 4)
	else:
	    IO.Warn('PGP Error: '+self.outsum['error'][0],4)


    def ParseDecodingOutput(self):
	# set outsum, the summary dictionary.

	# Dictionary keys:  Should be checked in following order.
	# error - None if valid message, (error, info) if not
	# signature - 1 if signed, 0 if not, -1 if file is only a signature
	# goodsignature - 1 if signature was good, 0 if not, -1 if unknown
	# certified - 1 if signature was trusted, 0 if not
	# signer - userID of signer
	# encrypted - 1 if message was encrypted, 0 if not, -1 if no sec key
	# recipients - list of userIDs message was encrypted to

	# Check for a signature
	resig1 = regex.compile('^Signature by unknown keyid: \\(.*\\)$')
	resig2 = regex.compile('^Good signature made .* by key:\012' + \
			       '.*\012   \\(.*\\)$')
	resig3 = regex.compile('^BAD signature made .* by key:\012' + \
			       '.*\012   \\(.*\\)$')

	if resig1.search(self.output) >= 0:
	    self.outsum['signature'] = 1
	    self.outsum['signer'] = resig1.group(1)
	    self.outsum['goodsignature'] = -1
	elif resig2.search(self.output) >= 0:
	    self.outsum['signature'] = 1
	    self.outsum['signer'] = resig2.group(1)
	    self.outsum['goodsignature'] = 1
	elif resig3.search(self.output) >= 0:
	    self.outsum['signature'] = 1
	    self.outsum['signer'] = resig3.group(1)
	    self.outsum['goodsignature'] = 0
	else: self.outsum['signature'] = 0

	if self.outsum['signature']:
	    if regex.search('^WARNING: The signing key is not trusted', \
			    self.output) >= 0:
		self.outsum['certified'] = 0
	    else: self.outsum['certified'] = 1

	# Check for encryption
	if regex.search('^Message is encrypted.$', self.output) >= 0:

	    # PGP 5 diags are all over the place.  Not worth it to try
	    # to look for recipients!
	    self.outsum['recipients'] = ['Unknown']

	    if regex.search('^Cannot decrypt message.', self.output) >= 0:
		self.outsum['encrypted'] = -1
	    else: self.outsum['encrypted'] = 1

	else: self.outsum['encrypted'] = 0

	self.outsum['error'] = self.CheckError()
	if self.outsum['error']:
	    if self.outsum['error'][0] == 'other':
		IO.Warn('PGP Error: '+self.outsum['error'][1], 4)

	    elif not self.outsum['error'][0] == 'badpass':
		# Do not warn about a bad pass phrase since this is
		# handled later.
		IO.Warn('PGP Error: ' + self.outsum['error'][0] + \
			' ' + str(self.outsum['error'][1]), 4)
	    return

	IO.Say('PGP Parse Output results:\n'+str(self.outsum), 8)



class ExecInter:

    # This class depends on the lower level interface(s) above, but
    # may also communicate to the user.

    def ReturnInter(self, filename):
	# Create the interface object from Settings
	return Settings['ExecInterface'](filename)


    def BatchModeCheck(self, message, exitlevel):
	if Settings['Batchmode'] == 'Y':
	    IO.Fatal(message, exitlevel)


    def Decode(self, action, filename, cleartext = None):
	# Same as Encode, but for Decrypt or CheckSig.  Returns
	# filename of output and output summary in tuple

	Settings['Recipients'] = []
	P = self.ReturnInter(filename)

	if action == 'Decrypt':
	    P.Decrypt()
	elif action == 'CheckSig':
	    P.CheckSig(cleartext)

	P.ParseDecodingOutput()

	if P.outsum['error']:
	    if P.outsum['error'][0] == 'badpass':
		if P.outsum['recipients']:
		    IO.Warn('Invalid passphrase for ' +
			    P.outsum['recipients'][0],4)

		self.BatchModeCheck('Invalid passphrase.',18)
		IO.GetPass()
		del P
		return self.Decode(action, filename)
	    else: IO.Fatal('Unrecognized error from PGP',19)
	elif P.outsum['encrypted'] == -1:
	    IO.Fatal('Required secret key missing.  Message encrypted' +
		     ' to:\n' + P.outsum['recipients'],7)

	else:
	    if action == 'Decrypt':
		return P.outfile, P.outsum
	    else: return None, P.outsum


    def Encode(self, action, filename):
	# Performs action on filename, returns filename of output

	P = self.ReturnInter(filename)
	
	if action == 'MakeSig':
	    P.MakeSig()
	elif action == 'Encrypt':
	    P.Encrypt()
	elif action == 'EncryptSign':
	    P.EncryptSign()
	elif action == 'Sign':
	    P.ClearSign()

	P.ParseCodingOutput()

	if not P.outsum['error']:
	    return P.outfile
	else:
	    if P.outsum['error'][0] == 'badpass':
		self.BatchModeCheck('Invalid passphrase.',18)
		IO.GetPass()
	    elif P.outsum['error'][0] == 'badrecipient':
		self.BatchModeCheck('One or more bad recipients.',19)
		IO.GetRecKeys()
	    elif P.outsum['error'][0] == 'baduser':
		self.BatchModeCheck('Invalid signing key.',20)
		IO.GetSigner()
	    else: IO.Fatal('Unrecognized error from PGP',19)

	    # Error may be fixed, so try again.
	    del P
	    return self.Encode(action, filename)


    def CheckDecodeSettings(self):
	# Ask for passphrase if Settings['Action'] == 'Encrypt'

	if Settings['Action'] == 'Encrypt' \
	   and not Settings['PassPhrase'] and \
	   Settings['Interactive'] != 'Y':
	    self.BatchModeCheck('No passphrase specified.',25)
	    IO.GetPass()


    def CheckSettings(self):
	# Depending on the action, makes sure certain settings are
	# defined.

	if not Settings['Action']:
	    self.BatchModeCheck('No Action specified in batch mode.',21)
	    IO.GetAction()

	if not Settings['Protocol']:
	    self.BatchModeCheck('No protocol specified in batch mode.',24)
	    IO.GetProtocol()

	if Settings['Action'] == 'Encrypt' or \
	   Settings['Action'] == 'EncryptSign':

	    if Settings['Interactive'] != 'Y':
		if not Settings['RecKeys']:
		    self.BatchModeCheck('No recipient keys specified.',19)
		    IO.GetRecKeys()

	    if not Settings['AsymCipher']:
		self.BatchModeCheck('No public key algorithm specified.', 22)
		IO.GetAsymCipher()

	if Settings['Action'] == 'Sign' or \
	   Settings['Action'] == 'EncryptSign':

	    if not Settings['Signkey']:
		self.BatchModeCheck('No signing key specified.',23)
		IO.GetSigner()

	    if not Settings['PassPhrase'] and \
	       Settings['Interactive'] != 'Y':
		self.BatchModeCheck('No passphrase specified.',25)
		IO.GetPass()



######################################################################
####### Miscellaneous Code - Class Options and misc functions ########
#######   all global functions are here, including Main()     ########
######################################################################

class Options:

    # A class for storing command line option information

    def __init__(self, arglist):
	# Initialize the option list

	try: self.allopts = getopt.getopt(arglist,'DECSMNbifc:v:d:g:r:u:a:p:',
					  ['confirm'])
	except getopt.error:
	    IO.Fatal('Invalid arguments, see man page for details',1)

	self.args = self.allopts[1]
	self.opts = self.allopts[0]


    def GetArg(self, letter):
	# Returns the argument associated with a given letter.

	for i in self.opts:
	    if i[0]=='-'+letter:
		return i[1]


    def CheckOpt(self, letter):
	# Returns the letter if option was received, None if not.

	for i in self.opts:
	    if i[0]=='-'+letter:
		return letter


    def CheckLongOpt(self, option):
	# Returns argument if option was received, None if not.
	# Note: this may return '', which is different from None.

	for i in self.opts:
	    if i[0]=='--'+option:
		return i[1]


    def GetAction(self):
	# Returns the major mode of the program.  This may be
	# overridden by PGP-action lines, if any.

	if self.CheckOpt('E'):
	    if self.CheckOpt('S'):
		return 'EncryptSign'
	    else: return 'Encrypt'
	elif self.CheckOpt('S'):
	    return 'Sign'
	elif self.CheckOpt('C'):
	    return 'Prompt'
	elif self.CheckOpt('N'):
	    return 'None'
	else: return 'Decrypt'


    def GetFilename(self):

	# Returns the filename to process, error if too many

	if len(self.args) > 1:
	    IO.Fatal('Invalid Arguments, see man page for details.',1)
	elif len(self.args) == 1:
	    return self.args[0]



def ConfigSettings(filename):
    # Reads the configuration file and alters the Settings.  The
    # Configuration file is never read again.

    try: f = open(filename)
    except (IOError, TypeError):
	IO.Warn('Cannot open config file '+filename+', using defaults.',3)
	return

    linenum = 1
    curline = f.readline()
    while curline:
	curline=string.lstrip(curline)

	if curline and curline[0] != '#':
	    l = ParseLine(curline)
	    if len(l) <= 1:
		IO.Fatal('Parse error in config file line '+str(linenum),3)
	    if len(l) == 2:
		Settings[l[0]]=l[1]
	    else:
		Settings[l[0]]=l[1:]
	
	linenum=linenum+1
	curline = f.readline()

    f.close()


def SetSettings(arglist): 
    # Initializes the Settings dictionary with the configuration file
    # and the command line options.
    # Remember, if anything is a 'C', no value given to settings field.

    CL = Options(sys.argv[1:])

    # Set the configuration settings from the file, if available
    if CL.GetArg('c'):
	ConfigSettings(CL.GetArg('c'))
    elif os.environ.has_key('MAPILRC'):
	ConfigSettings(os.environ['MAPILRC'])
    else: ConfigSettings(os.environ['HOME']+'/.mapilrc')

    # Now start to set Settings from command line.

    if CL.GetArg('v'):
	Settings['Verbose'] = CL.GetArg('v')

    Settings['Action'] = CL.GetAction()

    if CL.GetArg('d'):
	Settings['Policiesfilename'] = CL.GetArg('d')

    if CL.GetArg('r'):
	Settings['Recipients'] = ParseLine(CL.GetArg('r'))
	Settings['RecFix'] = 'Y'

    if CL.GetArg('h'):
	Settings['Hash'] = CL.GetArg('h')

    if CL.CheckOpt('b'):
	Settings['Batchmode'] = 'Y'

    if CL.GetArg('u'):
	Settings['Signkey'] = CL.GetArg('u')

    if CL.CheckOpt('i'):
	Settings['Interactive'] = 'Y'

    if CL.GetArg('a'):
	Settings['AsymCipher'] = CL.GetArg('a')

    if CL.GetArg('p'):
	Settings['Protocol'] = CL.GetArg('p')

    if CL.GetArg('g'):
	Settings['PGPVersion'] = CL.GetArg('g')

    if CL.CheckOpt('f'):
	Settings['DisablePolicies'] = 'Y'

    if CL.CheckLongOpt('confirm') != None:
	Settings['Confirm'] = 'Y'

    # Parse command line arguments
    if len(CL.args) > 2:
	IO.Fatal('Too many command line arguments.',5)
    if len(CL.args) >= 1:
	Settings['Inputfilename'] = CL.args[0]
    else: IO.Fatal('At least one argument required.',30)

    if len(CL.args) == 2:
	Settings['Outputfilename'] = CL.args[1]
    else: Settings['Outputfilename'] = Settings['Inputfilename']

    # Set information from environment
    if os.environ.has_key('PGPPASS'):
	Settings['PassPhrase'] = os.environ['PGPPASS']

    # Now convert if option's internal representation is different
    # from the string required in the config file.

    # verbosity should be a number, not a letter
    if Settings['Verbose']:
	try: Settings['Verbose'] = string.atoi(Settings['Verbose'])
	except (ValueError, TypeError):
	    IO.Fatal('Invalid verbosity, must be a number 0-9',4)

    # If Recipients is set in the config file, may be string now,
    # not list as desired.
    if type(Settings['Recipients']) is types.StringType:
	Settings['Recipients'] = [Settings['Recipients']]

    # If an option is 'Prompt' make it none instead.
    for key in ['BlockCipher', 'AsymCipher', 'Hash', 'UpdateRecord',
		'Signkey', 'Protocol','Action']:
	if Settings[key] == 'Prompt':
	    Settings[key] = None

    # If a filename starts with ~, replace with $HOME
    for key in ['Policiesfilename', 'Backupfilename', 'TempDir',
		'PGPDirectory']:
	if Settings[key] and Settings[key][0] == '~':
	    Settings[key] = os.environ['HOME']+Settings[key][1:]

    # Now check all the options, since no checking done above.

    # Makes sure Settings[key] is in Choices[key] for some options.
    for key in ['PGPVersion', 'BlockCipher', 'AsymCipher',
		'Hash', 'Protocol', 'CheckQP', 'Action',
		'PGPVersion', 'TransferEncoding']:
	if Settings[key] and Settings[key] not in Choices[key]:
	    IO.Fatal('Invalid setting for '+key+'.\n',12)

    # Make sure some settings are strings and not lists
    for key in ['Policiesfilename', 'PGPDirectory', 'Backupfilename',
		'TempDir', 'Signkey']:
	if Settings[key] and type(Settings[key]) is not types.StringType:
	    IO.Fatal('Invalid setting for '+key+'.\n',12)

    # Now initialize some settings from Settings dict.

    if Settings['PGPVersion'] == 'PGP-2.6':
	Settings['ExecInterface'] = PGP2Interface
    else: Settings['ExecInterface'] = PGP5Interface

    if Settings['TempDir']:
	tempfile.tempdir = Settings['TempDir']


def ParseLine(line):
    # This is used in a few different places, like for recipients
    # given on the command line or the prompt, and for parsing the
    # policies and configuration file.  Returns a list of strings.

    outlist = []
    i = 0
    length = len(line)

    while i < length:
	field, i = GetField(line, i)
	if not field:
	    break
	outlist.append(field)

    return outlist


def GetField(str, i):
    # Get the quoted field used in the policies file starting at position
    # i of the string str.  Return a tuple of the field, and the next
    # position of the string after the field.  Bypasses initial
    # whitespace Returns None, 0 if parse error

    length = len(str)

    while str[i] in string.whitespace:
	i = i+1
	if i == length:
	    return None, 0
    start = i

    if str[i] in string.letters+string.digits+'~/':
	while not (i == length or \
		   str[i] in string.whitespace):
	    i = i+1
	return str[start:i], i
    else:
	# Field is quoted
	quotechar = str[i]
	i = i+1
	if i == length:
	    return None, 0

	while not str[i] == quotechar:
	    i = i+1
	    if i == length:
		return None, 0
	return str[start+1:i], i+1



def GetQuote(str):
    # Returns a quoting character not in string str, or '' if no
    # quote needed

    if str[0] in string.letters+string.digits+'/~':
	i = 0
	while i < len(string.whitespace):
	    if string.whitespace[i] in str:
		break
	    i = i+1
	else: return ''

    for quote in '\'"\\:|<>`!@#$%^&*()?=+-_[]{}':
	if quote not in str:
	    return quote


def QuoteList(list):
    # Given a list of strings, returns a string with all necessary
    # fields in the list quoted and separated by a space

    if not list:
	return ''

    quote = GetQuote(list[0])
    string = quote + list[0] + quote

    for i in range(1,len(list)):
	quote = GetQuote(list[i])
	string = string + ' ' + quote + list[i] + quote
    return string


def SetAddressInfo(D):
    # Look through the policies file for the addresses in
    # Settings['Recipients'].  Set the settings accordingly for each
    # one, and set Settings['RecKeys'] appropriately.

    addrec = []
    notfound = []

    # First built two lists: addrec will be a list of pairs (address,
    # record), notfound will be a list of the addresses in
    # ['Recipients'] not found in the policies file.

    for i in range(len(Settings['Recipients'])):
	rec = D.GetRecord(Settings['Recipients'][i])
	if rec:
	    addrec.append((Settings['Recipients'][i],rec))
	    IO.Say(Settings['Recipients'][i]+' found in policies file.',9)
	else:
	    notfound.append(Settings['Recipients'][i])

    # Add addresses in notfound to Settings['RecKeys'].
    if len(notfound):
	dict = GenMultDict(1)
	AddRecKeys(notfound)
    else:
	dict = GenMultDict(0)

    # Now set the settings by iterating over the list of records.  The
    # dictionary may be updated by each SetSettings.  The dictionary
    # shows everything as already having been set if there are any
    # addresses not found in the recipients file.
    for i in range(len(addrec)):
	addrec[i][1].SetSettings(addrec[i][0], dict)


def GenMultDict(init):
    # This is a subroutine of the above function.  To keep track of
    # which settings are defaults and which have been set a dictionary
    # will be used.  If the value of a key in the dictionary is 1, it
    # means the option has already been set.  If it is 0, it means it
    # is still free to be set.

    dict = {}
    for key in ['Protocol', 'Signkey', 'Action', 'AsymCipher']:
	dict[key] = init

    return dict


def AddRecipients(list):
    # Check Settings['RecFix'], if not, append list to ['Recipients'].
    # Make sure no recipient is empty.

    if Settings['RecFix'] != 'Y':
	length = len(Settings['Recipients'])
	Settings['Recipients'][length:length] = filter(lambda x: x, list)


def AddRecKeys(list):
    # similar to AddRecipients, but adds keys to ['RecKeys'].
    # if list is None, then blank out 'RecKeys'.

    if list == None:
	Settings['RecKeys'] = None
    else:
	length = len(Settings['RecKeys'])
	Settings['RecKeys'][length:length] = list


def MakeOutputfile(file):
    # Moves file to Settings['Outputfilename']

    try: os.stat(Settings['Outputfilename'])
    except os.error:
	pass
    else: IO.Warn('Deleting file ' + Settings['Outputfilename'] + \
		  ' to make room for output.',6)

    TF.Move(file, Settings['Outputfilename'])


def UpdatePolicies(D, M):
    # Write record to policies file depending on settings

    # To run any of this, the message can have only one recipient, and
    # UpdateRecord has to be either 'Prompt' (but no Batchmode for
    # this) or 'Y'
    addrs = M.GetAddresses()
    if len(addrs) == 1 and \
       (Settings['UpdateRecord'] == 'Y' or \
	(not Settings['UpdateRecord'] and Settings['Batchmode'] != 'Y')):

	address = addrs[0]
	R = Record(None, address)

	# If records are the same, no need to update
	R2 = D.GetRecord(R.address, 0, 1)
	if R2 and R.CompareRecord(R2):
	    IO.Say('Record already up to date, no change to policies file.',5)
	    Settings['UpdateRecord'] = 'N'

	# Ask user if 'UpdateRecord' is 'Prompt'
	if not Settings['UpdateRecord']:
	    IO.GetUpdate(R)

	if Settings['UpdateRecord'] == 'Y':
	    IO.Say('Updating Record',5)
	    if R2:
		D.ChangeRecord(address, R)
	    else: D.AddRecord(R)

    else: IO.Say('Policies file update disabled - no change to file.',7)


def Init(optionlist = sys.argv[1:]):
    # Initialize the Settings[] and return M as the message object.
    # For debugging, can just run Init in the interpreter.

    SetSettings(optionlist)

    IO.Say('Running mapil version '+Version+'.',5)
    IO.Say(str(Settings)+'\n',8)
    os.umask(077)

    return MimePart(Settings['Inputfilename'])


def MainDecrypt(M):
    # The function called when decoding a message.
    # Formerly in Main.  Returns the output file name

    # Make sure no X-Mapil headers already present
    M.XMapilCheck()

    # Set the protocol and the action
    M.GetProtocol()

    if Settings['Protocol'] != 'None':
	EI.CheckDecodeSettings()
    else:
	IO.Say('No PGP encoding found.',2)

    if Settings['Protocol'] == 'P/MIME':
	if Settings['Action'] == 'Sign':
	    return M.PMIMEchecksig()
	else: return M.PMIMEdecrypt()

    elif Settings['Protocol'] == 'dump':
	return M.DumpDecrypt()


def MainEncrypt(M):
    # Called when a message is to be encoded.  Formerly in Main().
    # Returns output file name.

    # Add recipients found in headers
    AddRecipients(M.GetAddresses())

    # Get Settings based on recipients.  If DisablePolicies, just copy
    # recipients to get keys.
    if Settings['DisablePolicies'] != 'Y':
	D = Policies()
	SetAddressInfo(D)
    else: Settings['RecKeys'] = Settings['Recipients']

    # If anything is missing, prompt user, update Settings
    EI.CheckSettings()

    # Display all current settings, prompt user for changes.
    if Settings['Verbose'] >= 5:
	IO.ReviewSettings()
    # Final settings for debugging purposes
    IO.Say('Final Settings: ' + str(Settings) + '\n',8)

    # Perform action based on Settings
    if Settings['Protocol'] == 'P/MIME':
	if Settings['Action'] == 'Sign':
	    outfile = M.PMIMEsign()
	elif Settings['Action'] == 'EncryptSign':
	    outfile = M.PMIMEencrypt()
	elif Settings['Action'] == 'Encrypt':
	    outfile = M.PMIMEencrypt(0)
    elif Settings['Protocol'] == 'dump':
	if Settings['Action'] != 'None':
	    outfile = M.Dump(Settings['Action'])

    # If not DisablePolicies, see if policies file needs to be updated.
    if Settings['DisablePolicies'] != 'Y':
	UpdatePolicies(D, M)
	D.Close()

    if Settings['Action'] != 'None':
	return outfile


def Main(optionlist = sys.argv[1:]):
    # This is the first function to execute.  Mapil exits when this
    # returns.

    M=Init(optionlist)

    if Settings['Action'] == 'Decrypt':
	outfile = MainDecrypt(M)
    else:
	outfile = MainEncrypt(M)

    if Settings['Action'] != 'None':
	MakeOutputfile(outfile)
    M.UnwriteParts()
    TF.DelTemps()


# Now start command section

# UserIO and ExecInter don't carry any data and just group useful
# functions together.  Initialize IO and EI so these functions will be
# available.
IO=UserIO()
EI=ExecInter()
TF=TempFile()

# Start program if script.
Main()
