/*
 * pgpkeyring.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "pgpkeyring.h"
#include "pgp2key.h"
#include "pgp2signature.h"

#include <kapp.h>


// ---------  static member  -----------

PgpKeyring* PgpKeyring::pgpkeyring = 0;

PgpKeyring* PgpKeyring::getPgpKeyring()
{
  if (PgpKeyring::pgpkeyring)
    return PgpKeyring::pgpkeyring;
  else
    {
      PgpKeyring::pgpkeyring = new PgpKeyring;
      return PgpKeyring::pgpkeyring;
    }

  return 0;
};

// -----------------------------------------------------------------
  
PgpKeyring::PgpKeyring() :
  pgp_config(PgpConfig::getPgpConfig()),
  pubring(pgp_config->getPubring())
{
  load();
};

PgpKeyring::~PgpKeyring()
{};

// ------------------------------------------------------------------

QList<PgpKey> 
PgpKeyring::findKey(QString uid, QString keyId, bool pub)
{
  return key_list;
};

PgpKey*       
PgpKeyring::findKeyId(const QString keyId) const
{
  return key_dict.find(keyId);
};

// updates a key. Use write() to make changes permanent
void          
PgpKeyring::updateKey(PgpKey *key)
{};

// removes a key from the internal key list. Use write() to make changes permanent
void           
PgpKeyring::removeKey(PgpKey *key)
{
  key_list.remove(key);
  key_dict.remove(key->keyId());
};

// updates all keys, uids and signatures with keyId
void         
PgpKeyring::updateKeys(const PgpKey &changedKey)
{
  PgpKey       *key;
  PgpUserId    *uid;
  PgpSignature *sig;
  int          i, ii;
  bool         sig_changed;

  // update all signatures wth key-id keyId
  for (key = key_list.first(); key != 0; key = key_list.next())
    {
      for (i=0; i<key->count(); i++)
	{
	  uid = key->getUserId(i);
	  sig_changed = FALSE;

	  for (ii=0; ii<uid->count(); ii++)
	    {
	      sig = uid->signature(ii);
	      
	      if (sig->keyId() == changedKey.keyId())
		{
		  sig->setTrust(changedKey.getTrust());
		  sig_changed = TRUE;
		}
	    }

	  if (sig_changed)
	    uid->updateTrust();
	}
    }
};

// reloads the key list
void          PgpKeyring::reload()
{
  key_dict.clear();
  key_list.clear();

  pubring = pgp_config->getPubring();

  load();
};

#include <iostream.h>

// writes keylist into keyring
void          
PgpKeyring::write()
{
  QDataStream    out_stream;
  QFile          file(pubring);
  PgpKey         *key;

  file.open(IO_WriteOnly);

  out_stream.setDevice(&file);

  for (key = key_list.first(); key != 0; key = key_list.next())
    key->store(out_stream);

  file.close();
};

// loads keys from keyring
void          
PgpKeyring::load()
{
  QString        myname = pgp_config->getDefaultKey();
  QFile          file(pubring);
  QDataStream    in_stream;
  UINT8          CTB;
  PgpKey         *key;
  PgpSignature   *revocation;
 
  file.open(IO_ReadOnly);

  in_stream.setDevice(&file);

  // load Keys
  in_stream >> CTB;

  while ((!in_stream.eof()) && (CTB != 0xff))
    {
      switch ((CTB >> 2) & 15)
	{
	case PGP2_KEY:
	  key = new Pgp2Key(in_stream, CTB);
	  if (key->isPrivate() && !key->isRevoked())
	    {
	      if (myname.isNull() || myname == "")
		{
		  // MYNAME not set. Make the first private key the default key
		  myname = key->getUserId()->userId();
		  pgp_config->setDefaultKey(myname);
		  key->setDefault(TRUE);
		} 
	      else if (key->getUserId()->userId().find(myname) != -1)
		{
		  key->setDefault(TRUE);
		}
	    }
	  key_dict.insert(key->keyId(), key);
	  key_list.append(key);
	  break;
	case PGP2_SIGNATURE:
	  debug("!!!!!!!!!!!!!!!!!!!revocation found");
	  revocation = new Pgp2Signature(in_stream, CTB);
	  revocation_list.append(revocation);
	  break;
	default:
	  debug("error, unknown or unexpected pgp2 packet");
	}
    }
};
