/* 
 * pgpconfigtab.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <iostream.h>
#include <stdlib.h>

#include <qdir.h>
#include <qlayout.h>
#include <qfileinf.h>
#include <qbttngrp.h>

#include <kmsgbox.h>
#include <kfiledialog.h>

#include "pgpconfigtab.h"
#include "pgpconfigtab.moc"


KPGPConfig::KPGPConfig (QWidget * parent, const char *name)
  : KConfigWidget (parent, name)
{
  QPushButton *btn;
 
  pgp_config = PgpConfig::getPgpConfig();

  QBoxLayout *base  = new QBoxLayout(this, QBoxLayout::Down, 5);
  QBoxLayout *line  = new QBoxLayout(QBoxLayout::LeftToRight);
  QBoxLayout *files = new QBoxLayout(QBoxLayout::Down);
  
  base->addLayout(line);
  
  line->addWidget(getCheckSettings());
  line->addLayout(files);
  
  files->addWidget(fileSettings(btn, "Public Key File", pubringEdit));
  connect(btn, SIGNAL(clicked()), this, SLOT(slotChoosePubring()));
  
  files->addWidget(fileSettings(btn, "Secret Key File", secringEdit));
  connect(btn, SIGNAL(clicked()), this, SLOT(slotChooseSecring()));
  
  files->addWidget(fileSettings(btn, "Seed File", seed_fileEdit));
  connect(btn, SIGNAL(clicked()), this, SLOT(slotChooseSeedFile()));
  
  base->addWidget(getTrustSettings());
  base->addWidget(getCommentSettings());

  loadSettings();
}

// appearance dialog
KPGPConfig::~KPGPConfig ()
{
}

void KPGPConfig::applySettings()
{
  pgp_config->setPubring(pubringEdit->text());
  pgp_config->setSecring(secringEdit->text());
  pgp_config->setRandseed(seed_fileEdit->text());
  pgp_config->setComment(commentEdit->text());
  pgp_config->setVerbose(VERBOSE);

  pgp_config->setFlag(PgpConfig::Interactive, checkINTERACTIVE->isChecked());
  pgp_config->setFlag(PgpConfig::Compress, checkCOMPRESS->isChecked());
  pgp_config->setFlag(PgpConfig::Keepbinary, checkKEEPBINARY->isChecked());
  pgp_config->setFlag(PgpConfig::Armor, checkARMOR->isChecked());
  pgp_config->setFlag(PgpConfig::Textmode, checkTEXTMODE->isChecked());
  pgp_config->setFlag(PgpConfig::Clearsig, checkCLEARSIG->isChecked() );
  pgp_config->setFlag(PgpConfig::Showpass, checkSHOWPASS->isChecked());
  pgp_config->setFlag(PgpConfig::EncryptToSelf, checkENCRYPTTOSELF->isChecked() );

  pgp_config->setMarginalsNeeded(atoi(marginalEdit->text()));
  pgp_config->setCompletesNeeded(atoi(completeEdit->text()));
  pgp_config->setCertDepth(atoi(depthEdit->text()));

  pgp_config->apply();
}

void KPGPConfig::loadSettings()
{
  QString str;

  pubringEdit->setText(pgp_config->getPubring());
  secringEdit->setText(pgp_config->getSecring());
  seed_fileEdit->setText(pgp_config->getRandseed());
  commentEdit->setText(pgp_config->getComment());

  VERBOSE  = pgp_config->getVerbose();

  checkINTERACTIVE->setChecked(pgp_config->getFlag(PgpConfig::Interactive));
  checkCOMPRESS->setChecked(pgp_config->getFlag(PgpConfig::Compress));
  checkKEEPBINARY->setChecked(pgp_config->getFlag(PgpConfig::Keepbinary));
  checkARMOR->setChecked(pgp_config->getFlag(PgpConfig::Armor));
  checkTEXTMODE->setChecked(pgp_config->getFlag(PgpConfig::Textmode));
  checkCLEARSIG->setChecked(pgp_config->getFlag(PgpConfig::Clearsig));
  checkSHOWPASS->setChecked(pgp_config->getFlag(PgpConfig::Showpass));
  checkENCRYPTTOSELF->setChecked(pgp_config->getFlag(PgpConfig::EncryptToSelf));

  marginalEdit->setText(str.sprintf("%i", pgp_config->getMarginalsNeeded()));
  completeEdit->setText(str.sprintf("%i", pgp_config->getCompletesNeeded()));
  depthEdit->setText(str.sprintf("%i", pgp_config->getCertDepth()));
}

/// Resets the default values
void  KPGPConfig::defaultSettings()
{ 
  // pgp_config->setDefault();
  debug("KPGPConfig::defaultValues()"); 
};
 
QWidget* KPGPConfig::fileSettings(QPushButton* &btn, QString name, QLineEdit * &ledit)
{
  QGroupBox   *group = new QGroupBox(name, this, "group");
  QBoxLayout  *line  = new QBoxLayout(group, QBoxLayout::LeftToRight, 10);
  
  ledit = new QLineEdit(group);
  ledit->adjustSize();
  ledit->setMinimumSize(100, ledit->height());
  ledit->setMaximumSize(1000, ledit->height());
  
  line->addWidget(ledit, 0, AlignBottom);
  line->addSpacing(10);
  
  btn = new QPushButton("...", group);
  btn->setFixedSize(btn->sizeHint().width(), 20);
  
  line->addWidget(btn, 0, AlignBottom);
  
  group->adjustSize();
  group->setMinimumSize(150, ledit->height()+30);
  
  return group;
};

QWidget* KPGPConfig::getCheckSettings()
{
  QGroupBox   *group = new QGroupBox("PGP Variables", this, "group_check");
  QBoxLayout  *row   = new QBoxLayout(group, QBoxLayout::Down, 10);
  
  row->addSpacing(10);
  
  checkTEXTMODE = new QCheckBox("TEXTMODE", group);
  checkTEXTMODE->adjustSize();
  row->addWidget(checkTEXTMODE);
  
  checkARMOR = new QCheckBox("ARMOR", group);
  checkARMOR->adjustSize();
  row->addWidget(checkARMOR);
  
  checkKEEPBINARY = new QCheckBox("KEEPBINARY", group);
  checkKEEPBINARY->adjustSize();
  row->addWidget(checkKEEPBINARY);
  
  checkCOMPRESS = new QCheckBox("COMPRESS", group);
  checkCOMPRESS->adjustSize();
  row->addWidget(checkCOMPRESS);
  
  checkSHOWPASS = new QCheckBox("SHOWPASS", group);
  checkSHOWPASS->adjustSize();
  row->addWidget(checkSHOWPASS);
  
  checkCLEARSIG = new QCheckBox("CLEARSIG", group);
  checkCLEARSIG->adjustSize();
  row->addWidget(checkCLEARSIG);
  
  checkENCRYPTTOSELF = new QCheckBox("ENCRYPTTOSELF", group);
  checkENCRYPTTOSELF->adjustSize();
  row->addWidget(checkENCRYPTTOSELF);
  
  checkINTERACTIVE = new QCheckBox("INTERACTIVE", group);
  checkINTERACTIVE->adjustSize();
  row->addWidget(checkINTERACTIVE);
  
  group->adjustSize();
  
  return group;
};

QWidget* KPGPConfig::getTrustSettings()
{
  QLabel    *label;
  
  QGroupBox  *group = new QGroupBox(this, "group_trust");
  QBoxLayout *box   = new QBoxLayout(group, QBoxLayout::Down, 7);
  QBoxLayout *line;
  QString     str;
     
  
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
     
     label = new QLabel( "Number of Completly Trusted Introducers Needed", group );
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     line->addWidget(label);
     
     completeEdit = new QLineEdit( group );
     line->addWidget(completeEdit);
     
     completeEdit->setFixedSize(30, completeEdit->sizeHint().height());
     line->addSpacing(10);
        
   
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
  
     label = new QLabel( "Number of Marginally Trusted Introducers Needed", group );
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     line->addWidget(label);
  
     marginalEdit = new QLineEdit( group );
     marginalEdit->setFixedSize(30, marginalEdit->sizeHint().height());
     line->addWidget(marginalEdit);
     line->addSpacing(10);
     
  line = new QBoxLayout(QBoxLayout::LeftToRight, 7);
  
     box->addLayout(line);
  
     label = new QLabel( "How Deep May Introducers be Nested", group );
     line->addWidget(label);
     label->adjustSize();
     label->setMinimumSize(label->width(), label->height());
     
     depthEdit = new QLineEdit( group );
     depthEdit->setFixedSize(30, marginalEdit->sizeHint().height());
     line->addWidget(depthEdit);
     line->addSpacing(10);
  
  group->adjustSize();
  group->setMinimumSize(group->width(), 105);
  
  return group;
};

QWidget* KPGPConfig::getCommentSettings()
{
  QGroupBox   *group = new QGroupBox("Default PGP Comment", this, "grp_comment");
  QBoxLayout  *line  = new QBoxLayout(group, QBoxLayout::LeftToRight, 10);
  
  commentEdit = new QLineEdit(group);
  commentEdit->adjustSize();
  commentEdit->setMinimumSize(100, commentEdit->height());
  commentEdit->setMaximumSize(1000, commentEdit->height());
  
  line->addWidget(commentEdit, 0, AlignBottom);
  
  group->adjustSize();
  group->setMinimumSize(120, commentEdit->height()+30);
  
  return group;
};



void KPGPConfig::resizeEvent(QResizeEvent *)
{
}

 
void KPGPConfig::slotChoosePubring()
{
  QString new_pubring = KFileDialog::getOpenFileName(PgpConfig::getPgpConfig()->pgpPath());
  
  if ((new_pubring != pubringEdit->text()) && !new_pubring.isNull())
    {
      pubringEdit->setText(new_pubring);
      pgp_config->setPubring(new_pubring);
      emit signalPubringChanged();
    }
}

void KPGPConfig::slotChooseSecring()
{
  QString new_secring = KFileDialog::getOpenFileName(PgpConfig::getPgpConfig()->pgpPath());
  
  if ((new_secring != secringEdit->text()) && !new_secring.isNull())
    {
      pubringEdit->setText(new_secring);
      pgp_config->setPubring(new_secring);
    }
}

void KPGPConfig::slotChooseSeedFile()
{
  QString new_seed = KFileDialog::getOpenFileName(PgpConfig::getPgpConfig()->pgpPath());
  
  if ((new_seed != seed_fileEdit->text()) && !new_seed.isNull())
    {
      seed_fileEdit->setText(new_seed);
      pgp_config->setPubring(new_seed);
    }
}
