/* pu_lib.c						*/
/* Routines to read and write pic structures to disk	*/
/* Ian King 1994 					*/

#include "pu_defs.h"
#include <stdio.h>
#include <ctype.h>

unsigned char checksum;
FILE *inputfile;

int getachar()
{
	int c;

	do
		c = fgetc(inputfile);
	while (c == '\r');		/* strip LF out of MSDOS files */

	return c;
}

unsigned char getbyte()
{
	unsigned char byte;
	unsigned int  data;

	fscanf(inputfile,"%02x",&data);

	byte = data & 0xff;
	checksum += byte;	/* all the bytes are used in the checksum */
				/* so here is the best place to update it */
	return byte;
}

int readihex8m(char *ihx8mfname, PICDEFN *pic, int *tophx)
/*
 * record format for inhx8m = colon - #words - address - type - data - checksum
 */
{
	unsigned int address;
	unsigned int maxAddress = 0;
	int linetype = 0;
	int wordsthisline;
	int i;
	int lineCount = 1;
	unsigned short *memory;
	int csby;
	unsigned char hi, lo;

	memory = pic->picmemmap;

	if ((inputfile = fopen(ihx8mfname,"r")) == NULL)
	{
		printf("Could not open input file %s\n",ihx8mfname);
		return 1;
	}
      
	while (1)
	{
		if (getachar() != ':' )
		{
			printf("Need a colon as first character in each line\n");
			printf("Colon missing in line %d\n", lineCount);
			return 1;
		}

		checksum = 0;

		wordsthisline = getbyte()/2;
		hi = getbyte();
		lo = getbyte();
		address = (hi<<8 | lo) / 2;

		/* wierdness of INHX8M! address different */
		/* endian-ness and 2x too big */

		linetype = getbyte();	/* 0 for data, 1 for end  */

		if(linetype == 1)	/* lets get out of here hit the end */
			break;

		for(i=0; i<wordsthisline; i++)
		{
			if (address >= MAXPICSIZE)
			{
				printf("Address out of range\n");
				printf("Got 0x%04x maximum is 0x%04x at line %d\n",address,MAXPICSIZE, lineCount);
				return 1;
			}

			lo = getbyte();
			hi = getbyte();
			memory[address] = (unsigned short)((hi<<8) | lo);
			++address;
			if (address > maxAddress)
				maxAddress = address;
		}

		csby = getbyte();	/* get the checksum byte */
					/* this should make the checksum zero */
					/* due to side effect of getbyte */

		if (checksum)
		{
			printf("Checksum error in input file.\n");
			printf("Got 0x%02x want 0x%02x at line %d\n",csby, (0-checksum) & 0xff, lineCount);
			return 1;
		}

		(void)getachar();		/* lose <return> */

		lineCount++;
	}

	fclose(inputfile);

	*tophx = maxAddress;
	
	return 0;
}
void PU_Clear(PICDEFN *pic)
{
	int i;

	for(i=0; i<MAXPICSIZE; i++)
		pic->picmemmap[i] = 0xffff;

	pic->pictype	= 84;
	pic->picid[0]	= 0xffff;
	pic->picid[1]	= 0xffff;
	pic->picid[2]	= 0xffff;
	pic->picid[3]	= 0xffff;
	pic->osctype	= 3;
	pic->clock	= 4.0e6;
	pic->cp_fuse	= 1;
	pic->wd_fuse	= 1;
	pic->pu_fuse	= 1;
}

int PU_Read(const char *filename, PICDEFN *pic, int *top)
{
	PU_Clear(pic);

	if (readihex8m(filename, pic, top) == 0)
		return PU_OK;
	else
		return PU_FAIL;
}

int PU_WriteHeader(char *filename, PICDEFN *pic, char *comment)
{
	FILE *filehandle;

	filehandle = fopen(filename, "w");
	if (filehandle == NULL)
	{
		printf("Could not open file %s for writing\n",filename);
		return PU_FAIL;
	}

	fprintf(filehandle,"# File %s\n", filename);
	fprintf(filehandle,"# %s\n", comment);
	fprintf(filehandle,"T%d\n", pic->pictype);
	fprintf(filehandle,"W%d\n", pic->wd_fuse);
	fprintf(filehandle,"P%d\n", pic->cp_fuse);
	fprintf(filehandle,"C%d %e\n", pic->osctype, pic->clock);
	fprintf(filehandle,"U%d\n", pic->pu_fuse);
	fprintf(filehandle,"I%04x %04x %04x %04x\n",pic->picid[0],pic->picid[1],
						 pic->picid[02],pic->picid[3]);

	fclose(filehandle);
	return PU_OK;
}

int PU_WriteBlock(char *filename, PICDEFN *pic, int from, int to)
{
	FILE *filehandle;
	int length;
	int i;
	int numberleft;
	int current;
	int	mask;

	length = (to-from) + 1;
	if (length<=0)
	{
		printf("Error stupid block size! from %04x to %04x (size %d)\n",
				from, to, length);
		return PU_FAIL;
	}

	if ((from>MAXPICSIZE) || (to>MAXPICSIZE))
	{
		printf("Error from or to Out of Range from=%04x to=%04x\n",from, to);
		return PU_FAIL;
	}

	if ((from < 0) || (to < 0))
	{
		printf("Error Negative address specified!\n");
		return PU_FAIL;
	}

	filehandle = fopen(filename, "a");
	if (filehandle == NULL)
	{
		printf("Could not open file %s for writing\n",filename);
		return PU_FAIL;
	}

	fprintf(filehandle,"A%04x\n", from);

	current = from;

	switch (pic->pictype) {
	  case 54:
		mask = 0x0fff;
		break;
	  case 84:
		mask = 0x3fff;
		break;
	  default:
		mask = ~0;
		break;
	}

	while(current < to)
	{
		numberleft = to - current;
		if (numberleft > 8)
			numberleft = 8;

		fprintf(filehandle,"D%d", numberleft);

		for(i=0;i<numberleft;i++)
			fprintf(filehandle," %04x", pic->picmemmap[current+i] & mask);

		fprintf(filehandle,"\n");

		current += numberleft;
	}

	fclose(filehandle);
	return PU_OK;
}

int PU_WriteTrailer(char *filename, PICDEFN *pic, char *comment)
{
	FILE *filehandle;
	int i;
	unsigned short checksum;

	filehandle = fopen(filename, "a");
	if (filehandle == NULL)
	{
		printf("Could not open file %s for writing\n",filename);
		return PU_FAIL;
	}

	checksum = 0;

	for(i=0; i<MAXPICSIZE; i++)
		checksum ^= pic->picmemmap[i];	/* simple checksum  */
						/* easily fooled!   */

	fprintf(filehandle, "S%04x\n", checksum);
	fprintf(filehandle, "# %s\n", comment);
	fprintf(filehandle, "# ... The End ...\n");

	fclose(filehandle);
	return PU_OK;
}

/* ... The End ... */
