/* ----------------------------------------------------------------------------
 * Copyright (C) 2000-2005 by Karim Kaschani
 * 
 * This file is part of the software xIncode developed to support the design
 * of interconnects in semiconductor technologies.
 * 
 * xIncode is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xIncode is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ------------------------------------------------------------------------- */

#include "xincode.h"
#include "misc.h"





/* ----------------------------------------------------------------- globals */

static char	*file_name;
static char	*file_mode;
static char	*item_name;
static char	*prompt_string;





/* ----------------------------------------------------------------------------
 * isEmpty - check if text widget is empty
 * ------------------------------------------------------------------------- */

int isEmpty(Widget w)
{
	char	*tmp, *end;
	int	status;

	tmp = XmTextGetString(w);

	if (StrLen(tmp) == 0) {
	   status = TRUE;
	} else {
	   status = FALSE;
	}

	XtFree(tmp);

	return status;
}





/* ----------------------------------------------------------------------------
 * getDouble - get double value from text widget
 * ------------------------------------------------------------------------- */

double getDouble(Widget w, double limit)
{
	char	*tmp, *end;
	double	value;

	tmp = XmTextGetString(w);
	value = str2ENG(tmp, &end);

	if (tmp == end || value < limit) {
	   value = limit;
	}

	XtFree(tmp);

	return value;
}





/* ----------------------------------------------------------------------------
 * closeCB - close message form callback
 * ------------------------------------------------------------------------- */

void CloseCB(Widget w, int client_data, caddr_t call_data)
{
    XtDestroyWidget(XtParent(w));
}





/* ----------------------------------------------------------------------------
 * ErrorDialogCB - callback for error dialog
 * ------------------------------------------------------------------------- */

void ErrorDialogCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status = (int *) client_data;
	XmCommandCallbackStruct *cb = (XmCommandCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) {
	   *status = DIALOG_OK;
	} else
	   *status = DIALOG_CANCEL;
}





/* ----------------------------------------------------------------------------
 * PromptDialogCB - callback for prompt dialog
 * ------------------------------------------------------------------------- */

void PromptDialogCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status             = (int *) client_data;
	XmCommandCallbackStruct *cb = (XmCommandCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) {
	   XmStringGetLtoR(cb->value, XmFONTLIST_DEFAULT_TAG, &prompt_string);
	   *status = DIALOG_OK;
	   if (StrLen(prompt_string) == 0) {
	      XtFree(prompt_string);
	      prompt_string = NULL;
	   }
	} else {
	   prompt_string = NULL;
	   *status = DIALOG_CANCEL;
	}
}





/* ----------------------------------------------------------------------------
 * QuestionDialogCB - callback for question dialog
 * ------------------------------------------------------------------------- */

void QuestionDialogCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status             = (int *) client_data;
	XmCommandCallbackStruct *cb = (XmCommandCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) {
	   *status = DIALOG_OK;
	} else {
	   *status = DIALOG_CANCEL;
	}
}





/* ----------------------------------------------------------------------------
 * ChoiceDialogCB - callback for choice dialog
 * ------------------------------------------------------------------------- */

void ChoiceDialogCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status             = (int *) client_data;
	XmCommandCallbackStruct *cb = (XmCommandCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) 
	   *status = DIALOG_OK;
	else if (cb->reason == XmCR_CANCEL)
	   *status = DIALOG_CANCEL;
	else
	   *status = DIALOG_HELP;
}





/* ----------------------------------------------------------------------------
 * CreateMessage - create a message form
 * ------------------------------------------------------------------------- */

void CreateMessage(Widget parent, char *message, char *title,
                   int rows, int columns, int line)
{
   Arg			args[20];
   register int		n;
   Widget		close_button;
   Widget		message_box;
   Widget		message_text;
   Widget		separator_line;
   XmString		title_string;
   XmString		button_string;
   Position		X, Y, x;
   static Position	y;
   Dimension		W, H, w, h;
   extern Widget	app_shell;
 
  /* ---------------------------------------------------- create message_box */

   button_string  = XmStringCreateLtoR ("Close", XmSTRING_DEFAULT_CHARSET);
   title_string   = XmStringCreateLtoR (title, XmSTRING_DEFAULT_CHARSET);
 
   n = 0;
   XtSetArg (args[n], XmNdialogTitle, title_string);  n++;
   XtSetArg (args[n], XmNdefaultPosition, False);  n++;
   message_box = XmCreateFormDialog(XtParent(parent), "MessageDialog", args, n);

   /* ------------------------------------------- create 'Close' push button */

   n = 0;
   XtSetArg(args[n], XmNhighlightThickness, 0);  n++;
   XtSetArg(args[n], XmNlabelString, button_string); n++;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   close_button = XmCreatePushButtonGadget(message_box, "Close", args, n);
   XtAddCallback(close_button, XmNactivateCallback, (XtCallbackProc)CloseCB,
                 NULL);
   XtManageChild(close_button);

   n = 0;
   XtSetArg(args[n], XmNdefaultButton, close_button);  n++;
   XtSetValues(message_box, args, n);

   /* ------------------------------------------------ create separator line */

   n = 0;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);  n++;
   XtSetArg(args[n], XmNbottomWidget, close_button);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   separator_line = XmCreateSeparator(message_box, "separator", args, n);

   XtManageChild(separator_line);
   
   /* ------------------------------------------------- create scrolled text */

   n = 0;
   XtSetArg(args[n], XmNrows, rows);  n++;
   XtSetArg(args[n], XmNcolumns, columns);  n++;
   XtSetArg(args[n], XmNtraversalOn, False); n++;

   if (columns >= line) {
      XtSetArg(args[n], XmNscrollHorizontal, False);  n++;
      XtSetArg(args[n], XmNwordWrap, True);  n++;
   } else {
      XtSetArg(args[n], XmNscrollHorizontal, True);  n++;
      XtSetArg(args[n], XmNwordWrap, False);  n++;
   }

   XtSetArg(args[n], XmNscrollVertical, True);  n++;
   XtSetArg(args[n], XmNeditable, False);  n++;
   XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);  n++;
   XtSetArg(args[n], XmNvalue, message);  n++;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);  n++;
   XtSetArg(args[n], XmNbottomWidget, separator_line);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   message_text = XmCreateScrolledText(message_box, "MessageText", args, n);

   XtManageChild(message_text);

   XmStringFree (title_string);
   XmStringFree (button_string);

   /* ---------------------------------------------------- place message box */

   XtVaSetValues(XtParent(message_box), XmNmappedWhenManaged, False, NULL);
   XtManageChild(message_box);

   XtVaGetValues(app_shell,
                 XmNx, &X,
                 XmNy, &Y,
                 XmNwidth, &W,
                 XmNheight, &H,
                 NULL);

   XtVaGetValues(message_box,
                 XmNwidth, &w,
                 XmNheight, &h,
                 NULL);

   x = X+W/2-w/2;                   /* center the message box in x direction */
   y += H/20;                       /* continuously increment the y position */

   /* message box off screen ? */

   if (x < 0)
      x = W/5;
   if (x+w > WidthOfScreen(XtScreen(message_box)))
      x = WidthOfScreen(XtScreen(message_box)) - w - W/5;
   
   /* message box off toplevel shell ? */
 
   if ((y < Y + H/10) || (y + h > Y+H-H/10))
      y = Y + H/10;
   
   XtVaSetValues(message_box,
                 XmNx, x,
                 XmNy, y,
                 NULL);

   XtMapWidget(XtParent(message_box));
   XtVaSetValues(XtParent(message_box), XmNmappedWhenManaged, True, NULL);
}





/* ----------------------------------------------------------------------------
 * CreateErrorDialog - create error dialog
 * ------------------------------------------------------------------------- */

void CreateErrorDialog(Widget parent, char *title, char *message,
                       char *button)
{
	Widget			dialog;
	XmString		message_string, title_string, button_string;
	Arg			al[10];
	register int		ac;
	int			status;
	extern XtAppContext	app_context;

	fprintf(stderr, "\007"); /* ring bell */

	message_string = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
	title_string = XmStringCreateLocalized(title);
	button_string = XmStringCreateLocalized(button);

	ac = 0;
	XtSetArg(al[ac], XmNdialogTitle, title_string); ac++;
	XtSetArg(al[ac], XmNmessageString, message_string); ac++;
	XtSetArg(al[ac], XmNokLabelString, button_string); ac++;
	XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
	dialog = XmCreateErrorDialog(parent, "ErrorDialog", al, ac);

	XtAddCallback(dialog, XmNokCallback, ErrorDialogCB, &status);

	XmStringFree(message_string);
	XmStringFree(title_string);
	XmStringFree(button_string);

	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

	XtManageChild(dialog);

	status = DIALOG_WAIT;

	while (status == DIALOG_WAIT)
	      XtAppProcessEvent(app_context, XtIMAll);

	XtDestroyWidget(dialog);
}





/* ----------------------------------------------------------------------------
 * CreatePromptDialog - create prompt dialog
 * ------------------------------------------------------------------------- */

char *CreatePromptDialog(Widget parent, char *title, char *message)
{
	Widget			dialog, button;
	XmString		message_string, title_string;
	Arg			al[10];
	register int		ac;
	int			status;
	extern XtAppContext	app_context;

	message_string = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
	title_string = XmStringCreateLocalized(title);

	ac = 0;
       	XtSetArg(al[ac], XmNdialogTitle, title_string);  ac++;
	XtSetArg(al[ac], XmNselectionLabelString, message_string); ac++;
	XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
	dialog = XmCreatePromptDialog(parent, "PromptDialog", al, ac);

	XmStringFree(message_string);
	XmStringFree(title_string);

	XtAddCallback(dialog, XmNokCallback, PromptDialogCB, &status);
	XtAddCallback(dialog, XmNcancelCallback, PromptDialogCB, &status);

	button = XmSelectionBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	XtManageChild(dialog);

	status = DIALOG_WAIT;

	while (status == DIALOG_WAIT)
	      XtAppProcessEvent(app_context, XtIMAll);

	XtDestroyWidget(dialog);

	return prompt_string;
}





/* ----------------------------------------------------------------------------
 * CreateQuestionDialog - create query dialog
 * ------------------------------------------------------------------------- */

int CreateQuestionDialog(Widget parent, char *title, char *message)
{
	Widget			dialog, button;
	XmString		message_string, title_string;
	XmString		ok_string, cancel_string;
	Arg			al[10];
	register int		ac;
	int			status;
	extern XtAppContext	app_context;

	message_string = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
	title_string = XmStringCreateLocalized(title);
	ok_string = XmStringCreateLocalized("Yes");
	cancel_string = XmStringCreateLocalized("No");

	ac = 0;
       	XtSetArg(al[ac], XmNdialogTitle, title_string);  ac++;
	XtSetArg(al[ac], XmNmessageString, message_string); ac++;
	XtSetArg(al[ac], XmNokLabelString, ok_string); ac++;
	XtSetArg(al[ac], XmNcancelLabelString, cancel_string); ac++;
	XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
	dialog = XmCreateQuestionDialog(parent, "QuestionDialog", al, ac);

	XmStringFree(message_string);
	XmStringFree(title_string);
	XmStringFree(ok_string);
	XmStringFree(cancel_string);

	XtAddCallback(dialog, XmNokCallback, QuestionDialogCB, &status);
	XtAddCallback(dialog, XmNcancelCallback, QuestionDialogCB, &status);

	button = XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);

	XtManageChild(dialog);

	status = DIALOG_WAIT;

	while (status == DIALOG_WAIT)
	      XtAppProcessEvent(app_context, XtIMAll);

	XtDestroyWidget(dialog);

	if (status == DIALOG_OK)
	   return True;
	else
	   return False;
}





/* ----------------------------------------------------------------------------
 * CreateChoiceDialog - create choice dialog
 * ------------------------------------------------------------------------- */

int CreateChoiceDialog(Widget parent, char *title, char *message,
                       char *string1, char *string2, char *string3)
{
	Widget			dialog;
	XmString		message_string, title_string;
	XmString		ok_string, cancel_string, help_string;
	Arg			al[10];
	register int		ac;
	int			status;
	extern XtAppContext	app_context;

	message_string = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
	title_string = XmStringCreateLocalized(title);
	ok_string = XmStringCreateLocalized(string1);
	cancel_string = XmStringCreateLocalized(string2);
	help_string = XmStringCreateLocalized(string3);

	ac = 0;
       	XtSetArg(al[ac], XmNdialogTitle, title_string);  ac++;
	XtSetArg(al[ac], XmNmessageString, message_string); ac++;
	XtSetArg(al[ac], XmNokLabelString, ok_string); ac++;
	XtSetArg(al[ac], XmNcancelLabelString, cancel_string); ac++;
	XtSetArg(al[ac], XmNhelpLabelString, help_string); ac++;
	XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
	dialog = XmCreateQuestionDialog(parent, "QuestionDialog", al, ac);

	XmStringFree(message_string);
	XmStringFree(title_string);
	XmStringFree(ok_string);
	XmStringFree(cancel_string);
	XmStringFree(help_string);

	XtAddCallback(dialog, XmNokCallback, ChoiceDialogCB, &status);
	XtAddCallback(dialog, XmNcancelCallback, ChoiceDialogCB, &status);
	XtAddCallback(dialog, XmNhelpCallback, ChoiceDialogCB, &status);

	XtManageChild(dialog);

	status = DIALOG_WAIT;

	while (status == DIALOG_WAIT)
	      XtAppProcessEvent(app_context, XtIMAll);

	XtDestroyWidget(dialog);

	if (status == DIALOG_OK)
	   return 1;
	else if (status == DIALOG_CANCEL)
	   return 2;
	else
	   return 3;
}





/* ----------------------------------------------------------------------------
 * FileSelectionCB - callback for file selection dialog
 * ------------------------------------------------------------------------- */

void FileSelectionCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status             = (int *) client_data;
	XmFileSelectionBoxCallbackStruct *cb =
	                        (XmFileSelectionBoxCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) {

	   /* ----------------------------------------------------------- OK */

	   if (file_name != NULL) {
	      XtFree(file_name);
	      file_name = NULL;
	   }

	   XmStringGetLtoR(cb->value, XmFONTLIST_DEFAULT_TAG, &file_name);

	   if (!IsFile(file_name)) {
	      CreateErrorDialog(w, "File Select Error",
	                        "No file selected.", "Dismiss");

	      XtFree(file_name);
	      file_name = NULL;
	   } else {
	      if (FileExists(file_name)) {
	         switch (CreateChoiceDialog(w, "File Access Dialog",
	                     "The file you have chosen already exists.",
	                     "Overwrite", "Append", "Cancel")) {
	                case 2 : StrAdd(&file_mode, "a");
	                         *status = DIALOG_OK;
	                         break;
	                case 3 : free(file_mode);
	                         file_mode = NULL;
	                         XtFree(file_name);
	                         file_name = NULL;
	                         break;
	                default: StrAdd(&file_mode, "w");
	                         *status = DIALOG_OK;
	                         break;
	         }
	      } else {
	         StrAdd(&file_mode, "w");
	         *status = DIALOG_OK;
	      }
	   }

	} else {

	   /* ------------------------------------------------------- Cancel */

	   file_name = NULL;
	   *status = DIALOG_CANCEL;
	}
}





/* ----------------------------------------------------------------------------
 * OpenFile - create file selection dialog and open file
 * ------------------------------------------------------------------------- */

int OpenFile(FILE **fp, Widget parent)
{
	Widget			dialog;
	XmString		title_string, pattern_string;
	Arg			al[10];
	register int		ac;
	int			status;
	extern XtAppContext	app_context;
	char			*tmp;

	/* --------------------------------------- create file selection box */

	title_string = XmStringCreateLocalized("File Selection");
	pattern_string = XmStringCreateLocalized("*.sql");

	ac = 0;
       	XtSetArg(al[ac], XmNdialogTitle, title_string);  ac++;
	XtSetArg(al[ac], XmNpattern, pattern_string); ac++;
	XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
	dialog = XmCreateFileSelectionDialog(parent, "FileSelectionDialog",
	         al, ac);

	XmStringFree(title_string);
	XmStringFree(pattern_string);

	XtAddCallback(dialog, XmNokCallback, FileSelectionCB, &status);
	XtAddCallback(dialog, XmNcancelCallback, FileSelectionCB, &status);

	XtUnmanageChild(XmFileSelectionBoxGetChild(dialog,
	                XmDIALOG_HELP_BUTTON));

	if (file_name != NULL) {
	   pattern_string = XmStringCreateLocalized(file_name);
	   XmFileSelectionDoSearch(dialog, pattern_string);
	   XmStringFree(pattern_string);
	}

	XtManageChild(dialog);

	/* --------------------------------------------- wait for completion */

	status = DIALOG_WAIT;

	while (status == DIALOG_WAIT)
	      XtAppProcessEvent(app_context, XtIMAll);

	XtDestroyWidget(dialog);

	if (status == DIALOG_CANCEL) {
	   return -1;
	}

	*fp = fopen(file_name, file_mode);

	free(file_mode);
	file_mode = tmp = NULL;

	return 0;
}





/* ----------------------------------------------------------------------------
 * SelectionCB - callback for selection dialog
 * ------------------------------------------------------------------------- */

void SelectionCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int *status             = (int *) client_data;
	XmSelectionBoxCallbackStruct *cb =
	                        (XmSelectionBoxCallbackStruct *) call_data;

	if (item_name != NULL) {
	   XtFree(item_name);
	   item_name = NULL;
	}

	if (cb->reason == XmCR_OK) {

	   /* ----------------------------------------------------------- OK */

	   XmStringGetLtoR(cb->value, XmFONTLIST_DEFAULT_TAG, &item_name);
	   *status = DIALOG_OK;

	} else {

	   /* ------------------------------------------------------- Cancel */

	   item_name = NULL;
	   *status = DIALOG_CANCEL;
	}
}





/* ----------------------------------------------------------------------------
 * CreateSelectionDialog- create selection dialog
 * ------------------------------------------------------------------------- */

int CreateSelectionDialog(Widget parent, char *title, line *list, int items)
{
    static Widget	dialog;
    XmString		title_string, *list_item;
    Arg			al[10];
    register int	ac;
    int			i, status;
    extern XtAppContext	app_context;

    if (dialog == NULL) {

       /* --------------------------------------------- create selection box */

       title_string = XmStringCreateLocalized(title);

       list_item = (XmString *) XtMalloc(items * sizeof(XmString));

       for (i=0; i<items; i++) {
           list_item[i] = XmStringCreateLocalized(list[i]);
       }

       ac = 0;
       XtSetArg(al[ac], XmNdialogTitle, title_string);  ac++;
       XtSetArg(al[ac], XmNlistItems, list_item);  ac++;
       XtSetArg(al[ac], XmNlistItemCount, items);  ac++;
       XtSetArg(al[ac], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); ac++;
       dialog = XmCreateSelectionDialog(parent, "SelectionDialog", al, ac);

       XtAddCallback(dialog, XmNokCallback, SelectionCB, &status);
       XtAddCallback(dialog, XmNcancelCallback, SelectionCB, &status);

       XtUnmanageChild(XmSelectionBoxGetChild(dialog,
                       XmDIALOG_HELP_BUTTON));
       XtUnmanageChild(XmSelectionBoxGetChild(dialog,
                       XmDIALOG_APPLY_BUTTON));
       XtUnmanageChild(XmSelectionBoxGetChild(dialog,
                       XmDIALOG_TEXT));
       XtUnmanageChild(XmSelectionBoxGetChild(dialog,
                       XmDIALOG_SELECTION_LABEL));

       XmStringFree(title_string);

       while (--i >= 0) {
             XmStringFree(list_item[i]);
       }

       XtFree((char *)list_item);

       XtManageChild(dialog);

       /* ---------------------------------------------- wait for completion */

       status = DIALOG_WAIT;

       while (status == DIALOG_WAIT)
             XtAppProcessEvent(app_context, XtIMAll);

       XtDestroyWidget(dialog);
       dialog = NULL;

       if (status == DIALOG_CANCEL) {
          return -1;
       } else {
          i = 0;
          while (i<items && StrCmp(item_name, list[i]) != 0) {
                i++;
          }
       }

       return i;
    } else {
       return -1;
    }
}
