/* ----------------------------------------------------------------------------
 * Copyright (C) 2000-2005 by Karim Kaschani
 * 
 * This file is part of the software xIncode developed to support the design
 * of interconnects in semiconductor technologies.
 * 
 * xIncode is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xIncode is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ------------------------------------------------------------------------- */

#include "xincode.h"
#include "calc.h"
#include "error.h"
#include "help.h"
#include "misc.h"
#include "xutils.h"
#include "y.tab.h"

#if defined (HAVE_XPM)
#   include <xpm.h>
#   include "xincode.xpm"
#else
#   include "xincode.xbm"
#endif





/* -------------------------------------------------------- global variables */

Widget		technologyW;			/* technology field widget */
Widget		failureW;			/* failure field widget */
Widget		lifetimeW;			/* lifetime field widget */
Widget		temperatureW;			/* temperature field widget */
Widget		currentW;			/* current field widget */
Widget		layerwidthW;			/* layerwidth field widget */
Widget		tempincW;			/* temp. incr. field widget */
Widget		viaxsizeW;			/* x-size of via */
Widget		viaysizeW;			/* y-size of via */
Widget		viacurrW;			/* maximum via current */
Widget		layerW;				/* layer selection widget */
Widget		paramW;				/* parameter field widget */
Widget		resultW;			/* result field widget */
Widget		cbuttonW;			/* calculation button Widget */

Pixel		failureW_hlc;			/* failureW highlight color */
Pixel		failureW_fgc;			/* failureW foreground color */
Pixel		lifetimeW_hlc;			/* lifetimeW highlight color */
Pixel		lifetimeW_fgc;			/* lifetimeW foreground */
Pixel		temperatureW_hlc;		/* temperatureW highlight */
Pixel		temperatureW_fgc;		/* temperatureW foreground */
Pixel		currentW_hlc;			/* currentW highlight color */
Pixel		currentW_fgc;			/* currentW foreground color */
Pixel		layerwidthW_hlc;		/* layerwidthW highlight */
Pixel		layerwidthW_fgc;		/* layerwidthW foreground */
Pixel		tempincW_hlc;			/* tempincW highlight color */
Pixel		tempincW_fgc;			/* tempincW foreground color */
Pixel		viaxsizeW_hlc;			/* viaxsizeW highlight color */
Pixel		viaxsizeW_fgc;			/* viaxsizeW foreground */
Pixel		viaysizeW_hlc;			/* viaysizeW highlight color */
Pixel		viaysizeW_fgc;			/* viaysizeW foreground */

static char	*failureL     = "FIT";
static char	*lifetimeL    = "Lifetime [h]";
static char	*temperatureL = "Temp. [C]";
static char	*currentL     = "Current [A]";
static char	*layerwidthL  = "Width [m]";
static char	*tempincL     = "Temp. Inc.";
static char	*viaxsizeL    = "X/Y-Size [m]";
static char	*viaysizeL    = "Y-Size [m]";
static char	*viacurrL     = "Current";
static char	*linesL       = "Conduction Lines";
static char	*viasL        = "Bottommost Contacts/Vias";

double		LLfailure = 0;			/* lower limit of failure */
double		LLlifetime = 0;			/* lower limit of lifetime */
double		LLtemperature = -273;		/* lower limit of temperature */
double		LLcurrent = 0;			/* lower limit of current */
double		LLlayerwidth = 0;		/* lower limit of layerwidth */
double		LLviasize = 0;			/* lower limit of via size */

DBData		*db;				/* database data */
static char	*TF_ENV = "XINCODETECHFILE"; 	/* techfile environment */
static char	*RC_ENV = "XINCODERCPATH"; 	/* resource file environment */
static char	*RC_FILE = "/.xIncoderc"; 	/* resource file template */
static char	rc_file[MAX_CHARS];		/* resource file */
static char	*DB_FILE = "/xIncode.db";	/* database file template */
static char	db_file[MAX_CHARS];		/* database file */
static char	*tech_file;			/* technology file */
static int	listNo;				/* selected techfile number */
static char	*tbd = "<tbd>";			/* to be defined string */
static int	LastField;			/* last field having focus
						 * before calculation */
static int	LastLayerCalculation;		/* kind of last layer
						 * calculation */
static int	LastViaCalculation;		/* kind of last via
						 * calculation */

extern int	lineNo;				/* parsed line number */
extern char	yytext[];			/* parser messages */
extern int	yyparse();			/* parser */
extern FILE	*yyin, *yyout;			/* parsers file descriptors */





/* ------------------------------------------------------ fallback resources */

String fallbackResources[] = {
	"xIncode*fontList: 7x14",
	"xIncode*foreground: black",
	"xIncode*background: grey80",
	"xIncode*highlightColor: red",
	"xIncode*Annotations*background: grey70",
	"xIncode*Parameters*background: LightBlue3",
	"xIncode*Results*background: LightPink2",
	"xIncode*Results*fontList: 7x14bold",
	"xIncode*scrolledList*background: DarkSeaGreen3",
	"xIncode*SelectionDialog*background: grey80",
	"xIncode*SelectionDialog*ItemsList*background: grey70",
	"xIncode*MenuBar*MessageDialog*background: grey80",
	"xIncode*MenuBar*MessageDialog*MessageText*background: grey70",
	"xIncode*MenuBar*background: grey70",
	"xIncode*MenuBar*Load.accelerator: Ctrl<Key>t",
	"xIncode*MenuBar*Load.acceleratorText: Ctrl+T",
	"xIncode*MenuBar*Read.accelerator: Ctrl<Key>r",
	"xIncode*MenuBar*Read.acceleratorText: Ctrl+R",
	"xIncode*MenuBar*Save.accelerator: Ctrl<Key>s",
	"xIncode*MenuBar*Save.acceleratorText: Ctrl+S",
	"xIncode*MenuBar*Minimize.accelerator: Ctrl<Key>m",
	"xIncode*MenuBar*Minimize.acceleratorText: Ctrl+M",
	"xIncode*MenuBar*Quit.accelerator: Ctrl<Key>c",
	"xIncode*MenuBar*Quit.acceleratorText: Ctrl+C",
	NULL
};





/* ----------------------------------------------------------------------------
 * xincodeexit - terminate xIncode
 * ------------------------------------------------------------------------- */

void xincodeexit(int signo)
{
	exit(0);
}





/* ---------------------------------------------------------------------------- 
 * initResources - initialize resources
 * ------------------------------------------------------------------------- */

void initResources()
{
	char *	home;
	
	db = NULL;
	tech_file = NULL;

	LastField            = LAST_NONE;
	LastLayerCalculation = LAST_NONE;
	LastViaCalculation   = LAST_NONE;

	/* --------------------------------------------------- setup db_file */

	strcpy(db_file, DBDIR);
	strcat(db_file, DB_FILE);

	/* ------------------------------------------------- setup tech_file */

	tech_file = getenv(TF_ENV);

	/* --------------------------------------------------- setup rc_file */

	if ((home = getenv(RC_ENV)) == NULL) {
	   if ((home = getenv("HOME")) == NULL) {
	      fprintf(stderr, "FATAL ERROR: HOME directory not found\n");
	      exit(2);
	   }
	}

	strcpy(rc_file, home);
	strcat(rc_file, RC_FILE);

	/* ----------------------------------- initialize Tec data structure */

	Tec.layer = (LAYData *) malloc(sizeof(LAYData));
	Tec.name = NULL;
	Tec.layers = 0;

	/* -------------------------- initialize layer & via size parameters */

	C.nL = 0;
	C.sL = NULL;
	C.xsize = 0;
	C.ysize = 0;
}





/* ---------------------------------------------------------------------------- 
 * readResourceFile - read resource file
 * ------------------------------------------------------------------------- */

void readResourceFile()
{
	FILE	*fp;

	if ((fp = fopen(rc_file, "r")) != NULL) {
	   fscanf(fp, "%s\n", tmp1);
	   XmTextSetString(failureW, tmp1);

	   fscanf(fp, "%s\n", tmp1);
	   XmTextSetString(lifetimeW, tmp1);
	   
	   fscanf(fp, "%s\n", tmp1);
	   XmTextSetString(temperatureW, tmp1);
	   
	   fclose(fp);

	   C.fit       = getDouble(failureW, LLfailure);
	   C.lifetime  = getDouble(lifetimeW, LLlifetime);
	   C.temp      = getDouble(temperatureW, LLtemperature);
	}
}





/* ---------------------------------------------------------------------------- 
 * freeTecData - free technology data structure
 * ------------------------------------------------------------------------- */

void freeTecData()
{
	free(Tec.name);
	Tec.name = NULL;
	
	if (Tec.layers > 0) {
	   while (--Tec.layers >= 0) {
	         free(Tec.layer[Tec.layers].name);
	         Tec.layer[Tec.layers].name = NULL;
	   }

	   free(Tec.layer);
	   Tec.layer = NULL;
	}

	/* ----------------------------------- initialize Tec data structure */

	Tec.layer = (LAYData *) malloc(sizeof(LAYData));
	Tec.name = NULL;
	Tec.layers = 0;
}





/* ----------------------------------------------------------------------------
 * yyerror - parser error message handler
 * ------------------------------------------------------------------------- */

void yyerror(char *s)
{
	sprintf(tmp1, "%s [%d]:\n%s at \"%s\"",
	        tech_file, lineNo, s, yytext);
	CreateErrorDialog(main_win, "Parser Error", tmp1, "Dismiss");
}





/* ----------------------------------------------------------------------------
 * LoadTechnologyFile - load technology file data
 * ------------------------------------------------------------------------- */

int LoadTechnologyFile(char *file)
{
	int	status, i;

	/* ---------------------------- check for (and open) technology file */

	if ((yyin = fopen(file, "r")) == NULL) {
	   CreateErrorDialog(main_win, "File Access Error",
	                     strerror(errno), "Dismiss");
	   status = FALSE;
	} else {

	   /* ------------------------ check technology file for correctness */

	   DEBUG = TRUE;
	   lineNo = 1;

	   if (!yyparse()) {
	      rewind(yyin);
	      freeTecData();

	      /* ----------------------------- parse correct technology file */

	      DEBUG = FALSE;
	      yyparse();
	      fclose(yyin);
	      status = TRUE;

	      if (LOG) {
	         fprintf(stderr, "----------------------------------------------------------------------\n");
	         fprintf(stderr, "TECHNOLOGY FILE PARAMETERS\n");
	         fprintf(stderr, "----------------------------------------------------------------------\n");
	         fprintf(stderr,"Technology = %s\n", Tec.name);
	         fprintf(stderr,"Type = %d\n",Tec.type);
	         fprintf(stderr,"Layers = %d\n",Tec.layers);
	         for (i=0; i<Tec.layers; i++) {
	             fprintf(stderr,"Name = %s\n",Tec.layer[i].name);
	             fprintf(stderr,"\tMaterial = %d\n",Tec.layer[i].material);
	             fprintf(stderr,"\tInterconnection Type = %d\n",Tec.layer[i].viatype);
	             fprintf(stderr,"\tThickness = %g\n",Tec.layer[i].thickness);
	             fprintf(stderr,"\tMinimum Layer Width = %g\n",Tec.layer[i].minlayerwidth);
	             fprintf(stderr,"\tMinimum Via Size = %g\n",Tec.layer[i].minviasize);
	             fprintf(stderr,"\tVia Type Coefficient = %g\n",Tec.layer[i].viacoefficient);
	             fprintf(stderr,"\tt50n = %g\n",Tec.layer[i].t50n);
	             fprintf(stderr,"\tReference Current = %g\n",Tec.layer[i].refcur);
	             fprintf(stderr,"\tReference Temperature = %g\n",Tec.layer[i].reftemp);
	             fprintf(stderr,"\tReference Width = %g\n",Tec.layer[i].refwidth);
	             fprintf(stderr,"\tMaximum Current Ratio = %g\n",Tec.layer[i].maxcurratio);
	             fprintf(stderr,"\tMinimum Current Ratio = %g\n",Tec.layer[i].mincurratio);
	             fprintf(stderr,"\tSigma = %g\n",Tec.layer[i].sigma);
	             fprintf(stderr,"\tActivation Energy = %g\n",Tec.layer[i].energy);
	             fprintf(stderr,"\tExponent = %g\n",Tec.layer[i].exp);
	             fprintf(stderr,"\tMaximum Power Density = %g\n",Tec.layer[i].maxpower);
	             fprintf(stderr,"\tSpecific Resistance = %g\n",Tec.layer[i].rho);
	             fprintf(stderr,"\tTemperature Coefficient = %g\n",Tec.layer[i].tc);
	             fprintf(stderr,"\tTotal Thickness of Isolation Layers = %g\n",Tec.layer[i].dist2si);
	             fprintf(stderr,"\tEffective Thermal Conductivity of Isolation Layers = %g\n",Tec.layer[i].lambda);
	             fprintf(stderr,"\tOverheating Threshold Temperature = %g\n",Tec.layer[i].templimit);
	         }
	      }
	   } else {
	      fclose(yyin);
	      status = FALSE;
	   }
	}

	return status;
}





/* ---------------------------------------------------------------------------- 
 * freeDBdata - free database data structure
 * ------------------------------------------------------------------------- */

void freeDBdata(DBData *data)
{
	free(data->label);
	free(data->file);
	free(data);

	tech_file = NULL;
}





/* ---------------------------------------------------------------------------- 
 * readDatabase - read database file
 * ------------------------------------------------------------------------- */

DBData *readDatabase(Widget parent, char *file)
{
	DBData	*data;
	FILE	*fp;
	int	mem, inc = 10;

	mem = inc;

	data = (DBData *) malloc(sizeof(DBData));
	data->label = (line *) malloc(sizeof(line) * mem);
	data->file  = (line *) malloc(sizeof(line) * mem);
	data->size  = 0;

	if ((fp = fopen(file, "r")) != NULL) {
	   while(fgets(tmp1, MAX_CHARS, fp) != NULL) {
	        (void) sscanf(tmp1, "%s %s",
	                      data->label[data->size], data->file[data->size]);
	        data->size++;

	        if (data->size >= mem) {
	           mem += inc;
	           data->label = (line *) realloc(data->label,
	                                          sizeof(line) * mem);
	           data->file  = (line *) realloc(data->file,
	                                          sizeof(line) * mem);
	        }
	   }

	   if (data->size == 0) {
	      CreateErrorDialog(parent, "Database Error", DBreadError, "Quit");
	      freeDBdata(data);
	      data = NULL;
	   }

	   fclose(fp);
	} else {
	   CreateErrorDialog(parent, "Database Error", DBaccessError, "Quit");
	   freeDBdata(data);
	   data = NULL;
	}

	return data;
}





/* ---------------------------------------------------------------------------- 
 * initTechnology - initialize dialog
 * ------------------------------------------------------------------------- */

void initTechnology()
{
	XmString	*list;
	int		i;

	/* -------------------------------------- initialize technology name */

	XmTextSetString(technologyW, Tec.name);

	/* ------------------------------------------- initialize layer list */

	XmListDeleteAllItems(layerW);

	list = (XmString *) XtMalloc(Tec.layers * sizeof(XmString));

	for (i=0; i<Tec.layers; i++) {
	    list[i] = XmStringCreateLocalized(Tec.layer[i].name);
	}

	XmListAddItems(layerW, list, Tec.layers, 1);

	while (--i >= 0) {
	      XmStringFree(list[i]);
	}

	XtFree((char *)list);
}






/* ---------------------------------------------------------------------------- 
 * saveDefaults - save reliability data
 * ------------------------------------------------------------------------- */

void saveDefaults()
{
	char	*text;
	FILE	*fp;
	
	if ((fp = fopen(rc_file, "w")) != NULL) {
	   text = XmTextGetString(failureW);
	   fprintf(fp, "%s\n",text);
	   XtFree(text);

	   text = XmTextGetString(lifetimeW);
	   fprintf(fp, "%s\n",text);
	   XtFree(text);
	
	   text = XmTextGetString(temperatureW);
	   fprintf(fp, "%s\n",text);
	   XtFree(text);

	   fclose(fp);
	} else {
	   sprintf(tmp1, RCaccessError, rc_file);
	   CreateErrorDialog(main_win, "Resource File Error", tmp1, "Quit");
	}
}





/* ---------------------------------------------------------------------------- 
 * resetLayerFields - reset all three layer fields
 * ------------------------------------------------------------------------- */

void resetLayerFields(RESDataL RL)
{
	strcpy(tmp1, "");

	XmTextSetString(currentW, tmp1);
	XmTextSetString(layerwidthW, tmp1);
	XmTextSetString(tempincW, tmp1);

	XtVaSetValues(currentW, XmNforeground, currentW_fgc, NULL);
	XtVaSetValues(layerwidthW, XmNforeground, layerwidthW_fgc, NULL);
	XtVaSetValues(tempincW, XmNforeground, tempincW_fgc, NULL);

	RL.current = 0;
	RL.width = 0;
	RL.temp = 0;
}





/* ---------------------------------------------------------------------------- 
 * resetViaFields - reset all three VIA fields
 * ------------------------------------------------------------------------- */

void resetViaFields(RESDataV RV)
{
	strcpy(tmp1, "");

	XmTextSetString(viaxsizeW, tmp1);
	XmTextSetString(viaysizeW, tmp1);
	XmTextSetString(viacurrW, tmp1);

	XtVaSetValues(viaxsizeW, XmNforeground, viaxsizeW_fgc, NULL);
	XtVaSetValues(viaysizeW, XmNforeground, viaysizeW_fgc, NULL);

	RV.xsize = 0;
	RV.ysize = 0;
	RV.viacurr = 0;
}





/* ---------------------------------------------------------------------------- 
 * showResults - display calculated results
 * ------------------------------------------------------------------------- */

void showResults(RESDataL RL, RESDataV RV)
{
	/* -------------------------------- metal & polysilicon line results */

	if (RL.current > LLcurrent) {
	   sprintENG(tmp1, RL.current, "A");
	   XmTextSetString(currentW, tmp1);
	}

	if (RL.width >= LLlayerwidth) {
	   sprintENG(tmp1, RL.width, "m");
	   XmTextSetString(layerwidthW, tmp1);
	}

	if (RL.temp > 0) {
	   sprintENG(tmp1, RL.temp, "K");
	   XmTextSetString(tempincW, tmp1);

	   if (RL.templimit > 0) { /* handle overheating */
	      sprintENG(tmp1, RL.templimit, "K");
	      sprintf(tmp2, overheatingWarning, tmp1);
	      CreateErrorDialog(main_win, "Warning", tmp2, "Dismiss");

	      XtVaSetValues(tempincW,
	                    XmNforeground, tempincW_hlc,
	                    NULL);
	   } else {
	      XtVaSetValues(tempincW,
	                    XmNforeground, tempincW_fgc,
	                    NULL);
	   }
	}

	/* ------------------------------------ contact hole and via results */

	if (RV.xsize >= LLviasize) {
	   sprintENG(tmp1, RV.xsize, "m");
	   XmTextSetString(viaxsizeW, tmp1);
	}

	if (RV.ysize >= LLviasize) {
	   sprintENG(tmp1, RV.ysize, "m");
	   XmTextSetString(viaysizeW, tmp1);
	}

	if (RV.viacurr > LLcurrent) {
	   sprintENG(tmp1, RV.viacurr, "A");
	   XmTextSetString(viacurrW, tmp1);
	}
}





/* ----------------------------------------------------------------------------
 * SelectLayerCB - callback for layer selection list
 * ------------------------------------------------------------------------- */

void SelectLayerCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	int	*plist, items;

	/* ----------------------------- determine number of selected layers */

	if (XmListGetSelectedPos(layerW, &plist, &items))
	   XtFree((char *) plist);

	/* ---------------------------------- enable/disable parameter entry */

	if (items > 0) {
	   XtSetSensitive(paramW, True);
	   XtSetSensitive(resultW, True);
	   XtSetSensitive(cbuttonW, True);
	} else {
	   XtSetSensitive(paramW, False);
	   XtSetSensitive(resultW, False);
	   XtSetSensitive(cbuttonW, False);
	}
}





/* ----------------------------------------------------------------------------
 * MenuCB - callback from push buttons in pulldown menus
 * ------------------------------------------------------------------------- */

void MenuCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
Position	X,Y;

	switch ((int) client_data) {
		case MENU_LOAD: 
		     if (db != NULL) freeDBdata(db);
		     if ((db = readDatabase(main_win, db_file)) != NULL) {
		        listNo = CreateSelectionDialog(main_win,
		                 "Select Technology", db->label, db->size);
		        if (listNo >= 0) {
		           tech_file = db->file[listNo];
		           if (LoadTechnologyFile(tech_file)) {
		              initTechnology();
	                      SelectLayerCB(NULL, NULL, NULL);
		           }
		        }
		     }
		     break;

		case MENU_SAVE:
		     saveDefaults();
		     break;

		case MENU_READ:
	             readResourceFile();
		     break;

		case MENU_MINIMIZE:
	             XtVaSetValues (app_shell, XmNiconic, True, NULL);
		     break;

		case MENU_EXIT: 
		     xincodeexit(0);
		     break;

		default:
		     printf ("MenuCB: UNEXPECTED TAG.\n");
		     break;
	}
}





/* ----------------------------------------------------------------------------
 * HelpCB - callback for help button
 * ------------------------------------------------------------------------- */

void HelpCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
        int	index = (int) client_data;

        CreateMessage(w, HelpText[index], HelpTitle[index], 15, 75, 75);
}





/* ----------------------------------------------------------------------------
 * CreateMenuBar - create menu bar
 * ------------------------------------------------------------------------- */

Widget CreateMenuBar(Widget parent)
{
	Widget		menu_bar;
	Widget		cascade;
	Widget		file_menu;
	Widget		help_menu;
	Widget		button, separator;
	Arg		al[10];
	register int	ac;
	int		i;
	XmString	label;
 
 
	/* ------------------------------------------------- create menu bar */
 
	menu_bar = XmCreateMenuBar (parent, "MenuBar", NULL, 0);
	file_menu = XmCreatePulldownMenu (menu_bar, "FileMenu", NULL, 0);
 
	/* ------------------------------------- create "File" pulldown menu */
 
	label = XmStringCreateLtoR ("Load Technology", XmSTRING_DEFAULT_CHARSET);
	button = XtVaCreateManagedWidget("Load",
	              xmPushButtonWidgetClass, file_menu,
	              XmNlabelString, label,
	              NULL);
	XtAddCallback (button, XmNactivateCallback, MenuCB,
	              (XtPointer)MENU_LOAD);

	separator   = XtVaCreateManagedWidget("separator",
	              xmSeparatorWidgetClass, file_menu,
	              NULL);

	label = XmStringCreateLtoR ("Read Defaults", XmSTRING_DEFAULT_CHARSET);
	button = XtVaCreateManagedWidget("Read",
	              xmPushButtonWidgetClass, file_menu,
	              XmNlabelString, label,
	              NULL);
	XtAddCallback (button, XmNactivateCallback, MenuCB,
	              (XtPointer)MENU_READ);
 
	label = XmStringCreateLtoR ("Save Defaults", XmSTRING_DEFAULT_CHARSET);
	button = XtVaCreateManagedWidget("Save",
	              xmPushButtonWidgetClass, file_menu,
	              XmNlabelString, label,
	              NULL);
	XtAddCallback (button, XmNactivateCallback, MenuCB,
	              (XtPointer)MENU_SAVE);
 
	separator   = XtVaCreateManagedWidget("separator",
	              xmSeparatorWidgetClass, file_menu,
	              NULL);

	button = XmCreatePushButton (file_menu, "Minimize", NULL, 0);
	XtAddCallback (button, XmNactivateCallback, MenuCB,
	              (XtPointer)MENU_MINIMIZE);
	XtManageChild (button);

	separator   = XtVaCreateManagedWidget("separator",
	              xmSeparatorWidgetClass, file_menu,
	              NULL);

	button = XmCreatePushButton (file_menu, "Quit", NULL, 0);
	XtAddCallback (button, XmNactivateCallback, MenuCB,
	              (XtPointer)MENU_EXIT);
	XtManageChild (button);
 
	ac = 0;
	XtSetArg (al[ac], XmNsubMenuId, file_menu);  ac++;
	cascade = XmCreateCascadeButton (menu_bar, "File", al, ac);
	XtManageChild (cascade);
 
	/* ------------------------------------- create "Help" pulldown menu */
 
	help_menu = XmCreatePulldownMenu (menu_bar, "HelpMenu", NULL, 0);
 
	for (i=0; i<NUMHELP; i++) {
	    label = XmStringCreateLtoR ((char *) HelpTitle[i], 
	            XmSTRING_DEFAULT_CHARSET);
 
	    ac = 0;
	    XtSetArg (al[ac], XmNlabelString, label);  ac++;
	    button = XmCreatePushButton (help_menu, "HelpText", al, ac);
	    XtAddCallback (button, XmNactivateCallback, HelpCB, (XtPointer) i);
	    XtManageChild (button);
	    XmStringFree (label);
	}
 
	ac = 0;
	XtSetArg (al[ac], XmNsubMenuId, help_menu);  ac++;
	cascade = XmCreateCascadeButton (menu_bar, "Help", al, ac);
	XtManageChild (cascade);
 
	ac = 0;
	XtSetArg (al[ac], XmNmenuHelpWidget, cascade);  ac++;
	XtSetValues (menu_bar, al, ac);
 
	return (menu_bar);
}





/* ----------------------------------------------------------------------------
 * GetSelectedLayers - get selcted layers from layer selection list
 * ------------------------------------------------------------------------- */

void GetSelectedLayers() 
{
	int	i, bot;

	bot = 0;

	/* --------------------------------------- determine number of layer */

	if (C.sL != NULL) XtFree((char *) C.sL);

	XmListGetSelectedPos(layerW, &C.sL, &C.nL);

	for (i=0; i<C.nL; i++) {
	    C.sL[i] -= 1;
	    LLlayerwidth = MAX(Tec.layer[C.sL[i]].minlayerwidth, 0);

	    if (Tec.layer[C.sL[i]].dist2si < Tec.layer[C.sL[bot]].dist2si)
	       bot = i;
	}

	LLviasize = MAX(Tec.layer[C.sL[bot]].minviasize, 0);

	/* ----------------------------------- set-up calculation parameters */

	C.fit       = getDouble(failureW, LLfailure);
	C.lifetime  = getDouble(lifetimeW, LLlifetime);
	C.temp      = getDouble(temperatureW, LLtemperature);
	C.current   = getDouble(currentW, LLcurrent);
	C.width     = getDouble(layerwidthW, 0);
	C.xsize     = getDouble(viaxsizeW, 0);
	C.ysize     = getDouble(viaysizeW, 0);
}





/* ----------------------------------------------------------------------------
 * CheckFailure - check failure text field
 * ------------------------------------------------------------------------- */

int CheckFailure() 
{
	int	status = TRUE;

	C.fit = getDouble(failureW, LLfailure);

	if (C.fit == LLfailure) {
	   sprintf(tmp1, "\"%s\": %s %1.3e",
	           failureL, ParameterError, LLfailure);
	   CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	   XtVaSetValues(failureW, XmNforeground, failureW_hlc, NULL);
	   status = FALSE;
	} else {
	   XtVaSetValues(failureW, XmNforeground, failureW_fgc, NULL);
	}

	return status;
}





/* ----------------------------------------------------------------------------
 * CheckLifetime - check lifetime text field
 * ------------------------------------------------------------------------- */

int CheckLifetime() 
{
	int	status = TRUE;

	C.lifetime = getDouble(lifetimeW, LLlifetime);

	if (C.lifetime == LLlifetime) {
	   sprintf(tmp1, "\"%s\": %s %1.3e",
	           lifetimeL, ParameterError, LLlifetime);
	   CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	   XtVaSetValues(lifetimeW, XmNforeground, lifetimeW_hlc, NULL);
	   status = FALSE;
	} else {
	   XtVaSetValues(lifetimeW, XmNforeground, lifetimeW_fgc, NULL);
	}

	return status;
}





/* ----------------------------------------------------------------------------
 * CheckTemperature - check temperature text field
 * ------------------------------------------------------------------------- */

int CheckTemperature() 
{
	int	status = TRUE;

	C.temp = getDouble(temperatureW, LLtemperature);

	if (C.temp == LLtemperature) {
	   sprintf(tmp1, "\"%s\": %s %1.3e",
	           temperatureL, ParameterError, LLtemperature);
	   CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	   XtVaSetValues(temperatureW, XmNforeground, temperatureW_hlc, NULL);
	   status = FALSE;
	} else {
	   XtVaSetValues(temperatureW, XmNforeground, temperatureW_fgc, NULL);
	}

	return status;
}





/* ----------------------------------------------------------------------------
 * CurrentCB - callback for current text field
 * ------------------------------------------------------------------------- */

void CurrentCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	LastField            = LAST_CURRENT;
	LastLayerCalculation = LAST_CURRENT;
}





/* ----------------------------------------------------------------------------
 * LayerWidthCB - callback for layerwidth text field
 * ------------------------------------------------------------------------- */

void LayerWidthCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	LastField            = LAST_WIDTH;
	LastLayerCalculation = LAST_WIDTH;
}





/* ----------------------------------------------------------------------------
 * ViaXSizeCB - callback for x-size via field
 * ------------------------------------------------------------------------- */

void ViaXSizeCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	LastField          = LAST_XSIZE;
	LastViaCalculation = LAST_XSIZE;
}





/* ----------------------------------------------------------------------------
 * ViaYSizeCB - callback for y-size via field
 * ------------------------------------------------------------------------- */

void ViaYSizeCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	LastField          = LAST_YSIZE;
	LastViaCalculation = LAST_YSIZE;
}





/* ----------------------------------------------------------------------------
 * CalculateCB - callback for calculate button
 * ------------------------------------------------------------------------- */

void CalculateCB(Widget w, XtPointer client_data, XtPointer call_data) 
{
	RESDataL	RL;
	RESDataV	RV;
	int		OK;

	/* -------------------------------------------------------- defaults */

	OK = TRUE;

	RL.current = LLcurrent;
	RL.width   = 0;
	RL.temp    = 0;

	RV.xsize   = 0;
	RV.ysize   = 0;
	RV.viacurr = LLcurrent;

	/* ---------------------------- update layer parameters if necessary */

	if (LastField != LAST_NONE) GetSelectedLayers();

	/* ----------------------------------------------- update parameters */

	if (!CheckFailure()) OK = FALSE;
	if (!CheckLifetime()) OK = FALSE;
	if (!CheckTemperature()) OK = FALSE;

	/* ------------- layer calculation depending on the last entry field */

	if (LastLayerCalculation == LAST_CURRENT && OK) {
	   if (isEmpty(currentW)) {
	      resetLayerFields(RL);
	   } else {
	      C.current = getDouble(currentW, LLcurrent);

	      if (C.current == LLcurrent) {
	         sprintf(tmp1, "\"%s\": %s %1.3e",
	                 currentL, ParameterError, LLcurrent);
	         CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	         XtVaSetValues(currentW, XmNforeground, currentW_hlc, NULL);
	         OK = FALSE;
	      } else {
	         RL = computeMinLayerWidth(Tec, C);
	         RL.current = C.current;
	         XtVaSetValues(currentW, XmNforeground, currentW_fgc, NULL);
	         XtVaSetValues(layerwidthW, XmNforeground, layerwidthW_fgc, NULL);
	      }
	   }
	}

	if (LastLayerCalculation == LAST_WIDTH && OK) {
	   if (isEmpty(layerwidthW)) {
	      resetLayerFields(RL);
	   } else {
	      C.width = getDouble(layerwidthW, 0);

	      if (C.width < LLlayerwidth) {
	         sprintf(tmp1, "\"%s\": %s %1.3e",
	                 layerwidthL, WidthError, LLlayerwidth);
	         CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	         XtVaSetValues(layerwidthW, XmNforeground, layerwidthW_hlc, NULL);
	         OK = FALSE;
	      } else {
	         RL = computeMaxCurrent(Tec, C);
	         RL.width = C.width;
	         XtVaSetValues(layerwidthW, XmNforeground, layerwidthW_fgc, NULL);
	         XtVaSetValues(currentW, XmNforeground, currentW_fgc, NULL);
	      }
	   }
	}

	/* --------------- via calculation depending on the last entry field */

	if (LastViaCalculation == LAST_XSIZE && OK) {
	   if (isEmpty(viaxsizeW)) {
	      resetViaFields(RV);
	   } else {
	      C.xsize = getDouble(viaxsizeW, 0);

	      if (C.xsize < LLviasize) {
	         sprintf(tmp1, "\"%s\": %s %1.3e",
	                 viaxsizeL, SizeError, LLviasize);
	         CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	         XtVaSetValues(viaxsizeW, XmNforeground, viaxsizeW_hlc, NULL);
	         OK = FALSE;
	      } else {
	         C.ysize = C.xsize;
	         RV.viacurr = computeViaCurrent(Tec, C);
	         RV.xsize = C.xsize;
	         RV.ysize = C.ysize;
	         XtVaSetValues(viaxsizeW, XmNforeground, viaxsizeW_fgc, NULL);
	         XtVaSetValues(viaysizeW, XmNforeground, viaysizeW_fgc, NULL);
	      }
	   }
	}

	if (LastViaCalculation == LAST_YSIZE && OK) {
	   if (isEmpty(viaysizeW)) {
	      resetViaFields(RV);
	   } else {
	      C.ysize = getDouble(viaysizeW, 0);
	      C.xsize = getDouble(viaxsizeW, 0);

	      if (C.ysize < LLviasize) {
	         sprintf(tmp1, "\"%s\": %s %1.3e",
	                 viaysizeL, SizeError, LLviasize);
	         CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	         XtVaSetValues(viaysizeW, XmNforeground, viaysizeW_hlc, NULL);
	         OK = FALSE;
	      } else {
	         XtVaSetValues(viaysizeW, XmNforeground, viaysizeW_fgc, NULL);
	      }

	      if (C.xsize < LLviasize) {
	         sprintf(tmp1, "\"%s\": %s %1.3e",
	                 viaxsizeL, SizeError, LLviasize);
	         CreateErrorDialog(main_win, "Parameter Error", tmp1, "Dismiss");
	         XtVaSetValues(viaxsizeW, XmNforeground, viaxsizeW_hlc, NULL);
	         OK = FALSE;
	      } else {
	         XtVaSetValues(viaxsizeW, XmNforeground, viaxsizeW_fgc, NULL);
	      }

	      if (OK) {
	         RV.viacurr = computeViaCurrent(Tec, C);
	         RV.xsize = C.xsize;
	         RV.ysize = C.ysize;
	      }
	   }
	}

	/* --------------------------------------------- show results if any */

	if (LOG) {
	   fprintf(stderr, "----------------------------------------------------------\n");
	   fprintf(stderr, "CONDUCTION LINE DESIGN: Results\n");
	   fprintf(stderr, "----------------------------------------------------------\n");
	   fprintf(stderr, ":: %-30s = %12.5e\n", currentL, C.current);
	   fprintf(stderr, "=> %-30s = %12.5e\n", currentL, RL.current);
	   fprintf(stderr, ":: %-30s = %12.5e\n", layerwidthL, C.width);
	   fprintf(stderr, "=> %-30s = %12.5e\n", layerwidthL, RL.width);
	   fprintf(stderr, "=> %-30s = %12.5e\n", tempincL, RL.temp);
	   
	   fprintf(stderr, "\nVIA DESIGN: Results\n");
	   fprintf(stderr, ":: %-30s = %12.5e\n", viaxsizeL, C.xsize);
	   fprintf(stderr, "=> %-30s = %12.5e\n", viaxsizeL, RV.xsize);
	   fprintf(stderr, ":: %-30s = %12.5e\n", viaysizeL, C.ysize);
	   fprintf(stderr, "=> %-30s = %12.5e\n", viaysizeL, RV.ysize);
	   fprintf(stderr, "=> %-30s = %12.5e\n", viacurrL, RV.viacurr);
	}

	if (LastField != LAST_NONE && OK) showResults(RL, RV);

	/* ------------------ traverse back to last entry field having focus */

	switch (LastField) {
	       case LAST_CURRENT:
	            XmProcessTraversal(currentW, XmTRAVERSE_CURRENT);
	            break;
	       case LAST_WIDTH:
	            XmProcessTraversal(layerwidthW, XmTRAVERSE_CURRENT);
	            break;
	       case LAST_XSIZE:
	            XmProcessTraversal(viaxsizeW, XmTRAVERSE_CURRENT);
	            break;
	       case LAST_YSIZE:
	            XmProcessTraversal(viaysizeW, XmTRAVERSE_CURRENT);
	            break;
	       default:
	            XmProcessTraversal(cbuttonW, XmTRAVERSE_CURRENT);
	            break;
	}
}





/* ----------------------------------------------------------------------------
 * CreateMainForm - create main form
 * ------------------------------------------------------------------------- */

Widget CreateMainForm(Widget parent)
{
	Widget		work_area;
	Widget		label;
	Widget		form, frame, separator;
	Widget		sub_area, list_area, row_area;
	Dimension	w;
	Arg		al[10];
	register int	ac;

	work_area    = XtVaCreateWidget("workArea",
	               xmFormWidgetClass, parent,
	               NULL);

	/* ------------------------------------------------- create top form */

	form         = XtVaCreateWidget("form",
	               xmFormWidgetClass, work_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNtopOffset, 10,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	/* ............................................ create scrolled list */
	
	list_area    = XtVaCreateWidget("listArea",
	               xmRowColumnWidgetClass, form,
	               XmNorientation, XmVERTICAL,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	/* selected technology */

	sub_area     = XtVaCreateWidget("textArea",
	               xmFormWidgetClass, list_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               "Technology  ",
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	XtVaGetValues(label, XmNwidth, &w, NULL);

	technologyW  = XtVaCreateManagedWidget("Annotations",
	               xmTextWidgetClass, sub_area,
	               XmNeditable, False,
	               XmNcursorPositionVisible, False,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	XtManageChild(sub_area);

        /* scrolled list */

	sub_area     = XtVaCreateWidget("textArea",
	               xmFormWidgetClass, list_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, sub_area,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);
	
	label        = XtVaCreateManagedWidget(
	               "Layer",
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);
	layerW     = XmCreateScrolledList(sub_area, "scrolledList", NULL, 0);

	XtVaSetValues(layerW,
	               XmNscrollBarDisplayPolicy, XmSTATIC,
	               XmNvisibleItemCount, 4,
	               XmNselectionPolicy, XmMULTIPLE_SELECT,
	               NULL);

	XtVaSetValues(XtParent(layerW),
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNscrolledWindowMarginWidth, 2,
	               XmNscrolledWindowMarginHeight, 2,
	               NULL);

	XtManageChild(layerW);

	XtManageChild(sub_area);

	XtManageChild(list_area);

	XtAddCallback(layerW, XmNmultipleSelectionCallback, SelectLayerCB,
	              NULL);

	/* .......................................... create parameter field */

	paramW       = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, form,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_WIDGET,
	               XmNleftWidget, list_area,
	               XmNleftOffset, 5,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNrightOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNsensitive, False,	/* for initialization only */
	               NULL);

	/* 1st parameter */

	sub_area     = XtVaCreateWidget("textArea",
	               xmFormWidgetClass, paramW,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               failureL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	failureW     = XtVaCreateManagedWidget("Parameters",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNwidth, w,
	               XmNvalue, tbd,
	               NULL);

	XtManageChild(sub_area);
	
	/* 2nd parameter */

	sub_area     = XtVaCreateWidget("textArea",
	               xmFormWidgetClass, paramW,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, sub_area,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               lifetimeL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	lifetimeW    = XtVaCreateManagedWidget("Parameters",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNwidth, w,
	               XmNvalue, tbd,
	               NULL);

	XtManageChild(sub_area);
	
	/* 3rd parameter */

	sub_area     = XtVaCreateWidget("textArea",
	               xmFormWidgetClass, paramW,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, sub_area,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               temperatureL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	temperatureW = XtVaCreateManagedWidget("Parameters",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNwidth, w,
	               XmNvalue, tbd,
	               NULL);

	XtManageChild(sub_area);
	
	XtManageChild(paramW);

	XtManageChild(form);

	/* ---------------------------------------------- create result form */

	form         = XtVaCreateWidget("form",
	               xmFormWidgetClass, work_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, form,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               NULL);

	/* .................................................. create 1st row */

	resultW      = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, form,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNnumColumns, 2,
	               XmNorientation, XmHORIZONTAL,
	               XmNpacking, XmPACK_COLUMN,
	               XmNmarginWidth, 10,
	               XmNsensitive, False,
	               NULL);

	frame        = XtVaCreateWidget("textArea",
	               xmFrameWidgetClass, resultW,
	               XmNshadowType, XmSHADOW_ETCHED_IN,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               linesL,
	               xmLabelGadgetClass, frame,
	               XmNchildType, XmFRAME_TITLE_CHILD,
	               XmNchildHorizontalAlignment, XmALIGNMENT_CENTER,
	               NULL);

	row_area     = XtVaCreateWidget("form",
	               xmRowColumnWidgetClass, frame,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNnumColumns, 1,
	               XmNorientation, XmHORIZONTAL,
	               XmNpacking, XmPACK_COLUMN,
	               NULL);

	/* 1st item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               currentL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	currentW     = XtVaCreateManagedWidget("Results",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtAddCallback(currentW, XmNfocusCallback,
	               (XtCallbackProc) CurrentCB, NULL);
	
	XtManageChild(sub_area);

	/* 2nd item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_WIDGET,
	               XmNleftWidget, sub_area,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               layerwidthL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	layerwidthW  = XtVaCreateManagedWidget("Results",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtAddCallback(layerwidthW, XmNfocusCallback,
	              (XtCallbackProc) LayerWidthCB, NULL);
	
	XtManageChild(sub_area);

	/* 3rd item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_WIDGET,
	               XmNleftWidget, sub_area,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNrightOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               tempincL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	tempincW     = XtVaCreateManagedWidget("Annotations",
	               xmTextWidgetClass, sub_area,
	               XmNeditable, False,
	               XmNcursorPositionVisible, False,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtManageChild(sub_area);

	XtManageChild(row_area);

	XtManageChild(frame);

	/* .................................................. create 2nd row */

	frame        = XtVaCreateWidget("textArea",
	               xmFrameWidgetClass, resultW,
	               XmNshadowType, XmSHADOW_ETCHED_IN,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               viasL,
	               xmLabelGadgetClass, frame,
	               XmNchildType, XmFRAME_TITLE_CHILD,
	               XmNchildHorizontalAlignment, XmALIGNMENT_CENTER,
	               NULL);

	row_area     = XtVaCreateWidget("form",
	               xmRowColumnWidgetClass, frame,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNnumColumns, 1,
	               XmNorientation, XmHORIZONTAL,
	               XmNpacking, XmPACK_COLUMN,
	               NULL);

	/* 1st item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               viaxsizeL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	viaxsizeW    = XtVaCreateManagedWidget("Results",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtAddCallback(viaxsizeW, XmNfocusCallback,
	               (XtCallbackProc) ViaXSizeCB, NULL);
	
	XtManageChild(sub_area);

	/* 2nd item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_WIDGET,
	               XmNleftWidget, sub_area,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               viaysizeL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	viaysizeW    = XtVaCreateManagedWidget("Results",
	               xmTextWidgetClass, sub_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtAddCallback(viaysizeW, XmNfocusCallback,
	              (XtCallbackProc) ViaYSizeCB, NULL);
	
	XtManageChild(sub_area);

	/* 3rd item */

	sub_area     = XtVaCreateWidget("textArea",
	               xmRowColumnWidgetClass, row_area,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_WIDGET,
	               XmNleftWidget, sub_area,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNrightOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNorientation, XmVERTICAL,
	               XmNpacking, XmPACK_TIGHT,
	               NULL);

	label        = XtVaCreateManagedWidget(
	               viacurrL,
	               xmLabelGadgetClass, sub_area,
	               XmNalignment, XmALIGNMENT_BEGINNING,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               NULL);

	viacurrW     = XtVaCreateManagedWidget("Annotations",
	               xmTextWidgetClass, sub_area,
	               XmNeditable, False,
	               XmNcursorPositionVisible, False,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, label,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 10,
	               XmNwidth, w,
	               NULL);

	XtManageChild(sub_area);

	XtManageChild(row_area);

	XtManageChild(frame);

	XtManageChild(resultW);

	XtManageChild(form);

	/* --------------------------------------- create calculation button */

	form         = XtVaCreateWidget("form",
	               xmFormWidgetClass, work_area,
	               XmNtopAttachment, XmATTACH_WIDGET,
	               XmNtopWidget, form,
	               XmNtopOffset, 5,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNleftOffset, 5,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNrightOffset, 5,
	               XmNbottomAttachment, XmATTACH_FORM,
	               NULL);

	cbuttonW     = XtVaCreateManagedWidget("Calculate",
	               xmPushButtonWidgetClass, form,
	               XmNtopAttachment, XmATTACH_FORM,
	               XmNleftAttachment, XmATTACH_FORM,
	               XmNrightAttachment, XmATTACH_FORM,
	               XmNbottomAttachment, XmATTACH_FORM,
	               XmNshowAsDefault, (Dimension) 2,
	               XmNsensitive, False,
	               NULL);

	XtAddCallback(cbuttonW, XmNactivateCallback,
	              (XtCallbackProc) CalculateCB, NULL);

	XtManageChild(form);

	XtVaSetValues(work_area,
	               XmNdefaultButton, cbuttonW,
	               NULL);

	return(work_area);
}





/* ----------------------------------------------------------------------------
 * main - main program
 * ------------------------------------------------------------------------- */
 
int main(int argc, char **argv)
{
	Widget		menu_bar;
	Pixmap		icon, shape;
	extern int	opterr;
	int		error = FALSE;
	int		c;

	/* ------------------------------------------------------ get option */

	opterr = 0;

	while ((c = getopt(argc, argv, "d")) != EOF) {
	      switch (c) {
	             case 'd': LOG = TRUE;
	                       break;
	             case '?': error = TRUE;
	                       break;
	      }
	}

	if (error) {
	   fprintf(stderr, "usage: xincode [-d]\n");
	   exit(2);
	}

	/* ------------------------------------------------------ initialize */

	strcpy(argv[0],"xIncode");

	signal(SIGINT, xincodeexit);
	signal(SIGTERM, xincodeexit);

	initResources();

	/* ---------------------------------------- set-up language settings */

	XtSetLanguageProc(NULL, NULL, NULL);

	/* ------------------------------------ initialize application shell */

	app_shell = XtVaAppInitialize (&app_context, "xIncode", NULL, 0,
	                               &argc, argv, fallbackResources, NULL);

	/* ---------------------------------------------- create main window */

	main_win = XtVaCreateManagedWidget("mainWin",
	                                   xmMainWindowWidgetClass, app_shell,
	                                   NULL);

	/* ---------------------------------------------- create icon bitmap */

#if defined (HAVE_XPM)
	XpmCreatePixmapFromData(XtDisplay(app_shell), 
	       RootWindowOfScreen(XtScreen (app_shell)),
	       xincode_xpm, &icon, &shape, NULL);
#else
	icon = XCreateBitmapFromData(XtDisplay(app_shell), 
	       RootWindowOfScreen(XtScreen (app_shell)),
	       (char *) xincode_bits, xincode_width, xincode_height);
	shape = 0;
#endif

	XtVaSetValues(app_shell, XmNiconPixmap, icon, XmNiconMask, shape, 0);

	/* ------------------------------------------------- create menu bar */

	menu_bar = CreateMenuBar(main_win);
	XtManageChild(menu_bar);

	/* ------------------------------------------------ create main form */

	main_form = CreateMainForm(main_win);
	XtManageChild(main_form);

	/* ----------------------------------------- realize toplevel widget */

	XtRealizeWidget(app_shell);

	/* ----------------------------------- get colors to indicate errors */

	XtVaGetValues(failureW,
	              XmNforeground, &failureW_fgc,
	              XmNhighlightColor, &failureW_hlc,
	              NULL);

	XtVaGetValues(lifetimeW,
	              XmNforeground, &lifetimeW_fgc,
	              XmNhighlightColor, &lifetimeW_hlc,
	              NULL);

	XtVaGetValues(temperatureW,
	              XmNforeground, &temperatureW_fgc,
	              XmNhighlightColor, &temperatureW_hlc,
	              NULL);

	XtVaGetValues(currentW,
	              XmNforeground, &currentW_fgc,
	              XmNhighlightColor, &currentW_hlc,
	              NULL);

	XtVaGetValues(layerwidthW,
	              XmNforeground, &layerwidthW_fgc,
	              XmNhighlightColor, &layerwidthW_hlc,
	              NULL);

	XtVaGetValues(tempincW,
	              XmNforeground, &tempincW_fgc,
	              XmNhighlightColor, &tempincW_hlc,
	              NULL);

	XtVaGetValues(viaxsizeW,
	              XmNforeground, &viaxsizeW_fgc,
	              XmNhighlightColor, &viaxsizeW_hlc,
	              NULL);

	XtVaGetValues(viaysizeW,
	              XmNforeground, &viaysizeW_fgc,
	              XmNhighlightColor, &viaysizeW_hlc,
	              NULL);

	/* -------------------------------------- read techfile if specified */

	if (tech_file != NULL) {
	   if (LoadTechnologyFile(tech_file)) {
	      initTechnology();
	   }
	}

	/* --------------------------------------- read resource file if any */

	readResourceFile();

	/* -------------------------------------------------- process events */

	XtAppMainLoop(app_context);
}
