/* ----------------------------------------------------------------------------
 * Copyright (C) 2000-2005 by Karim Kaschani
 * 
 * This file is part of the software xIncode developed to support the design
 * of interconnects in semiconductor technologies.
 * 
 * xIncode is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xIncode is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ------------------------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include "xincode.h"
#include "calc.h"





/* ----------------------------------------------------------------- globals */

static char	units[] = "afpnm kMGTPE?";

int	StrLen(char *);
int	StrCmp(char *, char *);
char	*StrAdd(char **, char *);
char	*StrNCpy(char *, char *, int);





/* ----------------------------------------------------------------------------
 * StrLen - my own 'strlen' to handle NULL pointers
 * --------------------------------------------------------------------------*/

int StrLen(char *s)
{
	int	l;

	l = 0;

	if (s != NULL && StrCmp(s, "") != 0) {
	   for (l=0; *s != '\0'; s++)
	       l++;
	}

	return l;
}





/* ----------------------------------------------------------------------------
 * StrAdd - strcat (strcpy) + appropriate memory allocation
 * --------------------------------------------------------------------------*/

char *StrAdd(char **s1, char *s2)
{
	int	l;

	l = StrLen(*s1) + StrLen(s2) + 1;

	if (*s1 == NULL) {
	   if ((*s1 = (char *) realloc(*s1, sizeof(char))) != NULL) {
	      strcpy(*s1, "\0");
	   } else {
	      return NULL;
	   }
	}

	if ((*s1 = (char *) realloc(*s1, sizeof(char) * l)) == NULL) {
	   return NULL;
	} else {
	   return strcat(*s1, s2);
	}
}





/* ----------------------------------------------------------------------------
 * StrCmp - my own 'strcmp' to handle NULL pointers
 * --------------------------------------------------------------------------*/

int StrCmp(char *s1, char *s2)
{
	if (s1 == NULL || s2 == NULL)
	   return(-1);
	else
	   return(strcmp(s1, s2));
}





/* ----------------------------------------------------------------------------
 * StrNCpy - my own 'strncpy' to work around a strange bug of Solaris 2.4x86
 * --------------------------------------------------------------------------*/
 
char *StrNCpy(char *s1, char *s2, int n)
{
	char	*p1, *p2;

	p1 = s1;
	p2 = s2;

	while (p2 != NULL && n > 0) {
	      *p1 = *p2;
	      p1++;
	      p2++;
	      n--;
	}

	*p1 = '\0';	/* terminate copied string properly */

	return s1;
}





/* ----------------------------------------------------------------------------
 * str2ENG - convert ASCII string to double using engineer notation
 * ------------------------------------------------------------------------- */

double str2ENG(const char *str, char **end)
{
	char	*pos;
	double	value;

	value = strtod(str, &pos);

	switch (pos[0]) {
	       case 'a': value *= 1e-18; pos++; break;
	       case 'f': value *= 1e-15; pos++; break;
	       case 'p': value *= 1e-12; pos++; break;
	       case 'n': value *= 1e-09; pos++; break;
	       case '':
	       case 'u': value *= 1e-06; pos++; break;
	       case 'm': value *= 1e-03; pos++; break;
	       case 'k': value *= 1e+03; pos++; break;
	       case 'M': value *= 1e+06; pos++; break;
	       case 'G': value *= 1e+09; pos++; break;
	       case 'T': value *= 1e+12; pos++; break;
	       case 'P': value *= 1e+15; pos++; break;
	       case 'E': value *= 1e+18; pos++; break;
	       default: break;
	}

	if (end != NULL) *end = pos;

	return value;
}





/* ----------------------------------------------------------------------------
 * sprintENG - print a double in engineer notation in a string
 * ------------------------------------------------------------------------- */

void sprintENG(char *s, double x, char *u)
{
	int		order, index, commas;
	static	char	format[10];

	/* --------------------------------------------- calculate eng order */

	order = ((int) floor(nlog10(x)/3.0)) * 3;
	index = order/3 + 6;
	if (index < 0 || index > 12) index=13;

	/* strip order from oroginal value */

	x = x / pow(10.0, order);
	commas = 3 - floor(nlog10(x));
	sprintf(format, "%%5.%if", commas);

	sprintf(s, format, x);
	strncat(s, &units[index], 1);
	strncat(s, &u[0], 1);
}





/* ----------------------------------------------------------------------------
 * IsFile - check filename
 * --------------------------------------------------------------------------*/

int IsFile(char *filename)
{
	int		file = 0;
	struct stat	buf;

	if (StrLen(filename) == 0) {
	   file = -1;
	} else {
	   if (stat(filename, &buf) == 0) {
	      if (S_ISDIR(buf.st_mode)) file = -1;
	      if (S_ISCHR(buf.st_mode)) file = -1;
	      if (S_ISBLK(buf.st_mode)) file = -1;
	      if (S_ISFIFO(buf.st_mode)) file = -1;
	      if (S_ISLNK(buf.st_mode)) file = -1;
	      if (S_ISSOCK(buf.st_mode)) file = -1;
	   }
	}

	return (file == 0);
}





/* ----------------------------------------------------------------------------
 * FileExists - check if file exists
 * --------------------------------------------------------------------------*/

int FileExists(char *filename)
{
	return (access(filename, F_OK) == 0);
}
