/* ----------------------------------------------------------------------------
 * Copyright (C) 2000-2005 by Karim Kaschani
 * 
 * This file is part of the software xIncode developed to support the design
 * of interconnects in semiconductor technologies.
 * 
 * xIncode is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xIncode is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ------------------------------------------------------------------------- */

#include <float.h>
#include <math.h>
#include "xincode.h"
#include "xutils.h"





/* ----------------------------------------------------------------- globals */

static double	T0  = 273.16;		/* 0C temperature in Kelvin */
static double	VT0 = 0.023539245;	/* thermal voltage at 0C */

static char *InvGaussError = {
"InvGauss: Double underflow.\n\
Check 'FIT' and 'Lifetime' specifications."
};





/* ----------------------------------------------------------------------------
 * nlog10 - new log10 function
 * ------------------------------------------------------------------------- */

double nlog10(double x)
{
	double	r;

	if (x == 0)
	   r = 0.0;
	else
	   r = log10(x) + 1e-15;

	return r;
}





/* ----------------------------------------------------------------------------
 * rho - calculate temperature dependent specific resistance
 * ------------------------------------------------------------------------- */

double rho(double rho0, double Tc, double T)
{
	double	rho;

	rho = rho0 * (1.0 + Tc * (T - 25.0));

	if (LOG) {
	   fprintf(stderr, "      %-27s = %12.5e\n",
	           "Temperature", T);
	   fprintf(stderr, "      %-27s = %12.5e\n",
	           "Specific Resistance", rho);
	}

	return rho;
}





/* ----------------------------------------------------------------------------
 * InvGauss - find percentage points of the normal distribution
 *
 *            R.E. Odeh, J.O. Evans
 *            "The Percentage Points of the Normal Distribution"
 *            Applied Statistics
 * ------------------------------------------------------------------------- */

double InvGauss(double frate, double sigma)
{
	const double	p0 = -0.322232431088;
	const double	p1 = -1.0;
	const double	p2 = -0.342242088547;
	const double	p3 = -0.204231210245e-1;
	const double	p4 = -0.453642210148e-4;
	const double	q0 =  0.993484626060e-1;
	const double	q1 =  0.588581570495;
	const double	q2 =  0.531103462366;
	const double	q3 =  0.103537752850;
	const double	q4 =  0.385607006340e-2;

	double		ps, yi, ig;

	ps = frate;

	if (ps > 0.5) ps = 1.0 - ps;

	if (ps < FLT_MIN) {
	   CreateErrorDialog(main_win, "Calculation Error", InvGaussError,
	                               "Dismiss");
	   return 0.0;
	}

	if (ps == 0.5) return 0.0;

	yi = sqrt(log(1.0/(ps*ps)));

	ig = yi + ((((yi * p4 + p3) * yi + p2) * yi + p1) * yi + p0) /
	          ((((yi * q4 + q3) * yi + q2) * yi + q1) * yi + q0);

	if (frate < 0.5) ig = -ig;

	ig *= sigma;

	if (LOG) fprintf(stderr, "      %-27s = %12.5e\n",
	                 "Inverse Gauss", ig);

	return ig;
}





/* ----------------------------------------------------------------------------
 * FailureRate2MTF - calculate mean time to failure from specified failure rate
 * ------------------------------------------------------------------------- */

double FailureRate2MTF(double fit, double lifetime, double sigma)
{
	double	t50;

	t50 = lifetime * exp(- sigma * InvGauss(fit * 1e-9 * lifetime, sigma));

	if (LOG) fprintf(stderr, "      %-27s = %12.5e\n",
	                 "Mean-Time-To-Failure", t50);
	
	return t50;
}





/* ----------------------------------------------------------------------------
 * MaxCurrentDensity - Calculate maximum current density
 * ------------------------------------------------------------------------- */

double MaxCurrentDensity(LAYData L, double t50, MIGRData C)
{
	double	Jmax;

	Jmax = exp(-L.energy/(L.exp*VT0)*(T0/(T0+L.reftemp)-T0/(T0+C.temp)));
	Jmax *= L.refcur * pow(L.t50n/t50, 1/L.exp);

	if (LOG) fprintf(stderr, "      %-27s = %12.5e\n",
	                 "Design Current Density", Jmax);

	return Jmax;
}





/* ----------------------------------------------------------------------------
 * computeMaxCurrent - compute maximum current
 * ------------------------------------------------------------------------- */

RESDataL computeMaxCurrent(TECData T, MIGRData C)
{
	RESDataL	R;
	LAYData		Li;
	double		t50, Jmax, J, Ratio, Ii, Vsq, Temp, dSum, rhoT;
	int		i;

	if (LOG) {
	   fprintf(stderr, "----------------------------------------------------------\n");
	   fprintf(stderr, "CONDUCTION LINE DESIGN: Current Calculation\n");
	   fprintf(stderr, "----------------------------------------------------------\n");
	}

	R.width = C.width;

	/* ---------- calculate minimum voltage across square for each layer */

	Vsq = FLT_MAX; /* initialization */

	for (i=0; i<C.nL; i++) {
	    Li = T.layer[C.sL[i]]; /* define respective layer */

	    if (LOG) fprintf(stderr, "   %-30s : %s\n", "Layer", Li.name);

	    rhoT = rho(Li.rho, Li.tc, C.temp); /* temperature dependent rho */

	    if (Li.material == POLY) { /* PolySilicon */
	       Ii = C.width * sqrt(Li.maxpower * Li.thickness / rhoT);
	    } else { /* Metal */
	       t50   = FailureRate2MTF(C.fit, C.lifetime, Li.sigma);
	       Jmax  = MaxCurrentDensity(Li, t50, C);
	       Ratio = pow(C.width/Li.refwidth, 1/Li.exp);
	       Ii    = Jmax * Ratio * Li.thickness * C.width;

	       /* ------------------------------------- extremely wide lines */

	       if (Ratio > Li.maxcurratio) {
	          Ii = Li.maxcurratio * Jmax * Li.thickness * C.width;
	       }

	       /* ----------------------------------- extremely narrow lines */

	       if (Ratio < Li.mincurratio) {
	          Ii = Li.mincurratio * Jmax * Li.thickness * C.width;
	       }
	    }

	    if (LOG) {
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Current Density", Ii/(Li.thickness*C.width));
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Current", Ii);
	    }

	    Vsq = MIN(Vsq, Ii*rhoT/Li.thickness);
	}

	/* ------------------ calculate total current & temperature increase */

	R.current     = 0; /* initialization */
	R.temp = dSum = 0; /* initialization */

	for (i=0; i<C.nL; i++) {
	    Li = T.layer[C.sL[i]]; /* define respective layer */

	    if (LOG) fprintf(stderr, "   %-30s : %s\n", "Layer", Li.name);

	    rhoT = rho(Li.rho, Li.tc, C.temp); /* temperature dependent rho */
	    Ii = Vsq * Li.thickness/rhoT;
	    J  = Ii / (Li.thickness * C.width);

	    if (LOG) {
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Relaxed Current Density", J);
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Relaxed Current", Ii);
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Eff. Distance to Silicon", Li.dist2si - dSum);
	    }

	    Temp = (J*J) * rhoT * Li.thickness
	           * (Li.dist2si - dSum) / Li.lambda;

	    /* ----------------------------------------- result = sum of all */

	    R.current += Ii;
	    R.temp    += Temp;

	    /* ---- distinguish between planarized ond non-planarized layers */

	    if (T.type == PLANARIZED) {
	       dSum   += Li.thickness;
	    }
	}

	/* ------------------------------------- check for temperature alert */

	if (R.temp > Li.templimit)
	   R.templimit = Li.templimit;
	else
	   R.templimit = 0;
	
	return R;
}





/* ----------------------------------------------------------------------------
 * computeMinLayerWidth - compute minimum layer width
 * ------------------------------------------------------------------------- */

RESDataL computeMinLayerWidth(TECData T, MIGRData C)
{
	RESDataL	R;
	LAYData		Li;
	double		t50, Jmax, J, Wi, Temp, invRsq, Ii, dSum, rhoT;
	int		i;

	if (LOG) {
	   fprintf(stderr, "----------------------------------------------------------\n");
	   fprintf(stderr, "CONDUCTION LINE DESIGN: Width Calculation\n");
	   fprintf(stderr, "----------------------------------------------------------\n");
	}
	
	R.current = C.current;

	/* --------------------- determine sum of inverse square resistances */

	invRsq = 0;

	for (i=0; i<C.nL; i++) {
	    if (LOG) fprintf(stderr, "   %-30s : %s\n", "Layer",
	                     T.layer[C.sL[i]].name);

	    rhoT = rho(T.layer[C.sL[i]].rho, T.layer[C.sL[i]].tc, C.temp);
	    invRsq += T.layer[C.sL[i]].thickness / rhoT;
	}

	/* ------------------------- calculate width for each selected layer */

	R.width = FLT_MIN; /* initialization */

	for (i=0; i<C.nL; i++) {
	    Li = T.layer[C.sL[i]]; /* define respective layer */

	    if (LOG) fprintf(stderr, "   %-30s : %s\n", "Layer", Li.name);

	    rhoT = rho(Li.rho, Li.tc, C.temp); /* temperature dependent rho */
	    Ii = C.current * (Li.thickness / rhoT) / invRsq;

	    if (LOG) fprintf(stderr, "      %-27s = %12.5e\n", "Current", Ii);

	    if (Li.material == POLY) { /* PolySilicon */
	       Wi = Ii * sqrt(rhoT / (Li.maxpower * Li.thickness));
	    } else { /* Metal */
	       t50  = FailureRate2MTF(C.fit, C.lifetime, Li.sigma);
	       Jmax = MaxCurrentDensity(Li, t50, C);
	       Wi   = Ii /(Li.thickness * Jmax);
	       Wi   = pow(Wi/Li.refwidth, Li.exp/(Li.exp + 1.0));
	       Wi  *= Li.refwidth;

	       J = Ii / (Li.thickness * Wi);

	       /* ------------------------------------- extremely wide lines */

	       if (J > Li.maxcurratio * Jmax) {
	          Wi = Ii / (Li.maxcurratio * Jmax * Li.thickness);
	       }

	       /* ----------------------------------- extremely narrow lines */

	       if (J < Li.mincurratio * Jmax) {
	          Wi = Ii / (Li.mincurratio * Jmax * Li.thickness);
	       }
	    }

	    if (LOG) {
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Current Density", Ii/(Li.thickness*Wi));
	       fprintf(stderr, "      %-27s = %12.5e\n", "Width", Wi);
	    }

	    /* ---------------------------------------- honour minimum width */

	    Wi = MAX(Wi, Li.minlayerwidth);

	    /* ------------------------------- result = maximum width of all */

	    R.width = MAX(R.width, Wi);
	}

	/* -------------------------------------------- temperature increase */

	R.temp = dSum = 0; /* initialization */

	for (i=0; i<C.nL; i++) {
	    Li = T.layer[C.sL[i]]; /* define respective layer */

	    if (LOG) fprintf(stderr, "   %-30s : %s\n", "Layer", Li.name);

	    rhoT = rho(Li.rho, Li.tc, C.temp); /* temperature dependent rho */
	    Ii = C.current * (Li.thickness / rhoT) / invRsq;

	    J = Ii / (Li.thickness * R.width);

	    if (LOG) {
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Relaxed Width", R.width);
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Relaxed Current Density", J);
	       fprintf(stderr, "      %-27s = %12.5e\n",
	                       "Eff. Distance to Silicon", Li.dist2si - dSum);
	    }

	    Temp = (J*J) * rhoT * Li.thickness
	           * (Li.dist2si - dSum) / Li.lambda;

	    /* ----------------------------------------- result = sum of all */

	    R.temp += Temp;

	    /* ---- distinguish between planarized ond non-planarized layers */

	    if (T.type == PLANARIZED) {
	       dSum   += Li.thickness;
	    }
	}

	/* ------------------------------------- check for temperature alert */

	if (R.temp > Li.templimit)
	   R.templimit = Li.templimit;
	else
	   R.templimit = 0;

	return R;
}





/* ----------------------------------------------------------------------------
 * computeViaCurrent - compute maximum via current
 * ------------------------------------------------------------------------- */

double computeViaCurrent(TECData T, MIGRData C)
{
	double		t50, Jmax, Current, rhoT;
	LAYData		L;
	int		i, bot;

	if (LOG) {
	   fprintf(stderr, "----------------------------------------------------------\n");
	   fprintf(stderr, "VIA DESIGN: Current Calculation\n");
	   fprintf(stderr, "----------------------------------------------------------\n");
	}

	/* ----------------------------------------------- find bottom layer */

	bot = 0;

	for (i=1; i<C.nL; i++) {
	    if (T.layer[C.sL[i]].dist2si < T.layer[C.sL[bot]].dist2si)
	       bot = i;
	}

	L = T.layer[C.sL[bot]];

	if (LOG) fprintf(stderr, "      %-27s : %s\n", "Bottom Layer", L.name);

	/* --------------------------------------- calculate maximum current */

	if (L.material == POLY) { /* PolySilicon */
	   rhoT = rho(L.rho, L.tc, C.temp); /* temperature dependent rho */
	   Current = 2.0 * (C.xsize + C.ysize)
	             * sqrt(L.maxpower * L.thickness / rhoT);
	} else { /* Metal */
	   t50  = FailureRate2MTF(C.fit, C.lifetime, L.sigma);
	   Jmax = MaxCurrentDensity(L, t50, C);

	   Current = Jmax * 2.0 * (C.xsize + C.ysize) * L.thickness;
	   Current /= L.viacoefficient;
	}

	if (LOG) fprintf(stderr, "      %-27s = %12.5e\n", "Current", Current);

	return Current;
}
