//Program: trfunc.cc
//Written by Paul Landes (landes@jove.acs.unt.edu)
//on Wed 10/15/97 22:45:52

//Copyright (C) 1997 by Paul Landes. This code/program is free 
//software; you can redistribute it and/or modify it under the terms 
//of the GNU General Public License as published by the Free Software 
//Foundation; either version 2 of the License, or (at your option) 
//any later version. 

#include <trtab.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

void printIntro(FILE *std)
{
  fprintf(std, "
trtab %s (Paul Landes)
Truth table generator
Copyright (C) 1998 Free Software Foundation, Inc.
This is free software with ABSOLUTELY NO WARRANTY.
For details type `warranty'.
\n", VERSION);
}

void printWarranty()
{
  printf("
    trtab %s (Paul Landes)
    Truth table generator

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License , or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, write to the Free Software
    Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
\n", VERSION);
}

void printHelp()
{
  printf("
Valid commands are: 
  exit		Exit the program
  warranty	GNU public license blurb
  help		This useless message

If the above commands are not used, you must enter a boolean
equation of the truth table to be shown (i.e. a*b).  For more
information regarding boolean equations see the man page.
\n");
}

void printUsage(int ecode, char *msg, bool showIntro)
{
  if (msg) cerr << "trtab: " << msg << endl;
  if (showIntro == true) printIntro(stderr);
  cout << USAGE << endl;
  exit(ecode);
}

int doEquation(char *equationPtr, char *ops, const trtabOp *op)
{
  char *err;
  int intErr;
  TruthTable testTable(ops);

  intErr = testTable.prepareEquations(LONGEQU, equationPtr);

  if (intErr) {
    cerr << testTable.getErrorMsg() << endl;
    return 1;
  }

  testTable.prepareMatrices();
  cout << testTable.getTruthTable(op->width);
  err = testTable.getErrorMsg();

  if (err) {
    cerr << err << endl;
    delete [] err;
  }
  return 0;
}

void copyVar(char *cstring, string str)
{
  strcpy(cstring, str.data());
  cstring[str.length()] = 0;
}

int generateOpFile(char *filename)
{
  FILE *out;

  if ( (out = fopen(filename, "w+")) == NULL) return errno;
  fputs("implies=->
or=+
and=*
equals===
notequals=<>
not=!
width=79
prompt=trtab$ 
quiet=false
", out);

  fclose(out);
  return -1;
}

int readOptions(char *filename, trtabOp *op)
{
  FILE *in;
  char line[LINE_SIZE], *first, *second;
  string str;
  
  map< string, string, less<string> > vhash;

  vhash[ "prompt" ] = "trtab$ ";
  vhash[ "quiet" ] = "false";

  in = fopen(filename, "r");
  if (!in) return errno;

  while ( fgets(line, LINE_SIZE, in) != NULL) {
    first = strtok(line, "=");
    if (!first) {
      cerr << "Syntax found in file " << filename 
	   << ".  No equal found" << endl;
      return 1;
    }
    second = strtok(NULL, "\0");
    second[ strlen(second) - 1] = 0;
    vhash[ first ] = second;
  }

  str = vhash[ "implies" ] + "`" + vhash[ "or" ] + "`" + vhash[ "and" ]
    + "`" + vhash[ "equals" ] + "`"+ vhash[ "notequals" ] + "`"
    + vhash[ "not" ];

  copyVar(op->operators, str);
  copyVar(op->prompt, vhash["prompt"]);
  if (op->width == -1) {
    if (vhash[ "width" ].empty() == false) 
      op->width = atoi( vhash[ "width" ].c_str() );
    else op->width = DEFAULT_SCREEN_WIDTH;
  }
  if (vhash["quiet"] == "true") op->quiet = true;
  fclose(in);

  return -1;
}

char *setupFiles(trtabOp *op)
{
  struct stat buf;
  int er;
  char *usroperators;

  strcpy(op->prompt, "trtab$ ");

  if ( stat(op->dirN.c_str(), &buf) == -1 ) {
    cout << "directory '" << op->dirN 
	 << "' not found--creating..." << endl;
    if ( mkdir(op->dirN.c_str(), 0777) == -1) 
      printUsage(1, strerror(errno));
  }
    
  if ( (readOptions((char*)op->optionsFn.c_str(), op)) != -1) {
    cout << "file '" << op->optionsFn 
	 << "' not found, generating..." << endl;

    if ( (er = generateOpFile((char*)op->optionsFn.c_str())) != -1) {
      cerr << "can't generate file: " << strerror(er) << endl
	   << "using default operators." << endl;
    }
    else {
      if (op->quiet == false) cout << "loaded options file '" 
				   << op->optionsFn << "'" << endl;
    }//if (err = ... / else
    readOptions((char*)op->optionsFn.c_str(), op);
    usroperators = op->operators;
  }//if ( (read...
  else usroperators = op->operators;
  return usroperators;
}

trtabOp::trtabOp(char **argv):ZCmdop(),help(USAGE),ver(VERSION)
{
  string home, error;
  home = getenv("HOME");
  if (!home.length()) printUsage(1, "home directory not found");
  dirN = home + string("/") + DEFAULT_DIR;

  add('q', "quiet", TBOOL, &quiet);
  quiet = false;
  add('w', "width", TINT, &width);
  width = -1;
  add('e', "equation", TSTRING, &equation);
  add('b', "batchfile", TSTRING, &batchFn);
  add('d', "dir", TSTRING, &dirN);

  add('h', "help", TPRINTEXIT, &help);
  add('v', "version", TPRINTEXIT, &ver);

  if ( parse(argv, error) < 0 ) printUsage(1, (char*)error.c_str());

  if ( dirN[ dirN.length() - 1 ] != '/') dirN += "/";
  historyFn = dirN + DEFAULT_HIS;
  optionsFn = dirN + DEFAULT_OPT;
}

int executeFile(const char *filename, const char *usroperators,
		const trtabOp *op)
{
  FILE *in;
  char line[LINE_SIZE];
  bool good;

  in = fopen(filename, "r");
  if (!in) printUsage(1, strerror(errno), false);
  while ( fgets(line, LINE_SIZE, in) != NULL) {
    line[ strlen(line) - 1 ] = 0;
    good = false;
    for(size_t i = 0; i < strlen(line); i++) 
      if (line[i] != ' ') good = true;
    if (good == true) doEquation((char*)line, (char*)usroperators, op);
  }
  fclose(in);
  return 0;
}
