/*
------------------------------------------------------------
Header file "linkedl.h"  linked list objects
written by Paul Landes
10-29-96
This header file contains linked list objects.

NOTE:
You NEED a pointer called *next in your PUBLIC data
(class or struct) for these objects to work.

This object only works with classes and structs.

Unless otherwise stated at the beginning of each member function,
all functions return standard error status: 1 on fail, 0 on success.

Includes:
  erralloc.h
------------------------------------------------------------
*/

#ifndef _LINKEDL_H
#define _LINKEDL_H

#include <erralloc.h>

//________(abstract)_________class LinkedList_____________________________
template <class T>
class LinkedList
{
public:
  //initialize the node count to 0 and set head to NULL
  LinkedList() { head = 0; nodeCount = 0;}
  virtual ~LinkedList() {} //virtual for non-parent destructor invoking

  //standard insert/delete node functions
  virtual int insertNode(T *, int) = 0;   //insert delete are standard
  virtual int deleteNode(int) = 0;
  virtual int insertNodeAtEnd(T *) = 0;   //insert node at end
  virtual int getNode(T &, int) = 0;      //memberwise copy of node
  int getNodeCount() { return nodeCount; } //return number of nodes

protected:
  virtual T *getNodePtr(int) = 0;  //get pointer to node (limitied access)
  virtual void deleteAll(T *) = 0; //delete all nodes
  T *head;  //starting address of lists (root, head, etc...)
  int nodeCount;        //keeps track of number of nodes
};

//___________________________class LinearLinkedList_______________________
template <class T>
class LinearLinkedList : public LinkedList<T>
{
public:
  LinearLinkedList():LinkedList<T>() {}
  ~LinearLinkedList();

  int insertNode(T *, int);
  int deleteNode(int);
  int insertNodeAtEnd(T *);  
  int getNode(T &, int);

  //copy all nodes with overloaded =
  LinearLinkedList &operator=(LinearLinkedList<T> const &);

protected:
  T *getNodePtr(int);   
  void deleteAll(T *); 
  
  T *ptri[3];
  int inti[3];
};

template <class T>
LinearLinkedList<T>::~LinearLinkedList()
{
  //if the list was started, delete all nodes
  if (head) deleteAll(head);  
}

template <class T>
void LinearLinkedList<T>::deleteAll(T *nextNode)
{
  //recursive function deletes all nodes
  if (nextNode) deleteAll(nextNode->next);
  delete nextNode;
}

//Insert a new node in linked list, *newNode is a pointer to the
//node to be inserted, position is the index in list.
//return 1 on fail, 0 on success
template <class T>
int LinearLinkedList<T>::insertNode(T *newNode, int position)
{
  T *scanNode = head, *followNode;
  int i;

  if ( (!newNode) || (position < 0) ) return 1;	//exit fail

  //every new node's next pointer is set to null to indicate the end of list
  newNode->next = 0;

  //if this is the first node to be inserted, the *head pointer points to it
  if (!head)
    {
      head = newNode;
      nodeCount++;
      return 0;	//exit success
    }

  //if inserting node in front, update head and next pointers
  if (!position)
    {
      newNode->next = head;
      head = newNode;
      nodeCount++;
      return 0;
    }

  //scan out to nth node
  for (i = 0; i < position; i++)  
    {
      if (!scanNode) return 1;  //index is after last node, exit fail
      followNode = scanNode;
      scanNode = scanNode->next;
    }

  if (!scanNode) followNode->next = newNode; //if last node
  else
    {
      newNode->next = scanNode;	//middle node
      followNode->next = newNode;
    }

  nodeCount++;  //increment number of nodes inserted
  return 0;
}

//Inserts a node at the end of the list.  No index is needed, *newNode
//points to a node to be inserted in the list.
//return 1 on fail, 0 on success
template <class T>
int LinearLinkedList<T>::insertNodeAtEnd(T *newNode)
{
  int i;

  newNode->next = 0;

  //test for first node
  if (!head)
    {
      head = newNode;
      nodeCount++;
      return 0;	//exit success
    }

  i = getNodeCount();
  return insertNode(newNode, i);
}

//delete a node from list with index 'position'
//return 1 on success, 0 on fail
template <class T>
int LinearLinkedList<T>::deleteNode(int position)
{
  T *scanNode = head, *followNode;
  int i;

  //if list has not been started or the index passed is not valid, exit fail
  if ( (!head) || (position < 0) ) return 1;

  //delete first node
  if (!position)
    {
      //scanNode = head;  //should not need this since init abve
      head = head->next;  //update head to next node
      delete scanNode;    //delete the old head
      nodeCount--;        //update the number of nodes
      return 0;           //exit success
    }

  //scan out to nth position node
  for (i = 0; i < position; i++)  
    {
      followNode = scanNode;
      scanNode = scanNode->next;
      if (!scanNode) return 1;
    }

  if (!scanNode->next) //if last node, delete last node
    {
      followNode->next = 0;
      delete scanNode;
    }
  else  //if middle node, reconnect linkes, delete node
    {
      followNode->next = scanNode->next;
      delete scanNode;
    }

  nodeCount--;  //dec the node counter
  return 0;
}

//does a memberwise copy on the returnNodes on the 'position' indexed node
//note: this function will not work without overloaded=, does a memberwise
//on structs...so be VERY careful using this function!
template <class T>
int LinearLinkedList<T>::getNode(T &returnNode, int position)
{
  T *obtainNode;

  //get the node pointer
  obtainNode = getNodePtr(position);

  //if node not found, invalid position so return fail
  if (!obtainNode) return 1;

  //do the memberwise/overloaded= copy
  returnNode = *obtainNode;

  return 0;
}

//returns the pointer of a node at index 'position' to friends and children
//returns pointer set to NULL on error, the node pointer on success
template <class T>
T *LinearLinkedList<T>::getNodePtr(int position)
{
    T *scanNode = head;
    int i;

    //return fail if invalid index or if there is no list started
    if ((position > nodeCount) || (!head)) return 0;

    //scan out for the node
    for (i = 0; i < position; i++) scanNode = scanNode->next;

    return scanNode;
}

//overloaded = for copying two linked lists--copy 'this' from passed object
//returns the linked list that was just copied to this object
template <class T>
LinearLinkedList<T> &LinearLinkedList<T>::
  operator=(LinearLinkedList<T> const &copyFrom)
{
  T *follow, *newNode, *scanNode;

  //if our local data has nodes, delete them
  if (head)
    {
      deleteAll(head);
      head = 0;
    }

  //start at head of passed linked list
  scanNode = copyFrom.head;

  //do the following code if the passed object has data in it
  if (scanNode)
    {
      //this first chunk of code gets it started (ie making head)
      newObjectAlloc(newNode);  //new a new node
      *newNode = *scanNode;     //do a memberwise/overloaded= copy
      follow = head = newNode;  //our head is now set
      scanNode = scanNode->next;//increment position in list

      //do until the end of the list from the passed object
      while (scanNode != 0)
  	{
	  newObjectAlloc(newNode); //new a new node
	  *newNode = *scanNode;    //memberwise/overloaded copy
	  follow->next = newNode;  //update position pointers
	  follow = newNode;
	  scanNode = scanNode->next;
  	}
      //last node of this object data is NULL for end of list marker
      newNode->next = 0; 
    }

  nodeCount = copyFrom.nodeCount;

  return *this;
}

#endif
