/*	includes this header file needs:
			evalhelp.h
			varlist.h
			stack.h
			stringn.h
*/

#ifndef _EQUSOLVE_H
#define _EQUSOLVE_H

#include <evalhelp.h>
#include <varlist.h>
#include <stack.h>

const char orderDBool[] = "~`=!`&`|`@";
const char realDefOrDBool[] = "@`|`&`=`!`~";
const char usrDefOrDBool[] = "->`+`*`==`<>`!";

const char orderD[] = "~_`/*`+-`<>{}`=!`&`|";
const char realDefOrD[] = "|`&`=`!`{`}`<`>`-`+`*`/`_`~";
const char usrDefOrD[] = "||`&&`==`!=`<=`>=`<`>`-`+`*`/`_`!";

typedef struct orderDef orderDef;

struct orderDef
{
    char order[30];
    char real[50];
    char user[100];
};

//returns a 0 on success, 1 on fail
template <class T>
int extractVarConst(char *&expression, unsigned char *&varDiscriptsP,
	T *&floatDiscriptsP)
{
  int i = 0, countNum = 0, pos2;
  char *numericString;
  unsigned char nextVar;
  unsigned char *varDiscripts;
  T *floatDiscripts;
  Stack<char *> *longNums;
  Stack<unsigned char> *chars;

  newObjectAlloc(longNums);
  newObjectAlloc(chars);

  for (i = 0; expression[i] != 0; i++)
    {
      if ( checkNumeric(expression[i]) && checkNumeric(expression[i + 1]) )
	{
	  pos2 = i;
	  numericString = extractValChar(expression, pos2);

	  nextVar = 255 - countNum;//vars->getNextHighConstVar();
	  longNums->push(numericString);
	  chars->push(nextVar);

	  replaceBlock(expression, nextVar, i, pos2 + i - 1);

	  countNum++;
	}
    }

  if (countNum == 0) return 1;

  varDiscripts = new unsigned char[countNum + 1];
  floatDiscripts = new T[countNum + 1];

  for (i = 0; i < countNum; i++)
    {
      T tmp;
      chars->pop(varDiscripts[i]);
      longNums->pop(numericString);
      tmp = (T)strtod(numericString, 0);
      floatDiscripts[i] = tmp;
    }

  varDiscripts[countNum] = 0;

  delete longNums;
  delete chars;

  varDiscriptsP = varDiscripts;
  floatDiscriptsP = floatDiscripts;

  return 0;
}

//_____(abstract)____________class EquSolve_______________________________
template <class T>
class EquSolve
{
public:
  EquSolve(const char * = 0, VarList<T> * = 0, StringNode * = 0,
	   orderDef * = 0, const char * = 0);

  virtual ~EquSolve();

  void getOrgExp(StringNode &result) const
    { result = originalExp; }
  void getEquName(StringNode &result) const
    { result = equName; }

  int parseEquation();
  int solveEquation(T &) const;

  EquSolve<T> *next;

protected:
  virtual int calOperators(T &, const char, const T &, const T &) const = 0;

  char *expression;
  orderDef *order;
  StringNode *errorMsg, errorPrompt, originalExp, equName;

  VarList<T> *vars;  //make this a non-pointer later to prevent new/del?
  VarList<T> *varsConst;
  int disabled;
  char postfixDone;
};

template <class T>
EquSolve<T>::EquSolve(const char *beginString, VarList<T> *varsP,
	StringNode *errorMsgP, orderDef *orderP, const char *equNameP)
{
  expression = 0;

  if ( (!beginString) || (!varsP) || (!errorMsgP) || (!orderP) )
    {
      disabled = 1;
    }
  else
    {
      order = orderP;

      next = 0;
      vars = 0;

      postfixDone = 0;
      disabled = 0;

      expression = newStringCopy(beginString);
      originalExp = beginString;

      if (equNameP) equName = equNameP;
      else equName = beginString;

      newObjectAlloc(varsConst);

      vars = varsP;
      errorMsg = errorMsgP;

      errorPrompt << "Error in '" << equName << "': ";
    }
}

template <class T>
EquSolve<T>::~EquSolve()
{
	if (expression) delete [] expression;
	if (varsConst) delete varsConst;
}

//exit success = 1; exit failure = 0
template <class T>
int EquSolve<T>::parseEquation()
{
  char *err, *varLetters, *tmpStr;

  if (disabled == 1)
    fatalErr("Object disabled because constructor() called.");

  if (postfixDone)
    {
      *errorMsg << errorPrompt << "Equations are parsed already.\n";
      return 0;
    }

  *errorMsg = 0; //clear and set errorMsg to 0

  postfixDone = 0;
  removeWhiteSpace(expression);

  err = replaceOperators(expression, order->real, order->user);
  if (err != 0)
    {
      *errorMsg << errorPrompt << err;
      delete [] err;
      return 0;
    }

  tmpStr = originalExp.getString();
  err = preInfix(expression, tmpStr, order->order);
  delete [] tmpStr;

  if (err != 0) {
    *errorMsg << errorPrompt << err;
    delete [] err;
    return 0;
  }

  infix(expression, order->order);
  postfix(expression, order->order);
  postfixDone = 1;

  //------------------

  varLetters = extractVariables(expression);
  if (varLetters)
    vars->initialize((unsigned char *)varLetters, NONCONST);

  return 1;
}

//return 0 on fail, 1 on success
template <class T>
int EquSolve<T>::solveEquation(T &result) const
{
    Stack<T> stack;
    int i, pos = 0;

    if (disabled == 1)
	fatalErr("Object disabled because constructor() called.");
    if (!postfixDone) return 1;

    while (expression[pos] != 0)
    {
	i = findChar(expression, order->order, pos);

	if (i == 0)
	{
	    T num, result;
	    stack.pop(num);

	    calOperators(result, expression[pos], num, 0);
	    stack.push(result);
	}

	else if (i > 0)   //operator
	{
	    T num1, num2, result;

	    stack.pop(num2);
	    stack.pop(num1);

	    calOperators(result, expression[pos], num1, num2);
	    stack.push(result);
	}
	else //not an operator
	{
	    if ( checkNumeric(expression[pos]) )  //1 digit interger
	    {
		T tmp = (int)(expression[pos] - '0');
		stack.push( (T)tmp );  //interger
	    }
	    else	//operand (letter)
	    {
		T tmp;

		if ( checkAlpha(expression[pos]) )		//if is normal letter var
		    vars->getValue(expression[pos], tmp);
		else	//if is const type
		    varsConst->getValue(expression[pos], tmp);

		stack.push(tmp);  //operand (letter)
	    }
	}
	pos++;
    }

    if (stack.pop(result) == 0)
	fatalErr ("Push/Pop parse error in T eval().");

    return 0;

}

//___________________________class EquSolveBool____________________________
template <class T>
class EquSolveBool : public EquSolve<T>
{
	public:
	EquSolveBool(const char *beginStringP = 0, VarList<T> *varListP = 0,
		StringNode *errorMsgP = 0, orderDef *orderP = 0, const char *equNameP = 0)
		:EquSolve<T>(beginStringP, varListP, errorMsgP, orderP, equNameP) {}

	~EquSolveBool() {};

  EquSolveBool<T> *next;

	protected:
	virtual int calOperators(T &, const char, const T &, const T &) const;
	int calBoolOperators(T &, const char, const T &, const T &) const;
};

//returns 1 on fail, 0 on success
template <class T>
int EquSolveBool<T>::calOperators
	(T &result, const char op, const T &num1, const T &num2) const
{
	return calBoolOperators(result, op, num1, num2);
}

//returns 1 on fail, 0 on success
template <class T>
int EquSolveBool<T>::calBoolOperators
	(T &result, const char op, const T &num1, const T &num2) const
{
    switch (op)
    {
    case '~':		//unirary
	result = !num1;
	break;

    case '<':
	if (num1 < num2) result = 1;
	else result = 0;
	break;

    case '>':
	if (num1 > num2) result = 1;
	else result = 0;
	break;

    case '}':
	if (num1 >= num2) result = 1;
	else result = 0;
	break;

    case '{':
	if (num1 <= num2) result = 1;
	else result = 0;
	break;

    case '=':
	if (num1 == num2) result = 1;
	else result = 0;
	break;

    case '!':
	if (num1 != num2) result = 1;
	else result = 0;
	break;

    case '&':
	if (num1 && num2) result = 1;
	else result = 0;
	break;

    case '|':
	if (num1 || num2) result = 1;
	else result = 0;
	break;

    case '@':
	if (num1 && !num2) result = 0;
	else result = 1;
	break;

    default:
	return 1;
    }

    return 0; //exit success
}

//___________________________class EquSolveArith____________________________
template <class T>
class EquSolveArith : public EquSolveBool<T>
{
public:
    EquSolveArith(const char * = 0, VarList<T> * = 0, VarList<T> * = 0,
		  StringNode * = 0, orderDef * = 0, const char * = 0);

    ~EquSolveArith() {};
    EquSolveArith<T> *next;

protected:
    virtual int calOperators(T &, const char, const T &, const T &) const;
    int calArithOperators(T &, const char, const T &, const T &) const;
};

template <class T>
EquSolveArith<T>::EquSolveArith(const char *beginStringP, VarList<T>
	*varListP, VarList<T> *varsConstP, StringNode *errorMsgP,
	orderDef *orderP, const char *equNameP)
	:EquSolveBool<T>(beginStringP, varListP, errorMsgP, orderP, equNameP)
{
    varsConst = varsConstP;
}

//returns 1 on fail, 0 on success
template <class T>
int EquSolveArith<T>::calOperators
	(T &result, const char op, const T &num1, const T &num2) const
{
    int err;
    err = calArithOperators(result, op, num1, num2);
    if (err) err = EquSolveBool<T>::calOperators(result, op, num1, num2);
    return err;
}

//returns 1 on fail, 0 on success
template <class T>
int EquSolveArith<T>::calArithOperators
	(T &result, const char op, const T &num1, const T &num2) const
{
    switch (op)
    {
    case '_':
	result = -num1;
	break;

    case '+':
	result = num1 + num2;
	break;

    case '-':
	result = num1 - num2;
	break;

    case '*':
	result = num1 * num2;
	break;

    case '/':
    	if (!num2) *errorMsg << "Divide by 0\n";
	else result = num1 / num2;
	break;

    default:
	return 1;
    }

    return 0;  //exit success
}

//___________________________Container Classes_____________________________


//__________(abstract)_______class SolveBase_______________________________
template <class T>
class SolveBase
{
public:
    SolveBase();
    virtual ~SolveBase();

    int errorExists()
    { return errorMsg->stringExists(); }
    void getErrorMsg(StringNode &result) const
    { result = *errorMsg; }

    int getVariableValue(const char letter, T &value) const
    { return vars->getValue(letter, value); }

    int setValue(const char, const T &);
    int setValue(const char *, const T *);

protected:
    StringNode *errorMsg;
    VarList<T> *vars;
    orderDef order;
};


template <class T>
SolveBase<T>::SolveBase()
{
    newObjectAlloc(vars);
    newObjectAlloc(errorMsg);
}

template <class T>
SolveBase<T>::~SolveBase()
{
    delete errorMsg;
    delete vars;
}

template <class T>
int SolveBase<T>::setValue(const char letter, const T &value)
{
    int err;
    err = vars->setValue(letter, value);

    if (err)
	*errorMsg << "Variable '" << letter << "' not found.\n";

    return err;
}

template <class T>
int SolveBase<T>::setValue(const char *letter, const T *value)
{
    int err;
    err = vars->setValue(letter, value);

    if (err) *errorMsg << "One of the variables '"
		       << letter << "' not found.\n";

    return err;
}

//_________(abstract)________class SingleEquSolve____________________________
template <class T>
class SingleEquSolve : public SolveBase<T>
{
public:
    SingleEquSolve(void):SolveBase<T>() {}
    ~SingleEquSolve() {}

    int solveEquation(T &);
    T *solveEquation();

    void getEquName(StringNode &result) { equ->getEquName(result); }
    void getOrgExp(StringNode &result) { equ->getOrgExp(result); }

protected:
    EquSolve<T> *equ;
    T *answer;
};

//return 0 on fail, 1 on success
template <class T>
int SingleEquSolve<T>::solveEquation(T &result)
{
    return equ->solveEquation(result);
}

//return 0 on fail, result on success
template <class T>
T *SingleEquSolve<T>::solveEquation()
{
    int err;
    T *result;
    newObjectAlloc(result);

    err = equ->solveEquation(*result);
    if (err) return result;
    else delete result;

    return 0;
}

//___________________________class BoolSolve_______________________________
template <class T>
class BoolSolve : public SingleEquSolve<T>
{
public:
    BoolSolve(const char *, const char * = 0, const orderDef * = 0);
    ~BoolSolve() { delete equ; }
};

template <class T>
BoolSolve<T>::BoolSolve(const char *beginStringP, const char *equNameP,
	const orderDef *orderP):SingleEquSolve<T>()
{
    if (!orderP)
    {
	strcpy(order.order, orderDBool);
	strcpy(order.real, realDefOrDBool);
	strcpy(order.user, usrDefOrDBool);
    }
    else order = *orderP;

    if (!(equ = new EquSolveBool<T>(beginStringP, vars,
				    errorMsg, &order, equNameP) )) fatalErr();

    equ->parseEquation();
}

//___________________________class ArithSolve_______________________________
template <class T>
class ArithSolve : public SingleEquSolve<T>
{
public:
    ArithSolve(const char *, const char * = 0, orderDef * = 0);
    ~ArithSolve();

protected:
    VarList<T> *varsConst;
};

template <class T>
ArithSolve<T>::ArithSolve(const char *beginStringP, const char *equNameP,
	orderDef *orderP):SingleEquSolve<T>()
{
    unsigned char *varDiscripts = 0;
    T *floatDiscripts = 0;
    char *manipulate;

    if (!orderP)
    {
	strcpy(order.order, orderD);
	strcpy(order.real, realDefOrD);
	strcpy(order.user, usrDefOrD);
    }
    else order = *orderP;

    manipulate = newStringCopy(beginStringP);
    newObjectAlloc(varsConst);

    if ( !extractVarConst(manipulate, varDiscripts, floatDiscripts) )
	varsConst->initialize(varDiscripts, CONST, floatDiscripts);

    if (!(equ = new EquSolveArith<T>(manipulate, vars, varsConst, errorMsg,
				     &order, equNameP) )) fatalErr();

    equ->parseEquation();

    if (varDiscripts) delete [] varDiscripts;
    if (floatDiscripts) delete [] floatDiscripts;
}

template <class T>
ArithSolve<T>::~ArithSolve()
{
    delete equ;
}

#endif
