//Program: cmdop.cc
//Written by Paul Landes <landes@unt.edu>
//on Sat 5/16/98 21:15:33

//Copyright (C) 1998 by Paul Landes. This code/program is free 
//software; you can redistribute it and/or modify it under the terms 
//of the GNU General Public License as published by the Free Software 
//Foundation; either version 2 of the License, or (at your option) 
//any later version. 

#include <cmdop.h>

//add a command line option entry.  shortop is the one letter option
//(must have), long op is the corresponding long option (not
//necessary), type is the kind of variable of data, data is the
//variable that will be filled in.
void ZCmdop::add(const char shortop, const string &longop, 
		 const type_t type, const void *data)
{
  CmdNode a;
  a.shortop = shortop;
  a.longop = longop;
  a.type =  type;
  a.data = (void*)data;
  mCmdVec.push_back(a);
}

//Get the command line node with letter as the short option
ZCmdop::CmdNode *ZCmdop::getIdx(const char letter)
{
  for(cmdvector::iterator i = mCmdVec.begin();
      i != mCmdVec.end(); i++)
    if ( (*i).shortop == letter ) return i;
  return NULL;
}

//Get the command line node with the string as a long option
ZCmdop::CmdNode *ZCmdop::getIdx(const char *str)
{
  for(cmdvector::iterator i = mCmdVec.begin();
      i != mCmdVec.end(); i++)
    if ( (*i).longop == str ) return i;
  return NULL;
}

//assign the data in the command line node to the operand op
void ZCmdop::assign(CmdNode *node, const char *op)
{
  switch(node->type) {
  case TINT: *(int*)node->data = atoi(op); break;
  case TDOUBLE: *(double*)node->data = strtod(op, NULL); break;
  case TSTRING: *(string*)node->data = op; break;
  case TBOOL: *(bool*)node->data = !*(bool*)node->data; break;

  //this is real handy for version or help messages where you just
  //print out a message and exit the program.  To use this, just put a
  //string in the inherited object that has the message
  case TPRINTEXIT: cout << *(string*)node->data << endl; exit(0);
  }
}

//parse every command string passed and assign the values given on the
//command line to the values in the command line nodes
int ZCmdop::parse(char **argv, string &error)
{
  CmdNode *node;
  string param;

  //do while there are no more strings in the string array
  while(argv[0]) {
    //the first char of the command is a dash (short op)
    if (argv[0][0] == '-') {

      //the second char of the command is dash (long op)
      if (argv[0][1] == '-') {
	//look for an equal sign.  If one is found, then parse it out
	//and get the command line node with the thing that is before
	//the equals and use the thing afterward as the command line
	//option.
	string op = argv[0] + 2;
	size_t pos;
	if ( (pos = op.find("=")) != (string::size_type)-1 ) {
	  param = op.substr(pos + 1, op.length());
	  op.erase(pos, op.length() - pos);
	}
	node = getIdx(op.c_str());
      }
      else {
	//if it is a short op (one dash) get the command node by
	//passing just the letter of the option
	if (argv[0][2] != 0) {
	  error = "Expecting a single letter option with token: " 
	    + string(argv[0]);
	  return -1;
	}
	node = getIdx(argv[0][1]);
      }

      //if we did not get a node, then this is a bad option
      if (!node) {
	error = "Invalid option `" + string(argv[0]) + "'.";
	return -1;
      }
      //make sure that we get an option at the end of the command line
      if ( !argv[1] && node->type != TBOOL && 
	   node->type != TPRINTEXIT) {
	error = "Expected parameter after `" + string(argv[0]) + "'.";
	return -1;
      }

      //if param is set, then this is a long option with an eqauls and
      //the value is stored in param, so use it for the value
      if (param.empty() == true) assign(node, argv[1]);
      else {
	assign(node, param.c_str());
	param = "";
      }
    }
    argv++;
  }
  return 0;
}
