//Filename: interface.cpp
//Purpose: A couple of routines used to draw the ncurses 
//  interface for ppc.
//$Id: interface.c,v 1.6 1997/07/29 21:12:18 rkirkpat Exp $

/*
 * Copyright (c) 1996-1997 by Ryan Kirkpatrick (rkirkpat@nag.cs.colorado.edu). 
 *   All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "interface.h"

void InitScreen()
//Init the screen.
{
	initscr();
	cbreak();
	refresh();

	nodelay(stdscr,TRUE);
	erase();

	refresh();
}

void Redraw(unsigned short base)
//Draw the screen.
{
	char *str;
	char *str2;

	//Create author/version string
	str = (char*)calloc(22+strlen(VERSION),sizeof(char));
	strcpy(str,"By Ryan Kirkpatrick, ");
	strcat(str,VERSION);

	//Create port string.
	str2 = (char*)calloc(30,sizeof(char));
	strcpy(str2,"Port: /dev/lp");
	switch(base) 
	{
		case 0x3bc: strcat(str2,"0"); break;
		case 0x378: strcat(str2,"1"); break;
		case 0x278: strcat(str2,"2"); break;
	}
	strcat(str2,"  Address: 0x");
	switch(base) 
	{
		case 0x3bc: strcat(str2,"3bc"); break;
		case 0x378: strcat(str2,"378"); break;
		case 0x278: strcat(str2,"278"); break;
	}
	
	erase();
	
	mvaddstr(0,30,"Parallel Port Control");
	mvaddstr(1,(80-strlen(str))/2,str);
	mvaddstr(2,(80-strlen(DATE))/2,DATE);
	mvaddstr(3,25,str2);
	free(str);
	free(str2);

	mvaddstr(5,0,"      Status Bits                 Data Bits                 Control Bits");
	mvaddstr(6,0,"Bit|    Name     |State       Bit|  Name  |State      Bit|  Name        |State");
	mvaddstr(7,0,"-----------------------       ------------------      ------------------------");
	mvaddstr(8,0," 0 | Reserved    |  ?          0 | Data 0 |  ?         0 | NOT Strobe   |  ?");
	mvaddstr(9,0," 1 | Reserved    |  ?          1 | Data 1 |  ?         1 | Auto Feed    |  ?");
	mvaddstr(10,0," 2 | IRQ Status  |  ?          2 | Data 2 |  ?         2 | NOT Init     |  ?");
	mvaddstr(11,0," 3 | Error       |  ?          3 | Data 3 |  ?         3 | Select Input |  ?");
	mvaddstr(12,0," 4 | Select      |  ?          4 | Data 4 |  ?         4 | PPort IRQ    |  ?");
	mvaddstr(13,0," 5 | Paper End   |  ?          5 | Data 5 |  ?         5 | Ext Mode Dir |  ?");
	mvaddstr(14,0," 6 | Acknowledge |  ?          6 | Data 6 |  ?         6 | Reserved     |  ?");
	mvaddstr(15,0," 7 | NOT Busy    |  ?          7 | Data 7 |  ?         7 | Reserved     |  ?");

	refresh();

	RedrawMenu();
}

void RedrawMenu()
//POST: Everything below the 14th row has been erased from the screen and
//        the menu has been redrawn.
{
	//Clear bottem part of screen.
	move(16,0);
	clrtobot();

	mvaddstr(17,38,"Menu");
	mvaddstr(18,14,"1. Read Status Bits             4. Toggle Control Bits");
	mvaddstr(19,14,"2. Read Data Bits               5. Help");
	mvaddstr(20,14,"3. Write Data Bits              6. Quit");
	mvaddstr(21,36,"Choice:");

	refresh();
}

void UpdateStatus (int data)
//PRE: data contains a integer less than 255, that describes the state of
//      the status port.
//POST: The on-screen display for the status bits has been updated.
{
	//Convert decimal to binary
	char *str = dec2bin(data);

	//Update the screen.
	int row = 8;
	int i;
	for (i = 7; i >= 0; i--)
		mvaddch(row++,20,str[i]);

	refresh();
}


void UpdateData (int data)
//PRE: data contains a integer less than 255, that describes the state of
//      the data port.
//POST: The on-screen display for the data bits has been updated.
{
	//Convert decimal to binary
	char *str = dec2bin(data);

	//Update the screen.
	int row = 8;
	int i;
	for (i = 7; i >= 0; i--)
		mvaddch(row++,45,str[i]);

	refresh();
}

void UpdateControl (int data)
//PRE: data contains a integer less than 255, that describes the state of
//      the control port.
//POST: The onscreen display for the control bits has been updated.
{
	//Convert decimal to binary
	char *str = dec2bin(data);

	//Update the screen.
	int row = 8;
	int i;
	for (i = 7; i >= 0; i--)
		mvaddch(row++,75,str[i]);

	refresh();
}

void UserHelp()
//POST: User help has been displayed.
{
	//Disable non-blocking mode for now.
	nodelay(stdscr,FALSE);

	//Clear the bottem part of the screen
	move(16,0);
	clrtobot();
	
	//Print first part of help.
	mvaddstr(17,0,"Read Status Bits: Updates the on screen display of control bits from the");
	mvaddstr(18,0,"                    port registers.");
	mvaddstr(19,0,"Read Data Bits:   Same as above, only updates the data bits instead.");
	mvaddstr(20,0,"Write Data Bits:  Allows the setting of the data pins of the parallel port");
	mvaddstr(21,0,"                    via binary, decimal, or binary coded decimal input.");
	mvaddstr(22,0,"                    Updates on screen display of data bits automatically.");

	//Wait for prompt to continue.
	mvaddstr(23,0,"Press any key to continue.");
	getch();
	
	//Clear the bottem part of the screen
	move(16,0);
	clrtobot();

	//Print second part of help.
	mvaddstr(17,0,"Toggle Control Bits: Allows the toggling of one of the control");
	mvaddstr(18,0,"                       bits. Updates on screen display of control bits");
	mvaddstr(19,0,"                       automatically.");
	mvaddstr(20,0,"Help:                This information!");
	mvaddstr(21,0,"Quit:                Exit the program.");

	//Wait for prompt to continue.
	mvaddstr(23,0,"Press any key to continue.");
	getch();	
	
	//Re-enable non-blocking mode.
	nodelay(stdscr,TRUE);
}

//EOF
