/*************************************************************************

    progutil - utility to program PIC Micros with picprog

    Version 0.1 			   

    File: programmer.cpp -- Programmer class
    Begin: 11. Nov 1999

    Copyright (C) 1999, 2000 by Raffael Stocker
    E-mail: raffael.stocker@stud.fh-deggendorf.de

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
    02111-1307, USA.

*************************************************************************/

#include <string>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <cstring>
#include <unistd.h>
#include "programmer.h"
#include "picprog.h"
#include "progutil.h"

Programmer::Programmer (string devicename)
{
  device = devicename;
  
  fd = open (device.c_str(), O_RDWR|O_SYNC);
  if (fd == -1)
    {
      throw ProgrammerFailure ("Couldn't open device");
    }
  dmmem = 0;
  cfgmem = 0;
  prgmem = 0;
}

Programmer::~Programmer ()
{
  if (fd) close (fd);
  delete [] prgmem;
  delete [] cfgmem;
  delete [] dmmem;
}

void
Programmer::erase (int what)
{
  if (!what) return;

  if (what & DATA)
    {
      if (ioctl (fd, PICPROG_BULKERDM) == -1)
	throw ProgrammerFailure ("Couldn't erase data-memory");
    }
  if (what & PROGRAM)
    {
      if (ioctl (fd, PICPROG_BULKERPM) == -1)
	throw ProgrammerFailure ("Couldn't erase program-memory");
    }
}

void
Programmer::disablecp ()
{
  if (ioctl (fd, PICPROG_DISPROT) == -1)
    throw ProgrammerFailure ("Couldn't disable code-protection");
}

void
Programmer::program (int what)
{
  if (!what) return;
  
  if (prgmem && (what & PROGRAM))
    {
      if (write (fd, (const void*)prgmem, PRGSIZE) == -1)
	throw ProgrammerFailure ("Couldn't write to program-memory");
    }
  if (dmmem && (what & DATA))
    {
      if (ioctl (fd, PICPROG_LOAD_DM, (unsigned long)dmmem) == -1)
	throw ProgrammerFailure ("Couldn't write to data-memory");
    }
  if (cfgmem && (what & CONFIG))
    {
      if (ioctl (fd, PICPROG_LOAD_CONFIG, (unsigned long)cfgmem) == -1)
	throw ProgrammerFailure ("Couldn't write to config-memory");
    }
}
      
void	  
Programmer::read (int what)
{
  if (!what) return;
  
  if (what & PROGRAM)
    {
      delete [] prgmem;
      try
	{
	  prgmem = new char[PRGSIZE];
	}
      catch (bad_alloc &e)
	{
	  cerr << ERRSTR << "Memory exhausted" << endl;
	  exit (1);
	}
      if (::read (fd, (void*)prgmem, PRGSIZE) == -1) 
	throw ProgrammerFailure ("Couldn't read program-memory");
    }
  if (what & DATA)
    {
      delete [] dmmem;
      try
	{
	  dmmem = new char[DATSIZE];
	}
      catch (bad_alloc &e)
	{
	  cerr << ERRSTR << "Memory exhausted" << endl;
	  exit (1);
	}
      if (ioctl (fd, PICPROG_READ_DM, (unsigned long)dmmem) == -1)
	throw ProgrammerFailure ("Couldn't read data-memory");
    }
  if (what & CONFIG)
    {
      delete [] cfgmem;
      try
	{
	  cfgmem = new char[CFGSIZE];
	}
      catch (bad_alloc &e)
	{
	  cerr << ERRSTR << "Memory exhausted" << endl;
	  exit (1);
	}
      if (ioctl (fd, PICPROG_READ_CONFIG, (unsigned long)cfgmem) == -1)
	throw ProgrammerFailure ("Couldn't read config-memory");
    }
}

void
Programmer::setprgdata (const char *data)
{
  prgmem = copydata (prgmem, data, PRGSIZE);
}

void
Programmer::setdmdata (const char *data)
{
  dmmem = copydata (dmmem, data, DATSIZE);
}

void
Programmer::setcfgdata (const char *data)
{
  cfgmem = copydata (cfgmem, data, CFGSIZE);
}


