/* 
    picprog - a development programmer driver
   
    Copyright (C) 1999, 2000 raffael.stocker@stud.fh-deggendorf.de (Raffael Stocker)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


    THIS CODE WAS DEBUGGED USING MY SELF-DEVELOPED EYE-AND-BRAIN (TM) DEBUGGER.
    SO IF THIS DOESN'T WORK ON YOUR MACHINE, YOU KNOW YOU HAVE TO BUY BETTER
    HARDWARE.

    13. Jul 1999   -   started coding
    22. Jul 1999  0.1  initial release
    25. Jan 2000  0.1  minor cleanups, added modversions
    19. Feb 2000  0.2  new release

*/

#include <linux/config.h>
#ifdef CONFIG_MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/kernel.h>
#include <linux/sched.h> 
#include <linux/module.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h> 
#include <linux/major.h> 
#include <linux/delay.h>
#include <asm/system.h> 
#include "picprog.h"



int io[4] = {0,0,0,0};
int picprog_major = 0;
MODULE_PARM( io, "1-4i" );
MODULE_PARM_DESC( io, "Portaddress of parallel ports to use (default=0x378)" );
MODULE_PARM( picprog_major, "i" );
MODULE_PARM_DESC( picprog_major, "Major number for device (default=dynamic)" );

MODULE_AUTHOR( "Raffael Stocker" );
MODULE_DESCRIPTION( "Device-Driver for PIC16X84 Development Programmer, " PICPROG_VERSION );


static int picp_nr_devs = 4;
static int num_dev;
static picp_t picp_devices[4];
static int used[4];
static struct timer_list wr_timer[4];
static int ports[4] = { 0x378, 0x278, 0x2bc, 0x3bc }; 


/* PICPROG - prototypes */

ssize_t picprog_read( struct file *, char *,
		      size_t, loff_t * );

ssize_t picprog_write( struct file *, const char *, 
		       size_t, loff_t * ); 

int picprog_open( struct inode *,
		  struct file * );

int picprog_release( struct inode *,
		     struct file * );

int picprog_ioctl( struct inode *,
		   struct file *,
		   unsigned int, unsigned long );

struct file_operations picprog_fops = {
  NULL,                  /* llseek */
  picprog_read,
  picprog_write,         
  NULL,                  /* readdir */
  NULL,                  /* poll */
  picprog_ioctl,
  NULL,                  /* mmap */
  picprog_open,
  NULL,                  /* flush */
  picprog_release
};


int comm_pic( int , int *, picp_t * );
int read_pmem( picp_t * );
void write_pmem( unsigned long );
void write_dmem( unsigned long );
void write_config( unsigned long );
void setup10( void (*)(unsigned long), picp_t * );
void do_bulk_er( picp_t * );

int picprog_init() 
{
  int i;
  int result;
  
  num_dev = 0;

  /* find number of port supplied with arguments */
  for ( i = 0; i < 4; i++ )
    { 
    if ( io[i] != 0 ) 
      {
	num_dev++;
      }
    }

  /* if no ports given, set default to 0x378 */
  if ( num_dev == 0 ) 
    {
      PICPROG_PRINT("No ports set! Assuming 0x378 as only port!\n");
      io[0] = 0x378;
      num_dev = 1;
    }  

  /* check if supplied ports are free, decrement num_dev if not */
  i = 0;
  while ( num_dev != 0 && io[i] != 0 ) 
    {
      if ( check_region( io[i], 3 ) != 0 ) 
	{
	  PICPROG_PRINT("IO 0x%X in use\n", io[i] );
	  num_dev--;
	  io[i] = 0;
	}
      i++;
    }
  
  /* if no ports are left, try to give up ;-) */
  if ( num_dev == 0) 
    {
      PICPROG_PRINT("No usable port found. Giving up!\n");
      return -ENODEV;
    }
  
  /* request io-spaces for all ports given */
  for ( i = 0; i < 4; i++) 
    {
      if ( io[i] != 0) 
	{
	  request_region( io[i], 3, PICPROG_NAME );
	}
    }

  /* set default major (in picprog.h) if not supplied */
  if (picprog_major == 0) picprog_major = PICPROG_MAJOR;

  /* register module at given major */
  if ( (result = register_chrdev( picprog_major, PICPROG_NAME, &picprog_fops )) < 0) 
    {
      PICPROG_PRINT( "unable to get major %d\n", picprog_major);
      return result;
    }

  /* if we allocated major number dynamically, remember the real major we got */
  if (picprog_major == 0) picprog_major = result; 
  
  PDEBUG("registered with major %d\n", picprog_major);
  
  PDEBUG("Initialisation was succesful!\n");
  
  return 0;
}

int picprog_open(struct inode *inode, struct file *filp) 
{
  picp_t *dev;     
  int i = 0;

  /* determine minor and run through initialisation procedure */
  int num = MINOR(inode->i_rdev);

  /* don't handle higher minors as devices supported */
  if ( (num >= picp_nr_devs) || (num_dev == 0) ) return -ENODEV;  

  /* minor already in use? */
  if ( used[num] == 1 ) return -EBUSY;

  /* find port associated with minor */
  for ( i = 0; i < 4; i++ ) 
    {
      if ( io[i] == ports[num] ) break;
    }
  
  /* If requested port is not there, return */

  if ( io[i] != ports[num] ) return -ENODEV;

  used[num] = 1;

  /* allocate dev-struct and initialize it */
  dev = &picp_devices[num];

  
  dev->port = io[i];
  /* io[i] = 0; */
  num_dev--;
  dev->num = num;
  
  /* this memory page will contain programming data */
  dev->data = (void *)__get_free_page( GFP_KERNEL );

  if ( !dev->data ) return -ENOMEM;

  /* initialize pointers to memory */
  dev->wp = dev->rp = dev->data;
  dev->end = dev->data +  2 * PICMEMSIZE + 2;

  filp->private_data = dev;

  /* initialize programmer */
  comm_pic( PROGRAM_MODE, 0, dev );
  
  MOD_INC_USE_COUNT;
  return (0);
}


int picprog_release( struct inode *inode, struct file *filp ) 
{
  picp_t *dev = filp->private_data;

  /* switch back to open drain outputs */
  comm_pic( RUN_MODE, 0, dev );

  /* free port, increment device number and set minor-usage to 0 */
  /*  for ( i = 0; i < 4; i++ )  */
  /*      { */
  /*        if ( io[i] == 0 )  */
  /*  	{ */
  /*  	  io[i] = dev->port; */
  num_dev++;
  used[dev->num] = 0;
  /*  	  break; */
  /*  	} */
  /*      } */
  
  /* free data memory */
  if ( dev->data )
    {
      free_page( (unsigned long)(dev->data) );
    }

  MOD_DEC_USE_COUNT;
  return 0;
}

ssize_t picprog_read( struct file *filp,
		      char *buffer, size_t count, loff_t *ppos ) 
{

  picp_t *dev = filp->private_data;

  /* read all? return 0 */
  if ( dev->rp == dev->end )
    {
      dev->rp = dev->data;
      return 0;
    }

  if ( (read_pmem(dev) < 0) ) return -EIO;

  /* same as in picprog_write(), but other direction */
  count = min( count, dev->end - dev->rp );
  copy_to_user( buffer, dev->rp, count );
  dev->rp += count;
  return count;
}

ssize_t picprog_write( struct file *filp,
		       const char *buffer, size_t count, loff_t *ppos )
{

  picp_t *dev = filp->private_data;
  dev->wr_queue = NULL;

  /* we copy all the data we get as long as we have enough space
     for it in the chip's program memory */
  count = min( count, dev->end - dev->wp );
  copy_from_user( dev->wp, buffer, count );
  dev->wp += count;

  if ( dev->wp == dev->end )
    {
      /* if all data is read from userspace, begin programming (blocking) */
      dev->progstate = 0;
      setup10( write_pmem, dev );
      interruptible_sleep_on(&dev->wr_queue);

      /* did we get an error? */
      if ( dev->progstate == -1 )
	{
	  PDEBUG( "temp: 0x%X\n", dev->temp );
	  return -EIO;
	}
    }
  return count;
}

int picprog_ioctl( struct inode *inode, struct file *filp,
		   unsigned int cmd, unsigned long arg ) 
{

  picp_t *dev = filp->private_data;
  int curaddr = 0;
  int data;      
  char *buffer = dev->end + 1; 
  int retval = -EINVAL, size = _IOC_SIZE( cmd );

  /* if we are writing to userspace or reading from it,
     we first check the permissions. That allows for 
     use of the nice double-underscore macros... */
  if ( _IOC_DIR(cmd) & _IOC_READ ) 
    {
      if ( !access_ok( VERIFY_WRITE, (void *)arg, size ) )
	return -EFAULT;
    }
  else if ( _IOC_DIR(cmd) & _IOC_WRITE )
    {
      if ( !access_ok( VERIFY_READ, (void *)arg, size ) )
	return -EFAULT;
    }
  
  if ( _IOC_TYPE( cmd ) != PICPROG_MAGIC ) return -EINVAL;
  if ( _IOC_NR( cmd ) > PICPROG_IOCMAX ) return -EINVAL;

  /* isn't this just pretty self-explanatory? */
  switch ( cmd ) 
    {
#ifdef PICPROG_DEBUG
    case PICPROG_HARDRESET:
      while ( GET_USE_COUNT (&__this_module ) > 1 )
	MOD_DEC_USE_COUNT;
      break;
#endif /* PICPROG_DEBUG */
    case PICPROG_BULKERPM:
      {
	do_bulk_er( dev );
	interruptible_sleep_on_timeout(&dev->wr_queue, HZ/50 );
	retval = 0;
	break;
      }
    case PICPROG_BULKERDM:
      {
	/* no comment... */
	comm_pic( PROGRAM_MODE, 0, dev );
	data = 0x3FFF;
	comm_pic( LOAD_DATA_DM, &data, dev ); 
	comm_pic( BEGIN_PROG, 0, dev );
	comm_pic( BULK_ER_DM, 0, dev );
	comm_pic( BEGIN_PROG, 0, dev );
	interruptible_sleep_on_timeout( &dev->wr_queue, HZ/50 );
	retval = 0;
	break;
      }
    case PICPROG_READ_DM:
      {
	comm_pic( PROGRAM_MODE, 0, dev );
	for ( curaddr = 0; curaddr <= PICDATASIZE; curaddr++ )
	  {
	    comm_pic( READ_DATA_DM, &data, dev );
	    *buffer = (char)data & 0xff;
	    PDEBUG( "*buffer: 0x%X, data: 0x%X\n", *buffer, data );
	    buffer++;
	    comm_pic( INC_ADDR, 0, dev );
	  } 
	*buffer = '\0';
	buffer = dev->end + 1;
	retval = __copy_to_user( (char*)arg, buffer, PICDATASIZE + 2 );
	break;
      }
    case PICPROG_LOAD_DM:
      {
	retval = __copy_from_user( dev->data, (char*)arg, PICDATASIZE + 2 );
	dev->wr_queue = NULL;
	dev->wp = dev->data;
	dev->end = dev->wp + PICDATASIZE + 1;
	setup10( write_dmem, dev );
	interruptible_sleep_on( &dev->wr_queue );
	if ( dev->progstate == -1 )
	  {
	    PDEBUG( "temp: 0x%X\n", dev->temp );
	    return -EIO;
	  }
	break;
      }
    case PICPROG_LOAD_CONFIG:
      {
	retval = __copy_from_user( dev->data, (char*)arg, 11 );
	dev->wr_queue = NULL;
	dev->wp = dev->data;
	dev->end = dev->wp + 10;
	setup10( write_config, dev );
	interruptible_sleep_on( &dev->wr_queue );
	if ( dev->progstate == -1 )
	  {
	    PDEBUG( "temp: 0x%X\n", dev->temp );
	    return -EIO;
	  }
	break;
      }
    case PICPROG_READ_CONFIG:
      {
	data = 0;
	comm_pic( LOAD_CONFIG, &data, dev );
	
	for ( curaddr = 0; curaddr <= 7; curaddr++ )
	  {
	    if ( curaddr == 4 ) 
	      {
		comm_pic( INC_ADDR, 0, dev );
		comm_pic( INC_ADDR, 0, dev );
		comm_pic( INC_ADDR, 0, dev );
		curaddr = 7;
	      }
	    comm_pic( READ_DATA, &data, dev );
	    PDEBUG( "data: 0x%X\n", data );
	    *buffer = (char)data & 0xff;
	    buffer++;
	    data >>= 8;
	    *buffer = (char)data & 0x3f;
	    buffer++;
	    comm_pic( INC_ADDR, 0, dev );
	  } 
	*buffer = '\0';
	buffer = dev->end + 1;
	retval = __copy_to_user( (char*)arg, buffer, 11 );
	break;
      }
    case PICPROG_DISPROT:
      {
	data = 0x3FF0;
	comm_pic( LOAD_CONFIG, &data, dev );
	
	for ( curaddr = 0; curaddr < 7; curaddr++ )
	  {
	    comm_pic( INC_ADDR, 0, dev );
	  }
	
	comm_pic( MAGIC_1, 0, dev );
	comm_pic( MAGIC_2, 0, dev );
	
	comm_pic( BEGIN_PROG, 0, dev );
	interruptible_sleep_on_timeout( &dev->wr_queue, HZ/50 );
	
	comm_pic( MAGIC_1, 0, dev );
	comm_pic( MAGIC_2, 0, dev );
	retval = 0;
	break;
      }
    }
  return retval;
}

void do_bulk_er( picp_t *dev )
{
  int data;
  /* the bulk erase procedure in Microchip's
     "EEPROM Memory Programming Specification" is the
     biggest lie of all */
  comm_pic( PROGRAM_MODE, 0, dev );
  comm_pic( BULK_ER_PM, 0, dev );
  data = 0x3FFF;
  comm_pic( LOAD_DATA, &data, dev );
  comm_pic( BEGIN_PROG, 0, dev );
  return;
}

void setup10( void (*function) (unsigned long), picp_t *dev)
{
  init_timer( &wr_timer[dev->num] );
  wr_timer[dev->num].function = function;
  wr_timer[dev->num].data = (unsigned long)dev;
  wr_timer[dev->num].expires = jiffies + HZ/100;
  add_timer( &wr_timer[dev->num] );
  return;
}

void write_config( unsigned long ptr )
{
  
  picp_t *dev = (void *)ptr;
  int data = 0;
  
  if ( dev->wp == dev->data )
    {
      data = 0;
      comm_pic( LOAD_CONFIG, &data, dev );
    }
  
  while ( dev->wp <= dev->end )
    {
      if ( dev->wp != dev->data )
	{
	  comm_pic( READ_DATA, &data, dev );
	  if ( data != dev->temp )
	    {
	      
	      dev->progstate = -1;
	      wake_up_interruptible(&dev->wr_queue);
	      return;
	    }
	  else if ( dev->wp == dev->end) 
	    {
	      dev->progstate = 0;
	      wake_up_interruptible(&dev->wr_queue);
	      return;
	    }
	  comm_pic( INC_ADDR, 0, dev );
	}
      if ( (dev->wp - dev->data)  == 8 )
	{
	  comm_pic( INC_ADDR, 0, dev );
	  comm_pic( INC_ADDR, 0, dev );
	  comm_pic( INC_ADDR, 0, dev );
	}
      data = (int)*dev->wp;
      data &= 0xFF;
      dev->wp++;
      data |= ( (int)*dev->wp << 8 );
      dev->wp++;
      data &= 0x3FFF;
      dev->temp = data;
      comm_pic( LOAD_DATA, &data, dev );
      comm_pic( BEGIN_PROG, 0, dev );
      
      setup10( write_config, dev );
      return; 
    }
 
  dev->end = dev->data +  2 * PICMEMSIZE + 2;
  dev->wp = dev->data;
  wake_up_interruptible(&dev->wr_queue);
}


void write_dmem( unsigned long ptr )
{
  
  picp_t *dev = (void *)ptr;
  int data = 0;
 

  while ( dev->wp < dev->end )
    {
      if ( dev->wp != dev->data )
	{
	  comm_pic( READ_DATA_DM, &data, dev );
	  PDEBUG( "data nach read: 0x%X\n", data );
	  if ( (data & 0xff) != dev->temp )
	    {
	      PICPROG_PRINT( "Don't understand smoke signals from controller, "
	                     "giving up...\n" );
	      wake_up_interruptible( &dev->wr_queue );
	      dev->progstate = -1;
	      return;
	    }
	  comm_pic( INC_ADDR, 0, dev );
	}
      data = (int)*dev->wp;
      /*  PDEBUG( "curaddr: %d *buffer: 0x%X\n", curaddr, *buffer ); */
      dev->wp++;
      data &= 0xff;
      dev->temp = data;
      /*   PDEBUG( "data: 0x%X\n", data ); */
      comm_pic( LOAD_DATA_DM, &data, dev );
      comm_pic( BEGIN_PROG, 0, dev );
      setup10( write_dmem, dev );
      return;
    }
  dev->progstate = 0;
  dev->end = dev->data +  2 * PICMEMSIZE + 2;
  dev->wp = dev->data;
  wake_up_interruptible( &dev->wr_queue );
}

int read_pmem( picp_t *dev )
{
  int data;
  int curaddr;

  /* reading is performed from the first byte to the last... */
  dev->rp = dev->data;

  /* a little reset is always good practize, so we know that
     we start at the beginning of the program memory */
  comm_pic( PROGRAM_MODE, 0, dev );

  for ( curaddr = 0; curaddr <= PICMEMSIZE; curaddr++ )
    {
      /* simply fetch all those 14-bit data words and write
	 them in little endian into our buffer */
      comm_pic( READ_DATA, &data, dev );
      *dev->rp = (char)data & 0xff;
      data >>= 8;
      dev->rp++;
      *dev->rp = (char)data & 0x3f;
      dev->rp++;
      comm_pic( INC_ADDR, 0, dev );
    }

  /* update read pointer */
  dev->rp = dev->data;
  return 0;

}

void write_pmem( unsigned long ptr )
{
  picp_t *dev = (void *)ptr;
  int data = 0;
  
  /* if we just started, do a bulk erase first
     and then do further programming */

  if ( dev->progstate == 0 )
    {
      dev->wp = dev->data;
      do_bulk_er( dev );
      dev->progstate = 1;

      /* erasing and programming takes 10 ms, so let's
	 set up a timer to do this */
      setup10( write_pmem, dev );
      return;
    }
  
  /* if we just came out of bulk erasing mode, 
     reset the chip, so it doesn't get confused */
  if ( dev->wp == dev->data )
    {
         comm_pic( PROGRAM_MODE, 0, dev );
    }
  
  /* only program as much data as we have... */
  while ( dev->wp < dev->end ) 
    {
      /* if this is not the first program cycle, let's
	 verify the data we've just written and increment
	 to the next address */
      if ( dev->wp != dev->data )
	{
	  comm_pic( READ_DATA, &data, dev );
	  PDEBUG( "data: 0x%X\n", data );
	  if ( data != dev->temp )
	    {
	      /* if something went wrong, wake up the writing
		 process and report the error through dev->progstate */
	      PICPROG_PRINT( "Programmer is burning!\n" ); 
	      dev->progstate = -1;
	      wake_up_interruptible( &dev->wr_queue );
	      return;
	    }
	  comm_pic( INC_ADDR, 0, dev );
	}

      /* here we fetch the next data word: little endian as
	 we are used to in intel formats */
      PDEBUG("*dev->wp: 0x%X\n", *dev->wp);
      data = (int)*dev->wp;
      data &= 0xFF;
      dev->wp++;
      PDEBUG("*dev->wp: 0x%X\n", *dev->wp);
      data |= ( (int)*dev->wp << 8 );
      dev->wp++;
      data &= 0x3FFF;
      dev->temp = data;
      /* now that we've just erased the whole chip above,
	 let's skip all those empty data bytes */
      if ( data != 0x3FFF )
	{ 
	  comm_pic( LOAD_DATA, &data, dev );
	  comm_pic( BEGIN_PROG, 0, dev );
	  /* again wait 10 ms for the programming cycle to 
	     perform */
	  setup10( write_pmem, dev );
	  return;
	}
    }

  /* reset write pointer and wake up waiting task */
  dev->wp = dev->data;
  dev->progstate = 0;
  wake_up_interruptible( &dev->wr_queue );
  
  return;
}

int comm_pic( int cmd, int *data, picp_t *dev ) 
{
  unsigned short port = (unsigned short)dev->port;
  int temp_cmd = cmd;
  int i;
  
  /* Are we shifting out a command to the controller or 
     do we just switch to a different programmer state? */

  if ( cmd < MAX_CMD ) 
    {
      /* Programming without Program-Mode-Selection-Voltage
	 is a bad thing! */

      outb_p( VPP_CONTR, port );
  

      /* Now let's shift out some commands. They are always
	 6 bits in length and are transmitted lsb first. */

      for ( i = 0; i < 6; i++)
	{
	  outb_p( (temp_cmd & DATA_OUT) | VPP_CONTR | CLOCK, port );
	  outb_p( (temp_cmd & DATA_OUT) | VPP_CONTR, port );
	  temp_cmd >>= 1;
	}
   
      
      /* If we have sent a command that's followed by some data,
	 let's handle that. */
      switch ( cmd ) 
	{
	case INC_ADDR:
	  PDEBUG( "INC_ADDR\n" );
	  break;
	case BEGIN_PROG:
	  PDEBUG( "BEGIN_PROG\n" );
	  break;
	case BULK_ER_PM:
	  PDEBUG( "BULK_ER_PM\n" );
	  break;
	case BULK_ER_DM:
	  PDEBUG( "BULK_ER_DM\n" );
	  break;
	case MAGIC_1:
	  PDEBUG( "MAGIC_1\n" );
	  break;
	case MAGIC_2:
	  PDEBUG( "MAGIC_2\n" );
	  break;
	case LOAD_DATA:
	case LOAD_CONFIG:
	case LOAD_DATA_DM:
	  {

	    /* data is always 14 bits, but we must send 16:
	       0, data[14], 0 */

	    udelay( 1 );
	    outb_p( VPP_CONTR | CLOCK, port);
	    outb_p( VPP_CONTR, port);
		    
	    PDEBUG( "before sending: *data: 0x%X\n", *data);

	    for ( i = 0; i < 14; i++ )
	      {
		outb_p( (*data & DATA_OUT) | VPP_CONTR | CLOCK, port );
		outb_p( (*data & DATA_OUT) | VPP_CONTR, port );
	     
		PDEBUG( "loop: %d *data: 0x%X\n", i, *data );
		*data >>= 1;
	
	      }

	    PDEBUG( "after sending: *data: 0x%X\n", *data );

	    outb_p( VPP_CONTR | CLOCK, port );
	    outb_p( VPP_CONTR, port );
	    

	    break;
	  }
	  
	case READ_DATA:
	case READ_DATA_DM:
	  {
	    /* same as above, but other direction now */
	    outb_p( VPP_CONTR | DATA_DIR, port );
	    udelay( 1 );
	    outb_p( VPP_CONTR | DATA_DIR | CLOCK, port );
	    outb_p( VPP_CONTR | DATA_DIR, port );
	    
	    *data = 0;
	    
 	    for ( i = 0; i < 14; i++ ) 
	      {
		*data >>= 1;
		outb_p( VPP_CONTR | DATA_DIR | CLOCK, port );
		outb_p( VPP_CONTR | DATA_DIR, port );
		if ( ( inb_p( port + 1 ) & DATA_IN ) ) *data |= 0x2000;
	      }
	    outb_p( VPP_CONTR | DATA_DIR | CLOCK, port );
	    outb_p( VPP_CONTR | DATA_DIR, port );
	    break;
	  }
	default:
	  return CMD_ERROR;
	}
    }
  else
    {
      /* now follow the commands to the programmer */
      switch( cmd )
	{
	case RESET:
	  {
	    /* reset controller: MCLR low, clock and data
	       in high impedance state */
	    outb_p( CLOCK_EN | MCLR_CONTR | DATA_DIR, port );
	    udelay( 100 );
	    outb_p( CLOCK_EN | DATA_DIR, port );
	    break;
	  }
	case PROGRAM_MODE:
	  {
	    /* reset and then pull up MCLR to 12 V --
	       the chip doesn't need the 12 V, but someone
	       at Arizona Microchip thought it would be a nice
	       idea to let SGS Thomson sell a few more voltage
	       regulators... */
	    outb_p( CLOCK_EN | MCLR_CONTR | DATA_DIR, port );
	    udelay( 100 );
	    outb_p( VPP_CONTR, port );
	    break;
	  }
	case RUN_MODE:
	  {
	    /* now back to run mode: MCLR = 5 V, clock and
	       data in high imdedance */
	    outb_p( CLOCK_EN | DATA_DIR, port );
	    break;
	  }
	default:
	  return CMD_ERROR;
	}
    }
  return 0;
}


#ifdef MODULE

int init_module( void ) 
{
  return( picprog_init() ); 
}

void cleanup_module( void ) 
{
  int i;

  /* free all the ports we used */
  for ( i = 0; i < 4; i++ )
    {
      if ( io[i] != 0 )
	{
	  release_region( io[i], 3 );
	}
    }
  
  unregister_chrdev( picprog_major, PICPROG_NAME );
  return;
}

#endif
