/*************************************************************************

    progutil - utility to program PIC Micros with picprog

    Version 0.1 			   

    File: mainclass.cpp -- Mainclass of progutil
    Begin: 11. Nov 1999

    Copyright (C) 1999, 2000 by Raffael Stocker
    E-mail: raffael.stocker@stud.fh-deggendorf.de

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place--Suite 330, Boston, MA
    02111-1307, USA.

*************************************************************************/

#include "mainclass.h"
#include "fileloader.h"
#include "progutil.h"
#include <unistd.h>
#include <stdexcept>

Mainclass::Mainclass (int argcount, char *argvector[])
{
  argc = argcount;
  argv = argvector;
  disablecp = false;
  help_wanted = false;
  ver_wanted = false;

  // Set default fileformat to override automatic format-recognition (by extension)
  //infilefmt = DEFAULTFORMAT;
  //outfilefmt = DEFAULTFORMAT;
  devicefile = DEFAULTDEVICE;
  
  // first do argument parsing using getopt
  parseargs ();

  // then initialize needed classes
  try
    {
      picprog = new Programmer (devicefile);
    }
  catch (const exception &failure)
    {
      cerr << ERRSTR << failure.what () << endl;
      exit (1);
    }
}

Mainclass::~Mainclass ()
{
  delete picprog;
}

void
Mainclass::parseargs ()
{
  if (argc == 1)
    {
      cerr << "Nothing happens." << endl;
      exit (42);
    }

  int option_char;
  program_mask = erase_mask = read_mask = 0;

  while ((option_char = getopt (argc, argv,"vhprecdo:I:O:PRECD:")) != -1)
    {
      switch (option_char)
	{
	case 'D': devicefile = optarg; break;
	case 'p': program_mask |= PROGRAM; break;
	case 'r': read_mask |= PROGRAM; break;
	case 'e': erase_mask |= PROGRAM; break;
	case 'c': program_mask |= CONFIG; break;
	case 'd': disablecp = true; break;
	case 'o': outputfile = optarg; break;
	case 'I': infilefmt = optarg; break;
	case 'O': outfilefmt = optarg; break;
	case 'P': program_mask |= DATA; break;
	case 'R': read_mask |= DATA; break;
	case 'E': erase_mask |= DATA; break;
	case 'C': read_mask |= CONFIG; break;
	case 'v': print_version (); exit (1); // ver_wanted = true; break;
	case 'h': // fall-through
	case '?': // fall-through
	case ':': print_usage (); exit (1); // help_wanted = true; break;
	default: cerr << ERRSTR << "Can't happen" << endl; exit(1);
	}
    }
  if (argv[optind] != 0)
    datafile = argv[optind];
}

// main method of class. Call with return <object>.exec() in main()
int
Mainclass::exec ()
{
/*  if (ver_wanted)
    {
      print_version ();
      return 1;
    }
  if (help_wanted)
    {
      print_usage ();
      return 1;
    } */
  if (disablecp)
    {
      try
	{
	  picprog->disablecp ();
	}
      catch (const exception &failure)
	{
	  cerr << ERRSTR << failure.what () << endl;
	  return 1;
	}
    }
  if (erase_mask)
    {
      try
	{
	  picprog->erase (erase_mask);
	}
      catch (const exception &failure)
	{
	  cerr << ERRSTR << failure.what () << endl;
	  return 1;
	}
    }
  if (program_mask)
    {
      if (datafile.empty ())
	{
	  cerr << ERRSTR << "No datafile given. Giving up." << endl;
	  return 1;
	}
      // construct FileLoader and get programdata (flash, eeprom, config)
      try
	{
	  FileLoader loader;
	  DataFile *file = loader.getfile (datafile, infilefmt); 

	  picprog->setprgdata (file->getprogmem ());
	  picprog->setdmdata (file->getdatamem ());
	  picprog->setcfgdata (file->getconfmem ());

	  picprog->program (program_mask);
	}
      catch (const exception &failure)
	{
	  cerr << ERRSTR << failure.what () << endl;
	  return 1;
	}
    }
  if (read_mask)
    {
      if (outputfile.empty ())
	{
	  cerr << ERRSTR << "No outputfile given. Giving up." << endl;
	  return 1;
	}
      // dito
      try
	{
	  FileLoader loader;
	  DataFile *file = loader.newfile (outputfile, outfilefmt); 

	  picprog->read (read_mask);

	  file->setprogmem (picprog->prgdata ());
	  file->setdatamem (picprog->dmdata ());
	  file->setconfmem (picprog->cfgdata ());

	  file->savefile ();
	}
      catch (const exception &failure)
	{
	  cerr << ERRSTR << failure.what () << endl;
	  return 1;
	}
    }
  return 0;
}

void
Mainclass::print_version ()
{
  cout << PRGNAME << " " << VERSION << endl;
  cout << "Copyright (C) 1999, 2000 " << AUTHOR << endl;
  cout << "Covered by GNU GPL--see COPYING for details" << endl;
}

void
Mainclass::print_usage ()
{
  cout << "Usage: progutil [vhprecdPREC] [D devicefile] [o outputfile]"
       << endl << "\t[I infilefmt] [O ofilefmt] datafile" << endl;
  cout << "\t-v:\tPrint version information" << endl;
  cout << "\t-h:\tPrint this usage information" << endl;
  cout << endl
       << "\t-p:\tWrite to program memory" << endl;
  cout << "\t-r:\tRead from program memory" << endl;
  cout << "\t-e:\tErase program memory" << endl;
  cout << "\t-c:\tWrite config memory" << endl;
  cout << "\t-d:\tDisable code protection (erases all memory)" << endl;
  cout << "\t-o:\tSet output filename" << endl;
  cout << "\t-I:\tForce input file format" << endl;
  cout << "\t-O:\tForce output file format" << endl;
  cout << "\t-P:\tWrite to data memory" << endl;
  cout << "\t-R:\tRead from data memory" << endl;
  cout << "\t-E:\tErase data memory" << endl;
  cout << "\t-C:\tRead from config memory" << endl;
  cout << endl
       << "\t-D devicefile:\tSpecify devicefile (default: " << DEFAULTDEVICE << ")" << endl;
  cout << "\tdatafile:\tSpecify filename for reading" << endl; 
  cout << endl
       << "Report bugs to " << EMAIL << endl;
}
