/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: rotate.c,v 2.1 94/09/28 14:27:01 nau Exp $";

/* functions used to rotate pins, elements ...
 */

#include <stdlib.h>

#include "global.h"

#include "crosshair.h"
#include "data.h"
#include "draw.h"
#include "misc.h"
#include "rotate.h"
#include "search.h"
#include "select.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	*RotateText(LayerTypePtr, TextTypePtr);
static	void	*RotateElementName(ElementTypePtr);
static	void	RotateArcLowLevel(ArcTypePtr, Position, Position, BYTE);

/* ----------------------------------------------------------------------
 * some local identifiers
 */
static	Position			CenterX,	/* center of rotation */
							CenterY;
static	BYTE				Number;		/* number of rotations */
static	ObjectFunctionType	RotateFunctions = {
	NULL,
	RotateText,
	NULL,
	NULL,
	NULL,
	RotateElementName,
	NULL,
	NULL };

/* ---------------------------------------------------------------------------
 * rotates a line in 90 degree steps
 */
void RotateLineLowLevel(LineTypePtr Line, Position X, Position Y, BYTE Number)
{
	ROTATE(Line->X1, Line->Y1, X, Y, Number);
	ROTATE(Line->X2, Line->Y2, X, Y, Number);

}

/* ---------------------------------------------------------------------------
 * rotates a text in 90 degree steps 
 * only the bounding box is rotated, text rotation itself
 * is done by the drawing routines
 */
void RotateTextLowLevel(TextTypePtr Text, Position X, Position Y, BYTE Number)
{
	RotateBoxLowLevel(&Text->BoundingBox, X, Y, Number);
	ROTATE(Text->X, Text->Y, X, Y, Number);

		/* set new direction, 0..3,
		 * 0-> to the right, 1-> straight up,
		 * 2-> to the left, 3-> straight down
		 */
	Text->Direction = ((Text->Direction +Number) & 0x03);
}

/* ---------------------------------------------------------------------------
 * rotates a polygon in 90 degree steps
 */
void RotatePolygonLowLevel(PolygonTypePtr Polygon,
	Position X, Position Y, BYTE Number)
{
	POLYGONPOINT_LOOP(Polygon,
		ROTATE(point->X, point->Y, X, Y, Number);
	);
	RotateBoxLowLevel(&Polygon->BoundingBox, X, Y, Number);
}

/* ---------------------------------------------------------------------------
 * rotates a text object and redraws it
 */
static void *RotateText(LayerTypePtr Layer, TextTypePtr Text)
{
	EraseText(Text);
	RotateTextLowLevel(Text, CenterX, CenterY, Number);
	DrawText(Layer, Text);
	return(Text);
}

/* ---------------------------------------------------------------------------
 * rotates an arc
 */
static void RotateArcLowLevel(ArcTypePtr Arc,
	Position X, Position Y,
	BYTE Number)
{
	Dimension	save;

		/* add Number*90 degrees to the startangle and check for overflow */
	Arc->StartAngle = (Arc->StartAngle +Number*90) % 360;
	ROTATE(Arc->X, Arc->Y, X, Y, Number);

		/* now change width and height */
	if (Number == 1 || Number == 3)
	{
		save = Arc->Width;
		Arc->Width = Arc->Height;
		Arc->Height = save;
	}
}

/* ---------------------------------------------------------------------------
 * rotate an element in 90 degree steps
 */
void RotateElementLowLevel(ElementTypePtr Element,
	Position X, Position Y,
	BYTE Number)
{
	ELEMENTLINE_LOOP(Element, RotateLineLowLevel(line, X, Y, Number););
	PIN_LOOP(Element, ROTATE_PIN_LOWLEVEL(pin, X, Y, Number););
	ARC_LOOP(Element, RotateArcLowLevel(arc, X, Y, Number););
	ROTATE(Element->MarkX, Element->MarkY, X, Y, Number);
	RotateBoxLowLevel(&Element->BoundingBox, X, Y, Number);
	RotateTextLowLevel(&CANONICAL_TEXT(Element), X, Y, Number);
	RotateTextLowLevel(&NAMEONPCB_TEXT(Element), X, Y, Number);
}

/* ----------------------------------------------------------------------
 * rotates the name of an element
 */
static void *RotateElementName(ElementTypePtr Element)
{
	EraseElementName(Element);
	RotateTextLowLevel(&CANONICAL_TEXT(Element), CenterX, CenterY, Number);
	RotateTextLowLevel(&NAMEONPCB_TEXT(Element), CenterX, CenterY, Number);
	DrawElementName(Element);
	return(Element);
}

/* ---------------------------------------------------------------------------
 * rotates a box in 90 degree steps 
 */
void RotateBoxLowLevel(BoxTypePtr Box,
	Position X, Position Y, BYTE Number)
{
	ROTATE(Box->X1, Box->Y1, X, Y, Number);
	ROTATE(Box->X2, Box->Y2, X, Y, Number);
	Box->X1 = MIN(Box->X1, Box->X2);
	Box->Y1 = MIN(Box->Y1, Box->Y2);
	Box->X2 = MAX(Box->X1, Box->X2);
	Box->Y2 = MAX(Box->Y1, Box->Y2);
}

/* ----------------------------------------------------------------------
 * rotates an objects at the cursor position as identified by its ID
 * the center of rotation is determined by the current cursor location
 */
void RotateObject(int Type, void *Ptr1, void *Ptr2,
	Position X, Position Y, BYTE Steps)
{
		/* setup default  global identifiers */
	CenterX = X;
	CenterY = Y;
	Number = Steps;
	ObjectOperation(&RotateFunctions, Type, Ptr1, Ptr2);
}

