/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/print.c,v 2.6 1994/11/04 12:47:26 nau Exp nau $";

/* printing routines
 */
#include <math.h>
#include <time.h>

#include "global.h"

#include "data.h"
#include "error.h"
#include "misc.h"
#include "print.h"
#include "rotate.h"

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	PrintHeader(Position, Position, Position, Position);
static	void	PrintArc(ArcTypePtr);
static	void	PrintText(TextTypePtr);
static	void	PrintBox(BoxTypePtr);
static	void	PrintLine(LineTypePtr);
static	void	PrintPolygon(PolygonTypePtr);
static	void	PrintPinOrVia(PinTypePtr, Cardinal);
static	void	PrintLayer(LayerTypePtr);
static	void	PrintElementPackage(ElementTypePtr);
static	void	SetPrintColor(Pixel);

/* ----------------------------------------------------------------------
 * some local identifiers
 *
 * PSFunctions is initializes by the standard PostScript code used by PCB.
 * All PS functions are declared in this section:
 *   P:   draws a filled pin-polygon 
 *   PV:  draws a pin or via
 *   PO:  draws a filled polygon 
 *   L0:  draws a line
 *   B:   draws a filled box
 *   A:   draws an element arc
 * additional information is available from the PS comments.
 *
 * some PS code for arcs comes from the output of 'Fig2Dev'. Thanks to
 * the author
 */
static	FILE	*FP;				/* output file */
static	char	*PSFunctions[] = {
	"",
	"/PcbDict 200 dict def",
	"PcbDict begin",
	"PcbDict /DictMatrix matrix put",
	"",
	"% some constants",
	"/White {1.0 setgray} def",
	"/TAN {0.207106781} def",
	"/MTAN {-0.207106781} def",
	"",
	"% draw a filled polygon",
	"% get (x,y)... and number of points from stack",
	"/PO {",
	"	/number exch def",
	"	newpath",
	"	moveto",
	"	number 1 sub { lineto } repeat",
	"	closepath fill stroke",
	"} def",
	"",
	"/P {",
	"% draw a pin-polygon,",
	"% get x, y, length (lines), thickness and rotation from stack",
	"	/r exch def /thickness exch def /l exch def /y exch def /x exch def",
	"	gsave x y translate thickness thickness scale r rotate",
	"	newpath",
	"	0.5  MTAN",
	"	TAN  -0.5",
	"	MTAN -0.5",
	"	-0.5 MTAN",
	"	-0.5 TAN  l add",
	"	MTAN 0.5  l add",
	"	TAN  0.5  l add",
	"	0.5  TAN  l add",
	"	8 PO grestore",
	"} def",
	"",
	"/PV {",
	"% pin or via, x, y and thickness are on the stack",
	"	/drillinghole exch def /thickness exch def /y exch def /x exch def",
	"	x y 0 thickness 0 P",
	"% draw drilling whole	",
	"	White x y 0 drillinghole 0 P",
	"% restore drawing color",
	"   PrintingColor",
	"} def",
	"",
	"/L {",
	"% line, get x1, y1, x2, y2 and thickness from stack",
	"	/thick exch def /y2 exch def /x2 exch def /y1 exch def /x1 exch def",
	"	gsave thick setlinewidth",
	"	x1 y1 moveto x2 y2 lineto stroke",
	"	grestore",
	"} def",
	"",
	"/B {",
	"% filled box, get x1, y1, x2 and y2 from stack",
	"	/y2 exch def /x2 exch def /y1 exch def /x1 exch def",
	"	newpath",
	"	x1 y1 moveto x2 y1 lineto x2 y2 lineto x1 y2 lineto",
	"	closepath fill stroke",
	"} def",
	"",
	"/A {",
	"% arc for elements, get x, y, width, height, thickness",
	"% startangle and delta-angle from stack",
	"/delta exch def /start exch def /thickness exch def",
	"/height exch def /width exch def /y exch def /x exch def",
	"% draw it",
	"	gsave thickness setlinewidth /save DictMatrix currentmatrix def",
	"% scaling is less then zero because the coord system has to be swapped",
	"	x y translate 0 width sub height scale",
	"	0 0 1 start start delta add arc save setmatrix stroke",
	"	grestore",
	"} def",
	"",
	"% setup some things",
	"0.0 setlinewidth",
	"1 setlinecap" };

/* ----------------------------------------------------------------------
 * prints PostScript header with function definition
 * bounding box corners are passed in
 */
static void PrintHeader(Position X1, Position Y1, Position X2, Position Y2)
{
	int		i;
	time_t	currenttime;

	currenttime = time(NULL);
	fputs("%!PS-Adobe-2.0 EPSF-1.2\n", FP);
	fputs("%%Creator: pcb\n", FP);
	fprintf(FP, "%%%%Title: %s\n", PCB->Name);
	fprintf(FP, "%%%%CreationDate: %s", asctime(localtime(&currenttime)));
	fprintf(FP, "%%%%BoundingBox: %i %i %i %i\n", X1, Y1, X2, Y2);
	fputs("%%EndComments\n", FP);

	for (i = 0; i < ENTRIES(PSFunctions); i++)
		fprintf(FP, "%s\n", PSFunctions[i]);
	fflush(FP);
}

/* ----------------------------------------------------------------------
 * prints an element arc
 */
static void PrintArc(ArcTypePtr Arc)
{
	fprintf(FP, "%d %d %d %d %d %d %d A\n",
		(int) Arc->X, (int) Arc->Y, (int) Arc->Width, (int) Arc->Height,
		(int) Arc->Thickness, Arc->StartAngle, Arc->Delta);
}

/* ----------------------------------------------------------------------
 * prints a line
 */
static void PrintLine(LineTypePtr Line)
{
	fprintf(FP, "%d %d %d %d %d L\n", (int) Line->X1, (int) Line->Y1,
		(int) Line->X2, (int) Line->Y2, (int) Line->Thickness);
}

/* ----------------------------------------------------------------------
 * prints a box
 */
static void PrintBox(BoxTypePtr Box)
{
	fprintf(FP, "%d %d %d %d B\n",
		(int) Box->X1, (int) Box->Y1, (int) Box->X2, (int) Box->Y2);
}

/* ----------------------------------------------------------------------
 * prints a text
 * the routine is identical to DrawText() in module draw.c except
 * that DrawLine() and DrawRectangle() are replaced by their corresponding
 * printing routines
 */
static void PrintText(TextTypePtr Text)
{
	Position		x = Text->X,
					y = Text->Y,
					x0 = 0;
	unsigned char	*string = (unsigned char *) Text->TextString;
	Cardinal		n;
	FontTypePtr		font = &PCB->Font;

		/* if text has to be mirrored, get x0 with the unrotated
		 * width of the surrounding box (which is the length of the string)
		 */
	if (TEST_FLAG(MIRRORFLAG, Text))
	{
		Dimension	width = Text->BoundingBox.X2 -Text->BoundingBox.X1,
					height = Text->BoundingBox.Y2 -Text->BoundingBox.Y1;

		if (Text->Direction == 0 || Text->Direction == 2)
			x0 = Text->X +width/2;
		else
			x0 = Text->X +height/2;
	}
	while (string && *string)
	{
			/* print lines if symbol is valid and data is present */
		if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
		{
			LineTypePtr		line = font->Symbol[*string].Line;
			LineType		newline;

			for (n = font->Symbol[*string].LineN; n; n--, line++)
			{
					/* create one line, scale and move it */
				newline = *line;
				newline.X1 = newline.X1 *Text->Scale /100 +x;
				newline.Y1 = newline.Y1 *Text->Scale /100 +y;
				newline.X2 = newline.X2 *Text->Scale /100 +x;
				newline.Y2 = newline.Y2 *Text->Scale /100 +y;
				newline.Thickness = newline.Thickness *Text->Scale /100;

					/* do some mirroring and rotations */
				if (TEST_FLAG(MIRRORFLAG, Text))
				{
					newline.X1 = 2*x0 -newline.X1;
					newline.X2 = 2*x0 -newline.X2;
				}
				RotateLineLowLevel(&newline,Text->X,Text->Y,Text->Direction);
				PrintLine(&newline);
			}
		}
		else
		{
				/* the default symbol is a filled box */
			BoxType	defaultsymbol = PCB->Font.DefaultSymbol;

			defaultsymbol.X1 = defaultsymbol.X1 *Text->Scale /100 +x;
			defaultsymbol.Y1 = defaultsymbol.Y1 *Text->Scale /100 +y;
			

				/* do some mirroring and rotations */
			if (TEST_FLAG(MIRRORFLAG, Text))
				defaultsymbol.X1 = 2*x0 -defaultsymbol.X1 -defaultsymbol.X2;
			RotateBoxLowLevel(&defaultsymbol,Text->X,Text->Y, Text->Direction);
			PrintBox(&defaultsymbol);
		}
				/* move on to next cursor position */
		x += ((font->Symbol[*string].Width +font->Symbol[*string].Delta)
			*Text->Scale /100);
		string++;
	}
}

/* ---------------------------------------------------------------------------
 * prints a filled polygon
 */
static void PrintPolygon(PolygonTypePtr Ptr)
{
	int i = 0;

	POLYGONPOINT_LOOP(Ptr,
		{
			if (i++ % 9 == 8)
				fputc('\n', FP);
			fprintf(FP, "%i %i ", (int) point->X, (int) point->Y);
		}
	);
	fprintf(FP, "%d PO\n", Ptr->PointN);
}

/* ----------------------------------------------------------------------
 * prints a via or pin
 */
static void PrintPinOrVia(PinTypePtr Ptr, Cardinal Count)
{
	for(; Count; Count--, Ptr++)
		fprintf(FP, "%d %d %d %d PV\n",
			(int) Ptr->X, (int) Ptr->Y,
			(int) Ptr->Thickness, (int) Ptr->DrillingHole);
}

/* ----------------------------------------------------------------------
 * prints layer data
 */
static void PrintLayer(LayerTypePtr Layer)
{
	LINE_LOOP(Layer, PrintLine(line));
	TEXT_LOOP(Layer, PrintText(text));
	POLYGON_LOOP(Layer, PrintPolygon(polygon));
	fflush(FP);
}

/* ----------------------------------------------------------------------
 * prints package outline
 */
static void PrintElementPackage(ElementTypePtr Element)
{
	ELEMENTLINE_LOOP(Element, PrintLine(line));
	ARC_LOOP(Element, PrintArc(arc));
	PrintText(&ELEMENT_TEXT(PCB, Element));
}

/* ----------------------------------------------------------------------
 * queries color from X11 database and generates PostScript command
 * set PostScript identifier to the calculated value or to black by 
 * default
 */
static void SetPrintColor(Pixel X11Color)
{
	XColor	rgb;
	int		result;

		/* query best matching color from X server */
	rgb.pixel = X11Color;
	result = XQueryColor(Dpy,
		DefaultColormapOfScreen(XtScreen(Output.Toplevel)),
		&rgb);

		/* create a PS command to set this color */
	if (result != BadValue && result != BadColor)
		fprintf(FP, "/PrintingColor {%.3f %.3f %.3f setrgbcolor} def "
			"PrintingColor\n",
			(float) rgb.red /65535.0,
			(float) rgb.green /65535.0,
			(float) rgb.blue /65535.0);
	else
			/* all colors default to black on error */
		fputs("/PrintingColor {0.0 setgray} def PrintingColor\n", FP);

	fflush(FP);
}

/* ----------------------------------------------------------------------
 * generates PostScript output of currently visible layers
 * mirroring, rotating and scaling was already determined by a dialog
 *
 * output is written either to a file or, if the command starts with
 * a pipe symbol, piped to a command
 * whitespaces have already been removed from 'Command'
 *
 * 'MirrorFlag' is negated because PostScript and X11 use different
 * directions for the Y axis
 *
 * 'PackageOnly' selects a special mode where only packages, pins and vias
 * are printed
 *
 * offsets are in mil
 */
int PrintPS(char *Command, float Scale,
	Boolean MirrorFlag, Boolean RotateFlag,
	Boolean ColorFlag, Boolean PackageOnly,
	Position OffsetX, Position OffsetY)
{
	BoxTypePtr	box;
	Position	x1, y1,
				x2, y2;
	char		*filename;
	Cardinal	i;

		/* first we open a file or pipe */
	if (*Command == '|')
	{
		if ((FP = popen(Command+1, "w")) == NULL)
		{
			PopenErrorMessage(Command+1);
			return(1);
		}
	}
	else
	{
			/* expand filename without additional directories */
		if ((filename = ExpandFilename(NULL, Command)) == NULL)
			filename = Command;
		if ((FP = fopen(filename, "w")) == NULL)
		{
			OpenErrorMessage(filename);
			return(1);
		}
	}

		/* 'MirrorFlag' has to be inverted because X11 and PostScript use
		 * different Y-axis
		 */
	MirrorFlag = !MirrorFlag;

		/* set bounding box coordinates; layout has already been checked
		 * to be non-empty
		 */
	box = GetDataBoundingBox(PCB->Data);
	x1 = (Position) ((float) OffsetX * 0.072) -1;
	y1 = (Position) ((float) OffsetY * 0.072) -1;
	if (!RotateFlag)
	{
		x2= (Position)((((float)(box->X2 -box->X1) *Scale) +OffsetX) *0.072) +1;
		y2= (Position)((((float)(box->Y2 -box->Y1) *Scale) +OffsetY) *0.072) +1;
	}
	else
	{
		x2= (Position)((((float)(box->Y2 -box->Y1) *Scale) +OffsetX) *0.072) +1;
		y2= (Position)((((float)(box->X2 -box->X1) *Scale) +OffsetY) *0.072) +1;
	}

		/* print PostScript header and function definitions */
	PrintHeader(x1, y1, x2, y2);

		/* add information about layout size, offset ... */
	fprintf(FP, "%% PCBMIN(%d,%d), PCBMAX(%d,%d)\n",
		(int) box->X1, (int) box->Y1, (int) box->X2, (int) box->Y2);
	fprintf(FP, "%% PCBOFFSET(%d,%d), PCBSCALE(%.5f)\n",
		OffsetX, OffsetY, Scale);

		/* print first label */
	fputs("% PCBSTARTDATA don't remove it -------------------------\n", FP);
	fputs("gsave\n", FP);

		/* now insert transformation commands (reverse order):
		 * - move upper/left edge of layout to (0,0)
		 * - mirror it to transform X to PostScript coordinates
		 * - move to (0,0) again
		 * - if rotation is required, rotate and move to (0,0)
		 * - apply user scaling
		 * - move to new offset
		 * - scale to PostScript (72 dots per inch)
		 */ 
	fputs("0.072 0.072 scale\n", FP);
	fprintf(FP, "%i %i translate\n", OffsetX, OffsetY);
	fprintf(FP, "%.3f %.3f scale\n", Scale, Scale);
	if (RotateFlag)
	{
		fprintf(FP, "%i 0 translate\n", box->Y2 -box->Y1);
		fputs("90 rotate\n", FP);
	}
	if (MirrorFlag)
	{
		fprintf(FP, "0 %i translate\n", box->Y2 -box->Y1);
		fputs("1 -1 scale\n", FP);
	}
	fprintf(FP, "%i %i translate\n", -box->X1, -box->Y1);

		/* black is default drawing color */
	fputs("/PrintingColor {0.0 setgray} def PrintingColor\n", FP);

		/* print signal layers only if allowed;
		 * print data, vias and pins at last
		 */
	for (i = MAX_LAYER-1; !PackageOnly && i != -1; i--)
		if ((LAYER_ON_STACK(i))->On)
		{
			if (ColorFlag)
				SetPrintColor((LAYER_ON_STACK(i))->Color);
			PrintLayer(LAYER_ON_STACK(i));
		}

	if (PCB->ElementOn || PackageOnly)
		ELEMENT_LOOP(PCB->Data,
			if (ColorFlag)
				SetPrintColor(PCB->ElementColor);
			PrintElementPackage(element);
		);
	if (PCB->PinOn || PackageOnly)
		ELEMENT_LOOP(PCB->Data,
			if (ColorFlag)
				SetPrintColor(PCB->PinColor);
			PrintPinOrVia(element->Pin, element->PinN);
		);
	if (PCB->ViaOn || PackageOnly)
		{
			if (ColorFlag)
				SetPrintColor(PCB->ViaColor);
			PrintPinOrVia(PCB->Data->Via, PCB->Data->ViaN);
		}

		/* print trailing commands */
	fputs("grestore\n", FP);
	fputs("% PCBENDDATA don't remove it -------------------------\n", FP);
	fputs("showpage\n\004", FP);

	return(*Command == '|' ? pclose(FP) : fclose(FP));
}
