/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/control.c,v 2.3 1994/10/29 17:33:50 nau Exp $";

/* control panel
 */

#include <stdio.h>
#include <stdlib.h>

#include "global.h"

#include "control.h"
#include "data.h"
#include "draw.h"
#include "misc.h"
#include "set.h"

#include <X11/Shell.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Toggle.h>

/* ---------------------------------------------------------------------------
 * include bitmap data
 */
#include "mode_icon.data"

/* ---------------------------------------------------------------------------
 * some local defines
 */
#define	BUTTONS_PER_ROW		3	/* number of mode buttons per row */

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct					/* description for a single mode button */
{
	char			*Name,		/* the widgets name and label */
					*Label;
	int				Mode;		/* mode ID */
	char			*Bitmap;	/* background bitmap data */
	unsigned int	Width,		/* bitmap size */
					Height;
	Widget			W;
} ModeButtonType, *ModeButtonTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Widget	OnOffWidgets[MAX_LAYER+3],	/* widgets for on/off selection */
				DrawingWidgets[MAX_LAYER];

static	ModeButtonType	ModeButtons[] = {
	{ "via", "via", VIA_MODE, ViaModeIcon_bits,
		ViaModeIcon_width, ViaModeIcon_height, NULL },
	{ "line", "line", LINE_MODE, LineModeIcon_bits,
		LineModeIcon_width, LineModeIcon_height, NULL },
	{ "rectangle", "rectangle", RECTANGLE_MODE, RectModeIcon_bits,
		RectModeIcon_width, RectModeIcon_height, NULL },
	{ "text", "text", TEXT_MODE, TextModeIcon_bits,
		TextModeIcon_width, TextModeIcon_height, NULL },
	{ "polygon", "polygon", POLYGON_MODE, PolygonModeIcon_bits,
		PolygonModeIcon_width, PolygonModeIcon_height, NULL },
	{ "buffer", "buffer", PASTEBUFFER_MODE, BufferModeIcon_bits,
		BufferModeIcon_width, BufferModeIcon_height, NULL },
	{ "remove", "remove", REMOVE_MODE, RemoveModeIcon_bits,
		RemoveModeIcon_width, RemoveModeIcon_height, NULL },
	{ "mirror", "mirror", MIRROR_MODE, MirrorModeIcon_bits,
		MirrorModeIcon_width, MirrorModeIcon_height, NULL },
	{ "rotate", "rotate", ROTATE_MODE, RotateModeIcon_bits,
		RotateModeIcon_width, RotateModeIcon_height, NULL }};

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	Widget		AddLabel(String, Widget, Widget, Widget);
static	void		AddDrawingLayerSelection(Widget);
static	void		AddOnOffSelection(Widget);
static	void		UpdateDrawingLayerSelection(void);
static	void		UpdateOnOffSelection(void);
static	Cardinal	GetGroupOfLayer(Cardinal);
static	int			ChangeGroupVisibility(Cardinal, Boolean, Boolean);
static	void		PushOnTopOfLayerStack(int);
static	void		CB_SetOnOff(Widget, XtPointer, XtPointer);
static	void		CB_SetDrawingLayer(Widget, XtPointer, XtPointer);
static	void		CB_SetMode(Widget, XtPointer, XtPointer);

/* ---------------------------------------------------------------------------
 * adds a label without border, surrounded by additional space
 * at the specified position
 */
static Widget AddLabel(String Label, Widget Parent, Widget Top, Widget Left)
{
	Widget		label;
	Dimension	height;

	label = XtVaCreateManagedWidget("label", labelWidgetClass,
		Parent,
		XtNlabel, Label,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNjustify, XtJustifyCenter,
		LAYOUT_TOP,
		NULL);
	XtVaGetValues(label, XtNheight, &height, NULL);
	XtVaSetValues(label, XtNheight, 3*height/2, NULL);
	return(label);
}

/* ---------------------------------------------------------------------------
 * adds the layer selection buttons to our dialog (downwards)
 */
static void AddDrawingLayerSelection(Widget Parent)
{
			Widget			last;
			int				i;
			XtTranslations	table;
			char			name[10];
	static	String			translations = 
		"<Btn1Down>,<Btn1Up>: set() notify() \n";

		/* install some changes in translation table to make sure that
		 * always ONE element is selected
		 */
	table = XtParseTranslationTable(translations);

		/* insert a label at the top */
	last = AddLabel("current", Parent, NULL, NULL);

	for (i = 0; i < MAX_LAYER; i++)
	{
		sprintf(name, "layer%-i", i+1);
		DrawingWidgets[i] = XtVaCreateManagedWidget(name, toggleWidgetClass,
			Parent,
			XtNforeground, PCB->Data->Layer[i].Color,
			XtNfromHoriz, NULL,
			XtNfromVert, last,
			LAYOUT_TOP,
			NULL);
		last = DrawingWidgets[i];

			/* the first entry identifies the radiogroup,
			 * no radio data is added because the identification
			 * may change (function is to be performed by UpdateDrawingLayer())
			 */
		XtVaSetValues(last,
			XtNradioGroup, DrawingWidgets[0],
			NULL);
		XtOverrideTranslations(last, table);
		XtAddCallback(last, XtNcallback, CB_SetDrawingLayer, (XtPointer) i);
	}
}

/* ---------------------------------------------------------------------------
 * adds the layer On/Off selection buttons to our dialog (downwards)
 * the label for the layer buttons are set by UpdateOnOffSelection()
 */
static void AddOnOffSelection(Widget Parent)
{
	Widget		last;
	int			i;
	char		*text,
				name[10];
	Pixel		color;

		/* insert a label at the top */
	last = AddLabel("on/off", Parent, NULL, NULL);

	for (i = 0; i < MAX_LAYER+3; i++)
	{
		switch(i)
		{
			case MAX_LAYER:			/* settings for elements */
				color = PCB->ElementColor;
				text = "pkg. outline";
				strcpy(name, "elements");
				break;

			case MAX_LAYER+1:		/* settings for pins */
				color = PCB->PinColor;
				text = "pkg. pins";
				strcpy(name, "pins");
				break;

			case MAX_LAYER+2:		/* settings for vias */
				color = PCB->ViaColor;
				text = "vias";
				strcpy(name, "vias");
				break;

			default:				/* layers */
				color = PCB->Data->Layer[i].Color;
				text = "";
				sprintf(name, "layer%-i", i+1);
				break;
		}

			/* create widget and install callback function */
		OnOffWidgets[i] = last = XtVaCreateManagedWidget(name,toggleWidgetClass,
			Parent,
			XtNlabel, text,
			XtNforeground, color,
			XtNfromHoriz, NULL,
			XtNfromVert, last,
			LAYOUT_TOP,
			NULL);
		XtAddCallback(last, XtNcallback, CB_SetOnOff, (XtPointer) i);
	}
}

/* ---------------------------------------------------------------------------
 * updates the layer selection buttons
 */
static void UpdateDrawingLayerSelection(void)
{
	int		i;

		/* set new radio data, PCB may have changed
		 * we have to make sure that no NULL is passed to XtNradioData
		 */
	for (i = 0; i < MAX_LAYER; i++)
		XtVaSetValues(DrawingWidgets[i],
			XtNlabel, UNKNOWN(PCB->Data->Layer[i].Name),
			XtNradioData, (XtPointer) &PCB->Data->Layer[i],
			NULL);

		/* switch one button ON and reset the flags that are set
		 * by the callback functions of the toggle widgets
		  */
	XawToggleSetCurrent(DrawingWidgets[0], (XtPointer) CURRENT);
	RedrawOnEnter = False;
}

/* ---------------------------------------------------------------------------
 * updates the layer On/Off selection buttons
 */
static void UpdateOnOffSelection(void)
{
	int		i;

		/* the buttons for elements, vias and pins never
		 * change their label, so we only check the layer buttons
		 */
	for (i = 0; i < MAX_LAYER; i++)
		XtVaSetValues(OnOffWidgets[i],
			XtNlabel, UNKNOWN(PCB->Data->Layer[i].Name),
			XtNstate, PCB->Data->Layer[i].On,
			NULL);

		/* now set the state of elements, pins and vias */
	XtVaSetValues(OnOffWidgets[i++], XtNstate, PCB->ElementOn, NULL);
	XtVaSetValues(OnOffWidgets[i++], XtNstate, PCB->PinOn, NULL);
	XtVaSetValues(OnOffWidgets[i++], XtNstate, PCB->ViaOn, NULL);
}

/* ----------------------------------------------------------------------
 * lookup the group to which a layer belongs to
 * returns MAX_LAYER if no group is found
 */
static Cardinal GetGroupOfLayer(Cardinal Layer)
{
	int	group,
		i;

	for (group = 0; group < MAX_LAYER; group++)
		for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
			if (PCB->LayerGroups.Entries[group][i] == Layer)
				return(group);
	return(MAX_LAYER);
}

/* ----------------------------------------------------------------------
 * changes the visibility of all layers in a group
 * returns the number of changed layers
 */
static int ChangeGroupVisibility(Cardinal Layer,
	Boolean On, Boolean ChangeStackOrder)
{
	int		group,
			i,
			changed = 1;	/* at least the current layer changes */

		/* decrement 'i' to keep stack in order of layergroup */
	if ((group = GetGroupOfLayer(Layer)) < MAX_LAYER)
		for (i = PCB->LayerGroups.Number[group]; i;) 
		{
			int	layer = PCB->LayerGroups.Entries[group][--i];

				/* dont count the passed member of the group */
			if (layer != Layer)
			{
				PCB->Data->Layer[layer].On = On;
	
					/* push layer on top of stack if switched on */
				if (On && ChangeStackOrder)
					PushOnTopOfLayerStack(layer);
				changed++;
			}
		}

		/* change at least the passed layer */
	PCB->Data->Layer[Layer].On = On;
	if (On && ChangeStackOrder)
		PushOnTopOfLayerStack(Layer);

		/* update control panel and exit */
	UpdateOnOffSelection();
	return(changed);
}

/* ---------------------------------------------------------------------------
 * move layer (number is passed in) to top of layerstack
 */
static void PushOnTopOfLayerStack(int NewTop)
{
	int		i;

		/* first find position of passed one */
	for (i = 0; i < MAX_LAYER; i++)
		if (LayerStack[i] == NewTop)
			break;

		/* bring this element to the top of the stack */
	for(; i; i--)
		LayerStack[i] = LayerStack[i-1];
	LayerStack[0] = NewTop;
}

/* ---------------------------------------------------------------------------
 * callback routine, called when current layer is changed
 *
 * this routine is either called from one of the toggle widgets or
 * from the output window by using accelerators
 */
static void CB_SetDrawingLayer(Widget W,
	XtPointer ClientData, XtPointer CallData)
{
	ChangeGroupVisibility((int) ClientData, True, True);
	RedrawOnEnter = True;
}

/* ---------------------------------------------------------------------------
 * callback routine, called when On/Off flag of layer is changed
 */
static void CB_SetOnOff(Widget W, XtPointer ClientData, XtPointer CallData)
{
	Boolean	state;
	int		layer = (int) ClientData;

		/* get new state of widget */
	XtVaGetValues(W, XtNstate, &state, NULL);

		/* set redraw flag if object exists */
	switch(layer)
	{
		case MAX_LAYER:			/* settings for elements */
			PCB->ElementOn = state;
			RedrawOnEnter = (PCB->Data->ElementN != 0);
			break;

		case MAX_LAYER+1:		/* settings for pins */
			PCB->PinOn = state;
			RedrawOnEnter = (PCB->Data->ElementN != 0);
			break;

		case MAX_LAYER+2:		/* settings for vias */
			PCB->ViaOn = state;
			RedrawOnEnter = (PCB->Data->ViaN != 0);
			break;

		default:				/* layers; current one can't be switched off */
		{
			int		i, group;

				/* check if current one is in that group;
				 * if YES, do nothing because it cannot be changed
				 */
			if ((group = GetGroupOfLayer(layer)) < MAX_LAYER)
				for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
					if (PCB->LayerGroups.Entries[group][i] == LayerStack[0])
					{
							/* reset state to 'On' */
						XtVaSetValues(W, XtNstate, True, NULL);
						return;
					}

				/* switch layer group on/off */
			ChangeGroupVisibility(layer, state, False);
			RedrawOnEnter = True;
			break;
		}
	}
}

/* ---------------------------------------------------------------------------
 * initializes dialog box to get settings like
 *   visible layers
 *   drawing layer
 * returns popup shell widget
 */
void InitControlPanel(Widget Parent, Widget Top, Widget Left)
{
	Widget	masterform,
			visible,
			drawing;

	masterform = XtVaCreateManagedWidget("controlMasterForm", formWidgetClass,
		Parent,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		LAYOUT_TOP,
		NULL);
	visible = XtVaCreateManagedWidget("onOffSelection", formWidgetClass,
		masterform,
		XtNfromVert, NULL,
		XtNfromHoriz, NULL,
		NULL);
	drawing = XtVaCreateManagedWidget("currentSelection", formWidgetClass,
		masterform,
		XtNfromVert, visible,
		XtNfromHoriz, NULL,
		NULL);

		/* we now add the other widgets to the two forms */
	AddOnOffSelection(visible);
	AddDrawingLayerSelection(drawing);
	Output.Control = masterform;
}

/* ---------------------------------------------------------------------------
 * updates label and sizes in control panel
 */
void UpdateControlPanel(void)
{
	UpdateOnOffSelection();
	UpdateDrawingLayerSelection();
}

/* ---------------------------------------------------------------------------
 * callback routine, called when current mode is changed
 */
static void CB_SetMode(Widget W, XtPointer ClientData, XtPointer CallData)
{
	ModeButtonTypePtr	button;
	
		/* the radio group is determined by the widget of button 0 */
	button = (ModeButtonTypePtr) XawToggleGetCurrent(ModeButtons[0].W);
	SetMode(button ? button->Mode : NO_MODE);
}

/* ---------------------------------------------------------------------------
 * initializes mode buttons
 */
Widget InitModeButtons(Widget Parent, Widget Top, Widget Left)
{
	Screen	*screen = XtScreen(Parent);
	int		i;

	for (i = 0; i < ENTRIES(ModeButtons); i++)
	{
		Pixmap	bitmap;

			/* create background pixmap */
		bitmap = XCreatePixmapFromBitmapData(Dpy, XtWindow(Parent),
			ModeButtons[i].Bitmap, ModeButtons[i].Width, ModeButtons[i].Height,
			BlackPixelOfScreen(screen), WhitePixelOfScreen(screen),
			DefaultDepthOfScreen(screen));

			/* create radio button; use label only if pixmap creation failed */
		if (bitmap != BadAlloc)
			ModeButtons[i].W = XtVaCreateManagedWidget(ModeButtons[i].Name,
				toggleWidgetClass,
				Parent,
				XtNbitmap, bitmap,
				NULL);
		else
			ModeButtons[i].W = XtVaCreateManagedWidget(ModeButtons[i].Name,
				toggleWidgetClass,
				Parent,
				XtNlabel, ModeButtons[i].Label,
				NULL);

			/* the first entry identifies the radiogroup,
			 * we have to make sure that no NULL is passed to XtNradioData
			 */
		XtVaSetValues(ModeButtons[i].W,
			XtNradioGroup, ModeButtons[0].W,
			XtNradioData, (XtPointer) &ModeButtons[i],
			XtNfromHoriz, Left,
			XtNfromVert, Top,
			LAYOUT_TOP,
			NULL);
		XtAddCallback(ModeButtons[i].W, XtNcallback,
			CB_SetMode, (XtPointer) &ModeButtons[i]);

			/* update position for next widget */
		if (i % BUTTONS_PER_ROW == BUTTONS_PER_ROW-1)
		{
			Left = NULL;
			Top = ModeButtons[i].W;
		}
		else
			Left = ModeButtons[i].W;
	}
	return(ModeButtons[i-1].W);
}

/* ---------------------------------------------------------------------------
 * updates the mode selection buttons to reflect the currently set mode
 */
void UpdateModeSelection(void)
{
		/* reset all buttons if no mode is selected;
		 * the first buttons widget identifies the radio group whereas
		 * radioData points to the array member itself
		 */
	if (Settings.Mode == NO_MODE)
		XawToggleUnsetCurrent(ModeButtons[0].W);
	else
	{
		int	i;

		for (i = 0; i < ENTRIES(ModeButtons); i++)
			if (Settings.Mode == ModeButtons[i].Mode)
			{
				XawToggleSetCurrent(ModeButtons[0].W,
					(XtPointer) &ModeButtons[i]);
				break;
			}
	}
}

/* ---------------------------------------------------------------------------
 * install accelerators for 'drawing layer selection
 * It has to be called speratly because the output widget is created
 * after the control panel
 */
void InstallControlAccelerators(void)
{
	int		i;

	for (i = 0; i < MAX_LAYER; i++)
		XtInstallAccelerators(Output.Output, DrawingWidgets[i]);
}

