/*
 * Copyright (c) 2002 Tony Bybell.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, 
 * and/or sell copies of the Software, and to permit persons to whom the   
 * Software is furnished to do so, subject to the following conditions: 
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,  
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL   
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING   
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "vcd_write.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

static const char *vcd_timebase_exponents_strings[] =
        {
        "s",
        "ms",
        "us",
        "ns",
        "ps",
        "fs"
        };

static const char *vcd_symtype_strings[] =
        {
        "UNKNOWN",

        "module",
        "begin",
        "task", 
        "function",
        "fork",

        "event",
        "parameter",
        "integer",
        "real",

        "reg",
        "supply0",
        "supply1",
        "tri",
        "triand",
        "trior",
        "trireg",
        "tri0",
        "tri1",
        "wand",
        "wire",
        "wor",
        "port"
        };


const char *vcd_convert_vcdid(struct vcd_context *context, int vcdid)
{
int i;

if(!context->vcd_port_style_ids)
	{
	for(i=0;;i++)
	       	{
	       	context->vcdid_converted[i]=(char)((vcdid%94)+33); /* for range 33..126 */
	       	vcdid/=94;
       		if(!vcdid) {context->vcdid_converted[i+1]=0; break;}
       		}
	}
	else
	{
	sprintf(context->vcdid_converted, "<%d", vcdid);
	}

return(context->vcdid_converted);
}


struct vcd_context *vcd_open(const char *name)
{
FILE *handle = fopen(name, "wb");
if(!handle) 
	{
	return(NULL);
	}
	else
	{
	struct vcd_context *vc = (struct vcd_context *)calloc(1, sizeof(struct vcd_context));
	vc->handle = handle;
	vc->curr_time = -1LL;
	return(vc);
	}
}


int vcd_set_timebase(struct vcd_context *context, int exponent)
{
int rc = ((context)&&(exponent>=VCD_TIMEBASE_S)&&(exponent<VCD_TIMEBASE_INVALID)&&(!context->is_finalized));

if(rc)
	{
	context->timebase = exponent;
	}

return(rc);
}


int vcd_push_scope(struct vcd_context *context, const char *name, int symtype)
{
if((context)&&(name)&&(!context->is_finalized)&&(symtype>=VCD_SYM_MODULE)&&(symtype<=VCD_SYM_FORK))
	{
	struct vcd_scope *vs = (struct vcd_scope *)calloc(1, sizeof(struct vcd_scope));
	vs->name = (char *)strdup(name);
	vs->len = strlen(name);

	if(context->scope_head)
		{
		context->scope_curr->next = vs;
		vs->prev = context->scope_curr;
		context->scope_curr = vs;
		}
		else
		{
		context->scope_head = context->scope_curr = vs;
		}

	context->scope_depth++;
	vcd_symbol_add(context, "", -1, -1, symtype);

	return(1);
 	}
	else
	{
	return(0);
 	}
}


int vcd_pop_scope(struct vcd_context *context)
{
if((context)&&(context->scope_curr)&&(!context->is_finalized))
	{
	struct vcd_scope *vst = context->scope_curr;
	context->scope_curr = vst->prev;
	free(vst->name);
	free(vst);

	if(context->scope_curr)
		{
		context->scope_curr->next = NULL;
		}
		else
		{
		context->scope_head = NULL;
		}

	context->scope_depth--;
	return(1);
	}
	else
	{
	return(0);
	}
}


int vcd_query_scope_depth(struct vcd_context *context)
{
if((context)&&(!context->is_finalized))
	{
	return(context->scope_depth);
	}
	else
	{
	return(0);
	}
}


char *vcd_query_scope(struct vcd_context *context)
{
if((context)&&(!context->is_finalized)&&(context->scope_head))
	{
	struct vcd_scope *vst = context->scope_head;
	int namlen = 0;
	char *pnt, *pnt2;

	while(vst)
		{
		namlen += (vst->len + 1);
		vst=vst->next;
		}

	pnt = pnt2 = malloc(namlen ? namlen : 1);
	*pnt2 = 0;
	vst = context->scope_head;
	while(vst)
		{
		strcpy(pnt2, vst->name);
		pnt2 += vst->len;
		vst = vst->next;
		if(vst)
			{
			*(pnt2++) = '.';
			}
		}

	return(pnt);
	}
	else
	{
	return(NULL);
	}
}


int vcd_symbol_add(struct vcd_context *context, const char *name, int msb, int lsb, int symtype)
{
if((context)&&(name)&&(symtype>VCD_SYM_INVALID0)&&(symtype<VCD_SYM_INVALID1)&&(!context->is_finalized))
	{
	int depth = vcd_query_scope_depth(context);
	char *flatname;
	struct vcd_symbol *vs = calloc(1, sizeof(struct vcd_symbol));

	if(depth)
		{
		char *pfx = vcd_query_scope(context);
		int namlen = strlen(name);
		if(namlen)
			{
			flatname = malloc(strlen(pfx)+1+strlen(name)+1);		
			sprintf(flatname, "%s.%s", pfx, name);
			free(pfx);
			}
			else
			{
			flatname=pfx;
			}
		}
		else
		{
		flatname = (char *)strdup(name);
		}

	switch(symtype)
		{
		case VCD_SYM_EVENT:
		case VCD_SYM_REAL:
			msb=-1; lsb=-1; break;
		case VCD_SYM_PARAMETER:
		case VCD_SYM_INTEGER:
			msb=31; lsb=0; break;
		default:
			break;
		}

	vs->flatname = flatname;
	vs->msb = msb;
	vs->lsb = lsb;
	vs->symtype = symtype;

	if(msb<lsb)
		{
		vs->siz = (lsb-msb)+1;
		}
		else
		{
		vs->siz = (msb-lsb)+1;
		}

	if((symtype>=VCD_SYM_EVENT)&&(symtype<=VCD_SYM_PORT)) vs->vcdid = ++(context->curr_vcdid);
	context->tree_vcdid = ds_insert(vs->vcdid, context->tree_vcdid, (void *)vs);
	context->numsyms++;

	if(context->vcdsym_curr)
		{
		context->vcdsym_curr->next = vs;
		context->vcdsym_curr = vs;
		}
		else
		{
		context->vcdsym_head = context->vcdsym_curr = vs;
		}

	return(vs->vcdid);
	}
	else
	{
	return(0);
	}
}


int vcd_symbol_alias(struct vcd_context *context, const char *name, int msb, int lsb, int symtype, int old_vcdid)
{
int rc = 0;

if((context)&&(name)&&(old_vcdid>0)&&(context->curr_vcdid)&&(!context->is_finalized))
	{
	struct vcd_symbol *ovs = (struct vcd_symbol *)(context->tree_vcdid = ds_splay(old_vcdid, context->tree_vcdid))->pnt;
	if((ovs)&&(ovs->vcdid == old_vcdid)&&(ovs->symtype>=VCD_SYM_EVENT)&&(ovs->symtype<=VCD_SYM_PORT)&&(symtype>=VCD_SYM_EVENT)&&(symtype<=VCD_SYM_PORT))
		{
		int siz;

		switch(symtype)
			{
			case VCD_SYM_EVENT:
			case VCD_SYM_REAL:
				msb=-1; lsb=-1; break;
			case VCD_SYM_PARAMETER:
			case VCD_SYM_INTEGER:
				msb=31; lsb=0; break;
			default:
				break;
			}

		if(msb<lsb)
			{
			siz = (lsb-msb)+1;
			}
			else
			{
			siz = (msb-lsb)+1;
			}

		if(siz == ovs->siz) 
			{
			int depth = vcd_query_scope_depth(context);
			char *flatname;
			struct vcd_symbol *vs = calloc(1, sizeof(struct vcd_symbol));

			if(depth)
				{
				char *pfx = vcd_query_scope(context);
				int namlen=strlen(name);
				if(namlen)
					{
					flatname = malloc(strlen(pfx)+1+namlen+1);		
					sprintf(flatname, "%s.%s", pfx, name);
					free(pfx);
					}
					else
					{
					flatname = pfx;
					}
				}
				else
				{
				flatname = (char *)strdup(name);
				}

			vs->flatname = flatname;
			vs->msb = msb;
			vs->lsb = lsb;
			vs->symtype = symtype;
			vs->siz = ovs->siz;
			vs->vcdid = ovs->vcdid;

			context->vcdsym_curr->next = vs;
			context->vcdsym_curr = vs;
			context->numsyms++;

			rc = 1;
			}
		}
	}

return(rc);
}


int vcd_set_time(struct vcd_context *context, long long timval)
{
int rc = 0;

if(context)
	{
	if(!context->is_finalized) vcd_header_finalize(context);

	if((timval>=0LL)&&(timval>context->curr_time))
		{
		fprintf(context->handle, "#%lld\n", context->curr_time = timval);
		rc = 1;
		}
	}

return(rc);
}


static const char *vcd_truncate_bitvec(const char *s)
{
char l, r;

r=*s;
if(r=='1')
	{
	return s;
	}
	else
	{
        s++; 
	}

for(;;s++) 
	{
        l=r; r=*s;
        if(!r) return (s-1);

        if(l!=r)
		{
                return(((l=='0')&&(r=='1'))?s:s-1);
		}
	}
}


int vcd_emit_value_port_string(struct vcd_context *context, int vcdid, const char *value, int str0, int str1)
{
int rc = 0;

if((context)&&(value)&&(vcdid>0)&&(vcdid<=context->curr_vcdid))
	{
	int vlen;
	struct vcd_symbol *vs;
	const char *outval;
	int vcdidtemp = vcdid;

	if(!context->is_finalized) vcd_header_finalize(context);

	context->tree_vcdid = ds_splay(vcdid, context->tree_vcdid);
	vs = (struct vcd_symbol *)(context->tree_vcdid->pnt);
	if(vs->symtype==VCD_SYM_PORT)
		{
		vlen = strlen(value);
		if(vlen>vs->siz)
			{
			value += (vlen-vs->siz);	/* truncate to right */
			}
			else
			{
			/* you're at the mercy of VCD port left-fill semantics if your string is too short! */
			}
	
		if(str0<0) str0=0; else if(str0>7) str0=7;
		if(str1<0) str1=0; else if(str1>7) str1=7;
		fprintf(context->handle, "p%s %d %d %s\n", value, str0, str1, vcd_convert_vcdid(context, vcdid));
	
		rc = 1;
		}
	}

return(rc);
}


int vcd_emit_value_bit_string(struct vcd_context *context, int vcdid, const char *value)
{
int rc = 0;

if((context)&&(value)&&(vcdid>0)&&(vcdid<=context->curr_vcdid))
	{
	int vlen;
	struct vcd_symbol *vs;
	const char *outval;
	int vcdidtemp = vcdid;

	if(!context->is_finalized) vcd_header_finalize(context);

	context->tree_vcdid = ds_splay(vcdid, context->tree_vcdid);
	vs = (struct vcd_symbol *)(context->tree_vcdid->pnt);
	if(((vs->symtype>=VCD_SYM_REG)&&(vs->symtype<VCD_SYM_PORT))||(vs->symtype>=VCD_SYM_PARAMETER)||(vs->symtype>=VCD_SYM_INTEGER))
		{
		vlen = strlen(value);
		if(vlen>vs->siz)
			{
			value += (vlen-vs->siz);	/* truncate to right */
			}
			else
			{
			/* you're at the mercy of VCD left-fill semantics if your string is too short! */
			}
	
		if(vs->siz!=1)
			{
			fprintf(context->handle, "b%s %s\n", vcd_truncate_bitvec(value), vcd_convert_vcdid(context, vcdid));
			}
			else
			{
			fprintf(context->handle, "%s%s\n", value, vcd_convert_vcdid(context, vcdid));
			}
	
		rc = 1;
		}
	}

return(rc);
}






int vcd_emit_value_integer(struct vcd_context *context, int vcdid, int value)
{
char buf[33];
int i;

buf[32]=0;
for(i=31;i>-1;i--)
	{
	buf[i]=(value&1)+'0';
	value>>=1;
	}

return(vcd_emit_value_bit_string(context, vcdid, buf));
}


int vcd_emit_value_real(struct vcd_context *context, int vcdid, double value)
{
int rc = 0;

if((context)&&(vcdid>0)&&(vcdid<=context->curr_vcdid))
	{
	int vlen;
	struct vcd_symbol *vs;
	const char *outval;
	int vcdidtemp = vcdid;

	if(!context->is_finalized) vcd_header_finalize(context);

	context->tree_vcdid = ds_splay(vcdid, context->tree_vcdid);
	vs = (struct vcd_symbol *)(context->tree_vcdid->pnt);
	if(vs->symtype==VCD_SYM_REAL)
		{
		fprintf(context->handle, "r%.16g %s\n", value, vcd_convert_vcdid(context, vcdid));
		rc = 1;
		}
	}

return(rc);
}


static int vcd_sigcmp(const void *vs1, const void *vs2)
{
struct vcd_symbol *va = *(struct vcd_symbol **)vs1;
struct vcd_symbol *vb = *(struct vcd_symbol **)vs2;
char *s1 = va->flatname, *s2 = vb->flatname;
unsigned char c1, c2;
int u1, u2;
                                
for(;;)
        {
        c1=(unsigned char)*(s1++);
        c2=(unsigned char)*(s2++);
                                
        if((!c1)&&(!c2)) 
		{
		return(va->vcdid - vb->vcdid);	/* strings equal, sort on vcdid */
		}

        if((c1<='9')&&(c2<='9')&&(c2>='0')&&(c1>='0'))
                {
                u1=(int)(c1&15);
                u2=(int)(c2&15);
                        
                while(((c2=(unsigned char)*s2)>='0')&&(c2<='9'))
                        {
                        u2*=10;
                        u2+=(unsigned int)(c2&15);
                        s2++;
                        }
                 
                while(((c2=(unsigned char)*s1)>='0')&&(c2<='9'))
                        {
                        u1*=10;
                        u1+=(unsigned int)(c2&15);
                        s1++;
                        }
 
                if(u1==u2) continue;
                        else return((int)u1-(int)u2);
                }               
                else
                {
                if(c1!=c2) return((int)c1-(int)c2);
                }
        }                       
}


static void vcd_diff_hier(struct vcd_context *context, struct vcd_namehier *nh1, struct vcd_namehier *nh2)
{
struct vcd_namehier *nhtemp;

if(!nh2)
        {
        while((nh1)&&(nh1->not_final))
                {
                fprintf(context->handle, "$scope module %s $end\n", nh1->name);
                nh1=nh1->next;
                }
        return;
        }

for(;;)
        {
        if((nh1->not_final==0)&&(nh2->not_final==0)) /* both are equal */
                {
                break;
                }

        if(nh2->not_final==0)   /* old hier is shorter */
                {
                nhtemp=nh1;
                while((nh1)&&(nh1->not_final))
                        {
                        fprintf(context->handle, "$scope module %s $end\n", nh1->name);
                        nh1=nh1->next;
                        }
                break;
                }

        if(nh1->not_final==0)   /* new hier is shorter */
                {
                nhtemp=nh2;
                while((nh2)&&(nh2->not_final))
                        {
                        fprintf(context->handle, "$upscope $end\n");
                        nh2=nh2->next;
                        }
                break;
                }

        if(strcmp(nh1->name, nh2->name))
                {
                nhtemp=nh2;                             /* prune old hier */
                while((nh2)&&(nh2->not_final))
                        {
                        fprintf(context->handle, "$upscope $end\n");
                        nh2=nh2->next;
                        }

                nhtemp=nh1;                             /* add new hier */
                while((nh1)&&(nh1->not_final))
                        {
                        fprintf(context->handle, "$scope module %s $end\n", nh1->name);
                        nh1=nh1->next;
                        }
                break;
                }

        nh1=nh1->next;
        nh2=nh2->next;
        }
}


static void vcd_output_varname(struct vcd_context *context, struct vcd_symbol *vs)
{
char *name=vs->flatname;
char *pnt, *pnt2;
char *s;
int len;
struct vcd_namehier *nh_head=NULL, *nh_curr=NULL, *nhtemp;

pnt=pnt2=name;

for(;;)
{
while((*pnt2!='.')&&(*pnt2)) pnt2++;
s=(char *)calloc(1,(len=pnt2-pnt)+1);
memcpy(s, pnt, len);
nhtemp=(struct vcd_namehier *)calloc(1,sizeof(struct vcd_namehier));
nhtemp->name=s;

if(!nh_curr)
        {
        nh_head=nh_curr=nhtemp;
        }
        else
        {
        nh_curr->next=nhtemp;
        nh_curr->not_final=1;
        nh_curr=nhtemp;
        }

if(!*pnt2) break;
pnt=(++pnt2);
}

vcd_diff_hier(context, nh_head, context->nhold);

/* output vars here */
while(context->nhold)
        {
        nhtemp=context->nhold->next;
        free(context->nhold->name);
        free(context->nhold);
        context->nhold=nhtemp;
        }
context->nhold=nh_head;

if((vs->symtype>=VCD_SYM_EVENT)&&(vs->symtype<=VCD_SYM_PORT))
	{
	if((vs->symtype==VCD_SYM_PARAMETER)||(vs->symtype==VCD_SYM_INTEGER))
		{
	        fprintf(context->handle, "$var %s 32 %s %s $end\n", vcd_symtype_strings[vs->symtype], vcd_convert_vcdid(context, vs->vcdid), nh_curr->name);
		}
	else
	if(vs->siz==1)
	        {
		if(vs->msb==-1)
			{
		        fprintf(context->handle, "$var %s 1 %s %s $end\n", vcd_symtype_strings[vs->symtype], vcd_convert_vcdid(context, vs->vcdid), nh_curr->name);
			}
			else
			{
		        fprintf(context->handle, "$var %s 1 %s %s[%d] $end\n", vcd_symtype_strings[vs->symtype], vcd_convert_vcdid(context, vs->vcdid), nh_curr->name, vs->msb);
			}
	        }
	        else
	        {
		fprintf(context->handle, "$var %s %d %s %s[%d:%d] $end\n", vcd_symtype_strings[vs->symtype], vs->siz,
			vcd_convert_vcdid(context, vs->vcdid), nh_curr->name, vs->msb, vs->lsb);
	        }
	}
	else	/* we're a scope identifier */
	{
	nhtemp=(struct vcd_namehier *)calloc(1,sizeof(struct vcd_namehier));
	fprintf(context->handle, "$scope %s %s $end\n", vcd_symtype_strings[vs->symtype], nh_curr->name);
	nh_curr->not_final = 1;
	nh_curr->next = nhtemp;
	nhtemp->name = strdup("*");	/* patch in a placeholder name */
	}
}


int vcd_set_header_version_information(struct vcd_context *context, char *text)
{
if((context)&&(!context->is_finalized)&&(!context->version_text))
	{
	context->version_text = strdup(text);
	return(1);
	}

return(0);
}


int vcd_use_port_style_identifiers(struct vcd_context *context, int truth_value)
{
if((context)&&(!context->is_finalized))
	{
	context->vcd_port_style_ids = truth_value;
	return(1);
	}

return(0);
}


int vcd_header_finalize(struct vcd_context *context)
{
int rc = 0;

if((context)&&(!context->is_finalized))
	{
	struct vcd_symbol **sorted = malloc(context->numsyms*sizeof(struct vcd_symbol *));
	struct vcd_symbol *vs = context->vcdsym_head;
	int i;
	struct vcd_namehier *nh, *nh2;
        time_t walltime;
   	struct vcd_scope *st = context->scope_head;
	struct vcd_scope *st2;

	while(st)
		{
		st2=st;
		st=st->next;
		free(st2->name);
		free(st2);
		}
	context->scope_head = context->scope_curr = NULL;

        time(&walltime);
        fprintf(context->handle, "$date\n");
        fprintf(context->handle, "\t%s",asctime(localtime(&walltime)));
        fprintf(context->handle, "$end\n");

        fprintf(context->handle, "$version\n\t%s%s"VCD_VERSION_STRING"\n$end\n"
                "$timescale\n\t1%s\n$end\n\n", 
			(!context->version_text) ? "" : context->version_text,
			(!context->version_text) ? "" : " via ",
			vcd_timebase_exponents_strings[context->timebase]);

	if(context->version_text)
		{
		free(context->version_text);
		context->version_text=NULL;
		}

	for(i=0;i<context->numsyms;i++)
		{
		sorted[i] = vs;
		vs=vs->next;
		}

	qsort((void *)sorted, context->numsyms, sizeof(struct vcd_symbol *), vcd_sigcmp);
	for(i=0;i<context->numsyms;i++)
		{
		vcd_output_varname(context, sorted[i]);
		free(sorted[i]->flatname);
		sorted[i]->flatname=NULL;
		}

	free(sorted);

	nh = nh2 = context->nhold;
	while(nh)
		{
		nh2 = nh;
		if(nh->not_final)
			{
			fprintf(context->handle, "$upscope $end\n");
			}
		nh=nh->next;
		free(nh2->name);
		free(nh2);
		}
	context->nhold = NULL;

	fprintf(context->handle, "$enddefinitions $end\n\n");

	context->is_finalized = 1;
	rc = 1;
	}

return(rc);
}


int vcd_close(struct vcd_context *context)
{
if(context)
	{
	struct vcd_symbol *vs = context->vcdsym_head;
	struct vcd_symbol *vs2;
        time_t walltime;
   
	while(context->tree_vcdid)
		{
		context->tree_vcdid = ds_delete(context->tree_vcdid->item, context->tree_vcdid);
		}	

	while(vs)
		{
		vs2=vs;
		vs=vs->next;
		free(vs2);
		}
	context->numsyms = 0;
	context->curr_vcdid = 0;

        time(&walltime);
        fprintf(context->handle, "\n$comment\n");
        fprintf(context->handle, "\tFinished dumping at %s",asctime(localtime(&walltime)));
        fprintf(context->handle, "$end\n");

	fclose(context->handle);
	context->handle = NULL;
	free(context);

	return(1);
	}

return(0);
}


int vcd_emit_arbitrary_string(struct vcd_context *context, char *string)
{
int rc=0;

if((context)&&(context->is_finalized))
	{
	fprintf(context->handle, "%s", string);
	rc=1;
	}

return(rc);
}


int vcd_emit_arbitrary_string_with_linefeed(struct vcd_context *context, char *string)
{
int rc=0;

if((context)&&(context->is_finalized))
	{
	fprintf(context->handle, "%s\n", string);
	rc=1;
	}

return(rc);
}


