/*
 *  b20fill - function to continue processing geometrical
 *            state diagram input and complete filling data structures.
 *
 *  Tom Mayo - 6-15-95
 */

#include "b20.h"

void b20fill(void)
{
	double d;
	int asynch_transitions = 0;
	int to_states = 0;
	int from_states = 0;
	int items = 0;
	char fatal_string[128];
	transition_s_t *lookahead_transition = (transition_s_t *)NULL;
	transition_s_t *state_transition = (transition_s_t *)NULL;
	string_s_t *lookahead_output = (string_s_t *)NULL;
	string_s_t *state_output = (string_s_t *)NULL;

/*
 *  Match a name to each state based upon proximity
 *  to the center of the state circle.
 *
 *  It might be better to reserve this assignment
 *  until all strings are associated with states,
 *  then check to see which ones are assignments.
 *  The remaining one would be the state name.
 */
	current_state = first_state;

	while(current_state != (state_s_t *)NULL)
	{
		current_state->name = (string_s_t *)NULL;

		current_string = first_string;

		while(current_string != (string_s_t *)NULL)
		{
			d = fabs(distance(&(current_state->center_xy),
				&(current_string->origin_xy)));

			if (d <= current_state->radius * STATE_SNAP_RATIO)
			{
				current_state->name = current_string;
				break;
			}

			current_string = current_string->next_overall;
		}

		if (current_state->name == (string_s_t *)NULL)
		{
			sprintf(fatal_string, "-- Cannot find name for state at "
				"(%2.3f, %2.3f).\n",
				current_state->center_xy.x,
				current_state->center_xy.y);

			fatal_error(fatal_string);
		}
		else
		{
			if (debug_fill)
				fprintf(stderr, "-- Name for state at (%2.3f, %2.3f) is "
					"\"%s\".\n", current_state->center_xy.x,
					current_state->center_xy.y,
					(current_state->name)->text);
		}

		current_state = current_state->next_overall;
	}

/*
 *  Check each transition.
 */
	asynch_transitions = 0;

	current_transition = first_transition;

	while(current_transition != (transition_s_t *)NULL)
	{
		current_transition->condition = (string_s_t *)NULL;

		current_string = first_string;

/*
 *  Match a condition to each transition.
 */
		while(current_string != (string_s_t *)NULL)
		{
			d = distance(&(current_transition->center_xy),
				&(current_string->origin_xy));

			if (d <= STRING_SNAP_DIST)
			{
				current_transition->condition = current_string;
				break;
			}

			current_string = current_string->next_overall;
		}

		if (current_transition->condition == (string_s_t *)NULL)
		{
			fprintf(outfp, "-- WARNING:\n"
				"--   No condition. Assuming default transition at "
				"(%2.3f, %2.3f).\n",
				current_transition->center_xy.x,
				current_transition->center_xy.y);
		}
		else
		{
			if (debug_fill)
				fprintf(stderr, "-- Condition for transition at "
					"(%2.3f, %2.3f) is \"%s\".\n",
					current_transition->center_xy.x,
					current_transition->center_xy.y,
					(current_transition->condition)->text);
		}

/*
 *  Match ends of the transition to states.
 */
		to_states = 0;
		from_states = 0;

		current_state = first_state;

		while (current_state != (state_s_t *)NULL)
		{
			if (distance(&(current_transition->from_xy),
				&(current_state->center_xy))
				< current_state->radius + TRANS_SNAP_DIST)
			{
				from_states++;

				current_transition->from_state = current_state;

				if (current_state->first_transition == (transition_s_t *)NULL)
				{
					current_state->first_transition = current_transition;

					if (debug_fill)
						fprintf(stderr, "-- Starting new transition chain "
							"for state \"%s\".\n",
							((current_state->name)->text));
				}
				else
				{
					lookahead_transition = current_state->first_transition;

					items = 0;

					do
					{
						state_transition = lookahead_transition;
						lookahead_transition =
							state_transition->next_this_state; 
						items++;
					}
					while (lookahead_transition != (transition_s_t *)NULL);

					state_transition->next_this_state = current_transition;

					if (debug_fill)
						fprintf(stderr, "-- Adding link %d to "
							"transition chain for state \"%s\".\n",
							items, ((current_state->name)->text));
				}

				if (debug_fill)
				{
					if (current_transition->condition == (string_s_t *)NULL)
					{
						fprintf(stderr, "-- Transition DEFAULT from state"
							" \"%s\".\n",
							((current_transition->from_state)->name)->text);
					}
					else		
					{
						fprintf(stderr, "-- Transition \"%s\" from state"
							" \"%s\".\n",
							(current_transition->condition)->text,
							((current_transition->from_state)->name)->text);
					}
				}
			}

			if (distance(&(current_transition->to_xy),
				&(current_state->center_xy))
				< current_state->radius + TRANS_SNAP_DIST)
			{
				current_transition->to_state = current_state;
				to_states++;

				if (debug_fill)
				{
					if (current_transition->condition == (string_s_t *)NULL)
					{
						fprintf(stderr, "-- Transition DEFAULT to state"
							" \"%s\".\n",
							((current_transition->to_state)->name)->text);
					}
					else		
					{
						fprintf(stderr, "-- Transition \"%s\" to state"
							" \"%s\".\n",
							(current_transition->condition)->text,
							((current_transition->to_state)->name)->text);
					}
				}
			}

			current_state = current_state->next_overall;
		}

/*
 *  Check destination.
 */
		if (to_states == 0)
		{
			sprintf(fatal_string, "Transition \"%s\" has no destination.",
				(current_transition->condition)->text);
			fatal_error(fatal_string);
		}
		else if (to_states > 1)
		{
			sprintf(fatal_string, "Transition \"%s\" has multiple"
				" destinations.", (current_transition->condition)->text);
			fatal_error(fatal_string);
		}

/*
 *  Find asynchronous (or synchronous) reset transition.
 *    Check source.
 */
		if (from_states == 0)
		{
			if (asynch_transitions == 1)
			{
				fatal_error("Too many asynchronous transitions.");
			}
			else
			{
				asynch_transitions = 1;
				asynch_transition = current_transition;

				if (debug_fill)
					fprintf(stderr, "-- Reset transition is \"%s\".\n",
						(current_transition->condition)->text);
			}
		}
		else if (from_states > 1)
		{
			sprintf(fatal_string, "Transition \"%s\" has multiple"
				" sources.", (current_transition->condition)->text);
			fatal_error(fatal_string);
		}

		current_transition = current_transition->next_overall;
	}

/*
 *  Find outputs for each state.
 */
	current_string = first_string;

	while (current_string != (string_s_t *)NULL)
	{
		current_state = first_state;

		while (current_state != (state_s_t *)NULL)
		{
			if (distance(&(current_string->origin_xy),
				&(current_state->center_xy))
				< current_state->radius - OUTPUT_MARGIN_DIST
				&& current_state->name != current_string)
			{
				if (current_state->first_output == (string_s_t *)NULL)
				{
					current_state->first_output = current_string;

					if (debug_fill)
						fprintf(stderr, "-- Starting new output chain "
							"for state \"%s\".\n",
							((current_state->name)->text));
				}
				else
				{
					lookahead_output = current_state->first_output;

					items = 0;

					do
					{
						state_output = lookahead_output;
						lookahead_output = state_output->next_this_state; 
						items++;
					}
					while (lookahead_output != (string_s_t *)NULL);

					state_output->next_this_state = current_string;

					if (debug_fill)
						fprintf(stderr, "-- Adding link %d to "
							"output chain for state \"%s\".\n",
							items, ((current_state->name)->text));
				}
			}
			current_state = current_state->next_overall;
		}

		current_string = current_string->next_overall;
	}
}
