/*$Id: d_mos.cc,v 18.17 2000/09/04 09:45:33 al Exp $ -*- C++ -*-
 * mos device basics
 * netlist syntax:
 * device:  mxxxx d g s b mname <device args> <model card args>
 * model:   .model mname NMOS <args>
 *	or  .model mname PMOS <args>
 */
#include "d_cap.h"
#include "d_admit.h"
#include "d_res.h"
#include "ap.h"
#include "d_mos_base.h"
//#include "d_mos.h" already included
/*--------------------------------------------------------------------------*/
int MOS_COMMON::_count = -1;
int DEV_MOS::_count = 0;
static EVAL_MOS_Cgb Eval_Cgb(CC_STATIC);
static EVAL_MOS_Cgd Eval_Cgd(CC_STATIC);
static EVAL_MOS_Cgs Eval_Cgs(CC_STATIC);
static MOS_COMMON Default_MOS(CC_STATIC);
/*--------------------------------------------------------------------------*/
MOS_COMMON::MOS_COMMON(int c)
  :COMPONENT_COMMON(c),
   sdp(0),
   lo(NOT_INPUT),
   wo(NOT_INPUT),
   ad_in(NOT_INPUT),
   as_in(NOT_INPUT),
   pd(0.0),
   ps(0.0),
   nrd(1.0),
   nrs(1.0),
   off(false),
   icset(false),
   sb(c),
   db(c)
{
  for (int i = 0;  i < mNUM_INIT_COND;  i++){
    ic[i] = 0.;
  }
  ++_count;
  /* not all is initialized.  remainder filled in by expand */
}
/*--------------------------------------------------------------------------*/
MOS_COMMON::MOS_COMMON(const MOS_COMMON& p)
  :COMPONENT_COMMON(p),
   sdp(0),
   lo(p.lo),
   wo(p.wo),
   ad_in(p.ad_in),
   as_in(p.as_in),
   pd(p.pd),
   ps(p.ps),
   nrd(p.nrd),
   nrs(p.nrs),
   off(p.off),
   icset(p.icset),
   sb(p.sb, CC_STATIC),
   db(p.db, CC_STATIC)
{
  for (int i = 0;  i < mNUM_INIT_COND;  i++){
    ic[i] = 0.;
  }
  ++_count;
  /* not all is initialized.  remainder filled in by expand */
}
/*--------------------------------------------------------------------------*/
MOS_COMMON::~MOS_COMMON()
{
  --_count;
  delete sdp;
}
/*--------------------------------------------------------------------------*/
void MOS_COMMON::parse(CS& cmd)
{
  assert(!has_model());

  parse_modelname(cmd);
  if (cmd.is_pfloat()){				/* accept W/L notation for */
    wo = cmd.ctopf() * MICRON2METER;		/* dimensions (microns)    */
    {if (cmd.match('/')){
      untested();
      cmd.skip();
      lo = cmd.ctopf() * MICRON2METER;
    }else{
      untested();
    }}
  }
  int here = cmd.cursor();
  do{
    get(cmd, "L",   &lo,    mPOSITIVE);
    get(cmd, "W",   &wo,    mPOSITIVE);
    get(cmd, "AD",  &ad_in, mPOSITIVE);
    get(cmd, "AS",  &as_in, mPOSITIVE);
    get(cmd, "PD",  &pd,    mPOSITIVE);
    get(cmd, "PS",  &ps,    mPOSITIVE);
    get(cmd, "NRD", &nrd,   mPOSITIVE);
    get(cmd, "NRS", &nrs,   mPOSITIVE);
  }while (cmd.more() && !cmd.stuck(&here));
  cmd.check(bWARNING, "what's this?");
}
/*--------------------------------------------------------------------------*/
void MOS_COMMON::print(OMSTREAM& where)const
{
  where << "  " << modelname();

  where.setfloatwidth(7);
  if (lo != NOT_INPUT)
    where << "  l="   << lo;
  if (wo != NOT_INPUT)
    where << "  w="   << wo;
  if (ad_in != NOT_INPUT)
    where << "  ad="  << ad_in;
  if (as_in != NOT_INPUT)
    where << "  as="  << as_in;
  if (pd != 0.)
    where << "  pd="  << pd;
  if (ps != 0.)
    where << "  ps="  << ps;
  where   << "  nrd=" << nrd;
  where   << "  nrs=" << nrs;
  
  if (icset){
    where << "  IC=";
    for (int i = 0;  i < mNUM_INIT_COND;  i++)
      where << ic[i] << ' ';
  }
  where << '\n';
}
/*--------------------------------------------------------------------------*/
const MODEL_MOS_BASE* MOS_COMMON::expand()
{
  const MODEL_MOS_BASE* m=dynamic_cast<const MODEL_MOS_BASE*>(attach_model());
  if (!m){
    untested();
    error(bERROR,"model " + modelname() + " is not a mosfet (NMOS or PMOS)\n");
  }

  if (sdp){
    delete sdp;
  }
  sdp = m->new_sdp(this);
  assert(sdp);
  const SDP_MOS_BASE* b = dynamic_cast<const SDP_MOS_BASE*>(sdp);
  assert(b);

  db.set_modelname(modelname());
  db.area = b->ad;
  db.perim = pd;
  db.is = b->idsat;
  db.cj = m->cbd;	/* if NOT_INPUT diode will calculate it */
  db.cjsw = NOT_INPUT;
  db.attach(model());
  db.off = true;

  sb.set_modelname(modelname());
  sb.area = b->as;
  sb.perim = ps;
  sb.is = b->issat;
  sb.cj = m->cbs;	/* if NOT_INPUT diode will calculate it */
  sb.cjsw = NOT_INPUT;
  sb.attach(model());
  sb.off = true;

  return m;
}
/*--------------------------------------------------------------------------*/
DEV_MOS::DEV_MOS()
{
  attach_common(&Default_MOS);
  std::fill_n(ids_y0, int(IDS_TERMS), 0.);
  std::fill_n(ids_y1, int(IDS_TERMS), 0.);
  std::fill_n(isub_y0, int(ISUB_TERMS), 0.);
  std::fill_n(isub_y1, int(ISUB_TERMS), 0.);
  ids = gm = gds = gmb = vgs = vds = vbs = vdsat = vgst = von = 0.;
  cutoff = subthreshold = saturated = reversed = dbfwd = sbfwd = 
  	punchthru = false;
  Rs = Rd = NULL;
  Ddb = Dsb = NULL;
  Cgs = Cgd = Cgb = NULL;
  G = NULL;
  ++_count;
}
/*--------------------------------------------------------------------------*/
DEV_MOS::DEV_MOS(const DEV_MOS& p):BASE_SUBCKT(p)
{
  std::fill_n(ids_y0, int(IDS_TERMS), 0.);
  std::fill_n(ids_y1, int(IDS_TERMS), 0.);
  std::fill_n(isub_y0, int(ISUB_TERMS), 0.);
  std::fill_n(isub_y1, int(ISUB_TERMS), 0.);
  ids = gm = gds = gmb = vgs = vds = vbs = vdsat = vgst = von = 0.;
  cutoff = subthreshold = saturated = reversed = dbfwd = sbfwd = 
  	punchthru = false;
  Rs = Rd = NULL;
  Ddb = Dsb = NULL;
  Cgs = Cgd = Cgb = NULL;
  G = NULL;
  ++_count;
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::parse(CS& cmd)
{
  const MOS_COMMON* cc = prechecked_cast<const MOS_COMMON*>(common());
  assert(cc);
  MOS_COMMON* c = new MOS_COMMON(*cc);
  assert(c);
  
  parse_Label(cmd);
  parse_nodes(cmd,numnodes(),numnodes());
  c->parse(cmd);
  attach_common(c);
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::print(OMSTREAM& where, int)const
{
  const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
  assert(c);

  where << short_label();
  printnodes(where,numnodes());
  c->print(where);
}
/*--------------------------------------------------------------------------*/
double DEV_MOS::tr_probe_num(CS& cmd)const
{
  const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
  assert(c);
  const SDP_MOS_BASE* b = prechecked_cast<const SDP_MOS_BASE*>(c->sdp);
  assert(b);
  const MODEL_MOS_BASE* m = prechecked_cast<const MODEL_MOS_BASE*>(c->model());
  assert(m);
  assert(subckt().exists());

  {if (cmd.pmatch("VDS")){
    return nDRAIN.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VGS")){
    return nGATE.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VBS")){
    return nBULK.v0() - nSOURCE.v0();
  }else if (cmd.pmatch("VDSInt")){
    return vds;
  }else if (cmd.pmatch("VGSInt")){
    return vgs;
  }else if (cmd.pmatch("VBSInt")){
    return vbs;
  }else if (cmd.pmatch("VGD")){
    return nGATE.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VBD")){
    return nBULK.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VSD")){
    return nSOURCE.v0() - nDRAIN.v0();
  }else if (cmd.pmatch("VDM")){
    return (nDRAIN.v0() - nSOURCE.v0()
	    + nDRAIN.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VGM")){
    return (nGATE.v0() - nSOURCE.v0()
	    + nGATE.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VBM")){
    return (nBULK.v0() - nSOURCE.v0()
	    + nBULK.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VSM")){
    return (nSOURCE.v0() - nSOURCE.v0()
	    + nSOURCE.v0() - nDRAIN.v0()) / 2.;
  }else if (cmd.pmatch("VDG")){
    return nDRAIN.v0() - nGATE.v0();
  }else if (cmd.pmatch("VBG")){
    return nBULK.v0() - nGATE.v0();
  }else if (cmd.pmatch("VSG")){
    return nSOURCE.v0() - nGATE.v0();
  }else if (cmd.pmatch("VDB")){
    return nDRAIN.v0() - nBULK.v0();
  }else if (cmd.pmatch("VGB")){
    return nGATE.v0() - nBULK.v0();
  }else if (cmd.pmatch("VSB")){
    return nSOURCE.v0() - nBULK.v0();
  }else if (cmd.pmatch("VD")){
    return nDRAIN.v0();
  }else if (cmd.pmatch("VG")){
    return nGATE.v0();
  }else if (cmd.pmatch("VB")){
    return nBULK.v0();
  }else if (cmd.pmatch("VS")){
    return nSOURCE.v0();
  }else if (cmd.pmatch("Id")){
    return (Rd)
      ?   CARD::probe(Rd,"I")
      :   CARD::probe(G,"I")
	- CARD::probe(Cgd,"I")
	+ CARD::probe(Ddb,"I") * m->polarity;
  }else if (cmd.pmatch("IS")){
    return (Rs)
      ?   CARD::probe(Rs,"I")
      : - CARD::probe(G,"I")
	- CARD::probe(Cgs,"I")
	+ CARD::probe(Dsb,"I") * m->polarity;
  }else if (cmd.pmatch("IG")){
    return CARD::probe(Cgs,"I")
         + CARD::probe(Cgd,"I")
	 + CARD::probe(Cgb,"I");
  }else if (cmd.pmatch("IB")){
    return - CARD::probe(Ddb,"I") * m->polarity
	   - CARD::probe(Dsb,"I") * m->polarity
	   - CARD::probe(Cgb,"I");
  }else if (cmd.pmatch("IBD")){
    return CARD::probe(Ddb,"I");
  }else if (cmd.pmatch("IBS")){
    return CARD::probe(Dsb,"I");
  }else if (cmd.pmatch("CGSOvl")){
    return CARD::probe(Cgs,"NV");
  }else if (cmd.pmatch("CGDOvl")){
    return CARD::probe(Cgd,"NV");
  }else if (cmd.pmatch("CGBOvl")){
    return CARD::probe(Cgb,"NV");
  }else if (cmd.pmatch("CGST")){
    return CARD::probe(Cgs,"EV");
  }else if (cmd.pmatch("CGDT")){
    return CARD::probe(Cgd,"EV");
  }else if (cmd.pmatch("CGBT")){
    return CARD::probe(Cgb,"EV");
  }else if (cmd.pmatch("CGSm")){
    return CARD::probe(Cgs,"EV") - CARD::probe(Cgs,"NV");
  }else if (cmd.pmatch("CGDm")){
    return CARD::probe(Cgd,"EV") - CARD::probe(Cgd,"NV");
  }else if (cmd.pmatch("CGBm")){
    return CARD::probe(Cgb,"EV") - CARD::probe(Cgb,"NV");
  }else if (cmd.pmatch("CBD")){
    return CARD::probe(Ddb,"Cap");
  }else if (cmd.pmatch("CBS")){
    return CARD::probe(Dsb,"Cap");
  }else if (cmd.pmatch("CGATE")){
    return b->cgate;
  }else if (cmd.pmatch("GM")){
    return gm;
  }else if (cmd.pmatch("GDS")){
    return gds;
  }else if (cmd.pmatch("GMB")){
    return gmb;
  }else if (cmd.pmatch("GBD")){
    return CARD::probe(Ddb,"G");
  }else if (cmd.pmatch("GBS")){
    return CARD::probe(Dsb,"G");
  }else if (cmd.pmatch("VGST")){
    return vgst;
  }else if (cmd.pmatch("VON")){
    return von;
  }else if (cmd.pmatch("VDSAT")){
    return vdsat * m->polarity;
  }else if (cmd.pmatch("VTH")){
    return von * m->polarity;
  }else if (cmd.pmatch("IDS")){
    return ids;
  }else if (cmd.pmatch("IDSTray")){
    return - CARD::probe(Cgd,"I")
           + CARD::probe(Ddb,"I") * m->polarity;
  }else if (cmd.pmatch("P")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"P");
    }      
    return power;
  }else if (cmd.pmatch("PD")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"PD");
    }      
    return fixzero(power, 1000000*tr_probe_num_str("P"));
  }else if (cmd.pmatch("PS")){
    double power = 0.;
    for (CARD_LIST::const_iterator
	   ci = subckt().begin(); ci != subckt().end(); ++ci){
      power += CARD::probe(*ci,"PS");
    }      
    return fixzero(power, 1000000*tr_probe_num_str("P"));
  }else if (cmd.pmatch("REgion")){
    return static_cast<double>(
	  (!cutoff)
	+ (!subthreshold * 2)
	+ (saturated * 4)
	+ (sbfwd * 10)
	+ (dbfwd * 20)
	+ (punchthru * 40)
    ) * ((reversed)? -1 : 1);
  }else if (cmd.pmatch("Status")){
    untested();
    return static_cast<double>(converged() * 2);
//  }else if (cmd.pmatch("DTNew")){
//    return timef - time0;
//  }else if (cmd.pmatch("DTOld")){
//    return time0 - time1;
//  }else if (cmd.pmatch("TIMEF")){
//    return timef;
//  }else if (cmd.pmatch("TIME")){
//    return time0;
//  }else if (cmd.pmatch("TIMEO")){
//    return time1;
  }else{
    return BASE_SUBCKT::tr_probe_num(cmd);
  }}
  /*NOTREACHED*/
}
/*--------------------------------------------------------------------------*/
bool DEV_MOS::tr_needs_eval()
{
  {if (is_q_for_eval()){
    untested();
    return false;
  }else if (!converged()){
    return true;
  }else{
    const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
    assert(c);
    const MODEL_MOS_BASE* m=prechecked_cast<const MODEL_MOS_BASE*>(c->model());
    assert(m);
    polarity_t polarity = m->polarity;
    node_t eff_source((reversed) ? drainnode : sourcenode);
    node_t eff_drain((reversed) ? sourcenode : drainnode);
    return !(conchk(vds,polarity*(eff_drain.v0()-eff_source.v0()),OPT::vntol)
	     && conchk(vgs, polarity*(nGATE.v0()-eff_source.v0()),OPT::vntol)
	     && conchk(vbs, polarity*(nBULK.v0()-eff_source.v0()),OPT::vntol));
  }}
}
/* on STATUS::iter[iSTEP] ..............
 * This forces nobypass on the first iteration of a new time step.
 * It hides a BUG, probably in review.
 * If an entire time step is bypassed, review doesn't realize it is ok
 * so it re-queues the same time (zero time step), even though the
 * numbers are ok.
 * The reason this is necessary is that a capacitor inside a mosfet may
 * need evaluation (integration) even if the whole device is bypassed.
 */
/*--------------------------------------------------------------------------*/
bool DEV_MOS::do_tr()
{
  const MOS_COMMON* c = prechecked_cast<const MOS_COMMON*>(common());
  assert(c);
  const MODEL_MOS_BASE* m = prechecked_cast<const MODEL_MOS_BASE*>(c->model());
  assert(m);
  assert(subckt().exists());

  double Vds, Vgs, Vbs;
  
  sourcenode.map();	/* BUG:: why is this here??.... */
  drainnode.map();	/* because these nodes get missed in the usual map */

  bool was_cutoff = cutoff;
  bool was_subthreshold = subthreshold;
  bool was_saturated = saturated;
  bool was_reversed = reversed;
  bool was_dbfwd = dbfwd;
  bool was_sbfwd = sbfwd;
  polarity_t polarity = m->polarity;

  {if (STATUS::iter[SIM::mode] <= 1){
    reversed = false;
    Vds = Vgs = Vbs = 0.;
    if (OPT::mosflags & 0100){
      untested();
      //Vds = m->vto;
    }
    if (OPT::mosflags & 0200){
      untested();
      //Vgs = m->vto;
    }
    if (OPT::mosflags & 0400  &&  nSOURCE.t != nBULK.t){
      untested();
      Vbs = -1;
    }
    if (nDRAIN.t == nGATE.t){
      Vds = Vgs;
    }
  }else if (reversed){
    Vds = polarity * volts_limited(sourcenode,drainnode);
    Vgs = polarity * volts_limited(nGATE,drainnode);
    Vbs = polarity * volts_limited(nBULK,drainnode);
  }else{
    Vds = polarity * volts_limited(drainnode,sourcenode);
    Vgs = polarity * volts_limited(nGATE,sourcenode);
    Vbs = polarity * volts_limited(nBULK,sourcenode);
  }}

  limit_mos(Vds, Vgs, Vbs);// also sets the new vds,vgs,vbs
  if (vds < 0){
    error(bTRACE, long_label() + ": reversing\n");
    error(bTRACE, "before: vds=%g vgs=%g vbs=%g\n", vds, vgs, vbs);
    reversed = !reversed;
    vgs -= vds;
    vbs -= vds;
    vds = -vds;
    error(bTRACE, "after: vds=%g vgs=%g vbs=%g\n", vds, vgs, vbs);
    if (OPT::dampstrategy & dsREVERSE){
      SIM::fulldamp = true;
      untested();
      error(bTRACE, long_label() + ":reverse damp\n");
    }
    {if (!(OPT::mosflags & 0040)){
      vbs = std::min(vbs,0.);
    }else{
      untested();
    }}
  }
  m->tr_eval(this);

  /* Normally, ids >= 0, vds >= 0, but this is not guaranteed.
   * See "cannot reverse" above.
   */
  ids_y0[IDS] = polarity * ids;
  ids_y0[GDS] = gds;
  ids_y0[GMF] = gm;
  ids_y0[GMR] = 0.;
  ids_y0[GMBF] = gmb;
  ids_y0[GMBR] = 0.;
  if (reversed){
    ids_y0[IDS] = -ids_y0[IDS];
    ids_y0[GMR] = ids_y0[GMF];
    ids_y0[GMF] = 0.;
    ids_y0[GMBR] = ids_y0[GMBF];
    ids_y0[GMBF] = 0.;
  }
  
  set_converged(subckt().do_tr());
  
  if (vds > 100){
    untested();
    error(bDEBUG,"%s:%d: ", long_label().c_str(), evaliter());
    error(bDEBUG,"vds=%e vgs=%e vbs=%e\n",
		  vds,   vgs,   vbs);
    error(bDEBUG,"+      ids=%e gm=%e gds=%e gmb=%e\n",
    			 ids,   gm,   gds,   gmb);
  }
  trace3(long_label().c_str(), vds, vgs, vbs);
  trace4("", ids, gm, gds, gmb);
  if (was_cutoff != cutoff  ||  was_subthreshold != subthreshold  
  	||  was_saturated != saturated  ||  was_reversed != reversed  
	||  was_dbfwd != dbfwd  ||  was_sbfwd != sbfwd){
    if (OPT::dampstrategy & dsDEVREGION){
      SIM::fulldamp = true;
    }
    #if defined(DO_TRACE)
      error(bTRACE,"%s:%d: region change\n", long_label().c_str(), evaliter());
    #endif
  }
  return converged();
}
/*--------------------------------------------------------------------------*/
/* limit_mos: do Spice style voltage limiting
 */
void DEV_MOS::limit_mos(double Vds, double Vgs, double Vbs)
{
					/* Spice style vgs limiting */
  {if (!(OPT::mosflags & 0010) && STATUS::iter[SIM::mode] > 1){
    //assert(vgst == vgs - von);
    {if (Vgs > vgs){			/* increasing */
      {if (vgst < 0){			/* was off */
	vgs = std::min(Vgs, von + .5);
      }else if (vgst < 3.5){		/* ??? */
	vgs = std::min(Vgs, von + 4.);
      }else{
	vgs = std::min(Vgs, 3.*vgs - 2.*von + 2.);
      }}
    }else{				/* decreasing */
      {if (vgst < 0){			/* off */
	vgs = std::max(Vgs, 3. * vgs - 2. * von - 2.);
      }else if (vgst < 3.5){
	vgs = std::max(Vgs, von - .5);
      }else{
	vgs = std::max(Vgs, von + 2.);
      }}
    }}
    //Vds += vgs - Vgs;			/* vds patch (per Spice) not done */
  }else{				/* because it usually makes it worse */
    vgs = Vgs;
  }}
  {if (nDRAIN.t == nGATE.t){		/* gd tied, limiting done */
    vds = Vds + (vgs - Vgs);		/* it seems that vds = vgs should */
    					/* work, but it will be a little off */
					/* if there is resistance */

					/* Spice style vds limiting */
  }else if (!(OPT::mosflags & 0020) && STATUS::iter[SIM::mode] > 1){
    {if (Vds <= vds){			/* decreasing */
      {if (vds < 3.5){
	vds = std::max(Vds,-.5);
      }else{
	vds = std::max(Vds,2.);
      }}
    }else{				/* increasing */
      {if (vds < 3.5){
	vds = std::min(Vds,4.);
      }else{
	vds = std::min(Vds, 3.*vds + 2.);
      }}
    }}
    //vgs += vds - Vds;
  }else{
    vds = Vds;
  }}

  {if (!(OPT::mosflags & 0040) && STATUS::iter[SIM::mode] > 1){
//    if (Vbs > 0.){
//      if (vbs >= 0.){
//        vbs = std::min(Vbs,vbs+.1);
//      }else{
//        vbs = 0.;
//      }
//    }else{
//      vbs = Vbs;
//    }
    vbs = std::min(Vbs,0.);
  }else{
    vbs = Vbs;
  }}
  if (OPT::dampstrategy & dsDEVLIMIT
      && (vgs != Vgs || vds != Vds || vbs != Vbs)){
    untested();
    SIM::fulldamp = true;
    error(bTRACE, long_label() + ":device limit damp\n");
  }
  trace1(long_label().c_str(), evaliter());
  trace3("prop", Vds, Vgs, Vbs);
  trace3("using", vds, vgs, vbs);
}
/*--------------------------------------------------------------------------*/
void DEV_MOS::expand()
{
  const MOS_COMMON* cc = prechecked_cast<const MOS_COMMON*>(common());
  assert(cc);
  MOS_COMMON* c = const_cast<MOS_COMMON*>(cc);
  assert(c);
  const MODEL_MOS_BASE* m = c->expand();
  assert(m);
  const SDP_MOS_BASE* b = prechecked_cast<const SDP_MOS_BASE*>(c->sdp);
  assert(b);

  /* It seems to matter what order these parts are placed in.
   * Changing "push_front" to "push_back" changes the results.
   * Differences are small, probably insignificant,
   * probably due to differences in accumulated round-off errors.
   * The differences show with values close to zero,
   * that probably should be zero, but have values ~~1e-40 either way.
   * Not sure what is the best ordering, or if this is significant.
   */

  /* It would be easier, and perhaps faster, to clear out the whole 
   * list and rebuild.  The reason for this apparent excess is to
   * preserve probes on these internal elements.
   */
							       /* resistors */
  {if (OPT::rstray && b->rs != 0.){
    //untested();
    sourcenode.t = STATUS::newnode_model();
    {if (!Rs){
      Rs = new DEV_RESISTANCE;
      subckt().push_front(Rs);
      //untested();
    }else{
      untested();
    }}
    Rs->set("Rs", this, NULL, b->rs, nSOURCE, sourcenode);
  }else{
    sourcenode = nSOURCE;
    if (Rs){
      subckt().erase(Rs);
      Rs = NULL;
      untested();
    }
  }}

  {if (OPT::rstray && b->rd != 0.){
    //untested();
    drainnode.t = STATUS::newnode_model();
    {if (!Rd){
      Rd = new DEV_RESISTANCE;
      subckt().push_front(Rd);
      //untested();
    }else{
      untested();
    }}
    Rd->set("Rd", this, NULL, b->rd, nDRAIN, drainnode);
  }else{
    drainnode = nDRAIN;
    if (Rd){
      subckt().erase(Rd);
      Rd = NULL;
      untested();
    }
  }}

  {if (nBULK.t != drainnode.t   &&   b->idsat != 0.){
    if (!Ddb){
      Ddb = new DEV_DIODE;
      subckt().push_front(Ddb);
    }
    switch (m->polarity){
    case pN:
      Ddb->set("Ddb", this, &(c->db), 0., nBULK, drainnode);
      break;
    case pP:
      Ddb->set("Ddb", this, &(c->db), 0., drainnode, nBULK);
      break;
    }
  }else{
    if (Ddb){
      subckt().erase(Ddb);
      Ddb = NULL;
      untested();
    }
  }}

  {if (nBULK.t != sourcenode.t   &&   b->issat != 0.){
    if (!Dsb){
      Dsb = new DEV_DIODE;
      subckt().push_front(Dsb);
    }
    switch (m->polarity){
    case pN:
      Dsb->set("Dsb", this, &(c->sb), 0., nBULK, sourcenode);
      break;
    case pP:
      Dsb->set("Dsb", this, &(c->sb), 0., sourcenode, nBULK);
      break;
    }
  }else{
    if (Dsb){
      subckt().erase(Dsb);
      Dsb = NULL;
      untested();
    }
  }}

  {if (OPT::cstray  &&  nGATE.t != sourcenode.t){
    if (!Cgs){
      Cgs = new DEV_CAPACITANCE;
      subckt().push_front(Cgs);
    }
    Cgs->set("Cgs", this, &Eval_Cgs, (m->cgso*b->we), nGATE, sourcenode);
  }else{
    if (Cgs){
      subckt().erase(Cgs);
      Cgs = NULL;
      untested();
    }
  }}

  {if (OPT::cstray  &&  nGATE.t != drainnode.t){
    if (!Cgd){
      Cgd = new DEV_CAPACITANCE;
      subckt().push_front(Cgd);
    }
    Cgd->set("Cgd", this, &Eval_Cgd, (m->cgdo*b->we), nGATE, drainnode);
  }else{
    if (Cgd){
      subckt().erase(Cgd);
      Cgd = NULL;
      untested();
    }
  }}

  {if (OPT::cstray  &&  nBULK.t != nGATE.t){
    if (!Cgb){
      Cgb = new DEV_CAPACITANCE;
      subckt().push_front(Cgb);
    }
    Cgb->set("Cgb", this, &Eval_Cgb, (m->cgbo*b->le), nGATE, nBULK);
  }else{
    {if (Cgb){
      subckt().erase(Cgb);
      Cgb = NULL;
      untested();
    }else{
      untested();
    }}
  }}

  ids_nodes[GDS*2-1]  = sourcenode;
  ids_nodes[GDS*2-2]  = drainnode;
  ids_nodes[GMF*2-1]  = sourcenode;
  ids_nodes[GMF*2-2]  = nGATE;
  ids_nodes[GMR*2-1]  = nGATE;
  ids_nodes[GMR*2-2]  = drainnode;
  ids_nodes[GMBF*2-1] = sourcenode;
  ids_nodes[GMBF*2-2] = nBULK;
  ids_nodes[GMBR*2-1] = nBULK;
  ids_nodes[GMBR*2-2] = drainnode;

  if (!G){
    G = new DEV_POLY_G("Ids", IDS_TERMS-1, ids_nodes, ids_y0, ids_y1, this);
    subckt().push_front(G);
  }

  assert(subckt().exists());
  subckt().expand();
  subckt().set_slave();

  assert(!constant()); /* because it is nonlinear */
}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
