/*$Id: d_diode.cc,v 18.17 2000/09/04 09:45:33 al Exp $ -*- C++ -*-
 * diode model.
 * netlist syntax:
 * device:  dxxxx n+ n- mname <area> <off> <ic=vd> <model-card-args>
 * model:   .model mname D <args>
 */
#include "e_aux.h"
#include "d_admit.h"
#include "d_cap.h"
#include "ap.h"
#include "d_diode.h"
/*--------------------------------------------------------------------------*/
// implemented here
class MODEL_DIODE;
class DIODE_COMMON;
class DEV_DIODE;
class EVAL_DIODE_Yj;
class EVAL_DIODE_Cj;
/*--------------------------------------------------------------------------*/
int DEV_DIODE::_count = 0;
int DIODE_COMMON::_count = -3;
int MODEL_DIODE::_count = 0;
static EVAL_DIODE_Cj Eval_Cj(CC_STATIC);
static EVAL_DIODE_Yj Eval_Yj(CC_STATIC);
static DIODE_COMMON Default_DIODE(CC_STATIC);
const double default_pb = 1.0;
const double default_mjsw = 0.33;
/*--------------------------------------------------------------------------*/
MODEL_DIODE::MODEL_DIODE()
  :MODEL_CARD(),
   js	    (1e-14),
   rs	    (0.0),
   n_factor (1.0),
   tt	    (0.0),
   cjo	    (NOT_INPUT),
   pb	    (NOT_INPUT),
   mj	    (0.5),
   eg	    (1.11),
   xti	    (3.0),
   kf	    (NOT_INPUT),
   af	    (NOT_INPUT),
   fc	    (0.5),
   bv	    (NOT_INPUT),	/* infinity */
   ibv	    (1e-3),
   cjsw	    (0.0),
   pbsw	    (NOT_INPUT),
   mjsw	    (NOT_INPUT),
   gparallel(0.0),
   flags    (USE_OPT),
   mos_level(0)
{
  ++_count;
}
/*--------------------------------------------------------------------------*/
bool MODEL_DIODE::parse_front(CS& cmd)
{
  cmd.skiparg();			/* skip known "d" */
  return true;
}
/*--------------------------------------------------------------------------*/
void MODEL_DIODE::parse_params(CS& cmd)
{
    get(cmd, "TNOM",	&_tnom,	    mOFFSET,	-ABS_ZERO);
    get(cmd, "IS",	&js, 	    mPOSITIVE);
    get(cmd, "RS",	&rs, 	    mPOSITIVE);
    get(cmd, "N",   	&n_factor,  mPOSITIVE);
    get(cmd, "TT",  	&tt, 	    mPOSITIVE);
    get(cmd, "CJo", 	&cjo,	    mPOSITIVE);
    get(cmd, "VJ",  	&pb, 	    mPOSITIVE);
    get(cmd, "PB",  	&pb, 	    mPOSITIVE);
    get(cmd, "Mj",  	&mj, 	    mPOSITIVE);
    get(cmd, "EGap",	&eg, 	    mPOSITIVE);
    get(cmd, "XTI", 	&xti,	    mPOSITIVE);
    get(cmd, "KF",  	&kf, 	    mPOSITIVE);
    get(cmd, "AF",  	&af, 	    mPOSITIVE);
    get(cmd, "FC",  	&fc, 	    mPOSITIVE);
    get(cmd, "BV",  	&bv,	    mPOSITIVE);
    get(cmd, "IBV", 	&ibv,	    mPOSITIVE);
    get(cmd, "CJSw",	&cjsw,	    mPOSITIVE);
    get(cmd, "PBSw",	&pbsw,	    mPOSITIVE);
    get(cmd, "MJSw",	&mjsw,	    mPOSITIVE);
    get(cmd, "GParallel",&gparallel);
    get(cmd, "FLAgs",	&flags,	    mOCTAL);
    set(cmd, "NOFLAGS",	&flags,	    static_cast<int>(USE_OPT));
}
/*--------------------------------------------------------------------------*/
void MODEL_DIODE::post_parse()
{
  {if (_tnom == NOT_INPUT) {
    _tnom = OPT::tnom;
  }else{
    untested();
  }}
  if (pb == NOT_INPUT) {
    untested();
    pb = default_pb;
  }
  {if (pbsw == NOT_INPUT) {
    pbsw = pb;
  }else{
    untested();
  }}
  {if (mjsw == NOT_INPUT) {
    mjsw = default_mjsw;
  }else{
    untested();
  }}
  if (cjo == NOT_INPUT) {
    untested();
    cjo = 0.0;
  }
  if (bv == 0.) {
    bv = NOT_INPUT;
  }
}
/*--------------------------------------------------------------------------*/
void MODEL_DIODE::print_params(OMSTREAM& where)const
{
  where	<<   "tnom="<< _tnom+ABS_ZERO;
  where << "  is="  << js;
  where << "  rs="  << rs;
  where << "  n="   << n_factor;
  where << "  tt="  << tt;
  where << "  cjo=" << cjo;
  where << "  vj="  << pb;
  where << "  m="   << mj;
  where << "  eg="  << eg;
  where << "  xti=" << xti;
  if (kf != NOT_INPUT)
    where << "  kf="  << kf;
  if (af != NOT_INPUT)
    where << "  af="  << af;
  where << "  fc="  << fc;
  if (bv != NOT_INPUT) {
    where << "  bv="  << bv;
    where << "  ibv=" << ibv;
  }
  if (cjsw != 0.) {
    where << "  cjsw=" << cjsw;
    where << "  pbsw=" << cjsw;
    where << "  mjsw=" << mjsw;
  }
  if (gparallel != 0) {
    where << "  gparallel=" << gparallel;
  }
  if (!(flags & USE_OPT)) {
    where << "  flags="<< flags;
  }
}
/*--------------------------------------------------------------------------*/
DIODE_COMMON::DIODE_COMMON(int c)
  :COMPONENT_COMMON(c),
   is	    (NOT_INPUT),
   rs	    (NOT_INPUT),
   cj	    (NOT_INPUT),
   cjsw	    (NOT_INPUT),
   area	    (1.0),
   perim    (0.0),
   gparallel(NOT_INPUT),
   ic	    (NOT_INPUT),
   off	    (false)
{
  calc.is = calc.rs = calc.cj = calc.cjsw = calc.gparallel = false;
  ++_count;
}
/*--------------------------------------------------------------------------*/
DIODE_COMMON::DIODE_COMMON(const DIODE_COMMON& p, int c)
  :COMPONENT_COMMON(p,c),
   is	    (p.is),
   rs	    (p.rs),
   cj	    (p.cj),
   cjsw	    (p.cjsw),
   area	    (p.area),
   perim    (p.perim),
   gparallel(p.gparallel),
   ic	    (p.ic),
   off	    (p.off),
   calc	    (p.calc)
{
  ++_count;
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::parse(CS& cmd)
{
  assert(!has_model());

  parse_modelname(cmd);
  if (cmd.is_pfloat()) {
    area = cmd.ctopf();
  }
  int here = cmd.cursor();
  do{
    get(cmd, "Area",	&area,      mPOSITIVE);
    get(cmd, "Perim",	&perim,     mPOSITIVE);
    get(cmd, "IC",	&ic);
    get(cmd, "OFF",	&off);
    get(cmd, "IS",	&is,        mPOSITIVE);
    get(cmd, "Rs",	&rs,        mPOSITIVE);
    get(cmd, "Cjo",	&cj,        mPOSITIVE);
    get(cmd, "CJSW",	&cjsw,	    mPOSITIVE);
    get(cmd, "GParallel",&gparallel);
  }while (cmd.more() && !cmd.stuck(&here));
  cmd.check(bWARNING, "what's this?");
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::print(OMSTREAM& where)const
{
  where << "  " << modelname();

  where.setfloatwidth(7);
  where << ' ' << area;
  if (perim != 0.)
    where << "  perim=" << perim;
  if (off)
    where << "  off";
  if (ic != NOT_INPUT)
    where << "  ic=" <<  ic;
  if (!calc.is  &&  is != NOT_INPUT)
    where << "  is=" <<  is;
  if (!calc.rs  &&  rs != NOT_INPUT)
    where << "  rs=" <<  rs;
  if (!calc.cj  &&  cj != NOT_INPUT)
    where << "  cj=" <<  cj;
  if (!calc.cjsw  &&  cjsw != NOT_INPUT)
    where << "  cjsw=" << cjsw;
  if (!calc.gparallel && gparallel != NOT_INPUT)
    where << "  gparallel=" << gparallel;
  if (calc.is  &&  calc.rs  &&  calc.cj  &&  calc.cjsw)
    where << "\n*+";
  if (calc.is  &&  is != NOT_INPUT)
    where << "  is=" <<  is;
  if (calc.rs  &&  rs != NOT_INPUT)
    where << "  rs=" <<  rs;
  if (calc.cj  &&  cj != NOT_INPUT)
    where << "  cj=" <<  cj;
  if (calc.cjsw  &&  cjsw != NOT_INPUT)
    where << "  cjsw=" << cjsw;
  if (calc.gparallel && gparallel != NOT_INPUT)
    where << "  gparallel=" << gparallel;
  where << '\n';
}
/*--------------------------------------------------------------------------*/
void DIODE_COMMON::expand()
{
  const MODEL_DIODE* m = dynamic_cast<const MODEL_DIODE*>(attach_model());
  if (!m) {
    untested();
    error(bERROR, "model " + modelname() + " is not a diode (D)\n");
  }
  
  if (calc.is  ||  is == NOT_INPUT) {
    is = m->js * area;
    calc.is = true;
  }
  {if (calc.rs  ||  rs == NOT_INPUT) {
    rs = m->rs / area;
    calc.rs = true;
  }else{
    untested();
  }}
  if (calc.cj  ||  cj == NOT_INPUT) {
    cj = m->cjo * area;
    calc.cj = true;
  }
  {if (calc.cjsw  ||  cjsw == NOT_INPUT) {
    cjsw = m->cjsw * perim;
    calc.cjsw = true;
  }else{
    untested();
  }}
  if (calc.gparallel || gparallel == NOT_INPUT) {
    gparallel = m->gparallel * area;
  }
}
/*--------------------------------------------------------------------------*/
DEV_DIODE::DEV_DIODE()
  :BASE_SUBCKT(),
   _region(UNKNOWN),
   _isat (NOT_VALID),
   _gd	 (NOT_VALID),
   _Yj	 (0),
   _Cj	 (0)
{
  attach_common(&Default_DIODE);
  ++_count;
}
/*--------------------------------------------------------------------------*/
DEV_DIODE::DEV_DIODE(const DEV_DIODE& p)
  :BASE_SUBCKT(p),
   _region(UNKNOWN),
   _isat (NOT_VALID),
   _gd	 (NOT_VALID),
   _Yj	 (0),
   _Cj	 (0)
{
  untested();
  ++_count;
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::parse(CS& cmd)
{
  const DIODE_COMMON* cc = prechecked_cast<const DIODE_COMMON*>(common());
  assert(cc);
  DIODE_COMMON* c = new DIODE_COMMON(*cc);
  assert(c);
  
  parse_Label(cmd);
  parse_nodes(cmd,numnodes(),numnodes());
  c->parse(cmd);
  attach_common(c);
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::print(OMSTREAM& where, int)const
{
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(common());
  assert(c);

  where << short_label();
  printnodes(where,numnodes());
  c->print(where);
}
/*--------------------------------------------------------------------------*/
void DEV_DIODE::expand()
{
  const DIODE_COMMON* cc = prechecked_cast<const DIODE_COMMON*>(common());
  assert(cc);
  DIODE_COMMON* c = const_cast<DIODE_COMMON*>(cc);
  assert(c);
  c->expand();

  {if (c->cj != 0.  ||  c->cjsw != 0.) {
    if (!_Cj) {
      _Cj = new DEV_CAPACITANCE;
      subckt().push_front(_Cj);
    }
    _Cj->set("Cj", this, &Eval_Cj, 0., _n[OUT1], _n[OUT2]);
  }else{
    if (_Cj) {
      subckt().erase(_Cj);
      _Cj = NULL;
      untested();
    }
  }}

  if (!_Yj) {
    _Yj = new DEV_ADMITTANCE;
    subckt().push_front(_Yj);
  }
  _Yj->set("Yj", this, &Eval_Yj, 0., _n[OUT1], _n[OUT2]);

  assert(subckt().exists());
  subckt().expand();
  assert(!constant()); /* because it is nonlinear */
}
/*--------------------------------------------------------------------------*/
double DEV_DIODE::tr_probe_num(CS& cmd)const
{
  if (cmd.pmatch("Vd")) {
    return _n[OUT1].v0() - _n[OUT2].v0();
  }else if (cmd.pmatch("Id")) {
    return CARD::probe(_Yj,"I") + CARD::probe(_Cj,"I");
  }else if (cmd.pmatch("IJ")) {
    untested();
    return CARD::probe(_Yj,"I");
  }else if (cmd.pmatch("IC")) {
    untested();
    return CARD::probe(_Cj,"I");
  }else if (cmd.pmatch("P")) {
    return CARD::probe(_Yj,"P") + CARD::probe(_Cj,"P");
  }else if (cmd.pmatch("PD")) {
    return CARD::probe(_Yj,"PD") + CARD::probe(_Cj,"PD");
  }else if (cmd.pmatch("PS")) {
    return CARD::probe(_Yj,"PS") + CARD::probe(_Cj,"PS");
  }else if (cmd.pmatch("PJ")) {
    untested();
    return CARD::probe(_Yj,"P");
  }else if (cmd.pmatch("PC")) {
    untested();
    return CARD::probe(_Cj,"P");
  }else if (cmd.pmatch("Capacitance")) {
    return CARD::probe(_Cj,"EV");
  }else if (cmd.pmatch("Req")) {
    return CARD::probe(_Yj,"R");
  }else if (cmd.pmatch("Geq")) {
    return CARD::probe(_Yj,"Y");
  }else if (cmd.pmatch("Y")) {
    return CARD::probe(_Yj,"Y") + CARD::probe(_Cj,"Y");
  }else if (cmd.pmatch("Z")) {
    return port_impedance(_n[OUT1], _n[OUT2], lu, tr_probe_num_str("Y"));
  }else if (cmd.pmatch("ZRAW")) {
    return port_impedance(_n[OUT1], _n[OUT2], lu, 0.);
  }else if (cmd.pmatch("REgion")) {
    return static_cast<double>(_region);
  }else{
    untested();
    return BASE_SUBCKT::tr_probe_num(cmd);
  }
}
/*--------------------------------------------------------------------------*/
/* this function is a mess */
void EVAL_DIODE_Yj::tr_eval(ELEMENT* d)const
{
  DEV_DIODE* p = prechecked_cast<DEV_DIODE*>(d->owner());
  assert(p);
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(p->common());
  assert(c);
  const MODEL_DIODE* m = prechecked_cast<const MODEL_DIODE*>(c->model());
  assert(m);

  assert(m->_tnom > 0);

  FPOLY1& y = d->_y0;
  double volts = y.x;
  double amps  = y.f0;
  trace2(d->long_label().c_str(), volts, amps);

  int flags = (m->flags & m->USE_OPT) ? OPT::diodeflags : m->flags;
  double tempratio = SIM::temp / m->_tnom;
  double vt = KoQ * SIM::temp * m->n_factor;
  region_t oldregion = p->_region;
  p->_isat = c->is * pow(tempratio, m->xti) * Exp((m->eg/vt) * (tempratio-1));
  trace4("", tempratio, vt, oldregion, p->_isat);

  if (m->mos_level > 0 || flags & 0040) { // Spice style limiting
    double vcrit = vt * log(vt/(kRT2*p->_isat));
    double vold = d->_y1.f0;
    {if((volts > vcrit) && (abs(volts - vold) > (vt + vt))) {
      {if(vold > 0) {
	double arg = 1 + (volts - vold) / vt;
	{if(arg > 0) {
	  volts = vold + vt * log(arg);
	}else{
	  volts = vcrit;
	}}
      }else{
	volts = vt *log(volts/vt);
      }}
    }else{
      // leave volts as is
    }}
  }

  {if (m->mos_level > 0 && m->mos_level < 8) {
    switch (m->mos_level) {
    case 1:
    case 2:
    case 3:
    case 6:
    case 4:
    case 5:
      {if (volts <= 0.) {
	p->_region = REVERSE;
	y.f1 = p->_isat / vt + OPT::gmin;
	y.f0 = y.f1 * volts;
      }else{
	p->_region = FORWARD;
	double ev = exp(volts/vt);
	y.f1 = p->_isat * ev / vt + OPT::gmin;
	y.f0 = p->_isat * (ev - 1) + OPT::gmin * volts;
      }}
      break;
    case 7:
      {if (volts < .5) {
	p->_region = REVERSE;
	double ev = exp(volts/vt);
	y.f1 = p->_isat * ev / vt + OPT::gmin;
	y.f0 = p->_isat * (ev - 1) + OPT::gmin * volts;
      }else{
	p->_region = FORWARD;
	double ev = exp(.5/vt);
	double t0 = p->_isat * ev / vt;
	y.f1 = t0 + OPT::gmin;
	y.f0 = p->_isat * (ev - 1) + t0 * (volts - .5) + OPT::gmin * volts;
      }}
      break;
    default:
      unreachable();
      y.f1 = OPT::gmin;
      y.f0 = volts * y.f1;
    }
  }else if (flags & 0040) { // exact Spice model
    {if (volts >= -3*vt) { // forward and weak reversed
      double evd = Exp(volts/vt);
      y.f0 = p->_isat * (evd-1);
      y.f1 = p->_isat * evd/vt;
    }else if (m->bv == NOT_INPUT || volts >= m->bv) {
      double arg = 3 * vt / (volts * kE); // strong reversed
      arg = arg * arg * arg;
      y.f0 = -p->_isat * (1+arg);
      y.f1 = p->_isat * 3 * arg / volts;
    }else{
      incomplete();
      double evrev = exp(-(m->bv+volts)/vt);
      y.f0 = -p->_isat * evrev;
      y.f1 = p->_isat * evrev / vt;
    }}
    y.f0 += OPT::gmin * volts;
    y.f1 += OPT::gmin;
  }else{
    {if (c->off  &&  STATUS::iter[SIM::mode] <= 1) { /* initially guess off */
      p->_region = INITOFF;
      y.f1 = 0.;
      y.f0 = 0.;
      if (flags & 0020) {
	untested();
	y.f1 = OPT::gmin;
      }
      trace2("initoff", y.f0, y.f1);
    }else if (volts <= 0. /* &&  amps < 0.*/) {    	  /* reverse biased */
      p->_region = REVERSE;	    		  /* x = volts, f(x) = amps */
      {if (flags & 0010) {
	untested();
	y.f1 = y.f0 = 0.;
      }else{
	double expterm = p->_isat * Exp(volts/vt);	
	y.f0 = expterm - p->_isat;  /* i = f(x) = _isat * (Exp(volts/vt)-1) */
	y.f1 = expterm / vt;	    /* f'(x) = (_isat/vt) * Exp(volts/vt)   */
      }}

      if (flags & 0002) {	// g = gmin, maintain actual current
	y.f1 += OPT::gmin;	// 3 is a resistor, R=1/gmin
	y.f0 += OPT::gmin * volts;
      }
      if (flags & 0004) {	// 5 is a resistor, R=vt/_isat
	double x = p->_isat / vt;
	y.f1 += x;
	y.f0 += x * volts;
      }
      if (flags & 0001) {
	//y.f0 = y.f1 * volts;	// a resistor, R=1/f1
      }

      trace2("reverse", y.f0, y.f1);
    }else if (volts >= 0.  &&  amps >= 0.) {		  /* forward biased */
				    /* x = amps, f(x) = volts */
      /* derivation: */		    /* if f(x) = log(u): f'(x)=(1/u)(du/dx) */
      /* poly1 r; */
      /* r.f0 = vt * log(amps/p->_isat +1.); */
      /* r.f1 = vt / (_isat + amps); */
      /* y.f1 = 1. / r.f1; */
      /* y.f0 = amps - r.f0*y.f1 + volts*y.f1; */
      
      p->_region = FORWARD;
      y.f1 = (p->_isat + amps) / vt;
      y.f0 = amps - log(amps/p->_isat +1.)*(p->_isat + amps) + volts*y.f1;
      trace2("forward", y.f0, y.f1);
    }else{			    /* non-converged, inconsistent	    */
      p->_region = UNKNOWN;	    /* volts and amps have different signs  */
      y.f1 = p->_isat/vt;	    /* guess that the voltage should be 0   */
      y.f0 = 0.;		    /* (it usually is very close)	    */
      if (flags & 0001) {	    /* use the correct value there	    */
	y.f0 = volts * y.f1;
      }
      trace2("unknown", y.f0, y.f1);
    }}
    y.f1 += c->gparallel;
    y.f0 += c->gparallel * volts;
    
    if (oldregion != p->_region  &&  OPT::dampstrategy & dsDEVLIMIT) {
      SIM::fulldamp = true;
      untested();
      error(bTRACE, p->long_label() + ":device limit damp\n");
    }
    if (flags & 0100) {		// twist g to guarantee g >= gmin
      {if (y.f1 < OPT::gmin) {	// without changing i
	y.f1 = OPT::gmin;
	untested();
      }else{
	untested();
      }}
    }
    if (flags & 0200) {		// add a gmin in parallel
      y.f1 += OPT::gmin;
      y.f0 += OPT::gmin * volts;
      untested();
    }
    if (flags & 0400) {		// linearize .. shift I to pass thru 0
      untested();
      y.f0 = y.f1 * volts;
    }
  }}
  trace3(d->long_label().c_str(), y.x, y.f0, y.f1);
  p->_gd = y.f1;
}
/*--------------------------------------------------------------------------*/
void EVAL_DIODE_Cj::tr_eval(ELEMENT* raw_d)const
{
  assert(raw_d);
  DEV_CAPACITANCE* d = prechecked_cast<DEV_CAPACITANCE*>(raw_d);
  assert(d);
  const DEV_DIODE* p = prechecked_cast<const DEV_DIODE*>(d->owner());
  assert(p);
  const DIODE_COMMON* c = prechecked_cast<const DIODE_COMMON*>(p->common());
  assert(c);
  const MODEL_DIODE* m = prechecked_cast<const MODEL_DIODE*>(c->model());
  assert(m);

  double& volts = d->_y0.x;
  trace1(d->long_label().c_str(), volts);

  double cb;
  {if (c->cj != 0.) {
    {if (volts < m->fc * m->pb) {
      cb = c->cj / pow(1. - (volts / m->pb),  m->mj);
    }else{
      cb = (c->cj / pow(1. - m->fc, 1. + m->mj))
	* (1. - m->fc*(1.+m->mj) + (volts/m->pb)*m->mj);
    }}
  }else{
    untested();
    cb = 0.;
  }}
  assert(cb >= 0.);

  double csw;
  {if (c->cjsw != 0.) {
    {if (volts < m->fc * m->pbsw) {
      csw = c->cjsw / pow(1. - (volts / m->pbsw),  m->mjsw);
    }else{
      csw = (c->cjsw / pow(1. - m->fc, 1. + m->mjsw))
	* (1. - m->fc*(1.+m->mjsw) + (volts/m->pbsw)*m->mjsw);
    }}
  }else{
    csw = 0.;
  }}
  assert(csw >= 0.);
  
  double ctt;
  {if (m->tt != 0.) {
    untested();
    ctt = p->_gd * m->tt;
  }else{
    ctt = 0.;
  }}
  assert(ctt >= 0.);
  
  trace4("", cb, csw, ctt, cb+csw+ctt);
  d->_y0.f1 = cb + csw + ctt;
  {if (SIM::phase == SIM::pTRAN) {
    double cap = (d->_y0.f1 + d->_q[1].f1) / 2;
    d->_y0.f0 = (d->_y0.x - d->_q[1].x) * cap + d->_q[1].f0;
  }else{
    d->_y0.f0 = d->_y0.x * d->_y0.f1;
  }}
  trace3(d->long_label().c_str(), d->_y0.x, d->_y0.f0, d->_y0.f1);
}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
