/*
 * $Id: learn.c,v 1.8 1997/11/19 17:51:20 ukrebeld Exp ukrebeld $
 *
 * ex:set sw=8 ai:
 */

/*
 * Copyright (C) Udo Krebelder <ukrebeld@nt.tuwien.ac.at> 1997.
 * All rights reserved.
 * 
 * Permission to use, copy, modify and freely distribute this software
 * for non-commercial and not-for-profit purposes is hereby granted
 * without fee, provided that both the above copyright notice and
 * this permission notice appear in all copies and in supporting
 * documentation.
 * The author makes no representations about the suitability of this
 * software for any purpose. It is provided "as is" without express
 * or implied warranty.
 */ 

/*
 * debugging codes:
 *	-1	turn on all messages
 *	0	turn off all messages
 *	1	function entry messages
 *	2	function leaving messages
 *	4	updatetable messages
 *	8	learn messages
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include <locale.h>
#include "misc.h"

int (*strcmpfp) () = NULL;	/* string comparison function */
char *version = "2.0";
char *copyright =
	"Copyright (C) 1997 Udo Krebelder <ukrebeld@nt.tuwien.ac.at>";

unsigned errline = 1;		/* current (error) line in ascii files */
extern int errno;		/* system error code */

/*
 * Update forgetrate table every 'updatetime' vocables.
 */
unsigned updatetime = 0;

/*
 * Default vocabulary is ${HOME}/DATABASE
 */
#define DATABASE "/.learnrc"

/*
 * Seconds per day.
 */
#define DAYSEC 86400

/*
 * Default updatetime.
 */
#define UPDATETIME 50

/*
 * Possibility to randomly ask the user a word.
 */
#define RANDRATE 0.005

/*
 * Possibility for new vocabules.
 */
#define NEWRATE 0.1

/*
 */
#define NEWCOUNT 2

/*
 *  Do not re-ask vocables for WAITTIME seconds.
 */
#define WAITTIME 120

#ifdef __hpux	/* HP-UX B.10.20 A 9000/735 bug: RAND_MAX == SHRT_MAX! */
#define RAND_MAX INT_MAX
#endif

#ifdef _AIX /* AIX bug: RAND_MAX == SHRT_MAX! */
#define RAND_MAX INT_MAX
#endif

/*
 * Buffer increment size.
 */
#define INCREMENT 512

int skip(FILE *fp)
/*
 * Skip comment and whitespace. Return next character or EOF.
 */
{
	int c;

	DEBUG((1, "entering skip()"));
	L: c = fgetc(fp);
	if (c == '\n') {
		++errline;
		goto L;
	}
	if (isspace(c))
		goto L;
	if (c == '#') {
		do {
			if ((c = fgetc(fp)) == EOF)
				return EOF;
		} while (c != '\n');
		++errline;
		goto L;
	}
	return c;
}

typedef struct box {
	double boxrate;		/* forgetrate for this box */
	unsigned boxnasked;	/* number of asked vocables */
	unsigned boxncorrect;	/* number of correct answers */
} BOX;

#define ENTRATE 0x4c4502f0	/* box table magic number */

/*
 * Number of learning boxes.
 */
#define BOXNUM 10

/*
 * Global box table and number of boxes.
 */
BOX *boxbuf = NULL;
unsigned boxlen = 0;

int initbox(unsigned n)
/*
 * Generate a default box table with 'n' boxes.
 */
{
	int i;
	BOX *p;
	double c, y, k, l;

	DEBUG((1, "entering initbox(%d)", n));
	if (n < 2)
		return E_INVAL;
	if ((p = realloc(boxbuf, n * sizeof(BOX))) == NULL)
		return E_NOMEM;
	boxbuf = p;
	boxlen = n;

#define Y0 3.0		/* start interval are Y0 days */
#define Y1 360.0	/* end interval are Y1 days */
#define M1 6.0		/* linear function part is 1/M1 */

	c = pow((Y1 / Y0) * (1.0 - 1.0 / M1) + 1.0, 1.0 / (n - 1));
	k = (Y1 / M1 - Y0) / (n - 1);
	l = Y0;
	i = 0;
	while (i < boxlen) {
		y = k * i + l;
		l *= c;
		boxbuf[i].boxrate = y * DAYSEC;
		boxbuf[i].boxnasked = 0;
		boxbuf[i++].boxncorrect = 0;
	}
	return E_NOERROR;
}

int writebox(FILE *database)
{
	int m;
	unsigned k;

	DEBUG((1, "entering writebox()"));
	if (fwrite(&boxlen, sizeof(boxlen), 1, database) != 1)
		return E_IO;
	m = ENTRATE;
	if (fwrite(&m, sizeof(m), 1, database) != 1)
		return E_IO;
	if (fwrite(boxbuf, sizeof(BOX), boxlen, database) != boxlen)
		return E_IO;
	k = boxlen * sizeof(BOX) + sizeof(int) + sizeof(boxlen);
	while (k % 16) {
		if (fputc('\0', database) == EOF)
			return E_IO;
		++k;
	}
	return E_NOERROR;
}

int printbox(FILE *database)
{
	int i;

	DEBUG((1, "entering printbox()"));
	if (fprintf(database, "### #boxes\n") < 1
	|| fprintf(database, "%u\n", boxlen) < 1
	|| fprintf(database, "### forgettime (days) #asked #correct\n") < 1)
	{
		return E_IO;
	}
	i = 0;
	while (i < boxlen) {
		if (fprintf(database, "%.9g %u %u\n",
		boxbuf[i].boxrate / DAYSEC, boxbuf[i].boxnasked,
		boxbuf[i].boxncorrect) < 1)
		{
			return E_IO;
		}
		++i;
	}
	return E_NOERROR;
}

int readbox(FILE *database)
{
	int c, m;
	unsigned k, l;
	BOX *q;

	DEBUG((1, "entering readbox()"));
	k = fread(&l, 1, sizeof(l), database);
	if (k != sizeof(l)) {
		if (ferror(database))
			return E_IO;
		if (k == 0)
			return E_EOF;
		return E_FORMAT;
	}
	k = fread(&m, 1, sizeof(m), database);
	if (k != sizeof(m) || m != ENTRATE) {
		if (ferror(database))
			return E_IO;
		return E_FORMAT;
	}
	if (boxlen < l) {
		if ((q = realloc(boxbuf, l * sizeof(BOX))) == 0)
			return E_NOMEM;
		boxbuf = q;
		boxlen = l;
	}
	k = fread(boxbuf, sizeof(BOX), l, database);
	if (k != l) {
		if (ferror(database))
			return E_IO;
		return E_FORMAT;
	}
	k = l * sizeof(BOX) + sizeof(int) + sizeof(boxlen);
	while (k % 16) {
		if ((c = fgetc(database)) == EOF) {
			if (ferror(database))
				return E_IO;
			return E_FORMAT;
		}
		++k;
	}
	return E_NOERROR;
}

int scanbox(FILE *database)
{
	int i, c;
	BOX b;

	DEBUG((1, "entering scanbox()"));
	c = skip(database);
	if (c == EOF) {
		if (ferror(database))
			return E_IO;
		return E_EOF;
	}
	if (ungetc(c, database) == EOF)
		return E_IO;
	if (fscanf(database, "%d\n", &i) != 1 || i <= 0)
		return E_FORMAT;
	if (initbox(i))
		return E_NOMEM;
	i = 0;
	while (i < boxlen) {
		c = skip(database);
		if (c == EOF) {
			if (ferror(database))
				return E_IO;
			return E_FORMAT;
		}
		if (ungetc(c, database) == EOF)
			return E_IO;
		if (fscanf(database, "%lg %u %u\n", &b.boxrate,
		&b.boxnasked, &b.boxncorrect) != 3 || b.boxrate <= 0)
		{
			if (ferror(database))
				return E_IO;
			return E_FORMAT;
		}
		b.boxrate *= DAYSEC;
		boxbuf[i++] = b;
	}
	return E_NOERROR;
}

double learnprob(int box, long stamp)
{
	double x, y;

	DEBUG((1, "entering learnprob(%d, %ld)", box, stamp));
	if (box >= boxlen)
		box = boxlen - 1;
	if (box < 0)
		box = 0;
	x = stamp / boxbuf[box].boxrate;
	y = 0.5 * (tanh(6.0 * (x - 1.0)) + 1.0);
	x = RANDRATE / (box + 1);
	if (y < x)
		y = x;
	DEBUG((2, "learnprob: probability=%g", y));
	return y;
}

/*
 * Database entry
 */
typedef struct entry {
	unsigned entsize;	/* total size in bytes,
				   multiple of 8, offset == 0! */
	int entmagic;		/* magic number (ENTLEARNED,ENTNEW,ENTFREE) */
	int entbox;		/* learning iteration number */
	time_t entstamp;	/* time stamp of the last learning trial */
	char entword[1];	/* word list; the first word
				   is the left entry side, the others
				   are possible translations;
				   words are separated by '\0',
				   the first word and the right side
				   list (the translations) are terminated
				   by an additional '\0' character */
} ENTRY;

#define ENTLEARNED 0x4c450200	/* magic for learned vocable entry */
#define ENTNEW 0x4c450203	/* magic for new vocable entry */
#define ENTFREE 0x4c45020f	/* magic for free vocable entry */

/*
 * Global entry buffer and buffer size in bytes.
 */
ENTRY *entbuf = 0;
unsigned entlen = 0;

/*
 * Global character buffer and buffer size.
 */
char *linebuf = 0;
unsigned linelen = 0;

int badentry(ENTRY *ent)
/*
 * Check for corrupt vocable entries.
 */
{
	char *p, *q;
	unsigned k, n;

	DEBUG((1, "entering badentry()"));
	if ((ent->entmagic != ENTLEARNED
	&& ent->entmagic != ENTFREE
	&& ent->entmagic != ENTNEW)
	|| ent->entbox < 0
	|| ent->entsize <= sizeof(ENTRY)
	|| ent->entsize % 16)
	{
		return E_FORMAT;
	}
	p = ent->entword;
	q = (char*)ent + ent->entsize;
	for (n = 0; n < 2; n++) {
		while (*p != '\0') {
			if (p >= q)
				return E_FORMAT;
			k = strlen(p) + 1;
			if (k < 2)
				return E_FORMAT;
			p += k;
		}
	}
	return E_NOERROR;
}

int shouldlearn(double x)
/*
 * Determine, if we should ask the user this entry.
 */
{
	double y;

	DEBUG((1, "entering shouldlearn()"));
	y = (double)random() / RAND_MAX;
	return y <= x;
}

int printentry(FILE *fp, ENTRY *ent)
/*
 * Write entry 'ent' in ASCII format.
 */
{
	char *p, *q;

	DEBUG((1, "entering printentry()"));
	switch (ent->entmagic) {
		case ENTNEW:
			p = "n ";
			break;
		default:
			p = "";
	}
	q = ent->entword + strlen(ent->entword) + 2;
	if (fprintf(fp, "%s%d %lu %s :: %s",p, ent->entbox,
	(unsigned long)(ent->entstamp), ent->entword, q) < 1)
	{
		return E_IO;
	}
	while (*(q += strlen(q) + 1) != '\0') {
		if (fprintf(fp, ", %s", q) < 1)
			return E_IO;
	}
	if (fputc('\n', fp) == EOF)
		return E_IO;
	return E_NOERROR;
}

int writeentry(FILE *fp, ENTRY *ent)
/*
 * Write entry 'ent' in binary format.
 */
{
	DEBUG((1, "entering writeentry()"));
	if (fwrite(ent, ent->entsize, 1, fp) != 1)
		return E_IO;
	return E_NOERROR;
}

int readentry(FILE *fp)
/*
 * Read global entry 'entbuf' in binary format.
 */
{
	int m;
	unsigned k, l;
	ENTRY *q;

	DEBUG((1, "entering readentry()"));
	k = fread(&l, 1, sizeof(l), fp);
	if (k != sizeof(l) || l < sizeof(ENTRY) || l % 16) {
		if (ferror(fp))
			return E_IO;
		if (k == 0)
			return E_EOF;
		return E_FORMAT;
	}
	k = fread(&m, 1, sizeof(m), fp);
	if (k != sizeof(m)
	|| (m != ENTLEARNED && m != ENTFREE && m != ENTNEW))
	{
		if (ferror(fp))
			return E_IO;
		return E_FORMAT;
	}
	if (entlen < l) {
		if ((q = realloc(entbuf, l)) == 0)
			return E_NOMEM;
		entbuf = q;
		entlen = l;
	}
	entbuf->entsize = l;
	entbuf->entmagic = m;
	l -= sizeof(l) + sizeof(m);
	k = fread(&entbuf->entmagic + 1, 1, l, fp);
	if (k != l || badentry(entbuf)) {
		if (ferror(fp))
			return E_IO;
		return E_FORMAT;
	}
	return E_NOERROR;
}

int skipentry(FILE *fp)
/*
 * Skip corrupt vocable entry.
 */
{
	int status, magic;
	unsigned k;
	long pos;

	DEBUG((1, "entering skipentry()"));
	if ((pos = ftell(fp)) == -1)
		return E_IO;
	pos = (pos + 0xf) & ~0xf; /* round up to 4 bit boundary */
	if (fseek(fp, pos, SEEK_SET))
		return E_IO;
	for (;;) {
		k = fread(&magic, 1, sizeof(magic), fp);
		if (k != sizeof(magic)) {
			if (ferror(fp))
				return E_IO;
			if (k == 0)
				return E_EOF;
			return E_FORMAT;
		}
		if (magic == ENTLEARNED
		|| magic == ENTFREE
		|| magic == ENTNEW)
		{
			break;
		}
	}
	if (fseek(fp, -(sizeof(int) + sizeof(unsigned)), SEEK_CUR))
		return E_IO;
	if ((pos = ftell(fp)) == -1)
		return E_IO;
	for (k = 0; k < 2; k++) {
		if ((status = readentry(fp)) != E_NOERROR) {
			if (status == E_EOF && k > 0)
				break;
			return status;
		}
	}
	if (fseek(fp, pos, SEEK_SET))
		return E_IO;
	return E_NOERROR;
}

int scanword(FILE *fp, int offset)
/*
 * Scan vocable from file and write it to global buffer
 * `linebuf' + `offset'.
 * Optional leading and trailing white space is skipped.
 * Word is terminated by ',' or '\n' or "::".
 * Empty words are returned as empty strings ("").
 */
{
	int i, c, cc;
	char *q;

	DEBUG((1, "entering scanword()"));
	/* skip initial whitespace */
	L: c = fgetc(fp);
	if (c == EOF) {
		if (ferror(fp))
			return E_IO;
		return E_EOF;
	}
	if (isspace(c) && c != '\n')
		goto L;
	i = offset;
	cc = '\0';
	for (;;) {
		if (linelen == 0 || i >= linelen - 1) {
			if ((q = realloc(linebuf, i + INCREMENT)) == 0)
				return E_NOMEM;
			linebuf = q;
			linelen = i + INCREMENT;
		}
		if (c == '\n' || c == ',' || (c == ':' && cc == ':')) {
			/*
			 * We do not want fseek() here!  So we skip one ':'
			 * and the terminator string "::" becomes ":".
			 */
			if (c == ':')
				--i;
			if (ungetc(c, fp) == EOF)
				return E_IO;
			break;
		}
		if (c == '\0')
			return E_FORMAT;
		linebuf[i++] = c;
		cc = c;
		if ((c = fgetc(fp)) == EOF) {
			if (ferror(fp))
				return E_IO;
			break;
		}
	}
	/* remove trailing whitespace */
	while (i != offset && isspace(linebuf[i - 1]))
		--i;
	linebuf[i] = '\0';
	return E_NOERROR;
}

int scanlist(FILE *fp, int *size)
/*
 * Scan comma (',') separated vocable list.
 * Data is stored in global buffer `linebuf'
 * and terminated by the empty string ("").
 * The `size' argument is set to the total
 * number of used bytes.
 */
{
	int c, status, offset;
	char *p;

	DEBUG((1, "entering scanlist()"));
	offset = 0;
	for (;;) {
		status = scanword(fp, offset);
		if (status) {
			if (status == E_EOF && offset != 0)
				status = E_FORMAT;
			return status;
		}
		if (linebuf[offset] == '\0') {
			if (offset == 0) {
				c = fgetc(fp);
				if (c != ',')
					break;
			}
			return E_FORMAT; /* empty word */
		}
		offset += strlen(linebuf + offset) + 1;
		c = fgetc(fp);
		if (c != ',')
			break;
	}
	if (c == EOF) {
		if (ferror(fp))
			return E_IO;
	} else {
		if (ungetc(c, fp) == EOF)
			return E_IO;
	}
	if (offset == linelen) {
		if ((p = realloc(linebuf, offset + INCREMENT)) == 0)
			return E_NOMEM;
		linebuf = p;
		linelen = offset + INCREMENT;
	}
	linebuf[offset++] = '\0';
	*size = offset;
	return E_NOERROR;
}

int scanentry(FILE *fp, ENTRY **ent, unsigned *len, int isdata)
/*
 * Scan entry.  A database entry has the form:
 * "[nN]? [0-9]+ [0-9]+ word :: a, b, c",
 * a vocable entry has the form: "a, b, c :: d, e, f".
 */
{
	int c, status, size;
	int size2, used, magic;
	long l, m;
	char *p;
	ENTRY *q;

	DEBUG((1, "entering scanentry(isdata = %d);", isdata));
	if (*len == 0) {
		if ((q = realloc(*ent, INCREMENT)) == 0)
			return E_NOMEM;
		*ent = q;
		*len = INCREMENT;
	}
	c = skip(fp);
	if (c == EOF) {
		if (ferror(fp))
			return E_IO;
		return E_EOF;
	}
	if (!isdata) {
		if (ungetc(c, fp) == EOF)
			return E_IO;
	}
	if (isdata) {
		if (c == 'N' || c == 'n')
			magic = ENTNEW;
		else {
			if (ungetc(c, fp) == EOF)
				return E_IO;
			magic = ENTLEARNED;
		}
		if (fscanf(fp, "%lu %lu ", &l, &m) != 2 || l < 0 || m < 0) {
			if (ferror(fp))
				return E_IO;
			return E_FORMAT;
		}
	} else {
		magic = ENTNEW;
		l = 0;
		m = time(0);
	}
	(*ent)->entmagic = magic;
	(*ent)->entbox = l;
	(*ent)->entstamp = m;
	used = (int)(&((ENTRY*)0)->entword);
	status = scanlist(fp, &size);
	if (status != E_NOERROR) {
		if (status == E_EOF)
			status = E_FORMAT;
		return status;
	}
	if (linebuf[0] == '\0')
		return E_FORMAT;
	if (isdata) {
		if (strlen(linebuf) + 2 != size)
			return E_FORMAT;	/* we want one word */
	}
	used += size;
	if (used >= *len) {
		if ((q = realloc(*ent, used + INCREMENT)) == 0)
			return E_NOMEM;
		*ent = q;
		*len = used + INCREMENT;
	}
	memcpy((*ent)->entword, linebuf, size);
	c = fgetc(fp);
	if (c == EOF) {
		if (ferror(fp))
			return E_IO;
		return E_FORMAT;
	}
	if (c != ':') {
		if (c == '\n')
			++errline;
		return E_FORMAT;
	}
	status = scanlist(fp, &size2);
	if (status != E_NOERROR) {
		if (status == E_EOF)
			status = E_FORMAT;
		return status;
	}
	if (linebuf[0] == '\0')
		return E_FORMAT;
	used += size2;
	if (used + sizeof(double) > *len) {
		if ((q = realloc(*ent, used + INCREMENT)) == 0)
			return E_NOMEM;
		*ent = q;
		*len = used + INCREMENT;
	}
	p = (*ent)->entword + size;
	memcpy(p, linebuf, size2);
	p += size2;
	while (used % 16) {
		*p++ = '\0';
		used++;
	}
	(*ent)->entsize = used;
	c = fgetc(fp);
	if (c == EOF) {
		if (ferror(fp))
			return E_IO;
		return E_NOERROR;
	}
	if (c != '\n')
		return E_FORMAT;
	++errline;
	return E_NOERROR;
}

int strcmp2(unsigned char *p, unsigned char *q)
/*
 * Compare argument strings,
 * ignore case and non visible characters.
 */
{
	int a, b;

	DEBUG((1, "entering strcmp(%s, %s)", p, q));
	do {
		a = *p++;
		if (isgraph(a) || a == '\0')
			a = tolower(a);
		else {
			while ((a = *p) && !isgraph(a))
				++p;
			a = -1;
		}
		b = *q++;
		if (isgraph(b) || b == '\0')
			b = tolower(b);
		else {
			while ((b = *q) && !isgraph(b))
				++q;
			b = -1;
		}
	} while (a && a == b);
	if (a && b)
		return a < b? 1 : a > b? -1 : 0;
	return b? 1 : a? -1 : 0;
}

int strcmp3(unsigned char *p, unsigned char *q)
/*
 * Compare argument strings,
 * ignore case and everything but letters.
 */
{
	int a, b;

	DEBUG((1, "entering strcmp(%s, %s)", p, q));
	do {
		a = *p++;
		if (isalpha(a) || a == '\0')
			a = tolower(a);
		else {
			while ((a = *p) && !isalpha(a))
				++p;
			a = -1;
		}
		b = *q++;
		if (isalpha(b) || b == '\0')
			b = tolower(b);
		else {
			while ((b = *q) && !isalpha(b))
				++q;
			b = -1;
		}
	} while (a && a == b);
	if (a && b)
		return a < b? 1 : a > b? -1 : 0;
	return b? 1 : a? -1 : 0;
}

int skipline(FILE *fp)
{
	int c;

	DEBUG((1, "entering skipline()"));
	for (;;) {
		c = fgetc(fp);
		if (c == EOF) {
			if (ferror(fp))
				return E_IO;
			return E_NOERROR;
		}
		if (c == '\n') {
			++errline;
			return E_NOERROR;
		}
	}
}

int askuser(void)
/*
 * Ask user for possible translations.
 * Return E_NOERROR for correct answer,
 * (-1) for incorrect ones.
 */
{
	int c, status, size;
	char *q, *r;

	DEBUG((1, "entering askuser()"));
	if (feof(stdin))	/* do not reread EOF */
		return E_EOF;
	r = entbuf->entword;
	if (fprintf(stdout, "%s? ", r) < 1)
		return E_IO;
	fflush(stdout);
	status = scanlist(stdin, &size);
	if (status != E_NOERROR) {
		if (status == E_EOF)
			fputc('\n', stdout);
		if (status != E_FORMAT)
			return status;
	}
	c = fgetc(stdin);
	if (c != '\n')
		status = E_FORMAT;
	++errline;
	if (status) {
		if (c != '\n' && (status = skipline(stdin)) != E_NOERROR)
			return status;
		q = "";
	}
	else
		q = linebuf;
	status = E_INVAL;
	while (*q != '\0') {
		r = entbuf->entword;
		r += strlen(r) + 2;
		do {
			if ((status = strcmpfp(q, r)) == 0)
				break;
			r += strlen(r) + 1;
		} while (*r != '\0');
		if (status)
			break;
		q += strlen(q) + 1;
	} 
	if (status == 0) {
		return E_NOERROR;
	}
	r = entbuf->entword;
	r += strlen(r) + 2;
	if (fprintf(stdout, "%s", r) < 1)
		return E_IO;
	while (*(r += strlen(r) + 1) != '\0') {
		if (fprintf(stdout, ", %s", r) < 1)
			return E_IO;
	}
	if (fprintf(stdout, "! ") < 1)
		return E_IO;
	fflush(stdout);
	if ((status = skipline(stdin)) != E_NOERROR)
		return status;
	if (feof(stdin))
		fputc('\n', stdout);
	return (-1);
}

int flushent(FILE *database)
{
	int status;

	DEBUG((1, "entering flushent()"));
	if (fseek(database, -entbuf->entsize, SEEK_CUR))
		return E_IO;
	if ((status=writeentry(database, entbuf)) != E_NOERROR)
		return status;
	if (fseek(database, 0, SEEK_CUR))
		return E_IO;
	return E_NOERROR;
}

int flushbox(FILE *database)
{
	long pos;

	DEBUG((1, "entering flushbox()"));
	if ((pos = ftell(database)) == -1)
		return E_IO;
	if (fseek(database, 0, SEEK_SET))
		return E_IO;
	if (writebox(database) != E_NOERROR)
		return E_IO;
	if (fseek(database, pos, SEEK_SET))
		return E_IO;
	return E_NOERROR;
}

void updatebox(int box)
{
	double d;

	DEBUG((1, "entering updatebox(%d)", box));
	d = (double)(boxbuf[box].boxncorrect)
		/ (double)(boxbuf[box].boxnasked);

#define C (8.0 / 10.0)
	d = (0.5 / C) * d + 0.5;

	d *= boxbuf[box].boxrate;
	if (box == 0) {
		if (d < DAYSEC)
			d = DAYSEC;
	} else  {
		if (d < boxbuf[box - 1].boxrate)
			d = boxbuf[box - 1].boxrate;
	}
	if (box == boxlen - 1) {
		if (d > 1800.0 * DAYSEC)	/* max are 5 years */
			d = 1800.0 * DAYSEC;
	} else {
		if (d > boxbuf[box + 1].boxrate)
			d = boxbuf[box + 1].boxrate;
	}
	DEBUG((4, "updatebox: setting boxrate of box %d to %g",
		box, d / DAYSEC));
	boxbuf[box].boxrate = d;
	boxbuf[box].boxnasked = 0;
	boxbuf[box].boxncorrect = 0;
}

int skipcount = 0;

void skipmessage(void)
{
	if (skipcount++ == 0) {
		errmsg("Skipping bad vocables");
	}
}

void statusmessage(unsigned nasked, unsigned ncorrect,
			time_t start, time_t stop)
{
	char *p;
	double d, e;

	if (nasked == 0)
		return;
	d = (double)ncorrect / (double)nasked;
	e = difftime(stop, start);
	if (e > 60.0)
		p = "Success rate: %u of %u (%.1f%%), %.0f minutes.\n";
	else
		p = "Success rate: %u of %u (%.1f%%).\n";
	fprintf(stderr, p, ncorrect, nasked, d * 100.0, e / 60.0);
}

int learn(FILE *database)
/*
 * Iterate through vocabulary and ask user words.
 */
{
	int box, status;
	unsigned nentry, nlast, nasked, ncorrect;
	long pos, stamp;
	time_t start, stop;
	double d;

	DEBUG((1, "entering learn()"));
	nentry = 0;
	nlast = UINT_MAX - 3;
	nasked = 0;
	ncorrect = 0;
	start = time(0);
	for (;;) {
		DEBUG((8, "learn: reading entry"));
		status = readentry(database);
		if (status == E_EOF) {
			if (nentry == 0)
				return E_FORMAT;
			if (nentry <= 3)
				nlast = UINT_MAX - 3;
			nentry = 0;
			/* skip box table */
			pos = sizeof(boxlen) + sizeof(int)
				+ boxlen * sizeof(BOX) + 0xf;
			pos &= ~0xf;
			DEBUG((8, "learn: rewinding file"));
			if (fseek(database, pos, SEEK_SET))
				return E_IO;
			continue;
		}
		if (status == E_FORMAT) {
			DEBUG((8, "learn: skipping bad entry"));
			if (skipentry(database) == E_NOERROR) {
				skipmessage();
				continue;
			}
		}
		if (status)
			return status;
		if (entbuf->entmagic == ENTFREE) {
			DEBUG((8, "learn: read free entry, continue"));
			continue;
		}
		box = entbuf->entbox;
		stamp = time(0) - entbuf->entstamp;
		if (entbuf->entmagic == ENTNEW) {
			DEBUG((8, "learn: new entry: box = %d, "
				"difftime = %g", box, (double)stamp/DAYSEC));
			/* this works only for new entries */
			if (stamp <= WAITTIME)
				d = RANDRATE;
			else
				d = NEWRATE / (box + 1);
		}
		else {
			DEBUG((8, "learn: learned entry: box = %d, "
				"difftime = %g", box, (double)stamp/DAYSEC));
			d = learnprob(box, stamp);
		}
		if (nentry < nlast + 2 && nentry + 2 > nlast) {
			DEBUG((8, "learn: adjacent entry"));
			d = RANDRATE;
		}
		DEBUG((8, "learn: probability = %g", d));
		if (shouldlearn(d) != 0) {
			DEBUG((8, "learn: asking user"));
			status = askuser();
			if (status != E_NOERROR && status != -1) {
				DEBUG((8, "learn: end of terminal or error"));
				if (flushbox(database) != E_NOERROR)
					return E_IO;
				if (status == E_EOF) {
					status = E_NOERROR;
					stop = time(0);
					statusmessage(nasked, ncorrect,
							start, stop);
				}
				return status;
			}
			if (status == -1) {
				DEBUG((8, "learn: wrong answer"));
				if (entbuf->entmagic == ENTNEW) {
					DEBUG((8, "learn: setting time"));
					entbuf->entstamp = time(0);
					DEBUG((8, "learn: resetting box"));
					entbuf->entbox = 0;
					status = flushent(database);
					if (status != E_NOERROR)
						return status;
				}
			} else if (status == E_NOERROR) {
				DEBUG((8, "learn: good answer"));
				ncorrect++;
				DEBUG((8, "learn: setting time"));
				entbuf->entstamp = time(0);
				if (entbuf->entmagic == ENTNEW) {
					if (++entbuf->entbox >= NEWCOUNT) {
						DEBUG((8, "learn: we have "
							"learned it"));
						entbuf->entbox = 0;
						entbuf->entmagic = ENTLEARNED;
						box = -1;
					}
				}
				else {
					boxbuf[box].boxncorrect++;
					if (entbuf->entbox < boxlen - 1)
					{
						DEBUG((8, "learn: "
							"incrementing box"));
						entbuf->entbox++;
					}
				}
				DEBUG((8, "learn: flushing entry"));
				if ((status = flushent(database)) != E_NOERROR)
					return status;
			}
			if (entbuf->entmagic == ENTLEARNED && box != -1 ) {
				if (++boxbuf[box].boxnasked >= updatetime)
				{
					DEBUG((8, "learn: updating box table"));
					updatebox(box);
				}
				if (boxbuf[box].boxnasked == 0
					|| nasked % 10 == 0)
				{
					DEBUG((8, "learn: flushing box table"));
					if (flushbox(database) != E_NOERROR)
						return E_IO;
				}
			}
			nlast = nentry;
			++nasked;
		}
		++nentry;
	}
}

int entcmp(ENTRY *a, ENTRY *b)
{
	char *p, *q;

	p = a->entword;
	q = b->entword;
	while (*p != '\0' && strcmpfp(p, q) == 0) {
		p += strlen(p) + 1;
		q += strlen(q) + 1;
	}
	if (*p++ != '\0' || *q++ != '\0')
		return -1;
	while (*p != '\0' && strcmpfp(p, q) == 0) {
		p += strlen(p) + 1;
		q += strlen(q) + 1;
	}
	if (*p != '\0' || *q != '\0')
		return -1;
	return 0;
}

int entmerge(ENTRY **ent, unsigned *len)
/*
 * Merge argument vocable entry 'ent' with global 'entbuf'.  Result is
 * stored in argument entry '*ent', 'entbuf' is not changed.  It is
 * assumed that the enties are OK and that the left sides are equal.
 */
{
	unsigned k, l, m, n;
	char *p, *q;
	ENTRY *t;

	DEBUG((1, "entering entmerge()"));

	/*
	 * We deal with two kinds of vocable entries: Learned entries
	 * and new entries.  Free entries are ignored.
	 */

	if (entcmp(*ent, entbuf) == 0) {
		/* we have equal entries: */

		/*
		 * The following switch statements assume that the merged
		 * entry is the one which is best learned.
		 */
		switch ((*ent)->entmagic) {
			case ENTLEARNED:
				switch (entbuf->entmagic) {
					case ENTLEARNED:
						if ((*ent)->entbox
						< entbuf->entbox)
						{
							(*ent)->entbox =
								entbuf->entbox;
							(*ent)->entstamp =
							entbuf->entstamp;
						}
						else if ((*ent)->entbox
						== entbuf->entbox)
						{
							(*ent)->entstamp =
							entbuf->entstamp;
						}
						break;
					case ENTNEW:
						break;
					default:
						return E_INVAL;
				}
				break;
			case ENTNEW:
				switch (entbuf->entmagic) {
					case ENTLEARNED:
						(*ent)->entmagic =
							ENTLEARNED;
						(*ent)->entbox =
							entbuf->entbox;
						(*ent)->entstamp =
							entbuf->entstamp;
						break;
					case ENTNEW:
						if ((*ent)->entbox
						< entbuf->entbox)
						{
							(*ent)->entbox =
								entbuf->entbox;
							(*ent)->entstamp =
							entbuf->entstamp;
						}
						else if ((*ent)->entbox
						== entbuf->entbox)
						{
							(*ent)->entstamp =
							entbuf->entstamp;
						}
						break;
					default:
						return E_INVAL;
				}
				break;
			default:
				return E_INVAL;
		}
		/* nothing to do */
		return E_NOERROR;
	}

	/* we unequal entries: */

#if 0
	/*
	 * It is not clear what to do with the magic entry in case of
	 * mixed entries.
	 * The following switch statements assume the worst and always
	 * degrade the merged entry from learned to unlearned in the
	 * mixed case.
	 */
	switch ((*ent)->entmagic) {
		case ENTLEARNED:
			switch (entbuf->entmagic) {
				case ENTLEARNED:
					if ((*ent)->entbox
					> entbuf->entbox)
					{
						(*ent)->entbox =
							entbuf->entbox;
						(*ent)->entstamp =
							entbuf->entstamp;
					}
					break;
				case ENTNEW:
					(*ent)->entmagic = ENTNEW;
					(*ent)->entbox = entbuf->entbox;
					(*ent)->entstamp = entbuf->entstamp;
					break;
				default:
					return E_INVAL;
			}
			break;
		case ENTNEW:
			switch (entbuf->entmagic) {
				case ENTLEARNED:
					break;
				case ENTNEW:
					if ((*ent)->entbox > entbuf->entbox)
					{
						(*ent)->entbox =
							entbuf->entbox;
						(*ent)->entstamp =
							entbuf->entstamp;
					}
					break;
				default:
					return E_INVAL;
			}
			break;
		default:
			return E_INVAL;
	}
#else
	/*
	 * The following switch statements assume that the merged
	 * entry is the best learned of the individuals.  This has the
	 * advantage that adding vocables twice does not change
	 * anything.
	 */
	switch ((*ent)->entmagic) {
		case ENTLEARNED:
			switch (entbuf->entmagic) {
				case ENTLEARNED:
					if ((*ent)->entbox
					< entbuf->entbox)
					{
						(*ent)->entbox =
							entbuf->entbox;
						(*ent)->entstamp =
							entbuf->entstamp;
					}
					else if ((*ent)->entbox
					== entbuf->entbox)
					{
						(*ent)->entstamp =
						entbuf->entstamp;
					}
					break;
				case ENTNEW:
					break;
				default:
					return E_INVAL;
			}
			break;
		case ENTNEW:
			switch (entbuf->entmagic) {
				case ENTLEARNED:
					(*ent)->entmagic = ENTLEARNED;
					(*ent)->entbox = entbuf->entbox;
					(*ent)->entstamp = entbuf->entstamp;
					break;
				case ENTNEW:
					if ((*ent)->entbox
					< entbuf->entbox)
					{
						(*ent)->entbox =
							entbuf->entbox;
						(*ent)->entstamp =
							entbuf->entstamp;
					}
					else if ((*ent)->entbox
					== entbuf->entbox)
					{
						(*ent)->entstamp =
						entbuf->entstamp;
					}
					break;
				default:
					return E_INVAL;
			}
			break;
		default:
			return E_INVAL;
	}
#endif

	/* merge entries right sides */
	k = 0;
	p = entbuf->entword;
	p += strlen(p) + 2;
	while (*p != '\0') {
		q = (*ent)->entword;
		q += strlen(q) + 2;
		while (*q != '\0') {
			if (strcmpfp(p, q) == 0)
				break;
			q += strlen(q) + 1;
		}
		if (*q == '\0') {
			m = strlen(p) + 1;
			n = q - (char*)(*ent) + 1;
			if (*len < n + m) {
				if ((t = realloc(*ent,
				n + m + INCREMENT)) == NULL)
				{
					return E_NOMEM;
				}
				*len = n + m + INCREMENT;
				*ent = t;
			}
			q = (*ent)->entword;
			q += strlen(q) + 2;
			for (l = 0; l < k; l++)
				q += strlen(q) + 1;
			n -= q - (char*)(*ent);
			memmove(q + m, q, n);
			memmove(q, p, m);
			q += m + n;
			m = q - (char*)(*ent);
			while (m++ % 16)
				*q++ = '\0';
			(*ent)->entsize = q - (char*)(*ent);
			++k;
		}
		p += strlen(p) + 1;
	}
	return E_NOERROR;
}

int mergeentry(FILE *database, ENTRY **ent, unsigned *len)
/*
 * Merge vocable entry 'ent' with database.
 * 'ent' may not be identical to global 'entbuf'.
 * This function uses a linear search method and is
 * therefore not efficient (but simple :-\).
 */
{
	int status, whence;
	unsigned freesize;
	long pos, freeoffset;

	DEBUG((1, "entering mergeentry()"));
	freesize = 0;
	freeoffset = 0;
	/* skip the forgetrate table */
	pos = (sizeof(boxlen) + sizeof(int)
		+ boxlen * sizeof(BOX) + 0xf) & ~0xf;
	if (fseek(database, pos, SEEK_SET))
		return E_IO;
	for (;;) {
		status = readentry(database);
		if (status != E_NOERROR) {
			if (status == E_EOF)
				break;
			if (status == E_FORMAT) {
				if (skipentry(database) == E_NOERROR) {
					skipmessage();
					continue;
				}
			}
			return status;
		}
		if (entbuf->entmagic == ENTFREE) {
			if (entbuf->entsize > freesize) {
				/* remember free entry */
				if ((pos = ftell(database)) != -1) {
					freesize = entbuf->entsize;
					freeoffset = pos - entbuf->entsize;
				}
			}
			continue;
		}
		if (strcmpfp((*ent)->entword, entbuf->entword) == 0) {
			if ((status = entmerge(ent, len)) != E_NOERROR)
				return status;
			entbuf->entmagic = ENTFREE;
			if (flushent(database) != E_NOERROR)
				return E_IO;
		}
	}
	if ((*ent)->entsize <= freesize) {
		(*ent)->entsize = freesize;
		whence = SEEK_SET;
	}
	else {
		freeoffset = 0;
		whence = SEEK_END;
	}
	if (fseek(database, freeoffset, whence))
		return E_IO;
	return writeentry(database, *ent);
}

int mergevoc(FILE *database)
/*
 * Scan database entries from 'stdin' and merge them with database.
 */
{
	int status;
	unsigned len;
	ENTRY *ent;

	DEBUG((1, "entering mergevoc()"));
	ent = NULL;
	len = 0;
	for (;;) {
		status = scanentry(stdin, &ent, &len, 1);
		if (status) {
			if (status == E_EOF)
				status = E_NOERROR;
			if (status == E_FORMAT) {
				errmsg("Line %u: Bad format", errline);
				skipline(stdin);
				continue;
			}
			break;
		}
		status = mergeentry(database, &ent, &len);
		if (status)
			break;
	}
	free(ent);
	return status;
}

int addvoc(FILE *database)
/*
 * Scan vocable entries from 'stdin' and merge them with 'database'.
 */
{
	int status;
	unsigned len1, len2;
	unsigned k, l, m, n;
	char *p, *q, *r;
	ENTRY *t, *ent1, *ent2;

	DEBUG((1, "entering addvoc()"));
	ent1 = ent2 = NULL;
	len1 = len2 = 0;
	for (;;) {
		status = scanentry(stdin, &ent1, &len1, 0);
		if (status != E_NOERROR) {
			if (status == E_EOF)
				status = E_NOERROR;
			if (status == E_FORMAT) {
				errmsg("Line %u: Bad format", errline);
				skipline(stdin);
				continue;
			}
			goto R;
		}
		if (len2 < len1) {
			if ((t = realloc(ent2, len1)) == NULL) {
				status = E_NOMEM;
				goto R;
			}
			ent2 = t;
			len2 = len1;
		}
		/* add all possible word combinations */
		for (n = 0; n < 2; n++) {
			q = p = ent1->entword;
			while (*q != '\0')
				q += strlen(q) + 1;
			r = ++q;
			while (*r != '\0')
				r += strlen(r) + 1;
			++r;
			if (n != 0) {
				l = q - p;
				p = q;
				q = ent1->entword;
			} else 
				l = r - q;
			r = ent2->entword;
			while (*p != '\0') {
				k = strlen(p) + 1;
				memcpy(r, p, k);
				r[k] = '\0';
				memcpy(r + k + 1, q, l);
				m = ent2->entword - (char*)ent2;
				m += k + 1 + l;
				while (m % 16)
					((char*)ent2)[m++] = '\0';
				ent2->entsize = m;
				ent2->entmagic = ENTNEW;
				ent2->entbox = 0;
				ent2->entstamp = time(0);
				status = mergeentry(database, &ent2, &len2);
				if (status != E_NOERROR)
					goto R;
				p += k;
			}
		}
	}
	R: free(ent1);
	free(ent2);
	return status;
}

int appendvoc(FILE *database)
/*
 * Scan database entries from 'stdin' and append them to 'database'.
 */
{
	int status;

	DEBUG((1, "entering appendvoc()"));
	if (fseek(database, 0, SEEK_END))
		return E_IO;
	for (;;) {
		status = scanentry(stdin, &entbuf, &entlen, 1);
		if (status != E_NOERROR) {
			if (status == E_EOF)
				status = E_NOERROR;
			if (status == E_FORMAT) {
				errmsg("Line %u: Bad format", errline);
				skipline(stdin);
				continue;
			}
			break;
		}
		status = writeentry(database, entbuf);
		if (status)
			break;
	}
	return status;
}

int printvoc(FILE *database)
/*
 * Write database in human readable form.  This format can be reread
 * with mergevoc().
 */
{
	int status;
	unsigned n;

	DEBUG((1, "entering printvoc()"));
	if (fputs("### box timestamp vocable\n", stdout) == EOF)
		return E_IO;
	n = 0;
	for (;;) {
		status = readentry(database);
		if (status != E_NOERROR) {
			if (status == E_EOF && n != 0)
				status = E_NOERROR;
			if (status == E_FORMAT) {
				if (skipentry(database) == E_NOERROR) {
					skipmessage();
					continue;
				}
			}
			break;
		}
		if (entbuf->entmagic == ENTFREE) {
			if (!(debuglevel & 2))
				continue;
			fputs("### ", stdout);
		}
		status = printentry(stdout, entbuf);
		if (status)
			break;
		n++;
	}
	return status;
}

int delvoc(FILE *database)
{
	DEBUG((1, "entering delvoc()"));
	errmsg("Sorry, `delete()\' not yet implemented");
	exit(E_INVAL);
}

char* strcat2(char *p, char *q)
{
	char *r;

	DEBUG((1, "entering strcat(%s, %s)", p, q));
	if ((r = malloc(strlen(p) + strlen(q) + 1)) == 0)
		return 0;
	strcpy(r, p);
	return strcat(r, q);
}

int printversion(void)
{
	DEBUG((1, "entering printversion()"));
	if (fprintf(stderr,"Learn %s. %s.\n", version, copyright) < 1)
		return E_IO;
	return E_NOERROR;
}

void usage(void)
{
	DEBUG((1, "entering usage()"));
	fprintf(stderr, "usage: learn [-anpdveo]"
	" [-s#] [-u#] ]-b#] [<database>]\n");
	exit(E_INVAL);
}

OPTION mainopt[] = {
	{ 'v',	"version" },
	{ 'p',	"print" },
	{ 'a',	"add" },
	{ 'd',	"delete" },
	{ 'n',	"new" },
	{ 1000,	"append" },
	{ 1001,	"debug" },
	{ 'e',	"exact" },
	{ 'o',	"loose" },
	{ 'u',	"update" },
	{ 'b',	"boxnum" },
	{ 's',	"seed" }
};

#define LEARN 1
#define VERSION 2
#define PRINT 3
#define NEW 4
#define DELETE 5
#define ADD 6
#define APPEND 7

int main(int argc, char **argv)
{
	int status, operation;
	unsigned boxnum;
	long seed, l;
	char *p, *q;
	OPTION *op;
	FILE *database;

	operation = 0;
	p = NULL;
	seed = -1;
	database = 0;
	boxnum = 0;

	/* language specific stuff */
	setlocale(LC_ALL, "");

	/* parse options */
	setoption(argc, argv, mainopt, optnum(mainopt));
	while ((op = getoption()) != 0) {
		switch (op->short_opt) {
			case 'n':
				if (operation)
					usage();
				operation = NEW;
				break;
			case 'p':
				if (operation)
					usage();
				operation = PRINT;
				break;
			case 'a':
				if (operation)
					usage();
				operation = ADD;
				break;
			case 'd':
				if (operation)
					usage();
				operation = DELETE;
				break;
			case 'v':
				if (operation)
					usage();
				operation = VERSION;
				break;
			case 1000:
				if (operation)
					usage();
				operation = APPEND;
				break;
			case 1001:
				if (getlparam(&l))
					usage();
				debuglevel = l;
				break;
			case 'u':
				if (getlparam(&l) || l <= 0)
					usage();
				updatetime = l;
				break;
			case 'b':
				if (getlparam(&l) || l <= 0)
					usage();
				boxnum = l;
				break;
			case 'e':
				strcmpfp = strcmp;
				break;
			case 'o':
				strcmpfp = strcmp3;
				break;
			case 's':
				if (getlparam(&seed) || seed < 0)
					usage();
				break;
			default:
				usage();
		}
		op->short_opt = 0;
	}

	/* get database */
	if (operation != VERSION && (p = getparam()) == 0) {
		if ((p = getenv("HOME")) == 0)
			p = "";
		if ((p = strcat2(p, DATABASE)) == 0) {
			errmsg("%s", strerror(errno));
			exit(E_NOMEM);
		}
	}

	/* check end of command line */
	if (!opteol())
		usage();

	/* default operation is learn */
	if (operation == 0)
		operation = LEARN;

	/* some checks */
	if (operation != LEARN) {
		if (seed >= 0)
			usage();
	}
	if (operation != NEW && operation != ADD && operation != APPEND) {
		if (boxnum != 0)
			usage();
	}

	if (boxnum == 0)
		boxnum = BOXNUM;

	/* init random generator */
	if (seed < 0)
		seed = time(0);
	srandom(seed);

	if (strcmpfp == NULL)
		strcmpfp = strcmp2;

	if (updatetime == 0)
		updatetime = UPDATETIME;

	/* open database */
	switch (operation) {
		case DELETE:
			q = "r+";
			break;
		case LEARN:
			q = "r+";
			break;
		case PRINT:
			q = "r";
			break;
		case NEW:
			q = "w+";
			break;
		case ADD:
		case APPEND:
			if ((database = fopen(p, "r+")) == 0)
				q = "w+";
			else
				q = 0;
			break;
		default:
			q = NULL;
	}

	if (q != NULL) {
		if ((database = fopen(p, q)) == 0) {
			errmsg("`%s\': %s", p, strerror(errno));
			exit(E_IO);
		}
	}

	switch (operation) {
		case DELETE:
		case LEARN:
		case PRINT:
			status = readbox(database);
			if (status)
				goto E;
		default:
			break;
	}

	switch (operation) {
		case ADD:
		case APPEND:
			status = readbox(database);
			if (status == E_NOERROR)
				break;
			if (status != E_EOF)
				goto E;
			if ((status = initbox(boxnum)) != E_NOERROR)
				goto E;
			fseek(database, 0, SEEK_SET);
			if ((status = writebox(database)) != E_NOERROR)
				goto E;
			if (fseek(database, 0, SEEK_CUR))
				status = E_IO;
			break;
		default:
			break;
	}

	if (operation != NEW && boxnum == 0) {
		errmsg("Program error. Line %d, file `%s\'",
		__LINE__, __FILE__);
		exit (E_INVAL);
	}

	/* do the real work */
	switch (operation) {
		case LEARN:
			status = learn(database);
			break;
		case NEW:
			if ((status = scanbox(stdin)) != E_NOERROR)
				break;
			fseek(database, 0, SEEK_SET);
			if ((status = writebox(database)) != E_NOERROR)
				goto E;
			if (fseek(database, 0, SEEK_CUR))
				status = E_IO;
			status = mergevoc(database);
			break;
		case PRINT:
			if ((status = printbox(stdout)) != E_NOERROR)
				break;
			status = printvoc(database);
			break;
		case APPEND:
			status = appendvoc(database);
			break;
		case VERSION:
			status = printversion();
			break;
		case ADD:
			status = addvoc(database);
			break;
		case DELETE:
			status = delvoc(database);
			break;
		default:
			errmsg("Program error. Line %d, file `%s\'",
			__LINE__, __FILE__);
			exit (E_INVAL);
	}

	/* error switch */
	E: switch (status) {
		case E_NOERROR:
			break;
		case E_IO:
			if (ferror(database))
				errmsg("`%s\': %s", p, strerror(errno));
			else
				errmsg("%s", strerror(errno));
			exit(E_IO);
		case E_EOF:
			errmsg("Unexpected end of file");
			exit(E_EOF);
		case E_FORMAT:
			errmsg("Bad Format");
			exit(E_FORMAT);
		case E_INVAL:
			errmsg("Invalid argument");
			exit(E_INVAL);
		case E_NOMEM:
			errmsg("%s", strerror(errno));
			exit(E_NOMEM);
		default:
			errmsg("Unknown error");
			exit(E_INVAL);
	}

	/* close files */
	if ((database && fclose(database)) || fclose(stdin) || fclose(stdout))
	{
		errmsg("%s", strerror(errno));
		exit(E_IO);
	}

	DEBUG((2, "leaving main()"));
	exit(E_NOERROR);
}
