// jxkill.cc -- Send an arbitrary WM protocol element to a window.
// For use in an xkill type application.
// Copyright (c) 1997 by James F. Carter.  1997-01-10, g++ v2.7.2

/*
 * The X Consortium, and any party obtaining a copy of these files from
 * the X Consortium, directly or indirectly, is granted, free of charge, a
 * full and unrestricted irrevocable, world-wide, paid up, royalty-free,
 * nonexclusive right and license to deal in this software and
 * documentation files (the "Software"), including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons who receive
 * copies from any such party to do so.  This license includes without
 * limitation a license to do the foregoing actions under any patents of
 * the party supplying this software to the X Consortium.
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE X CONSORTIUM OR THE AUTHOR BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Except as contained in this notice, the name of the X Consortium shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization
 * from the X Consortium.
 */


#include <X11/X.h>
#include <X11/Xlib.h>		//Needed by Xutil.h
#include <X11/Xutil.h>		//The only reason for this is XClassHint.
#include <tk.h>
#include <Tcl++.h>

// ===== Command class for jxkill (see Tcl++.h)

class JxkillCmd : public TclCommand {
  public:
    Tk_Window toplevel;

    JxkillCmd(TclCopy& tcl);
    ~JxkillCmd() { this->TclCommand::~TclCommand(); }
    static int Jxhandler(Display*, XErrorEvent*);
    static int JxIOhandler(Display*);
    /*virtual*/ int Cmd(int argc, char** argv);
    int SendProto(int argc, char** argv);
    int Family(int argc, char** argv);
    int Kill(int argc, char** argv);
    int ClassHint(int argc, char** argv);
};

// Constructor
JxkillCmd::JxkillCmd(TclCopy& tcl) : 
	TclCommand(tcl, "jxkill")
{
    toplevel = Tk_MainWindow(interp);
    Tk_MakeWindowExist(toplevel);	//But you still need "update" to get
					//a window ID out of it.
    XSetErrorHandler(&Jxhandler);
    XSetIOErrorHandler(&JxIOhandler);
}

// Error handler: ignore all errors.  When you're killing windows, they
// have a tendency to disappear at unexpected times.
/*static*/ int JxkillCmd::Jxhandler(Display*, XErrorEvent*) { return 0; }
/*static*/ int JxkillCmd::JxIOhandler(Display*) { exit(0); }

// Subcommand dispatcher
int JxkillCmd::Cmd(int argc, char* argv[]) {
    int rtn = TCL_ERROR;
    //Switch to call subcodes
    if (!strcmp(argv[1], "sendproto")) {
	rtn = SendProto(argc, argv);
    } else if (!strcmp(argv[1], "family")) {
	rtn = Family(argc, argv);
    } else if (!strcmp(argv[1], "kill")) {
	rtn = Kill(argc, argv);
    } else if (!strcmp(argv[1], "classhint")) {
	rtn = ClassHint(argc, argv);
    } else {
	AppendResult("Unrecognized subcode ", argv[1], NULL);
    }
    return rtn;
}

// The window arguments of the subcommands are tcl string integers (usually
// hex, e.g. 0x1c0002a) representing the X-server's window ID.

// SendProto -- Send a particular window manager protocol atom to a 
// designated window.  Args:
//	[jxkill sendproto $window $atom]
// The atom is specified as a char. string such as "WM_SAVE_YOURSELF".
// Returns 1 if the atom was sent, or 0 if not.  The window has to register
// to accept particular protocol atoms, and the atom is not sent to windows
// that can't handle it.
int JxkillCmd::SendProto(int argc, char** argv) {
    Window	window;
    Atom	atom = XInternAtom(Tk_Display(toplevel), argv[3], False);
;
    const char* rtn = "0";		//Preload a failure return value.
		// Convert the tcl argument strings to integers.
    Get(argv[2], (int*)(&window));
    Get(argv[3], (int*)(&atom));
		// Determine if the window can accept the atom.  Hunt for the 
		// given atom in the list of accepted protocols.  
    Atom	*protocols;
    int		num_prots;
    if (! XGetWMProtocols(Tk_Display(toplevel), window, 
				&protocols, &num_prots)) { goto bugout; }
    while (--num_prots >= 0) {
	if (protocols[num_prots] == atom) { break; }
    }
    XFree(protocols);
    if (num_prots < 0) { goto bugout; }

		// The window wants the atom.  Send an event to it.
    XEvent event;

    event.type = ClientMessage;
    event.xclient.display = Tk_Display(toplevel);
    event.xclient.window = window;

    event.xclient.message_type = 
	XInternAtom(Tk_Display(toplevel), "WM_PROTOCOLS", False);
    event.xclient.format = 32;		//All WM_PROTOCOLS are 32 bits.
    event.xclient.data.l[0] = atom;	//The atom
    event.xclient.data.l[1] = CurrentTime;

	//Send the event (display, window, propagate, event-mask, event)
    rtn = XSendEvent(Tk_Display(toplevel), window, False, 0L, &event) 
	? "1" : "0";
    XFlush(Tk_Display(toplevel));
		//"Destructor"
  bugout:
    SetResult(rtn);
    return TCL_OK;
}

// Get windows surrounding a particular window.  Args:
//	[jxkill family $window]
// Returns (as the tcl result) a list containing (in order) the root window, 
// the parent, and all the children (if any) of the argument window.  If
// the window is invalid, an empty list is returned.
int JxkillCmd::Family(int argc, char** argv) {
    Window	window;		//Window whose children are desired
    Get(argv[2], (int*)(&window));
    char	bfr[16];
    Window	parroot[2];
    Window*	lists[2];
    unsigned int nlists[2];
    lists[0] = parroot;
    nlists[0] = 2;
    if (! XQueryTree(Tk_Display(toplevel), window, 
		&(parroot[1]), &(parroot[0]), &(lists[1]), &(nlists[1]))) {
			//Most likely the window got killed.  Return empty list.
	return TCL_OK;
    }
    for (int i = 0; i < 2; ++i) {
	for (int j = nlists[i]; --j >= 0; ) {
	    sprintf(bfr, "0x%x", lists[i][j]);
	    AppendElement(bfr);
	}
    }
    XFree(lists[1]);
    return TCL_OK;
}

// Kill an X client.  See "man XKillClient" for what arguments may be given,
// but typically a window ID will be given.  
//	[jxkill kill $window]
int JxkillCmd::Kill(int argc, char** argv) {
    Window	window;		//Window whose children are desired
    Get(argv[2], (int*)(&window));
    XKillClient(Tk_Display(toplevel), window);
    return TCL_OK;
}

// Return the class hints of a window, a 2 component list giving the 
// application name and the class name, or "<no name>" if unavailable,
// as it is for most subwindows.  This is mainly for debugging.
//	[jxkill classhint $window]
int JxkillCmd::ClassHint(int argc, char** argv) {
    Window	window;		//Window whose hints are desired
    Get(argv[2], (int*)(&window));
    XClassHint     *class_hint = XAllocClassHint();
    if (XGetClassHint(Tk_Display(toplevel), window, class_hint)) {
	AppendElement(class_hint->res_name);
	AppendElement(class_hint->res_class);
	if (class_hint->res_class)
	    XFree(class_hint->res_class);
	if (class_hint->res_name)
	    XFree(class_hint->res_name);
    } else {
	SetResult("<no name>");
    }
    XFree(class_hint);
    return TCL_OK;
}

// ===== The main program and supporting class

int Tcl_AppInit(Tcl_Interp* interp) {
    Tcl tcl(interp);			//Includes Tcl_Init
    if (! *tcl) return TCL_ERROR;
	//argv/argc have been set into tcl $argv by Tk_Main.
    TCL_BARF( Tk_Init(interp) );
    long int xout;
    TCL_BARF( tcl.Expr("[lindex [set argv] 0] == {-F}", &xout) );
    JxkillCmd jxk(tcl);
    if (xout) {
	TCL_BARF( tcl.Eval("source [lindex [set argv] 1]") );
    } else {
	extern char tclpgm[];
	TCL_BARF( tcl.Eval(tclpgm) );
    }
    return TCL_OK;
}

extern void Tk_Main(int, char**, Tcl_AppInitProc);

int main(int argc, char** argv) {
    Tk_Main(argc, argv, Tcl_AppInit);
    return 0;
}
