// Tcl++.h -- Encapsulating tcl programmatic interface in a C++ class
// Copyright (c) 1996 by James F. Carter -- 1996-03-11, Gnu g++ v2.6.3

/*
 * The X Consortium, and any party obtaining a copy of these files from
 * the X Consortium, directly or indirectly, is granted, free of charge, a
 * full and unrestricted irrevocable, world-wide, paid up, royalty-free,
 * nonexclusive right and license to deal in this software and
 * documentation files (the "Software"), including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons who receive
 * copies from any such party to do so.  This license includes without
 * limitation a license to do the foregoing actions under any patents of
 * the party supplying this software to the X Consortium.
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE X CONSORTIUM OR THE AUTHOR BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Except as contained in this notice, the name of the X Consortium shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization
 * from the X Consortium.
 */

// This is not incr tcl.  This is a rendition of basic tcl in a form 
// compatible with C++ compilation.

/* Usage:

The tcl public entry points Tcl_Whatever are grouped into 6 parts:
	1.  TclCopy: core tcl operations (most of them);
	2.  Tcl: Same thing but its constructor initializes the interpreter;
	3.  TclList: Split up tcl lists
	4.  TclDString: tcl dynamic strings;
	5.  TclCommand: creation of user-defined tcl commands;
	6.  TclTrace: trace / trap on variables;
	7.  TclHash: hash tables.

In general, if there is a command Tcl_Whatever, then there will be a member
function of one of these classes called Whatever, with the same arguments
except minus the object pointer, which is provided implicitly due to being
a member function.  For DStrings, the prefix Tcl_DString is removed and
operator notation is used (= for assignment, += to append).  For Tcl_Hash,
that prefix is removed, and some functions involve imbedded classes
TclHash::Entry and TclHash::Search.  

Suggestions for converting existing code to work in C++:  Suppose the "C"
version has this form:

    #include <various.h>
    struct MYDATA {
	data members...
    };
    static struct MYDATA mydata[4];		//Your ClientData
	    //Forward references to public and static functions
    int MY_Glorfl_Cmd(ClientData, Tcl_Interp *, int, char* argv[]);
    static int MY_Read(Tcl_Interp* interp);	//An internal module
	    //Package initialization
    int MY_Init(Tcl_Interp* interp) {
	Tcl_CreateCommand(interp, "glorfl", &MY_Glorfl_Cmd,
	    (ClientData)mydata, (Tcl_CmdDeleteProc*)NULL);
	return TCL_OK;
    }
    
    int MY_Glorfl_Cmd(ClientData cd, Tcl_Interp *interp, 
						int argc, char* argv[]) {
	//body of command
    }
    etc. etc.
    
Assuming you use the "standard" symbol interp for the interpreter, you
can neglect to convert any of the function bodies (except for possible
optional changes to internal function names), simply turning it into a
class like this.  If you use the ClientData explicitly, simply change
the pointer to "this", or remove it, since the ClientData is now stored
in your class object.

    #include <various.h>
    
    class GlorflCmd : public TclCommand {
	struct MYDATA {
	    data members...
	};
	struct MYDATA mydata[4];		//Your ClientData
	
	GlorflCmd(Tcl* ainterp);		//Constructor (was MY_Init)
	    //Member functions.  "Cmd" is the main command; note change in args.
	virtual int Cmd(int argc, char* argv[]);
	int MY_Read();				//An internal module
    };	    //End of command object definition.

	    //Package initialization -- this becomes the constructor.
	    //Tcl_CreateCommand turns into TclCommand constructor.
    GlorflCmd::GlorflCmd(Tcl_Interp* ainterp) :
	    TclCommand(ainterp, "glorfl")
    {
	//Put any programmatic initialization here.  Lose return statement.
    }
    
    int GlorflCmd::Cmd(int argc, char* argv[]) {
	//body of command, no changes needed, it says here.
    }
    etc. etc.

If you rewrite (or newly write) your command in C++, TclCopy (providing
the interpreter pointer) is a base class of TclCommand (and also
TclTrace), and so you will rarely need to write interp-> explicitly.  Check
out the Resprintf and DSprintf extensions whereby you can do "sprintf"
"directly" into a DString or the tcl result rather than allocating a buffer
and copying its content.

*/

#ifndef TCLXX_H
#define TCLXX_H

// This macro makes the program return if a tcl operation fails.  The following
// semicolon is optional.
#define TCL_BARF(operation) if (TCL_OK != (operation)) return TCL_ERROR;

#include <tcl.h>

// Tcl basic class.  If the "C" interface has a command Tcl_Command, there
// will be a corresponding command here called Command, with the same arguments
// except minus interp.  If your own name shadows it, access as Tcl::Command. 
// Commands with variable argument lists are not reproduced because the 
// variable arg list cannot be transferred to the called function.
// Note: call the POSIX utilities (p. 352) directly.  
// Note: This class does not initialize the interpreter; there should be
// just one Tcl class (which does so), and several copies can point to the
// same interpreter.
class TclCopy {
  public:
		//Watch out for this -- it cues GetBoolean, etc.  It's assumed
		//that Boolean* and int* can be interconverted freely.
    typedef unsigned int Boolean; //Watch this -- it cues GetBoolean, etc.

    Tcl_Interp* interp;		//Logically should be a base class, but that
				//would require rewriting Tcl_CreateInterp
				//as a C++ constructor.

		//Constructor
    TclCopy(Tcl_Interp* ainterp) : interp(ainterp) { }

		//Auto readout the interpreter pointer.
    operator Tcl_Interp* ()
	{ return interp; }
    Tcl_Interp* operator -> ()
	{ return interp; }
    Tcl_Interp* operator * ()
	{ return interp; }

		//Script evaluation (p.288)  The script args are not const
		//char* because the interpreter fiddles with the scripts.
    int Eval(char* script)
	{ return Tcl_Eval(interp, script); }
    int EvalFile(char* filename)
	{ return Tcl_EvalFile(interp, filename); }
    int GlobalEval(char* script)		//Evaluated at global level
	{ return Tcl_Eval(interp, script); }
    int RecordAndEval(char* script, int flags)
	{ return Tcl_RecordAndEval(interp, script, flags); }
		//This one has to concatenate all the scripts, so isn't as
		//efficient as the "C" original.
    int VarEval(char* script, ...);

		//Creating commands -- see TclCommand class.

		//Setting the result(p. 294).  A const char* result is 
		//*assumed* to be static (not volatile).  Cast to (char*) to 
		//override.
    void SetResult(const char* res)
	{ Tcl_SetResult(interp, (char*)res, TCL_STATIC); }
		//Setting the result from a volatile buffer
    void SetResult(char* res)
	{ Tcl_SetResult(interp, res, TCL_VOLATILE); }
		//Setting the result from a dynamic source
    void SetResDyn(char* res)
	{ Tcl_SetResult(interp, res, TCL_DYNAMIC); }
		//Setting the result with a generic freeproc
    void SetResDyn(char* res, Tcl_FreeProc* freeproc)
	{ Tcl_SetResult(interp, res, freeproc); }
		//Append arbitrarily many strings to the result (end with NULL)
    void AppendResult(char* res1, ...);
		//Append a list element to a list result.
    void AppendElement(char* elem)
	{ Tcl_AppendElement(interp, elem); }
		//Clear the result.
    void ResetResult()
	{ Tcl_ResetResult(interp); }
		//EXTENSION: sprintf into the result.  As with regular sprintf,
		//the program trusts you, without checking, not to go over
		//TCL_RESULT_SIZE (= 200) bytes.
    void Resprintf(const char* format, ...);

		//Parsing (p. 314).  These take a string in input and store a 
		//numeric result in output, returning TCL_OK or TCL_ERROR 
		//having left an error message.  The type of *output controls 
		//what conversion is done.
    int Get(char* input, int* output)
	{ return Tcl_GetInt(interp, input, output); }
    int Get(char* input, double* output)
	{ return Tcl_GetDouble(interp, input, output); }
    int Get(char* input, Boolean* output)
	{ return Tcl_GetBoolean(interp, input, (int*)output); }

		//Expression evaluation (p. 314). Returns TCL_OK if the 
		//expression is OK.
		//For char*, the result appears in interp->result.
    int Expr(char* expression)
	{ return Tcl_ExprString(interp, expression); }
		//The rest of the Expr functions store a result in *output.
		//BEWARE, this one wants a long int*, not an int*.
    int Expr(char* expression, long int* output)
	{ return Tcl_ExprLong(interp, expression, output); }
    int Expr(char* expression, double* output)
	{ return Tcl_ExprDouble(interp, expression, output); }
    int Expr(char* expression, Boolean* output)
	{ return Tcl_ExprBoolean(interp, expression, (int*)output); }

		//Error code setting (p. 320)
    void AddErrorInfo(char* message)
	{ Tcl_AddErrorInfo(interp, message); }
		//Tcl_SetErrorCode: can't handle variable number of arguments.
		//Call it directly.

		//Variables (p. 326).  The scalar and array versions are
		//distinguished by the number of arguments.  char* return
		//values point to the content string (after setting).
		//Flag summary; "or" these together, or 0 for no flags:
		//  TCL_GLOBAL_ONLY	Use global symbol table (not local).
		//  TCL_LEAVE_ERR_MSG	Store error message in result if fails.
		//  TCL_APPEND_VALUE	Append to current content if exising.
		//  TCL_LIST_ELEMENT	Add {} and separators for a list.
    char* SetVar(char* name, char* newvalue, int flags)
	{ return Tcl_SetVar(interp, name, newvalue, flags); }
    char* SetVar(char* name, char* member, char* newvalue, int flags)
	{ return Tcl_SetVar2(interp, name, member, newvalue, flags); }
    char* GetVar(char* name, int flags)
	{ return Tcl_GetVar(interp, name, flags); }
    char* GetVar(char* name, char* member, int flags)
	{ return Tcl_GetVar2(interp, name, member, flags); }
    int UnsetVar(char* name, int flags)
	{ return Tcl_UnsetVar(interp, name, flags); }
    int UnsetVar(char* name, char* member, int flags)
	{ return Tcl_UnsetVar2(interp, name, member, flags); }
		//"C" to tcl variable linkage (p. 326); tcl writes change the 
		//variable value and upon tcl reads, the "C" variable is 
		//converted to give the value.  
    int LinkVar(char* name, int* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, TCL_LINK_INT); }
    int LinkVar(char* name, double* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, TCL_LINK_DOUBLE); }
    int LinkVar(char* name, char* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, TCL_LINK_STRING); }
    int LinkVar(char* name, Boolean* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, TCL_LINK_BOOLEAN); }
		//These variants do not allow tcl writes (readonly).
    int LinkVarRO(char* name, int* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, 
				TCL_LINK_READ_ONLY | TCL_LINK_INT); }
    int LinkVarRO(char* name, double* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, 
				TCL_LINK_READ_ONLY | TCL_LINK_DOUBLE); }
    int LinkVarRO(char* name, char* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, 
				TCL_LINK_READ_ONLY | TCL_LINK_STRING); }
    int LinkVarRO(char* name, Boolean* cvarbl)
	{ return Tcl_LinkVar(interp, name, (char*)cvarbl, 
				TCL_LINK_READ_ONLY | TCL_LINK_BOOLEAN); }
		//Erase a "C" variable link (recommended if the link is in a
		//structure that is being freed or is a local variable of
		//a procedure that is exiting).
    void UnlinkVar(char* varname)
	{ Tcl_UnlinkVar(interp, varname); }

		//Tracing: see class TclTrace.
};

// This class actually creates and initializes the interpreter.
class Tcl : public TclCopy {
  public:
		//Constructor and destructor.  For most apps, explicitly
		//deleting the interpreter is anally retentive.  (p.288)
    Tcl(Tcl_Interp* ainterp = Tcl_CreateInterp()) 
	: TclCopy(ainterp) { 
	    if (Tcl_Init(ainterp) != TCL_OK) interp = NULL;
	}
    virtual ~Tcl()
	{ Tcl_DeleteInterp(interp); }
		//Don't even think about trying to copy or assign this.
		//If you need to, you should be using a TclCopy.
  private:
    Tcl(const Tcl& t) : TclCopy(t) { }
    void operator = (const Tcl& t) { }
};

// Split up tcl lists.  
class TclList : public TclCopy {
    int count;			//Number of members in the list
    char** list;		//Pointers to the members; list[count] == NULL

  public:
		//Semi-default constructor
    TclList(TclCopy& interp)
	: TclCopy(interp), count(0), list(NULL)
	{ }	
		//Constructor to have specific number of slots that can
		//later be filled in.
    TclList(TclCopy& interp, const int N) 
	: TclCopy(interp), count(N), list(new (char*)[N+1])
	{ }
		//Constructor loading a pre-existing list.
    TclList(TclCopy& interp, char* data) 
	: TclCopy(interp), count(0), list(NULL)
	{ *this = data; }
		//Destructor
    ~TclList()
	{ delete list; }
		//Assign (split) a packed list.  Return value is TCL_OK
		//or TCL_ERROR.
    int operator = (char* data)
	{ delete list; return Tcl_SplitList(interp, data, &count, &list); }
		//Copy assignment can go awry since the referents of the 
		//pointers are (or may be) in the same block as b.list.
    void operator = (const TclList& b);
		//Number of elements (dimension) of the list.
    int dim() const
	{ return count; }
		//Element readout; the return value is modifiable.
    char*& operator [] (int i)
	{ return list[i]; }
		//Remerge the list.  NOTE, the char* is malloced and the
		//user must free/delete it.  See a TclDString assignment
		//operator for a more sanitary way.  Use SetResDyn if 
		//returning a list as a result, to get it freed.
    operator char* ()
	{ return Tcl_Merge(count, list); }
};

//Tcl dynamic strings (p. 346).  Note: call Tcl_CommandComplete and
//Tcl_StringMatch directly.
class TclDString : public Tcl_DString {
  public:
		//Constructors: default, and from content.
    TclDString()
	{ Tcl_DStringInit(this); }
    TclDString(TclDString& ds) { 
	Tcl_DStringInit(this); 
	Tcl_DStringAppend(this, ds, ds.Length()); 
    }
    TclDString(char* s)	{ 
	Tcl_DStringInit(this); 
	Tcl_DStringAppend(this, s, -1); 
    }
    ~TclDString()
	{ Tcl_DStringFree(this); }
		//Read out the content.
    operator char* ()
	{ return Tcl_DStringValue(this); }
		//Return (efficiently) the length of the string, not including
		//the ending '\0'
    int Length()
	{ return Tcl_DStringLength(this); }
		//Assign new content to a string.
    TclDString& operator = (TclDString& ds) {
	    Tcl_DStringFree(this); 
	    Tcl_DStringAppend(this, ds, ds.Length()); 
	    return *this;
	}
    TclDString& operator = (char* s) {
	    Tcl_DStringFree(this); 
	    Tcl_DStringAppend(this, s, -1); 
	    return *this;
	}
		//Merge a split string and assign to a dstring.  Too bad
		//you can't just transfer responsibility for the string.
    TclDString& operator = (TclList& tl) {
	    char* tlc = (char*)tl;
	    Tcl_DStringFree(this); 
	    Tcl_DStringAppend(this, tlc, -1); 
	    delete tlc;
	    return *this;
	}
		//Append to the end of a string.
    TclDString& operator += (TclDString& ds) {
	    Tcl_DStringAppend(this, ds, ds.Length()); 
	    return *this;
	}
    TclDString& operator += (char* s) {
	    Tcl_DStringAppend(this, s, -1); 
	    return *this;
	}
		//Append a list element, munging to ensure a valid list.
    char* AppendElement(char* s)
	{ return Tcl_DStringAppendElement(this, s); }
		//Put in the separator that starts a sublist
    void StartSublist()
	{ Tcl_DStringStartSublist(this); }
		//Put in the separator that ends a sublist
    void EndSublist()
	{ Tcl_DStringEndSublist(this); }
		//Truncate to a specific length (no effect if already shorter).
		//You can also expand the string to ensure that N bytes will
		//fit.
    void Trunc(int length)
	{ Tcl_DStringTrunc(this, length); }
		//Efficiently truncate to zero length.  Unfortunate name choice.
    void Free()
	{ Tcl_DStringFree(this); }
		//Move the content to the interpreter's result, leaving a
		//zero length content behind.
    void Result(TclCopy& interp)
	{ Tcl_DStringResult(interp, this); }
		//EXTENSION: sprintf into the DString.  The max length of
		//the data must be specified.
    void DSprintf(int length, const char* format, ...);
};


/* Command creation (p. 294).
    To create a tcl command, define a class derived from TclCommand.  Its
    Cmd member function will perform the command activities.  Use "interp"
    from TclCommand as your interp pointer if needed.  Example:

	class GlorflCmd : public TclCommand {
	    int DataMember;		//Common data for all glorfls.
	    GlorflCmd(TclCopy& tcl) : TclCommand(tcl, "glorfl") { constructor }
	    ~GlorflCmd() { TclCommand::~TclCommand(); }
	    virtual int Cmd(int argc, char** argv) {
		Tcl_WhateverFunc(interp, other args); //Whatever the cmd does
		interp.SetResult("its result");
		return TCL_OK;
	    }
	};

    The recommended name of the class is <Name>Cmd, same as the name of the
    "C" command proc would be.  

    If there are lots of glorfls, one of the data members would be a hash
    containing pointers to a separately defined structure.  A user-defined
    "constructor" would return a key (e.g. [glorfl open]), which would be
    given back as an argument to other sub-procedures (e.g. [glorfl print 
    $handle]).  

    On the other hand, if there can be only one glorfl, all its data can be
    in the GlorflCmd structure.
*/
class TclCommand : public TclCopy {
  private:  
    TclDString name;
  public:
    TclCommand(TclCopy& tcl, char* cmdname) :
	TclCopy(tcl), name(cmdname)
	{ Tcl_CreateCommand(interp, cmdname, &CCmd, this, &DCmd); }
		//The derived class destructor must call 
		//TclCommand::~TclCommand() explicitly to avoid a memory 
		//leak on the DString.  NOTICE, on a statically allocated
		//command, if the user's tcl script deletes it via 
		//[rename cmd {}], the virtual destructor will be called.
		//Then when the static allocation is cleared, it will be
		//destroyed again.  The user's virtual destructor should
		//check the length of member name and exit if zero.
    virtual ~TclCommand();
		//This is the actual command executing function.  
    virtual int Cmd(int argc, char** argv) = 0;
		//Inverse interface from tcl interp in "C" to the C++ executor.
    static int CCmd(ClientData clthis, Tcl_Interp* interp, 
						int argc, char** argv);
		//Command destruction callback.
    static void DCmd(ClientData clthis);
};

// Tracing (p. 327 and 332).  Define a derived class of a TclTrace object and 
// define the virtual Callback procedure to do the tracing activity.  Then
// use the TraceVar member function to apply the trace to any number of 
// variables.  This class is similar to TclCommand.
class TclTrace : public TclCopy {
  public:
		//Constructor
    TclTrace(TclCopy& tcl) : TclCopy(tcl) { }
		//Cause a trace on the specified variable (or var(member)).
    int TraceVar(char* varname, int flags)
	{ return Tcl_TraceVar(interp, varname, flags, &CCmd, this); }
    int TraceVar(char* varname, char* member, int flags)
	{ return Tcl_TraceVar2(interp, varname, member, flags, &CCmd, this); }
		//Cancel a previously requested trace.
    void UntraceVar(char* varname, int flags)
	{ Tcl_UntraceVar(interp, varname, flags, &CCmd, this); }
    void UntraceVar(char* varname, char* member, int flags)
	{ Tcl_UntraceVar2(interp, varname, member, flags, &CCmd, this); }
		//Search for traces: call Tcl_VarTraceInfo directly, specifying
		//&TclTrace::CCmd as the trace proc.
    
		//This is the actual command executing function.  
    virtual char* Callback(char* varname, char* member, int flags) = 0;
		//Inverse interface from tcl interp in "C" to the C++ executor.
    static char* CCmd(ClientData clthis, Tcl_Interp* interp, 
					char* varname, char* member, int flags)
	{ return ((TclTrace*)clthis)->Callback(varname, member, flags); }
};


// Tcl hash table (associative array) support (p. 338).  The hash table 
// contains entries each of which have a key and a pointer to a referent.
// The key can be a char*, an int, or multiple ints.  
template <class CONTENT> class TclHash {
  private:
    Tcl_HashTable hash;			//The actual table (pointers to lists)
    int isnewi;				//For newptr output.

  public:
		//Constructor.  The keytype is TCL_STRING_KEYS, 
		//TCL_ONE_WORD_KEYS (for keys of one int), or an int signifying
		//that the key is an array of so many ints.
    TclHash(int keytype = TCL_STRING_KEYS)
	{ Tcl_InitHashTable(&hash, keytype); }
    ~TclHash()
	{ Tcl_DeleteHashTable(&hash); }
		//Don't even think about assigning a hash table
  private:
    void operator = (const TclHash& bogus) { }
  public:
		//One entry from the hash table.  Declare as 
		//TclHash<CONTENT>::Entry.  Several functions below return 
		//pointers to Entry (or NULL if no Entry is available).
    struct Entry : public Tcl_HashEntry {
		//Having found a hash entry, get rid of it.  Idiotproof on a
		//NULL pointer.  Does NOT delete the referent.  Example: 
		//delete myhash.find(key);
	static void operator delete(void* entry)
	    { if (entry) Tcl_DeleteHashEntry((Tcl_HashEntry*)entry); }

		//Change the value of an entry.  Give it a reference to the
		//referent; the subroutine will store a pointer.  Example: 
		//*(myhash.find(key)) = newvalue;       (not &newvalue)
	void operator = (const CONTENT& newval)
	    { Tcl_SetHashValue(this, &newval); }

		//Read out the content of an entry.  Example:
		//SetResult(*(myhash.find(key)));
	operator CONTENT& () 
	    { return *(CONTENT*)Tcl_GetHashValue(this); }

    };
		//Functions for manipulating Entries.  The return value is
		//a pointer, and is NULL if no such entry exists.
		//Find the entry with that key; if there is none, create one.
    Entry* create(char* key)
	{ return (Entry*)Tcl_CreateHashEntry(&hash, key, &isnewi); }
    Entry* create(int* key)
	{ return (Entry*)Tcl_CreateHashEntry(&hash, 
					(char*)(void*)key, &isnewi); }
		//If the recently created entry is new, isnew() returns 1.
    int isnew()
	{ return isnewi; }
		//Find tne entry with that key, or NULL if not there.
    Entry* find(char* key)
	{ return (Entry*)Tcl_FindHashEntry(&hash, key); }
    Entry* find(int* key)
	{ return (Entry*)Tcl_FindHashEntry(&hash, (char*)(void*)key); }
		//Get back the key for an entry.  Explicitly cast to const int*
		//if that's the key type.  Don't modify the key or you'll never
		//find it again.  (Note, the book is wrong, Tcl_GetHashKey needs
		//both the table and the entry.)
    const char* key(Entry* ent)
	{ return Tcl_GetHashKey(&hash, ent); }

		//Storage area for enumerating all Entries of a hash table.
		//Default constructor.  Declare as TclHash<CONTENT>::Search.  
    struct Search : public Tcl_HashSearch {
		//"Assign" a specific hash table to it, to begin a search.
		//The first Entry pointer is returned (or NULL if empty).
	Entry* operator = (TclHash& ht)
	    { return (Entry*)Tcl_FirstHashEntry(&(ht.hash), this); }
		//Advance to the next entry (or NULL if all used).
	Entry* operator ++ ()
	    { return (Entry*)Tcl_NextHashEntry(this); }
    };

		//Return debug information about the hash table (see p. 344).
		//You must delete/free() the return value yourself.
    char* Stats()
	{ return Tcl_HashStats(&hash); }
};

#endif /*TCLXX_H*/
