// Tcl++.cc -- Encapsulating tcl programmatic interface in a C++ class
// Copyright (c) 1996 by James F. Carter -- 1996-03-11, Gnu g++ v2.6.3

/*
 * The X Consortium, and any party obtaining a copy of these files from
 * the X Consortium, directly or indirectly, is granted, free of charge, a
 * full and unrestricted irrevocable, world-wide, paid up, royalty-free,
 * nonexclusive right and license to deal in this software and
 * documentation files (the "Software"), including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons who receive
 * copies from any such party to do so.  This license includes without
 * limitation a license to do the foregoing actions under any patents of
 * the party supplying this software to the X Consortium.
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE X CONSORTIUM OR THE AUTHOR BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Except as contained in this notice, the name of the X Consortium shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization
 * from the X Consortium.
 */

// This is not incr tcl.  This is a rendition of basic tcl in a form 
// compatible with C++ compilation.

// Bodies for non-inline functions.  There aren't very many.  Mostly
// they have to do with varargs.
#include <Tcl++.h>
#include <stdarg.h>

//Evaluate arbitrarily many strings concatenated (end with NULL).  
int TclCopy::VarEval(char* part1, ...) {
    char* vcs;
    va_list va;
    va_start(va, part1);
    TclDString script(part1);
    while (NULL != (vcs = va_arg(va, char*))) {
	script += vcs;
    }
    return Tcl_Eval(interp, script);
}

//Append arbitrarily many strings to the result (end with NULL).  It would
//be more efficient to pass the vararg directly but if you look at the source,
//this technique isn't really that bad.
void TclCopy::AppendResult(char* res1, ...) {
    char* vcs;
    va_list va;
    va_start(va, res1);
    Tcl_AppendResult(interp, res1, NULL);	//There's always at least 1 arg
    while (NULL != (vcs = va_arg(va, char*))) {
	Tcl_AppendResult(interp, vcs, NULL);
    }
}

//EXTENSION: sprintf into the result.  As with regular sprintf,
//the program trusts you, without checking, not to go over
//TCL_RESULT_SIZE (= 200) bytes.
void TclCopy::Resprintf(const char* format, ...) {
    va_list va; 
    va_start(va, format); 
    vsprintf(interp->result, format, va);
    va_end(va);
}

// TclList copy assignment can go awry since the referents of the 
// pointers are (or may be) in the same block as b.list.
void TclList::operator = (const TclList& b) {
    if (dim() < b.dim()) {
	delete list;
	list = new (char*)[b.dim()+1];
    }
    memcpy(list, b.list, (b.dim()+1)*sizeof(char*));
    count = b.dim();
}

//EXTENSION: sprintf into a DString.
void TclDString::DSprintf(int length, const char* format, ...) {
    va_list va; 
    Trunc(length);
    va_start(va, format); 
    vsprintf((char*)(*this), format, va);
    va_end(va);    
}

//Destructor for TclCommand.  The command name length is used as an interlock;
//The destruct callback will call the destructor if this is nonzero.
TclCommand::~TclCommand() {
    name.Free();
    Tcl_DeleteCommand(interp, name);
}

//Inverse interface from tcl interp in "C" to the C++ executor.
/*static*/ int TclCommand::CCmd(ClientData clthis, Tcl_Interp* interp, 
						int argc, char** argv) {
    return ((TclCommand*)clthis)->Cmd(argc, argv); 
}

//Command destruction callback.
/*static*/ void TclCommand::DCmd(ClientData clthis) {
    if (((TclCommand*)clthis)->name.Length() > 0) 
	((TclCommand*)clthis)->TclCommand::~TclCommand(); 
}

