/*******************************************************************************
* Utility for X-Windows that uses the XLI or XLOADIMAGE utility to display a
* graphic image on the X-Windows background (root window).
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
			/* For random number generation 		      */
#include <sys/time.h>
			/* For error reporting routine			      */
#include <plh_error.h>
#include "xbackdrop.h"

			/* Simple result flags for routines and program       */
#define SUCCESS		1
#define FAILURE		0

			/* Macros to select between two comparable values     */
#define SMALLER(a,b)	(((a) > (b)) ? (b) : (a))
#define LARGER(a,b)	(((a) < (b)) ? (b) : (a))

			/* Macros to test for the existence of a file         */
struct stat dummy;
#define FileExists(a)	(!stat((a), &dummy))

                     	/* macros to switch bits in a 16-bit word for more    */
                     	/* variability in random number seed based on the     */
                     	/* system time 					      */
#define ONEMASK         0x0001
#define INVERT(a)      (((((a)     ) & ONEMASK) <<15) + \
                        ((((a) >> 1) & ONEMASK) <<14) + \
                        ((((a) >> 2) & ONEMASK) <<13) + \
                        ((((a) >> 3) & ONEMASK) <<12) + \
                        ((((a) >> 4) & ONEMASK) <<11) + \
                        ((((a) >> 5) & ONEMASK) <<10) + \
                        ((((a) >> 6) & ONEMASK) << 9) + \
                        ((((a) >> 7) & ONEMASK) << 8) + \
                        ((((a) >> 8) & ONEMASK) << 7) + \
                        ((((a) >> 9) & ONEMASK) << 6) + \
                        ((((a) >>10) & ONEMASK) << 5) + \
                        ((((a) >>11) & ONEMASK) << 4) + \
                        ((((a) >>12) & ONEMASK) << 3) + \
                        ((((a) >>13) & ONEMASK) << 2) + \
                        ((((a) >>14) & ONEMASK) << 1) + \
                        ((((a) >>15) & ONEMASK)     ))

                    	/* method to obtain a random binary variable          */
                    	/* 			0 <= X < 1 		      */
#define RAND_BINARY     (((double)random())/RAND_MAX)

			/* Various image setting codes		              */
#define NOISE_NONE	0
#define NOISE_DEFAULT	1
#define NOISE_VERBOSE	2
#define NOISE_MAX	2
#define DITHER_NONE	0
#define DITHER_BW	1
#define DITHER_COLOR	2
#define DITHER_MAX	2
#define COLOR_BW	0
#define COLOR_DEFAULT	1
#define COLOR_TRUE	2
#define COLOR_MAX	2
#define ROTATE_0	0
#define ROTATE_90	1
#define ROTATE_180	2
#define ROTATE_270	3
#define ROTATE_MAX	3
#define UPDATE_NONE	0
#define UPDATE_MINS	1
#define UPDATE_HRS	2
#define UPDATE_MAX	2
#define MZ_NOCLIP	0
#define MZ_CLIP		1
#define MZ_MAX		1

			/* Generic labels for setting a binary switch         */
#define ON		1
#define OFF		0

/*******************************************************************************
* Default values for initializing image and RC structure elements
*******************************************************************************/  
#define DEF_WEIGHT	1
#define DEF_BRIGHT	100
#define DEF_CNUM	0
#define DEF_COLOR	1
#define DEF_DITHER	0
#define DEF_SMOOTH	0
#define DEF_ZOOM	100
#define DEF_NORM	0
#define DEF_ROTATE	0
#define DEF_UNCOMP	0
#define DEF_BORDER	"black"
#define DEF_IMAGE	NULL
#define DEF_X		640
#define DEF_Y		480
#define DEF_UPDATE	0
#define DEF_DELAY	0
#define DEF_NOISE	0
#define DEF_MAXZOOM	0
#define DEF_EXE		"xli"
#define DEF_ZFIL	"zcat"

/*******************************************************************************
* Image and General RC structures for holding information taken from user files
*******************************************************************************/  
typedef
struct
 {
   int xres;			/* Pixel width of X-Windows display	      */
   int yres;			/* Pixel height of X-Windows display	      */
   char update;			/* Update mode for program		      */
   int delay;			/* Time delay for updating background         */
   int noise;			/* Noise level for invoking graphic utility   */
   char maxzoom;		/* Maximum image zoom method		      */
   char base[CBUFFER_SZ];	/* Name of executable to use for display      */
   char zfil[CBUFFER_SZ];	/* Name of executable filter for uncompress   */
 }
rcopts;

typedef
struct
 {
   unsigned int weight;		/* Point weight for selecting image           */
   unsigned int bright;		/* Percentage of brightness to use with image */
   unsigned int colornum;	/* Maximum number of colors to use with image */
   unsigned int color:2;	/* Color mode to display the image in 	      */
   unsigned int dither:2;	/* Dither mode to display the image with      */
   unsigned char smooth;	/* Smoothing degree to use with image         */
   unsigned int zoom;		/* Percentage of size to use with image       */
   unsigned int normalize:1;	/* Image normalization flag                   */
   unsigned int rotate:1;	/* Image rotation mode		              */
   unsigned int zfilter:1;	/* Image file uncompress filter flag          */
   char *border;		/* Border color for uncovered background      */   
   char *image;			/* name/path to image to display              */
 }
imageopts;

typedef
struct
 {
   unsigned int weight;		/* Point weight for selecting image           */
   char *imagefile;		/* Path to collection's image file	      */
   unsigned int imagenum;	/* Number of images held in collection        */
   unsigned int points;		/* Total number of points used in collection  */
   imageopts *images;		/* Pointer to image structures array          */
 }
collopts;
/*******************************************************************************
* Routine uses the background dimensions, image dimensions, rotation mode, and
* maximum zoom method to report the largest available zoom rate
*******************************************************************************/  
float MaxScaleFactor(int maxx, int maxy, int xdim, int ydim,
		     char rotate, int zoomtype)
 {
   if(rotate)		/* if image is rotated, swap the ratios to check      */
     if (zoomtype == MZ_NOCLIP)
       return SMALLER((float)maxy / xdim, (float)maxx / ydim);
     else		/* if clipping is allow, return larger of two ratios  */
       return LARGER((float)maxy / xdim, (float)maxx / ydim);
   else			/* image is not rotated, check normal ratios          */
     if (zoomtype == MZ_NOCLIP)
       return SMALLER((float)maxx / xdim, (float)maxy / ydim);
     else		/* if clipping is allow, return larger of two ratios  */
       return LARGER((float)maxx / xdim, (float)maxy / ydim);
 }
 
/*******************************************************************************
* Routine prints an error message based on the information passed to it and
* exits the program.
*******************************************************************************/  
void ShowError(char *message, char *routine, char *item)
 {
			/* Print basic error message with error identifier    */
   fprintf(stderr, "Error [%s]", message);

			/* Check for and print optional message components    */
   if(item != NULL)
     fprintf(stderr, " concerning [%s]", item);
   if(routine != NULL)
     fprintf(stderr, " in [%s]", routine);

			/* End error message and abort program                */
   fprintf(stderr, "\n");
   exit(FAILURE);
 }

/*******************************************************************************
* Routine takes the general runtime options and image options and builds the
* command line for executing the 3rd party graphic utility that displays the
* image on the X-Windows backdrop. The command is then passed to the system
* routine for execution.
*******************************************************************************/  
void LoadImage(imageopts options, rcopts genopts)
 {
   int result, x;
   char command[CBUFFER_SZ];
   char temp[SMALLBUFFER_SZ];

			/* Initialize command buffer with command base        */
   strcpy(command, genopts.base);
   strcat(command, " -onroot -center");

			/* Add the border color setting if it is present      */
   if(options.border != NULL)
    {
     sprintf(temp, " -border %s", options.border);
     strcat(command, temp);
    }

			/* Add normalize flag is the image options specify    */
   if(options.normalize)
     strcat(command, " -normalize");

			/* Add a smooth flag for each degree of smoothing     */
   for (x = 0; x < options.smooth; x++)
     strcat(command, " -smooth");

			/* Add flags to set the operation noise level         */
   switch(genopts.noise)
    {
      case NOISE_NONE:
      		strcat(command, " -quiet");
      		break;
      case NOISE_DEFAULT:
      		break;
      case NOISE_VERBOSE:
      		strcat(command, " -verbose");
      		break;
    }

			/* Add flags to perform dithering of image            */
   switch(options.dither)
    {
      case DITHER_BW:
      		strcat(command, " -dither");
      		break;
      case DITHER_NONE:
      		break;
      case DITHER_COLOR:
      		strcat(command, " -cdither");
      		break;
    }

			/* Set limit on number of colors if specified         */
   if(options.colornum != 0)
    {
     sprintf(temp, " -colors %d", options.colornum);
     strcat(command, temp);
    }

			/* Add flags for color mode to display image in       */
   switch(options.color)
    {
      case COLOR_BW:
      		strcat(command, " -gray");
      		break;
      case COLOR_DEFAULT:
      		break;
      case COLOR_TRUE:
      		strcat(command, " -expand");
      		break;
    }
    
    			/* Add flag for brightness level of image             */
   sprintf(temp, " -brighten %d", options.bright);
   strcat(command, temp);

			/* Add flag for rotating image if any is requested    */
   if (options.rotate > 0)
    {
     sprintf(temp, " -rotate %d", options.rotate * 90);
     strcat(command, temp);
    }
    
    			/* Add flag for scaling the image size                */
   sprintf(temp, " -zoom %d", options.zoom);
   strcat(command, temp);

			/* Add name/path of image to display                  */
   strcat(command, " ");
   strcat(command, options.image);

#ifdef DEBUG
   fprintf(stderr, "Executing command...\n%s\n", command);
#endif

			/* Execute command and check for errors               */
   if((result = system(command)) == -1)
     ShowError("Unable to find executable", "LoadImage", genopts.base);
   else if (result == 127)
     ShowError("Unable to execute shell command", "LoadImage", NULL);    
 }
 
/*******************************************************************************
* Routine to retrieve the dimensions of the image by interfacing with
* the 3rd party utility
*******************************************************************************/  
int GetDimensions(char *imagefile, int *xdim, int *ydim, rcopts opts)
 {
   char command[CBUFFER_SZ];
   char buffer[READBUFFER_SZ];
   char *trav;
   FILE *ident;

			/* Build identification command in buffer             */   
   sprintf(command, "%s -identify %s", opts.base, imagefile);
   
   			/* Open process in piped mode to access its output    */
   if(!(ident = popen(command, "r")))
     ShowError(ErrorMessage(E_PROCOPEN), "GetDimensions", "ident");

			/* Read output from utility and close pipe	      */
   fread(buffer, sizeof(char), READBUFFER_SZ, ident);
   pclose(ident);
   
			/* Find token in output string to locate dimensions   */
			/* Output reads '... is a widthxheight ...' 	      */
   trav = strstr(buffer, "is a");	/* Token not found - error occured    */
   if (trav == NULL)
     return FAILURE;
   else					/* Token found - scan for dimensions  */
    {
      trav += 5;			/* Skip up to beginning of width      */
      *xdim = atoi(trav);		/* Read the image width               */
      while(*(trav - 1) != 'x') trav++; /* Skip up until 'x' is passed        */
      *ydim = atoi(trav);		/* Read the image height              */

      return SUCCESS;
    }
 }

/*******************************************************************************
* Routine to split assignment statements read from an RC file into label and
* value. Sets pointers passed to routine to these two items starting point in
* the buffer.
*******************************************************************************/  
void SplitAssignment(char *buffer, char **name, char **value)
 {
   char *start1, *start2, *end;

			/* Locate and delimit the value                       */   
   start2 = buffer + strlen(buffer) - 1;
   while(isspace(*start2)) start2--;
   *(start2 + 1) = '\0';
   while(!isspace(*start2)) start2--;
   *value = start2 + 1;

			/* Locate and delimit the name                        */   
   *name = end = start1 = buffer;
   while(!isspace(*end)) end++;
   *end = '\0';
 }
 
/*******************************************************************************
* Routine reads the RC files, skipping comments (begins with #) and empty
* (begins with white space) lines, and processes the assignments made in the
* file.
*******************************************************************************/  
void ReadRCFile(char *rcfile, rcopts *settings)
 {
   FILE *rc = fopen(rcfile, "r");
   char buffer[READBUFFER_SZ];
   char *name, *value;

			/* Make sure that RC file opened correctly            */   
   if(rc == NULL)
     ShowError("Cannot Open RC File", "ReadRCFile", rcfile);

			/* Read lines from file, split them up, and assign    */
			/* values to general options structure		      */   
   while(fgets(buffer, READBUFFER_SZ, rc) != NULL)
   {
    if((buffer[0] != '#') && !isspace(buffer[0]))
     {
       SplitAssignment(buffer, &name, &value);

       if (!strcmp(name, XLABEL))
         settings->xres = atoi(value);
       else if (!strcmp(name, YLABEL))
         settings->yres = atoi(value);
       else if (!strcmp(name, UPDATELABEL))
         settings->update = atoi(value);
       else if (!strcmp(name, UDELAYLABEL))
         settings->delay = atoi(value);
       else if (!strcmp(name, NOISELABEL))
         settings->noise = atoi(value);
       else if (!strcmp(name, COMLABEL))
         strcpy(settings->base, value);
       else if (!strcmp(name, MZOOMLABEL))
         settings->maxzoom = atoi(value);
       else if (!strcmp(name, ZFILTER))
         strcpy(settings->zfil, value);
     }
   }

			/* Close RC file                                      */
   fclose(rc);
 }

/*******************************************************************************
* Routine to read the image collection entries and store them with the base
* address stored in the passed pointer. Also, tally the total number of points
* allocated to collection entries and store value.
*******************************************************************************/  
int ReadCollectionFile(char *rcfile, collopts **colls, unsigned int *points)
 {
   FILE *rc = fopen(rcfile, "r+");
   int entrycnt = 0;
   int entry;
   char buffer[READBUFFER_SZ];
   char *start;
   char *stop;
   char *tmp;
   
			/* open file, read number of entries, & rewind        */
   if(rc == NULL)
     ShowError("Cannot Open Archive File", "ReadImageFile", rcfile);
   while(fgets(buffer, READBUFFER_SZ, rc) != NULL)
     if((buffer[0] != '#') && !isspace(buffer[0])) entrycnt++;
   rewind(rc);

#ifdef VERBOSE     
   fprintf(stdout, "Reading %d entries from %s...\n", entrycnt, rcfile);
#endif

			/* allocate memory for entry structures               */  
   *colls = (collopts *)malloc(entrycnt * sizeof(collopts));

			/* initialize point counter and entry number          */
   entry = 0;
   *points = 0;
   
   			/* while not at the end of the archive file,          */
   			/* process the single line image entries	      */
   while(fgets(buffer, READBUFFER_SZ, rc) != NULL)
    {
			/* if line is not comment or empty, process it        */
     if((buffer[0] != '#') && !isspace(buffer[0]))
     {
			/* read collection weight, store it, and tally points */
      stop = start = buffer;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      (*colls)[entry].weight = atoi(start);
      if (atoi(start) < 1)
	ShowError("Weights cannot be < 1", "ReadImageFile", NULL);        
      *points += (*colls)[entry].weight;

			/* read image path and store it                       */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      tmp = (char *)malloc(strlen(start) + 1);
      if(tmp == NULL)
        ShowError(ErrorMessage(E_MEMALLOC), "ReadCollectionFile", "colls");
      strcpy(tmp, start);
      (*colls)[entry].imagefile = tmp;

			/* increment collection entry number                  */
      entry++;
     }
    }

			/* close collection file and return entry count       */
   fclose(rc);
   return entrycnt;
 }
 
/*******************************************************************************
* Routine to read the image archive entries and store them with the base
* address stored in the passed pointer. Also, tally the total number of points
* allocated to image entries and store value.
*******************************************************************************/  
int ReadImageFile(char *rcfile, imageopts **images, unsigned int *points)
 {
   FILE *rc = fopen(rcfile, "r+");
   int entrycnt = 0;
   int entry;
   char buffer[READBUFFER_SZ];
   char *start;
   char *stop;
   char *tmp;
   
			/* open file, read number of entries, & rewind        */
   if(rc == NULL)
     ShowError("Cannot Open Archive File", "ReadImageFile", rcfile);
   while(fgets(buffer, READBUFFER_SZ, rc) != NULL)
     if((buffer[0] != '#') && !isspace(buffer[0])) entrycnt++;
   rewind(rc);

#ifdef VERBOSE     
   fprintf(stdout, "Reading %d entries from %s...\n", entrycnt, rcfile);
#endif

			/* allocate memory for entry structures               */  
   *images = (imageopts *)malloc(entrycnt * sizeof(imageopts));

			/* initialize point counter and entry number          */
   entry = 0;
   *points = 0;
   
   			/* while not at the end of the archive file,          */
   			/* process the single line image entries	      */
   while(fgets(buffer, READBUFFER_SZ, rc) != NULL)
    {
			/* if line is not comment or empty, process it        */
     if((buffer[0] != '#') && !isspace(buffer[0]))
     {
			/* read image weight, store it, and tally points      */
      stop = start = buffer;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      (*images)[entry].weight = atoi(start);
      if (atoi(start) < 1)
	ShowError("Weights cannot be < 1", "ReadImageFile", NULL);        
      *points += (*images)[entry].weight;

			/* read image brightness and store it                 */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      if (atoi(start) < 1)
	ShowError("Brightness cannot be < 1", "ReadImageFile", NULL);        
      (*images)[entry].bright = atoi(start);

			/* read maximum number of colors and store it         */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      if (atoi(start) < 0)
	ShowError("Color number cannot be < 0", "ReadImageFile", NULL);        
      (*images)[entry].colornum = atoi(start);
      
			/* read image color mode and store it                 */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      switch(*start)
       {
	 case 'T':
         case 't':
         	(*images)[entry].color = COLOR_TRUE;
         	break;
	 case 'D':
         case 'd':
         	(*images)[entry].color = COLOR_DEFAULT;
         	break;
	 case 'N':
         case 'n':
         	(*images)[entry].color = COLOR_BW;
         	break;
         default:
      		ShowError("Color modes available: T,t,D,d,N,n",
      			  "ReadImageFile", NULL);        
       }

			/* read image dither mode and store it                */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      switch(*start)
       {
	 case 'B':
         case 'b':
         	(*images)[entry].dither = DITHER_BW;
         	break;
	 case 'C':
         case 'c':
         	(*images)[entry].dither = DITHER_COLOR;
         	break;
	 case 'N':
         case 'n':
         	(*images)[entry].dither = DITHER_NONE;
         	break;
         default:
      		ShowError("Dither modes available: B,b,C,c,N,n",
      			  "ReadImageFile", NULL);        
       }

			/* read image smoothing degree and store it           */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      if (atoi(start) < 0)
	ShowError("Smoothing must be a positive value", "ReadImageFile", NULL);        
      (*images)[entry].smooth = atoi(start);

			/* read image scaling and store it                    */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      if (atoi(start) < 0)
	ShowError("Scaling must be a positive value", "ReadImageFile", NULL);        
      (*images)[entry].zoom = atoi(start);

			/* read image normalize flag and store it             */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      switch(*start)
       {
	 case 'Y':
         case 'y':
         	(*images)[entry].normalize = ON;
         	break;
	 case 'N':
         case 'n':
         	(*images)[entry].normalize = OFF;
         	break;
         default:
      		ShowError("Normalize flag must be y or n",
      			  "ReadImageFile", NULL);        
       }

			/* read image rotation and store it                   */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      if ((atoi(start)/90) > ROTATE_MAX)
	ShowError("Rotations available: 0,90,180,270", "ReadImageFile", NULL);        
      (*images)[entry].rotate = atoi(start) / 90;
      
			/* read image compression flag and store it           */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      switch(*start)
       {
	 case 'Y':
         case 'y':
         	(*images)[entry].zfilter = ON;
         	break;
	 case 'N':
         case 'n':
         	(*images)[entry].zfilter = OFF;
         	break;
         default:
      		ShowError("Uncompress flag must be y or n",
      			  "ReadImageFile", NULL);        
       }

			/* read image border color and store it               */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      tmp = (char *)malloc(strlen(start) + 1);
      if(tmp == NULL)
        ShowError(ErrorMessage(E_MEMALLOC), "ReadImageFile", "border");
      strcpy(tmp, start);
      (*images)[entry].border = tmp;

			/* read image path and store it                       */
      start = stop + 1;
      while(isspace(*start)) start++;
      stop = start;
      while(!isspace(*stop)) stop++;
      *stop = '\0';
      tmp = (char *)malloc(strlen(start) + 1);
      if(tmp == NULL)
        ShowError(ErrorMessage(E_MEMALLOC), "ReadImageFile", "image");
      strcpy(tmp, start);
      (*images)[entry].image = tmp;

			/* increment image entry number                       */
      entry++;
     }
    }

			/* close image archive file and return entry count    */
   fclose(rc);
   return entrycnt;
 }

/*******************************************************************************
* Routine to check for the existence of a file
*******************************************************************************/  
/*int FileExists(char *name)
 {
   int retval;
   FILE *tester = fopen(name, "r");

   if (tester == NULL)
     retval = 0;
   else
     retval = 1;

   fclose(tester);

   return retval;   
 }*/
 
/*******************************************************************************
* Driver routine
*******************************************************************************/  
void main(int argc, char *argv[], char *env[])
 {
   FILE *log;				/* logfile for storing current image  */
   int result;				/* return value for system calls      */
   char *tempfile = TEMPFILE;		/* filename for uncompressing images  */
   char temp[CBUFFER_SZ];		/* buffer for system                  */
   char rcfile[CBUFFER_SZ];		/* path to rc file                    */
   char collfile[CBUFFER_SZ];		/* path to collections listing	      */
   int xdim,				/* Width of image                     */
       ydim;				/* Height of image                    */
   float maxscale;			/* Maximum scaling factor allowed     */
   rcopts genopts =			/* General runtime options            */
   	{ DEF_X,
   	  DEF_Y,
   	  DEF_UPDATE,
   	  DEF_DELAY,
   	  DEF_NOISE,
   	  DEF_MAXZOOM,
   	  DEF_EXE,
   	  DEF_ZFIL };
   int imagecnt = 0, points;		/* Count of images and used points    */
   int collcnt = 0, collpts;		/* Count of collections and points    */
   int count, colselect,
       choice, select;			/* Image selection variables          */
   collopts *collections;		/* Dynamic array of collection structs*/
   imageopts *images;			/* Dynamic array of image structures  */
   imageopts showopts =			/* Structure for displaying image     */
   	{ DEF_WEIGHT,
   	  DEF_BRIGHT,
   	  DEF_CNUM,
   	  DEF_COLOR,
   	  DEF_DITHER,
   	  DEF_SMOOTH,
   	  DEF_ZOOM,
   	  DEF_NORM,
   	  DEF_ROTATE,
   	  DEF_UNCOMP,
   	  DEF_BORDER,
   	  DEF_IMAGE };
   int userid;				/* UID of the invoking user           */
   struct passwd *passid;		/* Structure for user passwd info     */

   printf("xbackdrop v%s\n", VERSION);

			/* setup paths to default rc and image files          */
   strcpy(rcfile, DEF_BASEDIR);   
   strcat(rcfile, RCFILE);
   strcpy(collfile, DEF_BASEDIR);   
   strcat(collfile, COLFILE);
   
   			/* Get information pertaining to current user         */
   userid = getuid();
   passid = getpwuid(userid);

#ifdef VERBOSE
   fprintf(stdout, "Checking for files in the user's home directory...\n");
#endif
   
   sprintf(temp, "%s%s%s", passid->pw_dir, DEF_USRBASE, RCFILE);
   if(FileExists(temp))
    {
#ifdef VERBOSE
     fprintf(stdout, "Found general rc file.\n");
#endif
     strcpy(rcfile, temp);
    }

   sprintf(temp, "%s%s%s", passid->pw_dir, DEF_USRBASE, COLFILE);
   if(FileExists(temp))
    {
#ifdef VERBOSE
     fprintf(stdout, "Found collection file.\n");
#endif
     strcpy(collfile, temp);
    }
      
			/* Read general options from runtime commands file    */
   ReadRCFile(rcfile, &genopts);

#ifdef VERBOSE
   fprintf(stdout, "RC files options read:\n");
   fprintf(stdout, "\tResolution: %dx%d\n", genopts.xres, genopts.yres);
   fprintf(stdout, "\tUpdate Mode: %d\n", genopts.update);
   fprintf(stdout, "\tUpdate Delay: %d\n", genopts.delay);
   fprintf(stdout, "\tNoise Mode: %d\n", genopts.noise);
   fprintf(stdout, "\tAuto-Maximize Mode: %d\n", genopts.maxzoom);
   fprintf(stdout, "\tPath to XLI or XLOADIMAGE: [%s]\n", genopts.base);
   fprintf(stdout, "\tFilter Utility Path: [%s]\n", genopts.zfil);
#endif

			/* Read image entries from archive file               */
   collcnt = ReadCollectionFile(collfile, &collections, &collpts);

#ifdef VERBOSE
   fprintf(stdout, "Collection Entries: %d\nCollection Points: %d\n",
   		collcnt, collpts);
#endif

			/* Read image entries from archive file               */
   for (count = 0; count < collcnt; count++)
    {
     collections[count].imagenum =
         ReadImageFile(collections[count].imagefile,
                       &(collections[count].images),
                       &(collections[count].points));
#ifdef VERBOSE
     fprintf(stdout, "Imagefile: %s\n", collections[count].imagefile);
     fprintf(stdout, "\tEntries: %u\n", collections[count].imagenum);
     fprintf(stdout, "\tPoints: %u\n", collections[count].points);
#endif
    }                       

                 	/* seed random generator with inverted system time    */                       
   srandom(INVERT(times()));

   do
    {
                   	/* gives random number between 1 and collpts          */
     choice = (long)(RAND_BINARY * collpts + 1);

#ifdef VERBOSE
     fprintf(stdout, "Collection point %d selected.\n", choice);
#endif

			/* find collection selected by random number          */   
     count = 0;
     while(choice > 0)
      {
       if(choice <= collections[count].weight)
         colselect = count;
       choice -= collections[count++].weight;
      }

#ifdef VERBOSE
     fprintf(stdout, "Collection %d selected.\n", colselect + 1);
#endif

                   	/* gives random number between 0 and choice - 1       */
     choice = (long)(RAND_BINARY * collections[colselect].imagenum + 1);

#ifdef VERBOSE
     fprintf(stdout, "Image point %d selected.\n", choice);
#endif

			/* find image selected by random number               */   
     count = 0;
     while(choice > 0)
      {
       if(choice <= collections[colselect].images[count].weight)
         select = count;
       choice -= collections[colselect].images[count++].weight;
      }

#ifdef VERBOSE
     fprintf(stdout, "Image %d selected.\n", select + 1);
#endif

			/* log selection made                                 */
     if(!(log = fopen(LOGFILE, "w")))
       ShowError(ErrorMessage(E_FILEOPEN), "main", LOGFILE);
     fprintf(log, "Collection: %s\nImage: %s\n",
     		collections[colselect].imagefile,
     		collections[colselect].images[select].image);
     fclose(log);
			
			/* assign image entry values to display structure     */
     showopts = collections[colselect].images[select];

			/* Check for need to uncompress image before loading  */
     if(showopts.zfilter == ON)
      {
        sprintf(temp, "%s %s > %s", genopts.zfil, showopts.image, tempfile);
        if((result = system(temp)) == -1)
          ShowError("Unable to find executable", "main", genopts.zfil);
        else if (result == 127)
          ShowError("Unable to execute shell command", "main", NULL);    
        showopts.image = tempfile;
      }
            
			/* find the image dimensions                          */
     if (GetDimensions(showopts.image, &xdim, &ydim, genopts) == FAILURE)
       ShowError("Dimension Retrieval Failed", NULL, showopts.image);

			/* calculate maximum allowed scaling and store in     */
			/* show structure if requested                        */
     maxscale = MaxScaleFactor(genopts.xres, genopts.yres, xdim, ydim,
   		             showopts.rotate & 0x01, genopts.maxzoom);
     if(showopts.zoom == 0)
       showopts.zoom = (int)(maxscale * 100);

#ifdef VERBOSE 
     printf("Image [%s] has dimensions %dx%d using scale factor %d%%\n",
   		  showopts.image, xdim, ydim, showopts.zoom);     
#endif
 
 			/* call utility to display image                      */
     LoadImage(showopts, genopts);

#ifdef VERBOSE
     printf("Finished\n");
#endif

			/* Remove temporary file used for uncompression       */
     if(showopts.zfilter == ON)
        remove(tempfile);

			/* Sleep if running in daemon mode                    */
     if (genopts.update == UPDATE_MINS)
       if(sleep(genopts.delay * 60L) != 0)
        {
          fprintf(stderr, "Signal recieved during sleep period...exiting\n");
          exit(FAILURE);
        }
     else if (genopts.update == UPDATE_HRS)
       if(sleep(genopts.delay * 3600L) != 0)
        {
          fprintf(stderr, "Signal recieved during sleep period...exiting\n");
          exit(FAILURE);
        }

    }			/* continue selection unless in daemon mode           */
   while(genopts.update != UPDATE_NONE);

#ifdef VERBOSE
   printf("Exiting\n");
#endif
   exit(SUCCESS);
 }
 
