/*
 *  This file is part of MagiCapture.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * save.c: Image saving routines
 */

#include "config.h"

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <stdio.h>
#include <magick/api.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gdk_magick.h>

#include "common.h"
#include "util.h"
#include "stringtable.h"
#include "widgproc.h"
#include "save.h"

/* Global variables */

/* Default image file types */

static gchar *default_filetypes [] =
{
    "MIFF",
    "BMP",
    "GIF",
    "JPG",
    "PNG",
    "PPM",
    "XPM",
    NULL
};

/* File name strings */
static GString *save_filename = NULL;

/* Saved image data */
static Image *save_image = NULL;
static ImageInfo *save_image_info = NULL;
static MagickInfo *magick_info = NULL;

/* Dialog boxes and widgets */
static GtkWidget *save_dialog = NULL;
static GtkWidget *save_filetype_dialog = NULL;
static GtkWidget *label_filetype_selected = NULL;

/* File type data */
static gchar *save_clist_header [] = ARRAY_CLIST_HEADER;
static gchar *filetype_from_clist = NULL;
static gchar *filetype_selected = NULL;

/*********************************************************************
 *
 * Save window callback functions
 */

gint save_dialog_delete_event
(
    GtkWidget *widget,
    GdkEvent *event,
    gpointer data
)
{
    gtk_widget_hide (widget);
    return (TRUE);
}

void save_dialog_destroy_event (GtkWidget *widget, gpointer data)
{
    return;
}

gint save_filetype_delete_event
(
    GtkWidget *widget,
    GdkEvent *event,
    gpointer data
)
{
    gtk_widget_hide (widget);
    return (TRUE);
}

void save_filetype_destroy_event (GtkWidget *widget, gpointer data)
{
    return;
}

/*********************************************************************
 *
 * save_init: Initialize all data structures and strings related to
 * saving image data.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void save_init ()
{
    save_filename = g_string_new (NULL);
    save_image = NULL;
    save_image_info = NULL;
    magick_info = GetMagickInfo (NULL);

    save_create_dialog ();
    save_create_filetype_dialog ();
}

/*********************************************************************
 *
 * save_cleanup: Free all memory allocated to data structures relevant
 * to saving captured images. This function must be called after the
 * GTK main event loop.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void save_cleanup ()
{
    gtk_widget_destroy (save_dialog);
    gtk_widget_destroy (save_filetype_dialog);
}

/*********************************************************************
 *
 * save_create_dialog: Create the file selection control dialog for
 * saving images and initialize the file type selection menu within
 * the dialog.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void save_create_dialog ()
{
    MagickInfo *magick_index;
    GtkWidget *hbox;
    GtkWidget *filetype_menu;
    GtkWidget *filetype_option_menu;
    gchar *filetype;
    GString *filetype_desc;
    GString *filetype_selected;
    int i;

    /* Create a GTK save dialog box. */
    
    save_dialog = gtk_file_selection_new (TITLE_SAVE_WINDOW);
    gtk_window_set_modal (GTK_WINDOW (save_dialog), TRUE);
    
    gtk_signal_connect
    (
        GTK_OBJECT (save_dialog),
        "delete_event",
        GTK_SIGNAL_FUNC (save_dialog_delete_event),
        NULL
    );
    gtk_signal_connect
    (
        GTK_OBJECT (save_dialog),
        "destroy",
        GTK_SIGNAL_FUNC (save_dialog_destroy_event),
        NULL
    );

    /* Connect the OK and Cancel buttons to the appropriate operations. */

    gtk_signal_connect_object
    (
        GTK_OBJECT (GTK_FILE_SELECTION (save_dialog) -> ok_button),
        "clicked",
        (GtkSignalFunc) save_dialog_delete_event,
        GTK_OBJECT (save_dialog)
    );
    gtk_signal_connect
    (
        GTK_OBJECT (GTK_FILE_SELECTION (save_dialog) -> ok_button),
        "clicked",
        (GtkSignalFunc) save_ok,
        save_dialog
    );
    gtk_signal_connect_object
    (
        GTK_OBJECT (GTK_FILE_SELECTION (save_dialog) -> cancel_button),
        "clicked",
        (GtkSignalFunc) save_dialog_delete_event,
        GTK_OBJECT (save_dialog)
    );

    /* Generate a menu of default file types. */
        
    filetype_menu = gtk_menu_new ();

    add_menu_item
    (
        filetype_menu,
        GTK_SIGNAL_FUNC (save_filetype_menu_select),
        LABEL_FILETYPE_AUTOMATIC,
        (gpointer) "Automatic"
    );
    
    filetype_desc = g_string_new (NULL);
    for (i = 0; default_filetypes [i] != NULL; i++)
    {
        filetype = default_filetypes [i];
        for (magick_index = magick_info;
             magick_index != NULL;
             magick_index = magick_index -> next)
        {
            if (!strcmp (filetype, magick_index -> tag))
            {
                g_string_sprintf
                (
                    filetype_desc,
                    "%s (%s)",
                    magick_index -> tag,
                    magick_index -> description
                );
                
                add_menu_item
                (
                    filetype_menu,
                    GTK_SIGNAL_FUNC (save_filetype_menu_select),
                    filetype_desc -> str,
                    (gpointer) (magick_index -> tag)
                );
            }
        }        
    }
    g_string_free (filetype_desc, TRUE);

    add_menu_item
    (
        filetype_menu,
        GTK_SIGNAL_FUNC (save_filetype_menu_select),
        LABEL_FILETYPE_OTHER,
        (gpointer) "Other"
    );

    filetype_option_menu = gtk_option_menu_new ();
    gtk_option_menu_set_menu
    (
        GTK_OPTION_MENU (filetype_option_menu),
        filetype_menu
    );

    /* Add the file type menu to the save dialog box. */

    hbox = gtk_hbox_new (FALSE, 5);
    add_label (hbox, LABEL_CHOOSE_MENU);
    
    gtk_box_pack_start
    (
        GTK_BOX (hbox),
        filetype_option_menu,
        TRUE,
        TRUE,
        0
    );
    gtk_widget_show (filetype_option_menu);
    gtk_box_pack_start
    (
        GTK_BOX (GTK_FILE_SELECTION (save_dialog) -> main_vbox),
        hbox,
        TRUE,
        TRUE,
        0
    );
    gtk_widget_show (hbox);

    /*
     * Add a text label to the save dialog box which indicates the
     * image file type chosen.
     */
    
    filetype_selected = g_string_new (NULL);
    g_string_sprintf
    (
        filetype_selected,
        FMT_FILETYPE_SELECTED,
        "Automatic"
    );
    label_filetype_selected = gtk_label_new (filetype_selected -> str);
    gtk_misc_set_alignment (GTK_MISC (label_filetype_selected), 0, 0);
    gtk_box_pack_start
    (
        GTK_BOX (GTK_FILE_SELECTION (save_dialog) -> main_vbox),
        label_filetype_selected,
        FALSE,
        FALSE,
        5
    );
    gtk_widget_show (label_filetype_selected);
    g_string_free (filetype_selected, TRUE);
}

/*********************************************************************
 *
 * save_create_filetype_dialog: Create a dialog box which will allow
 * a user to choose from all the file types supported by the user's
 * ImageMagick installation.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void save_create_filetype_dialog ()
{
    GtkWidget *scrolled_window;
    MagickInfo *magick_index;
    GtkWidget *save_filetype_clist;
    gchar *clist_row [2];
    const gchar *clist_auto [] =
    {
        "Automatic",
        LABEL_FILETYPE_AUTOMATIC
    };
    
    /*
     * Create a dialog box which will list all the Imagemagick-supported
     * file types.
     */
    
    save_filetype_dialog = gtk_dialog_new ();
    gtk_window_set_modal (GTK_WINDOW (save_filetype_dialog), TRUE);
    
    gtk_window_set_title (GTK_WINDOW (save_filetype_dialog), TITLE_FILE_TYPE);
    gtk_container_set_border_width (GTK_CONTAINER (save_filetype_dialog), 5);

    add_button
    (
        GTK_DIALOG (save_filetype_dialog) -> action_area,
        GTK_OBJECT (save_filetype_dialog),
        GTK_SIGNAL_FUNC (save_filetype_delete_event),
        LABEL_OK
    );
    
    /*
     * Generate a clist containing all the ImageMagick-supported
     * file types.
     */

    save_filetype_clist = gtk_clist_new_with_titles (2, save_clist_header);
    
    gtk_clist_set_selection_mode
    (
        GTK_CLIST (save_filetype_clist),
        GTK_SELECTION_SINGLE
    );
    gtk_clist_set_shadow_type
    (
        GTK_CLIST (save_filetype_clist),
        GTK_SHADOW_NONE
    );
    gtk_clist_column_titles_show (GTK_CLIST (save_filetype_clist));
    gtk_clist_set_column_justification
    (
        GTK_CLIST (save_filetype_clist),
        0,
        GTK_JUSTIFY_LEFT
    );
    gtk_clist_set_column_justification
    (
        GTK_CLIST (save_filetype_clist),
        1,
        GTK_JUSTIFY_LEFT
    );

    gtk_clist_append (GTK_CLIST (save_filetype_clist), (gchar **) clist_auto);
    for (magick_index = magick_info;
         magick_index != NULL;
         magick_index = magick_index -> next)
    {
        clist_row [0] = (gchar *) magick_index -> tag;
        clist_row [1] = (gchar *) magick_index -> description;

        gtk_clist_append (GTK_CLIST (save_filetype_clist), clist_row);
    }    

    gtk_clist_columns_autosize (GTK_CLIST (save_filetype_clist));

    gtk_signal_connect
    (
        GTK_OBJECT (save_filetype_clist),
        "select_row",
        GTK_SIGNAL_FUNC (save_filetype_clist_select),
        NULL
    );
    gtk_clist_select_row (GTK_CLIST (save_filetype_clist), 0, 0);
    
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy
    (
        GTK_SCROLLED_WINDOW (scrolled_window),
        GTK_POLICY_NEVER,
        GTK_POLICY_ALWAYS
    );
    gtk_scrolled_window_add_with_viewport
    (
        GTK_SCROLLED_WINDOW (scrolled_window),
        save_filetype_clist
    );
    gtk_widget_show (save_filetype_clist);
    
    add_label
    (
        GTK_DIALOG (save_filetype_dialog) -> vbox,
        LABEL_CHOOSE_FILE_TYPE
    );   
    gtk_box_pack_start
    (
        GTK_BOX (GTK_DIALOG (save_filetype_dialog) -> vbox),
        scrolled_window,
        TRUE,
        TRUE,
        0
    );
    gtk_widget_set_usize (scrolled_window, 500, 200);
    gtk_widget_show (scrolled_window);
}

/*********************************************************************
 *
 * save_filetype_menu_select: GTK callback function called when the
 * user selects a file type using the menu in the save dialog box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   filetype - file type selected
 *
 * Return value:
 *   None.
 */

void save_filetype_menu_select (GtkWidget *widget, gchar *filetype)
{
    gchar *filetype_other;
    gint row, column;
    
    if (!strcmp (filetype, "Other"))
    {
        save_filetype_set (filetype_from_clist);
        gtk_widget_show (save_filetype_dialog);
        return;
    }
    else
    {
        save_filetype_set (filetype);
    }
}

/*********************************************************************
 *
 * save_filetype_clist_select: GTK callback function called when the
 * user selects a file type from the file type selection dialog box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   row - selected row in the clist
 *   column - selected column in the clist
 *   event - callback function event
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void save_filetype_clist_select
(
    GtkWidget *widget,
    gint row,
    gint column,
    GdkEventButton *event,
    gpointer data
)
{
    gtk_clist_get_text
    (
        GTK_CLIST (widget),
        row,
        0,
        &filetype_from_clist
    );

    save_filetype_set (filetype_from_clist);
}

/*********************************************************************
 *
 * save_filetype_set: Set the file type of the image to save. The file
 * type is chosen by the user via the menu in the save dialog box or
 * the clist in the file type selection box.
 *
 * Parameters:
 *   filetype - file type selected
 *
 * Return value:
 *   None.
 */

void save_filetype_set (gchar *filetype)
{
    GString *fmt_filetype_selected;

    filetype_selected = filetype;
    
    fmt_filetype_selected = g_string_new (NULL);
    g_string_sprintf
    (
        fmt_filetype_selected,
        FMT_FILETYPE_SELECTED,
        filetype
    );
    gtk_label_set_text
    (
        GTK_LABEL (label_filetype_selected),
        fmt_filetype_selected -> str
    );
    gtk_widget_show (label_filetype_selected);
    g_string_free (fmt_filetype_selected, TRUE);
}

/*********************************************************************
 *
 * save_do: Save the captured image into a file using
 * a GTK save dialog box.
 *
 * Parameters:
 *   image - image to save
 *   image_info - image info to use for saving the image
 *   None.
 *
 * Return value:
 *   None.
 */

void save_do (Image *image, ImageInfo *image_info)
{
    /* Set the save image to the image received. */
    
    save_image = image;
    save_image_info = image_info;
    
    if (save_image == NULL)
    {
        error_dialog (ERROR_MAGICK_IMAGE_SAVE);
        return;
    }

    /* Set the default filename to what was used previously. */
    
    if (save_filename != NULL)
    {
        gtk_file_selection_set_filename
        (
            GTK_FILE_SELECTION (save_dialog),
            save_filename -> str
        );
    }
    
    /* Show the save dialog box. */

    gtk_widget_show (save_dialog);
}

/*********************************************************************
 *
 * save_ok: Once the OK button is clicked on the save dialog box,
 * save the captured image into a file.
 *
 * Parameters:
 *   widget - the GTK widget involved
 *   filesel - the save dialog box
 *
 * Return value:
 *   None.
 */

void save_ok (GtkWidget *widget, GtkFileSelection *filesel)
{
    gchar *save_filename_text;
    
#ifdef HAVE_STAT
    
    struct stat stat_buffer;
    int stat_result;
    
#endif /* HAVE_STAT */
    
    /* Retrieve the filename from the file selection box. */
    
    save_filename_text = gtk_file_selection_get_filename
    (
        GTK_FILE_SELECTION (filesel)
    );
    g_string_assign (save_filename, save_filename_text);

    /* Determine if the file exists. If so, pop up a confirmation box. */

#ifdef HAVE_STAT

    stat_result = stat (save_filename -> str, &stat_buffer);
    if (stat_result == 0)
    {
        save_overwrite_dialog ();
    }
    else
    {
        save_progress (NULL, NULL);
    }
    
#else

    save_progress (NULL, NULL);
    
#endif /* HAVE_STAT */
}

/*********************************************************************
 *
 * save_overwrite_dialog: Open a dialog box which prompts the user
 * whether or not to overwrite an existing image file.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

#ifdef HAVE_STAT
void save_overwrite_dialog ()
{
    GtkWidget *overwrite_dialog;
    GString *overwrite_message;

    /*
     * Create a small dialog box which will prompt the user to
     * determine whether an existing file should be overwritten.
     */
    
    overwrite_message = g_string_new (NULL);    
    g_string_sprintf
    (
        overwrite_message,
        FMT_OVERWRITE,
        save_filename -> str
    );

    overwrite_dialog = gtk_dialog_new ();
    gtk_window_set_modal (GTK_WINDOW (overwrite_dialog), TRUE);
    gtk_window_set_title (GTK_WINDOW (overwrite_dialog), TITLE_OVERWRITE);
    gtk_container_set_border_width (GTK_CONTAINER (overwrite_dialog), 10);

    add_label
    (
        GTK_DIALOG (overwrite_dialog) -> vbox,
        overwrite_message -> str
    );
    add_button
    (
        GTK_DIALOG (overwrite_dialog) -> action_area,
        GTK_OBJECT (overwrite_dialog),
        GTK_SIGNAL_FUNC (save_progress),
        LABEL_YES
    );
    add_button
    (
        GTK_DIALOG (overwrite_dialog) -> action_area,
        GTK_OBJECT (overwrite_dialog),
        GTK_SIGNAL_FUNC (gtk_widget_destroy),
        LABEL_NO
    );    
        
    gtk_widget_show (overwrite_dialog);
    
    g_string_free (overwrite_message, TRUE);
}
#endif /* HAVE_STAT */

/*********************************************************************
 *
 * save_progress: Open a progress window indicating that the file
 * is being saved. Proceed to save the file.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback data passed to this function
 *
 * Return value:
 *   None.
 */

void save_progress (GtkWidget *widget, gpointer data)
{
    GtkWidget *progress_window;
    GtkWidget *box;
    GString *save_message;

    gboolean result;
    
    /*
     * Create a small dialog box which will display a message,
     * indicating that the image is being saved.
     */
    
    save_message = g_string_new (NULL);    
    g_string_sprintf
    (
        save_message,
        FMT_SAVING,
        save_filename -> str,
        filetype_selected
    );

    progress_window = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_window_set_modal (GTK_WINDOW (progress_window), TRUE);
    gtk_window_set_title (GTK_WINDOW (progress_window), save_message -> str);
    gtk_container_set_border_width (GTK_CONTAINER (progress_window), 20);

    box = gtk_vbox_new (FALSE, 0);
    add_label (box, save_message -> str);
    gtk_container_add (GTK_CONTAINER (progress_window), box);
    gtk_widget_show_now (box);

    while (gtk_events_pending ())
    {
        gtk_main_iteration ();
    }
    
    gtk_widget_show_now (progress_window);

    while (gtk_events_pending ())
    {
        gtk_main_iteration ();
    }

    /* Save the image. */

    result = save_image_to_file (save_filename -> str);

    /*
     * Delete the progress box. If the function was connected to an object,
     * delete the object as well.
     */
    
    gtk_widget_destroy (progress_window);
    if (widget != NULL)
    {
        gtk_widget_destroy (widget);
    }
    g_string_free (save_message, TRUE);
}

/*********************************************************************
 *
 * save_image_to_file: Save the captured image to a file using
 * ImageMagick's save capability.
 *
 * Parameters:
 *   save_path - path or file to save the image as
 *
 * Return value:
 *   TRUE if the operation completed successfully, FALSE otherwise
 */

gboolean save_image_to_file (gchar *save_path)
{
    guint save_status;
    GString *fmt_save_path;
    
    Image *disk_image;
    ImageInfo *disk_image_info;
    
    /*
     * Create a copy of the image that is suitable for saving.
     * This needs to be done because the WriteImage call modifies
     * the original image, which we want preserved.
     */
          
    disk_image = CloneImage
    (
        save_image,
        save_image -> columns,
        save_image -> rows,
        TRUE
    );
    disk_image_info = CloneImageInfo (save_image_info);
    if (disk_image == NULL || disk_image_info == NULL)
    {
        error_dialog (ERROR_CLONE);
        if (disk_image != NULL) DestroyImage (disk_image);
        if (disk_image_info != NULL) DestroyImageInfo (disk_image_info);
        return FALSE;
    }

    /* Prepend the image type to the filename. */

    fmt_save_path = g_string_new (NULL);
    if ((strcmp (filetype_selected, "Automatic") != 0)
        && (strcmp (filetype_selected, "Other") != 0))
    {
        g_string_sprintf
        (
            fmt_save_path,
            "%s:%s",
            filetype_selected, save_path
        );
    }
    else
    {
        g_string_assign (fmt_save_path, save_path);
    }
    
    /* Save the image. */

    strcpy (disk_image -> filename, fmt_save_path -> str);
    strcpy (disk_image_info -> filename, fmt_save_path -> str);
    SetImageInfo (disk_image_info, 1);
    save_status = WriteImage (disk_image_info, disk_image);
    if (!save_status)
    {
        error_dialog (ERROR_SAVE);
        if (disk_image != NULL) DestroyImage (disk_image);
        if (disk_image_info != NULL) DestroyImageInfo (disk_image_info);
        return FALSE;
    }
    
    if (disk_image != NULL) DestroyImage (disk_image);
    if (disk_image_info != NULL) DestroyImageInfo (disk_image_info);

    return TRUE;
}
