/*
 *  This file is part of MagiCapture.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * options.c: Configure MagiCapture options
 */

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
/*#include <magick/api.h>*/

#include "common.h"
#include "stringtable.h"
#include "widgproc.h"
#include "panel.h"
#include "options.h"
#include "util.h"

/* Global variables */

static OptionData new_option_data;
static GString *option_filename = NULL;

/*********************************************************************
 *
 * options_parse: Parse an option line and extract the parameter and
 * value from the line.
 *
 * Parameters:
 *   parameter - address of the character string which will receive
 *     the option parameter
 *   value - address of the character string which will receive
 *     the option value
 *   input - input string to parse
 *
 * Return value:
 *   None.
 */

void options_parse (gchar **parameter, gchar **value, gchar *input)
{
    size_t input_len;
    gchar *t_parameter;
    gchar *t_value;
    gint i, parameter_i, value_i;
    gboolean is_value, is_word_begin, is_line_begin;

    /* Allocate memory for the parameter and value strings. */
    
    input_len = strlen (input);
    t_parameter = g_malloc (sizeof (gchar) * input_len + 1);
    t_value = g_malloc (sizeof (gchar) * input_len + 1);

    /* Parse the input line and extract the parameter and its value. */

    parameter_i = 0;
    value_i = 0;
    is_value = 0;
    is_word_begin = TRUE;
    is_line_begin = TRUE;

    for (i = 0; i < input_len; i++)
    {
        /*
         * A pound sign at the beginning of a line (or after spaces)
         * indicates a comment.
         */
        
        if ((input [i] == '#') && (is_line_begin))
        {
            parameter_i = 0;
            value_i = 0;
            break;
        }

        /* Spaces before the parameter or value are ignored. */
        
        else if ((input [i] == ' ') && (is_word_begin))
        {
            continue;
        }

        /*
         * An equals sign indicates that the string that follows
         * denotes the value.
         */
        
        else if (input [i] == '=')
        {
            is_value = 1;
            is_word_begin = TRUE;
            is_line_begin = FALSE;
        }

        /* Extract the characters that comprise the parameter's value. */
        
        else if (is_value)
        {
            is_word_begin = FALSE;
            is_line_begin = FALSE;
            t_value [value_i] = input [i];
            value_i++;
        }

        /* Extract the characters that comprise the parameter. */
        
        else
        {
            is_word_begin = FALSE;
            is_line_begin = FALSE;
            t_parameter [parameter_i] = input [i];
            parameter_i++;
        }
    }

    /* Terminate the parameter and value with null characters. */
    
    t_parameter [parameter_i] = 0;
    t_value [value_i] = 0;

    /* Remove all trailing spaces and newlines from the parameter and value. */
    
    for (i = parameter_i - 1; i >= 0; i--)
    {
        if ((t_parameter [i] == ' ') || (t_parameter [i] == '\n'))
        {
            t_parameter [i] = 0;
        }
        else if (t_parameter [i] != 0)
        {
            break;
        }
    }
    for (i = value_i - 1; i >= 0; i--)
    {
        if ((t_value [i] == ' ') || (t_value [i] == '\n'))
        {
            t_value [i] = 0;
        }
        else if (t_value [i] != 0)
        {
            break;
        }
    }

    /* Return the parameter and value to the calling function. */
    
    *parameter = t_parameter;
    *value = t_value;
}

/*********************************************************************
 *
 * options_init: Process the user's configuration file an set the
 * application options based on the configuration values.
 *
 * Parameters:
 *   envp - array of characters containing the user's environment
 *
 * Return value:
 *   None.
 */

void options_init (gchar *envp [])
{
    FILE *option_file;
    char option_line [MAX_LINE];
    char* option_status;
    gint i;
    gchar *home_parameter, *home_value;
    gchar *parameter, *value;

    /* Define the default options. */
    
    new_option_data.preview = TRUE;
    new_option_data.hide_panel = TRUE;
    new_option_data.dither = TRUE;

    /* Retrieve the value of the HOME environment variable. */

    home_parameter = NULL;
    home_value = NULL;
    for (i = 0; envp [i] != NULL; i++)
    {
        if (!strncmp (envp [i], "HOME=", 5))
        {
            options_parse (&home_parameter, &home_value, envp [i]);
        }
    }
    if (home_value == NULL)
    {
        error_dialog (ERROR_OPTIONS_HOME);
        option_filename = NULL;
        return;
    }

    /* Derive the path of the MagiCapture configuration file. */

    option_filename = g_string_new (home_value);
    if (option_filename -> str [option_filename -> len - 1] == '/')
    {
        option_filename = g_string_append (option_filename, ".MagiCapture");
    }
    else
    {
        option_filename = g_string_append (option_filename, "/.MagiCapture");
    }
    g_free (home_parameter);
    g_free (home_value);
    
    /* Open the MagiCapture configuration file. */

    option_file = fopen ((const char *) (option_filename -> str), "r");
    
    if (option_file == NULL)
    {
        panel_store_options (&new_option_data);
        return;
    }

    /* Process the configuration file and read its values. */
    
    while (TRUE)
    {
        option_status = fgets (option_line, MAX_LINE, option_file);
        if (option_status == NULL)
        {
            break;
        }
        options_parse (&parameter, &value, option_line);

        /* Option "preview" */
        
        if (!strcmp (parameter, "preview"))
        {
            switch (value [0])
            {
                case 'T':
                case 't':
                case '1':
                    new_option_data.preview = TRUE;
                    break;
                default:
                    new_option_data.preview = FALSE;
                    break;
            }
        }

        /* Option "hide_panel" */
        
        if (!strcmp (parameter, "hide_panel"))
        {
            switch (value [0])
            {
                case 'T':
                case 't':
                case '1':
                    new_option_data.hide_panel = TRUE;
                    break;
                default:
                    new_option_data.hide_panel = FALSE;
                    break;
            }
        }

        /* Option "dither" */
        
        if (!strcmp (parameter, "dither"))
        {
            switch (value [0])
            {
                case 'T':
                case 't':
                case '1':
                    new_option_data.dither = TRUE;
                    break;
                default:
                    new_option_data.dither = FALSE;
                    break;
            }
        }
        g_free (parameter);
        g_free (value);
    }
        
    fclose (option_file);
    panel_store_options (&new_option_data);
}

/*********************************************************************
 *
 * options_do: Open a dialog box to configure options.
 *
 * Parameters:
 *   option_data - pointer to existing option data
 *
 * Return value:
 *   None.
 */

void options_do (OptionData *option_data)
{
    GtkWidget *window;
    GtkWidget *label;
    GtkWidget *notebook;
    GtkWidget *box;

    /* Set the new options to the existing options. */
    
    new_option_data.preview = option_data -> preview;
    new_option_data.hide_panel = option_data -> hide_panel;
    new_option_data.dither = option_data -> dither;

    /* Create an options dialog box. */
    
    window = gtk_dialog_new ();
    gtk_window_set_modal (GTK_WINDOW (window), TRUE);
    
    gtk_window_set_title (GTK_WINDOW (window), TITLE_OPTIONS);
    gtk_container_set_border_width (GTK_CONTAINER (window), 5);

    box = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (box), 5);

    /* Create a notebook which will contain all option tabs. */
    
    notebook = gtk_notebook_new ();
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);

    /* Add all the checkboxes for the general options tab. */
    
    add_checkbox
    (
        box,
        NULL, 
        GTK_SIGNAL_FUNC (option_preview_func),
        new_option_data.preview,
        LABEL_OPTION_PREVIEW
    );
    add_checkbox
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (option_hide_panel_func),
        new_option_data.hide_panel,
        LABEL_OPTION_HIDE_PANEL
    );
    add_checkbox
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (option_dither_func),
        new_option_data.dither,
        LABEL_OPTION_DITHER
    );

    gtk_widget_set_usize (box, 0, 200);
    gtk_widget_show (box);

    /* Create the general options tab. */
    
    label = gtk_label_new (LABEL_OPTION_TAB_GENERAL);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);

    /* Add the option dialog buttons. */

    add_button
    (
        GTK_DIALOG (window) -> action_area,
        GTK_OBJECT (window),
        GTK_SIGNAL_FUNC (options_save),
        LABEL_OPTION_SAVE
    );
    add_button
    (
        GTK_DIALOG (window) -> action_area,
        GTK_OBJECT (window),
        GTK_SIGNAL_FUNC (options_apply),
        LABEL_OPTION_APPLY
    );
    add_button
    (
        GTK_DIALOG (window) -> action_area,
        GTK_OBJECT (window),
        GTK_SIGNAL_FUNC (options_cancel),
        LABEL_OPTION_CANCEL
    );

    /* Add the notebook to the options dialog. */
    
    gtk_box_pack_start
    (
        GTK_BOX (GTK_DIALOG (window) -> vbox),
        notebook,
        TRUE,
        TRUE,
        0
    );
    gtk_widget_show (notebook);

    /* Show the options dialog. */
    
    gtk_widget_show (window);
}

/*********************************************************************
 *
 * options_init: Free all data associated with configuration options.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void options_cleanup ()
{
    if (option_filename != NULL)
    {
        g_string_free (option_filename, TRUE);
    }
}

/*********************************************************************
 *
 * options_save: GTK callback function called when the user clicks
 * on the "Save Options" button in the options dialog box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void options_save (GtkWidget *widget, gpointer data)
{
    FILE *option_file;

    if (option_filename == NULL)
    {
        error_dialog (ERROR_OPTION_SAVE);
        return;
    }
    
    option_file = fopen ((const char *) (option_filename -> str), "w");
    if (option_file == NULL)
    {
        error_dialog (ERROR_OPTION_SAVE);
        return;
    }

    fprintf
    (
        option_file,

        "# General Options\n"
        "\n"
        "preview=%s\n"
        "hide_panel=%s\n"
        "dither=%s\n",

        new_option_data.preview ? "TRUE" : "FALSE",
        new_option_data.hide_panel ? "TRUE" : "FALSE",
        new_option_data.dither ? "TRUE" : "FALSE"
    );
    
    fclose (option_file);

    panel_store_options (&new_option_data);
    gtk_widget_destroy (widget);

    return;
}

/*********************************************************************
 *
 * options_apply: GTK callback function called when the user clicks
 * on the "Apply" button in the options dialog box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void options_apply (GtkWidget *widget, gpointer data)
{
    /* Return the new options back to the main panel. */
    
    panel_store_options (&new_option_data);
    gtk_widget_destroy (widget);
    return;
}

/*********************************************************************
 *
 * options_apply: GTK callback function called when the user clicks
 * on the "Cancel" button in the options dialog box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void options_cancel (GtkWidget *widget, gpointer data)
{
    gtk_widget_destroy (widget);
    return;
}

/*********************************************************************
 *
 * options_preview_func: GTK callback function called when the user
 * selects the option "Preview screen capture immediately".
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void option_preview_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        new_option_data.preview = TRUE;
    }
    else
    {
        new_option_data.preview = FALSE;
    }
    return;
}

/*********************************************************************
 *
 * options_hide_panel_func: GTK callback function called when the user
 * selects the option "Hide main panel during capture".
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void option_hide_panel_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        new_option_data.hide_panel = TRUE;
    }
    else
    {
        new_option_data.hide_panel = FALSE;
    }
    return;
}

/*********************************************************************
 *
 * options_dither_func: GTK callback function called when the user
 * selects the option "Dither the screen capture preview image".
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void option_dither_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        new_option_data.dither = TRUE;
    }
    else
    {
        new_option_data.dither = FALSE;
    }
    return;
}
