/*
xpkgtool is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// wxWidgets
#include <wx/dir.h>
#include <wx/filedlg.h>
#include <wx/image.h>
#include <wx/imaglist.h>
#include <wx/utils.h>

// extern
#include "common.h"
#include "about.h"
#include "helpdlg.h"
#include "intcheck.h"
#include "maindlg.h"
#include "pkgtool.h"
#include "pkginfo.h"
#include "pkgprop.h"
#include "workdlg.h"
#ifdef _SLACKGRADE_EMBEDDED_
#include "../util/slackgrade/src/slackgrade.h"
#endif

// bitmaps
#include "xpm/help32.xpm"
#include "xpm/iconview32.xpm"
#include "xpm/intcheck32.xpm"
#include "xpm/lamp32.xpm"
#include "xpm/listview32.xpm"
#include "xpm/package16.xpm"
#include "xpm/package32.xpm"
#include "xpm/pkgprop32.xpm"
#include "xpm/remove32.xpm"
#include "xpm/upgrade32.xpm"
#include "xpm/xpkgtool32.xpm"
#ifdef _SLACKGRADE_EMBEDDED_
#include "../util/slackgrade/src/xpm/slackgrade32.xpm"
#endif

// control id's
#define ID_FRAME			1001	// window id
#define ID_LISTCTRL			2001	// list ctrl id
#define ID_STATUSBAR			3001	// status bar id

enum TOOLBAR
{
	ID_TOOLBAR = 4000,	// toolbar id
	ID_TB_ADDNEW,		// 'add new' toolbar id
	ID_TB_REMOVE,		// 'remove' toolbar id
	ID_TB_UPGRADE,		// 'upgrade' toolbar id
	ID_TB_PROPERTIES,	// 'properties' toolbar id
	ID_TB_REPORT,		// 'view details' toolbar id
	ID_TB_ICON,		// 'view icon' toolbar id
	ID_TB_HELP,		// 'help' toolbar id
	ID_TB_ABOUT,		// 'about' toolbar id
#ifdef _SLACKGRADE_EMBEDDED_
	ID_TB_UPDATE,		// 'slackgrade' button9 id
#endif
	ID_TB_INTCHECK,		// 'integrity check' toolbar id
};

enum MENU
{
	ID_MENU_ADDNEW = 5001,	// 'add new' menu id
	ID_MENU_REMOVE,		// 'remove' menu id
	ID_MENU_UPGRADE,	// 'upgrade' menu id
	ID_MENU_PROPERTIES,	// 'properties' menu id
	ID_MENU_EXIT,		// 'exit' menu id

	ID_MENU_ICON,		// 'view icon' menu id
	ID_MENU_REPORT,		// 'view details' menu id
	ID_MENU_INTCHECK,	// 'package integrity' menu id
#ifdef _SLACKGRADE_EMBEDDED_
	ID_MENU_UPDATE,		// 'slackgrade' menu id
#endif
	ID_MENU_HELP,		// 'help' menu id
	ID_MENU_ABOUT,		// 'about' menu id
};

// misc
#define ICON_PACKAGE			0

long GetID(wxString);

BEGIN_EVENT_TABLE(MainDlg,wxFrame)
	// list menu event map
	EVT_MENU(ID_MENU_ADDNEW,MainDlg::OnMenuAddNew)
	EVT_MENU(ID_MENU_REMOVE,MainDlg::OnMenuRemove)
	EVT_MENU(ID_MENU_UPGRADE,MainDlg::OnMenuUpgrade)
	EVT_MENU(ID_MENU_EXIT,MainDlg::OnClose)
	EVT_MENU(ID_MENU_PROPERTIES,MainDlg::OnMenuProperties)
	EVT_MENU(ID_MENU_ICON,MainDlg::OnMenuStyleIcon)
	EVT_MENU(ID_MENU_REPORT,MainDlg::OnMenuStyleReport)
	EVT_MENU(ID_MENU_INTCHECK,MainDlg::OnIntegrityCheck)
#ifdef _SLACKGRADE_EMBEDDED_
	EVT_MENU(ID_MENU_UPDATE,MainDlg::OnMenuUpdate)
#endif
	EVT_MENU(ID_MENU_HELP,MainDlg::OnHelp)
	EVT_MENU(ID_MENU_ABOUT,MainDlg::OnAbout)
	
	// toolbar event map
	EVT_TOOL(ID_TB_ADDNEW,MainDlg::OnMenuAddNew)
	EVT_TOOL(ID_TB_REMOVE,MainDlg::OnMenuRemove)
	EVT_TOOL(ID_TB_UPGRADE,MainDlg::OnMenuUpgrade)
	EVT_TOOL(ID_TB_PROPERTIES,MainDlg::OnMenuProperties)
	EVT_TOOL(ID_TB_REPORT,MainDlg::OnMenuStyleReport)
	EVT_TOOL(ID_TB_ICON,MainDlg::OnMenuStyleIcon)
	EVT_TOOL(ID_TB_HELP,MainDlg::OnHelp)
	EVT_TOOL(ID_TB_ABOUT,MainDlg::OnAbout)
#ifdef _SLACKGRADE_EMBEDDED_
	EVT_TOOL(ID_TB_UPDATE,MainDlg::OnMenuUpdate)
#endif
	EVT_TOOL(ID_TB_INTCHECK,MainDlg::OnIntegrityCheck)
	// listctrl event map
	EVT_LIST_ITEM_SELECTED(ID_LISTCTRL,MainDlg::OnListSelected)
	EVT_LIST_ITEM_DESELECTED(ID_LISTCTRL,MainDlg::OnListDeselected)
	EVT_LIST_ITEM_RIGHT_CLICK(ID_LISTCTRL,MainDlg::OnListRClick)
	EVT_LIST_KEY_DOWN(ID_LISTCTRL,MainDlg::OnListKeyDown)
	EVT_LIST_ITEM_ACTIVATED(ID_LISTCTRL,MainDlg::OnMenuProperties)
	EVT_CLOSE(MainDlg::OnClose)
END_EVENT_TABLE()

// Comparision callback for list sort
int wxCALLBACK ListCompare(long item1, long item2, long sortData)
{
    if (item1 < item2) return -1;
    if (item1 > item2) return 1;
    return 0;
}

// Main window - construction
MainDlg::MainDlg()
	: wxFrame((wxFrame*)NULL,ID_FRAME,"",wxDefaultPosition,wxSize(620,510))
{
	// calls "Loading" dialog
	WorkDlg *workdlg = new WorkDlg();
	workdlg->SetTitle("Loading...");
	workdlg->SetText1("XPKGTOOL is loading...");
	workdlg->SetText2("       Please wait");
	workdlg->SetBitmap(wxImage(xpkgtool32));
	workdlg->Show(TRUE);
	::wxYield();	// force gui update
	
	wxString title = APP_NAME;
	this->SetTitle(title.Append(" ").Append(APP_VERSION));
	SetIcon(wxIcon(xpkgtool32));
	
	// main menu initialization
	m_menubar = new wxMenuBar();

	m_menufile = new wxMenu();
	m_menufile->Append(ID_MENU_ADDNEW,"&Add package...");
	m_menufile->Append(ID_MENU_REMOVE,"&Remove package...");
	m_menufile->Append(ID_MENU_UPGRADE,"&Upgrade package...");
	m_menufile->AppendSeparator();
	m_menufile->Append(ID_MENU_PROPERTIES,"&Properties");
	m_menufile->AppendSeparator();
	m_menufile->Append(ID_MENU_EXIT,"E&xit");
	m_menubar->Append(m_menufile,wxString("&File"));
	
	m_menuview = new wxMenu();
	m_menuview->Append(ID_MENU_ICON,"&Icon","",wxITEM_RADIO);
	m_menuview->Append(ID_MENU_REPORT,"&Details","",wxITEM_RADIO);
	m_menuview->Check(ID_MENU_REPORT,TRUE);
	m_menubar->Append(m_menuview,wxString("&View"));

	m_menutool = new wxMenu();
	m_menutool->Append(ID_MENU_INTCHECK,"&Package Integrity");
#ifdef _SLACKGRADE_EMBEDDED_
	m_menutool->Append(ID_MENU_UPDATE,"Slack&Grade");
#endif
	m_menubar->Append(m_menutool,wxString("&Tools"));

	m_menuhelp = new wxMenu();
	m_menuhelp->Append(ID_MENU_HELP,"&Contents");
	m_menuhelp->AppendSeparator();
	m_menuhelp->Append(ID_MENU_ABOUT,"&About");
	m_menubar->Append(m_menuhelp,wxString("&Help"));

	SetMenuBar(m_menubar);
	
	// controls initialization
	m_list = new wxListCtrl(this,ID_LISTCTRL);
	// list menu setup (when user right-clicks the list)
	m_menulist = new wxMenu();
	m_menulist->Append(ID_MENU_ADDNEW,"&Add package...");
	m_menulist->Append(ID_MENU_REMOVE,"&Remove selected");
	m_menulist->Append(ID_MENU_UPGRADE,"&Upgrade package...");
	m_menulist->AppendSeparator();
	m_menulist->Append(ID_MENU_PROPERTIES,"&Properties");
	m_menulist->AppendSeparator();
	m_menulist->Append(ID_MENU_ICON,"&Icon View","",wxITEM_RADIO);
	m_menulist->Append(ID_MENU_REPORT,"&Details View","",wxITEM_RADIO);
#ifdef _SLACKGRADE_EMBEDDED_
	m_menulist->AppendSeparator();
	m_menulist->Append(ID_MENU_INTCHECK,"&Package Integrity");
	m_menulist->Append(ID_MENU_UPDATE,"Slack&Grade");
#endif
	m_menulist->Check(ID_MENU_REPORT,TRUE);
	// toolbar setup
	m_toolbar = new wxToolBar(this,ID_TOOLBAR,wxDefaultPosition,wxSize(200,34),wxTB_HORIZONTAL | wxTB_3DBUTTONS | wxSUNKEN_BORDER);
	SetToolBar(m_toolbar);
	m_toolbar->SetToolBitmapSize(wxSize(32,32));
	m_toolbar->AddTool(ID_TB_ADDNEW,"Add new package(s)...",wxBitmap(wxImage(package32)),"Add new package(s)");
	m_toolbar->AddTool(ID_TB_REMOVE,"Remove selected package(s)",wxBitmap(wxImage(remove32)),"Remove selected package(s)");
	m_toolbar->AddTool(ID_TB_UPGRADE,"Upgrade package(s)...",wxBitmap(wxImage(upgrade32)),"Upgrade package(s)");
	m_toolbar->AddSeparator();
	m_toolbar->AddTool(ID_TB_PROPERTIES,"Properties",wxBitmap(wxImage(pkgprop32)),"Package Properties");
	m_toolbar->AddSeparator();
	m_toolbar->AddRadioTool(ID_TB_REPORT,"List",wxBitmap(wxImage(listview32)),wxNullBitmap,"View as List");
	m_toolbar->AddRadioTool(ID_TB_ICON,"Icon",wxBitmap(wxImage(iconview32)),wxNullBitmap,"View as Icon");
#ifdef _SLACKGRADE_EMBEDDED_
	m_toolbar->AddSeparator();
	m_toolbar->AddTool(ID_TB_UPDATE,"SlackGrade - Slackware Update Wizard",wxBitmap(wxImage(slackgrade32)),"SlackGrade - Slackware Update Wizard");
#endif
	m_toolbar->AddTool(ID_TB_INTCHECK,"Package Integrity Check",wxBitmap(wxImage(intcheck32)),"Package Integrity Check");
	m_toolbar->AddSeparator();
	m_toolbar->AddTool(ID_TB_HELP,"Help",wxBitmap(wxImage(help32)),"Help");
	m_toolbar->AddTool(ID_TB_ABOUT,"About",wxBitmap(wxImage(lamp32)),"About");
	m_toolbar->Realize();
	
	// if not runnig as superuser, disable all package maintenance commands
	if(wxGetUserId()!="root")
	{
		m_toolbar->EnableTool(ID_TB_ADDNEW,FALSE);
		m_toolbar->EnableTool(ID_TB_REMOVE,FALSE);
		m_toolbar->EnableTool(ID_TB_UPGRADE,FALSE);
		m_menulist->Enable(ID_MENU_ADDNEW,FALSE);
		m_menulist->Enable(ID_MENU_REMOVE,FALSE);
		m_menulist->Enable(ID_MENU_UPGRADE,FALSE);
		m_menufile->Enable(ID_MENU_ADDNEW,FALSE);
		m_menufile->Enable(ID_MENU_REMOVE,FALSE);
		m_menufile->Enable(ID_MENU_UPGRADE,FALSE);
	}
	// statusbar setup
	m_status = new wxStatusBar(this,ID_STATUSBAR);
	SetStatusBar(m_status);
	// listctrl setup && scan installed packages
	RefreshList(wxLC_REPORT);
	delete workdlg;
}

// ::OnAbout - shows the about box
void MainDlg::OnIntegrityCheck()
{
	IntCheckDlg *m_intck = new IntCheckDlg();
	m_intck->Show(TRUE);
	::wxYield();
	m_intck->IntegrityCheck(m_list->GetItemText(SelItem));
	return;
}

// ::OnAbout - shows the about box
void MainDlg::OnAbout()
{
	AboutDlg *m_about = new AboutDlg();
	m_about->Show(TRUE);
}

// ::OnHelp - shows the help dialog
void MainDlg::OnHelp()
{
	HelpDlg *m_help = new HelpDlg();
	m_help->Show(TRUE);
}

// ::OnListRClick - the list ctrl was right clicked
void MainDlg::OnListRClick(wxListEvent& event)
{
	wxPoint menupos = wxGetMousePosition();
	if(ListStyle==wxLC_REPORT)
	{
		wxRect rect;
		if(m_list->GetItemRect(0,(wxRect&)rect))
		{
			menupos.y -= rect.GetHeight();
		}
	}
	m_list->PopupMenu(m_menulist,m_list->ScreenToClient(menupos));
}

// ::OnListKeyDown - a key was pressed over a list ctrl item
void MainDlg::OnListKeyDown(wxListEvent& event)
{
	if(event.GetKeyCode()==WXK_DELETE)
		OnMenuRemove();
}

// ::OnMenuAddNew - user clicked 'Add package' menu
void MainDlg::OnMenuAddNew()
{
	wxFileDialog *m_filedlg = new wxFileDialog(this,"Add Package(s)...","","",
					"Slackware Packages (*.tgz)|*.tgz",wxOPEN | wxMULTIPLE);
	if(m_filedlg->ShowModal()==wxID_OK)
	{
		int exitcode;
		wxString err;
		wxArrayString filenames,paths;
		m_filedlg->GetFilenames((wxArrayString&)filenames);
		m_filedlg->GetPaths((wxArrayString&)paths);
		WorkDlg *workdlg = new WorkDlg();
		workdlg->SetBitmap(wxImage(package32));
		workdlg->SetText1("Installing package...");
		workdlg->Show();
		PkgTool *pkgtool = new PkgTool();
		for(unsigned int i = 0;i<paths.GetCount();i++)
		{
			if(!::wxFileExists(LOCAL_PACKAGE_PATH + filenames.Item(i).BeforeLast('.')))
			{
				workdlg->SetText2(filenames.Item(i).BeforeLast('.'));
				pkgtool->Package = paths.Item(i);
				exitcode = pkgtool->InstallPkg();
				if(exitcode>PKGTOOL_SUCCESS)
				{
					err = pkgtool->Package;
					switch(exitcode)
					{
						case PKGTOOL_STATUS1:	// tar archive is corrupt
							err.Append("\ntar archive is corrupt!"); 
							break;
						case PKGTOOL_STATUS2:	// package is corrupt
							err.Append("\npackage is corrupt!"); 
							break;
						case PKGTOOL_STATUS3:	// package does not end in .tgz
							err.Append("\npackage does not end in .tgz!"); 
							break;
						case PKGTOOL_STATUS4:	// package is not a regular file
							err.Append("\npackage is not a regular file!"); 
							break;
						default:		// for some future status code
							err.Append("\nunknown error installing package!"); 
							break;
					}
					wxMessageBox(err,APP_NAME,wxICON_ERROR);
				}
			}
			else
			{
				wxString add_msg;
				add_msg = filenames.Item(i).BeforeLast('.');
				add_msg.Append("\nthis package is already installed!");
				wxMessageBox(add_msg,APP_NAME,wxOK | wxICON_INFORMATION);
			}
		}
		delete workdlg;
		RefreshList(ListStyle);
	}
}

// ::OnMenuRemove - user clicked 'Remove' menu
void MainDlg::OnMenuRemove()
{
	wxString remove_msg;
	if(m_list->GetSelectedItemCount()<1)
	{
		wxMessageBox("No package(s) selected to remove.",APP_NAME,wxICON_ERROR);
		return;
	}
	else if(m_list->GetSelectedItemCount()>1)
	{
		remove_msg.Append("Remove selected packages?\n");
	}
	else if(m_list->GetSelectedItemCount()==1)
	{
		remove_msg.Append("Remove the following package?\n");
		remove_msg.Append(m_list->GetItemText(SelItem));
	}
	if(wxMessageBox(remove_msg,APP_NAME,wxYES_NO | wxICON_QUESTION)==wxYES)
	{
		int exitcode;
		wxString err;
		WorkDlg *workdlg = new WorkDlg();
		workdlg->SetText1("Removing package...");
		workdlg->SetBitmap(wxImage(remove32));
		workdlg->Show();
		PkgTool *pkgtool = new PkgTool();
		int item = -1;
		// remove each selected package
		for(;;)
		{
			item = m_list->GetNextItem(item,wxLIST_NEXT_ALL,wxLIST_STATE_SELECTED);
			if(item==-1)
				break;
			workdlg->SetText2(m_list->GetItemText(item));
			pkgtool->Package = m_list->GetItemText(item);
			exitcode = pkgtool->RemovePkg();
			// removepkg returns only the exit status code 0 - no error handling possible.
			// anyway, the code below handle some future exit code
			if(exitcode>PKGTOOL_SUCCESS)
			{
				err = pkgtool->Package;
				err.Append("\nunknown error removing package!"); 
				wxMessageBox(err,APP_NAME,wxICON_ERROR);
			}
		}
		delete workdlg;
		RefreshList(ListStyle);
	}
}

// ::OnMenuUpgrade - user clicked 'Upgrade package' menu
void MainDlg::OnMenuUpgrade()
{
	wxFileDialog *m_filedlg = new wxFileDialog(this,"Upgrade Package...","","",
					"Slackware Packages (*.tgz)|*.tgz",wxOPEN | wxMULTIPLE);
	if(m_filedlg->ShowModal()==wxID_OK)
	{
		int exitcode;
		wxString err;
		wxArrayString filenames,paths;
		m_filedlg->GetFilenames((wxArrayString&)filenames);
		m_filedlg->GetPaths((wxArrayString&)paths);
		WorkDlg *workdlg = new WorkDlg();
		workdlg->SetText1("Upgrading package...");
		workdlg->SetBitmap(wxImage(upgrade32));
		workdlg->Show();
		PkgTool *pkgtool = new PkgTool();
		for(unsigned int i = 0;i<paths.GetCount();i++)
		{
			if(!::wxFileExists(LOCAL_PACKAGE_PATH + filenames.Item(i).BeforeLast('.')))
			{
				workdlg->SetText2(filenames.Item(i).BeforeLast('.'));
				pkgtool->Package = paths.Item(i);
				exitcode = pkgtool->UpgradePkg();
				// upgradepkg is ambiguous handling errors. only exit status 1 appears
				// for more than one error. this way, error cannot be specified for each case
				if(exitcode>PKGTOOL_SUCCESS)
				{
					err = pkgtool->Package;
					switch(exitcode)
					{
						case PKGTOOL_STATUS1:	// general
							err.Append("\nerror upgrading package!");
							break;
						default:		// for some future status code
							err.Append("\nunknown error upgrading package!");
							break;
					}
					wxMessageBox(err,APP_NAME,wxICON_ERROR);
				}
			}
			else
			{
				wxString add_msg;
				add_msg = filenames.Item(i).BeforeLast('.');
				add_msg.Append("\nthis package is already installed!");
				wxMessageBox(add_msg,APP_NAME,wxOK | wxICON_INFORMATION);
			}
		}
		delete workdlg;
		RefreshList(ListStyle);
	}
}

// ::OnMenuProperties - user clicked 'Properties' menu
void MainDlg::OnMenuProperties()
{
	if(m_list->GetSelectedItemCount()<1)
	{
		wxMessageBox("No package selected.",APP_NAME,wxICON_ERROR);
		return;
	}
	PropDlg *m_props = new PropDlg(m_list->GetItemText(SelItem));
	m_props->Show(TRUE);
}

// ::OnMenuStyleIcon - user clicked 'View Icon' menu/toolbar
void MainDlg::OnMenuStyleIcon()
{
	// toogle view mode toolbar buttons
	m_toolbar->ToggleTool(ID_TB_REPORT,FALSE);
	m_toolbar->ToggleTool(ID_TB_ICON,TRUE);
	// toogle view mode menus
	m_menuview->Check(ID_MENU_ICON,TRUE);
	m_menulist->Check(ID_MENU_ICON,TRUE);
	RefreshList(wxLC_ICON);
}

// ::OnMenuStyleReport - user clicked 'View Details' menu/toolbar
void MainDlg::OnMenuStyleReport()
{
	// toogle view mode toolbar buttons
	m_toolbar->ToggleTool(ID_TB_REPORT,TRUE);
	m_toolbar->ToggleTool(ID_TB_ICON,FALSE);
	// toogle view mode menus
	m_menuview->Check(ID_MENU_REPORT,TRUE);
	m_menulist->Check(ID_MENU_REPORT,TRUE);
	RefreshList(wxLC_REPORT);
}

#ifdef _SLACKGRADE_EMBEDDED_
// ::OnMenuProperties - user clicked 'Properties' menu
void MainDlg::OnMenuUpdate()
{
	SlackGradeDlg *m_update = new SlackGradeDlg();
	m_update->ShowModal();
	delete m_update;
	RefreshList(ListStyle);
}
#endif

// ::OnListSelected - a list ctrl item was selected
void MainDlg::OnListSelected(wxListEvent& event)
{
	// just save current selected item position
	if(SelItem==-1)
	{
		m_toolbar->EnableTool(ID_TB_REMOVE,TRUE);
		m_toolbar->EnableTool(ID_TB_PROPERTIES,TRUE);
		m_toolbar->EnableTool(ID_TB_INTCHECK,TRUE);
		m_menufile->Enable(ID_MENU_REMOVE,TRUE);
		m_menufile->Enable(ID_MENU_PROPERTIES,TRUE);
		m_menutool->Enable(ID_MENU_INTCHECK,TRUE);
		m_menulist->Enable(ID_MENU_REMOVE,TRUE);
		m_menulist->Enable(ID_MENU_PROPERTIES,TRUE);
		m_menulist->Enable(ID_MENU_INTCHECK,TRUE);
	}
	SelItem = event.GetIndex();
}

// ::OnListDeselected - a list ctrl item was deselected
void MainDlg::OnListDeselected()
{
	if(m_list->GetSelectedItemCount()==0)
	{
		m_toolbar->EnableTool(ID_TB_REMOVE,FALSE);
		m_toolbar->EnableTool(ID_TB_PROPERTIES,FALSE);
		m_toolbar->EnableTool(ID_TB_INTCHECK,FALSE);
		m_menufile->Enable(ID_MENU_REMOVE,FALSE);
		m_menufile->Enable(ID_MENU_PROPERTIES,FALSE);
		m_menutool->Enable(ID_MENU_INTCHECK,FALSE);
		m_menulist->Enable(ID_MENU_REMOVE,FALSE);
		m_menulist->Enable(ID_MENU_PROPERTIES,FALSE);
		m_menulist->Enable(ID_MENU_INTCHECK,FALSE);
		SelItem = -1;
	}
}

// ::RefreshList - rescan installed packages and redraw list ctrl
void MainDlg::RefreshList(long style)
{
	long counter=0;		// list index
	wxString filename;
	wxString status;
	ListStyle = style;
	m_list->ClearAll();
	m_list->SetWindowStyleFlag(style);
	if(style==wxLC_REPORT){
		m_imglist = new wxImageList(16,16,TRUE);
		m_imglist->Add(wxIcon(package16));
		m_list->SetImageList(m_imglist,wxIMAGE_LIST_SMALL);
		m_list->InsertColumn(0,"Package",wxLIST_FORMAT_LEFT,220);
		m_list->InsertColumn(1,"Description",wxLIST_FORMAT_LEFT,240);
		m_list->InsertColumn(2,"C Size",wxLIST_FORMAT_LEFT,70);
		m_list->InsertColumn(3,"U Size",wxLIST_FORMAT_LEFT,70);
	}
	else
	{
		m_imglist = new wxImageList(32,32,TRUE);
		m_imglist->Add(wxIcon(package32));
		m_list->SetImageList(m_imglist,wxIMAGE_LIST_NORMAL);
	}
	// pkg scanning
	wxDir *m_dir = new wxDir();
	wxListItem item;
	if(m_dir->Open(LOCAL_PACKAGE_PATH))
	{
		if(m_dir->GetFirst(&filename,"*",wxDIR_FILES))
		{
			PKGINFO pkg;
			m_list->InsertItem(counter,ICON_PACKAGE);
			m_list->SetItem(counter,0,(wxString&)filename,ICON_PACKAGE);
			if(style==wxLC_REPORT)
			{
				if(LoadPackageInfoFile(filename,&pkg,NO_FULL_DESCRIPTION))
				{
					m_list->SetItemData(counter,GetID(*pkg.Name));
					m_list->SetItem(counter,1,*pkg.ShortDescription);
					m_list->SetItem(counter,2,*pkg.CompressedSize);
					m_list->SetItem(counter,3,*pkg.UncompressedSize);
				}
			}
			while(m_dir->GetNext(&filename))
			{
				counter++;
				m_list->InsertItem(counter,ICON_PACKAGE);
				m_list->SetItem(counter,0,(wxString&)filename,ICON_PACKAGE);
				if(style==wxLC_REPORT)
				{
					if(LoadPackageInfoFile(filename,&pkg,NO_FULL_DESCRIPTION))
					{
						m_list->SetItemData(counter,GetID(*pkg.Name));
						m_list->SetItem(counter,1,*pkg.ShortDescription);
						m_list->SetItem(counter,2,*pkg.CompressedSize);
						m_list->SetItem(counter,3,*pkg.UncompressedSize);
					}
				}
			}
			status.Append(wxString::Format("%d",m_list->GetItemCount()));
			status.Append(" packages installed");
			m_status->SetStatusText((const wxString&)status);
		}
	}
	SortList();
	SelItem = -1;
	OnListDeselected();
}

// ::OnClose - user closed window
void MainDlg::OnClose()
{
	delete this;
}

// ::OnSortList - sorts package names in list
void MainDlg::SortList()
{
	m_list->SortItems(ListCompare, 0);
}

// ::NotImplemented - informs about a not implemented yet feature/function
void MainDlg::NotImplemented()
{
	wxMessageBox("Not implemented yet",APP_NAME,wxICON_INFORMATION);
	return;
}

// GetID - generates a llong id from a string (for sort)
long GetID(wxString string)
{
	long data;	// id value
	char letter;	// current character
	wxString id;	// id buffer
	letter = string.GetChar(0);
	data = toascii(letter);
	id.Append(wxString::Format("%d",(int)data));
	letter = string.GetChar(1);
	data += toascii(letter);
	id.Append(wxString::Format("%d",(int)data));
	letter = string.GetChar(2);
	data += toascii(letter);
	id.Append(wxString::Format("%d",(int)data));
	data = atol(id.c_str());
	return data;
}
