/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

YWindow::YWindow(YWindow *parent, Window win) {
    fParent = parent;
    fHandle = win;
    flags = 0;
    fX = fY = 0;
    fWidth = fHeight = 1;
    fPointer = leftPointer;
    fStyle = 0;
    fGraphics = 0;
    unmapCount = 0;
    
    if (win != 0) {
        flags |= wfCreated | wfAdopted;
        
        create();
    }
}

YWindow::~YWindow() {
    if (fGraphics) {
        delete fGraphics;
        fGraphics = 0;
    }
    if (flags & wfCreated)
        destroy();
}

void YWindow::setFocus() {
    XSetInputFocus(app->display(),
                   handle(),
                   RevertToParent,
                   CurrentTime);
}

void YWindow::setStyle(int aStyle) {
    fStyle = aStyle;
}

Graphics &YWindow::getGraphics() {
    if (fGraphics == 0)
        fGraphics = new Graphics(this);
    return *fGraphics;
}

void YWindow::repaint() {
    if ((flags & (wfCreated | wfVisible)) == (wfCreated | wfVisible)) {
        Graphics &g = getGraphics();

        paint(g, 0, 0, width(), height());
    }
}

void YWindow::create() {
    if (!(flags & wfCreated)) {
        flags |= wfCreated;

        XSetWindowAttributes attributes;
        unsigned long eventmask = /*CWBackPixel |*/ CWEventMask | CWCursor;

        attributes.cursor = fPointer;
       
        attributes.event_mask =
	    ExposureMask | StructureNotifyMask | 
            SubstructureRedirectMask | //SubstructureNotifyMask |
	    LeaveWindowMask | EnterWindowMask | FocusChangeMask |
            KeyPressMask | KeyReleaseMask |
            ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;

        if (fStyle & wsSaveUnder) {
            attributes.save_under = True;
            eventmask |= CWSaveUnder;
        }
        if (fStyle & wsOverrideRedirect) {
            attributes.override_redirect = True;
            eventmask |= CWOverrideRedirect;
        }

        fHandle = XCreateWindow(app->display(),
                                parent()->handle(),
                                x(), y(), width(), height(),
                                0,
                                CopyFromParent,
                                InputOutput,
                                CopyFromParent,
                                eventmask,
                                &attributes);

        MSG(("window created: 0x%lX at (%d:%d-%dx%d)", fHandle, x(), y(), width(), height()));
        if (flags & wfVisible)
            XMapWindow(app->display(), fHandle);
    } else {
        XWindowAttributes attributes;

        XGetWindowAttributes(app->display(),
                             fHandle,
                             &attributes);

        fX = attributes.x;
        fY = attributes.y;
        fWidth = attributes.width;
        fHeight = attributes.height;

        if (attributes.map_state != IsUnmapped)
            flags |= wfVisible;
        else
            flags &= ~wfVisible;

        if (parent() == 0) {
            XSelectInput(app->display(), fHandle,
                         SubstructureRedirectMask | 
                         SubstructureNotifyMask |
                         FocusChangeMask |
                         PropertyChangeMask |
                         LeaveWindowMask| EnterWindowMask | 
                         KeyPressMask | KeyReleaseMask |
                         ButtonPressMask | ButtonReleaseMask | ButtonMotionMask);
        } else {
            XSelectInput(app->display(), fHandle,
                         StructureNotifyMask | FocusChangeMask |
                         LeaveWindowMask | EnterWindowMask |
                         KeyPressMask | KeyReleaseMask |
                         PropertyChangeMask | ColormapChangeMask
                        );
        }
    }
    XSaveContext(app->display(), fHandle, windowContext, (XPointer)this);
}

void YWindow::destroy() {
    XDeleteContext(app->display(), fHandle, windowContext);
    if (flags & wfCreated) {
        if (!(flags & wfDestroyed)) {
            if (!(flags & wfAdopted)) {
                XDestroyWindow(app->display(), fHandle);
            } else {
                XSelectInput(app->display(), fHandle, NoEventMask); 
            }
        }
    }
}

void YWindow::reparent(YWindow *parent, int x, int y) {
    //flags &= ~wfVisible; // don't unmap when we get UnmapNotify
    if (flags & wfVisible) {
        flags |= wfUnmapped;
        unmapCount++;
    }

    fParent = parent;
    XReparentWindow(app->display(),
                    handle(),
                    parent->handle(),
                    x, y);
    fX = x;
    fY = y;
}

Window YWindow::handle() {
    if (!(flags & wfCreated))
        create();

    return fHandle;
}

void YWindow::show() {
    if (!(flags & wfVisible)) {
        flags |= wfVisible;
        XMapWindow(app->display(), handle());
    }
}

void YWindow::hide() {
    if (flags & wfVisible) {
        flags &= ~wfVisible;
        flags |= wfUnmapped;
        unmapCount++;
        XUnmapWindow(app->display(), handle());
    }
}

void YWindow::raise() {
    XRaiseWindow(app->display(), handle());
}

void YWindow::lower() {
    XLowerWindow(app->display(), handle());
}

void YWindow::handleEvent(const XEvent &event) {
    switch (event.type) {
    case KeyPress:
    case KeyRelease: handleKey(event.xkey); break;
    case ButtonPress:
    case ButtonRelease: handleButton(event.xbutton); break;
    case MotionNotify: 
         {
             XEvent new_event;

             if (XCheckMaskEvent(app->display(),
                                 KeyPressMask |
                                 KeyReleaseMask |
                                 ButtonPressMask | 
                                 ButtonReleaseMask | 
                                 ButtonMotionMask,
                                 &new_event) == True)
             {
                 if (event.type != new_event.type) {
                    XPutBackEvent(app->display(), &new_event);
                    handleMotion(event.xmotion);
                 } else
                    handleMotion(new_event.xmotion);
             } else 
                 handleMotion(event.xmotion);
         } 
         break;

    case EnterNotify:
    case LeaveNotify: handleCrossing(event.xcrossing); break;
    case FocusIn:
    case FocusOut: handleFocus(event.xfocus); break;
    case PropertyNotify: handleProperty(event.xproperty); break;
    case ColormapNotify: handleColormap(event.xcolormap); break;
    case MapRequest: handleMapRequest(event.xmaprequest); break;
    case ConfigureRequest: handleConfigureRequest(event.xconfigurerequest); break;
    case ConfigureNotify: handleConfigure(event.xconfigure); break;
    case DestroyNotify: handleDestroyWindow(event.xdestroywindow); break;
    case Expose: handleExpose(event.xexpose); break;
    case GraphicsExpose: handleGraphicsExpose(event.xgraphicsexpose); break;
    case MapNotify: handleMap(event.xmap); break;
    case UnmapNotify: handleUnmap(event.xunmap); break;
    case ClientMessage: handleClientMessage(event.xclient); break;
#ifdef SHAPE
    default:
        if (shapesSupported && event.type == (shapeEventBase + ShapeNotify))
            handleShapeNotify(*(XShapeEvent *)&event);
        break;
#endif
    }
}

void YWindow::handleExpose(const XExposeEvent &expose) {
    Graphics &g = getGraphics();
    XRectangle r;

    r.x = expose.x;
    r.y = expose.y;
    r.width = expose.width;
    r.height = expose.height;
    
    XSetClipRectangles(app->display(), g.handle(),
                       0, 0, &r, 1, Unsorted);
    paint(g,
          expose.x,
          expose.y,
          expose.width,
          expose.height);

    XSetClipMask(app->display(), g.handle(), None);
}

void YWindow::handleGraphicsExpose(const XGraphicsExposeEvent &graphicsExpose) {
    Graphics &g = getGraphics();
    XRectangle r;

    r.x = graphicsExpose.x;
    r.y = graphicsExpose.y;
    r.width = graphicsExpose.width;
    r.height = graphicsExpose.height;
    
    XSetClipRectangles(app->display(), g.handle(),
                       0, 0, &r, 1, Unsorted);
    paint(g,
          graphicsExpose.x,
          graphicsExpose.y,
          graphicsExpose.width,
          graphicsExpose.height);

    XSetClipMask(app->display(), g.handle(), None);
}

void YWindow::handleConfigure(const XConfigureEvent &/*configure*/) {
    /*if (configure.x != fX ||
        configure.y != fY ||
        (unsigned int)configure.width != fWidth ||
        (unsigned int)configure.height != fHeight)
    {
        fX = configure.x;
        fY = configure.y;
        fWidth = configure.width;
        fHeight = configure.height;

        this->configure(fX, fY, fWidth, fHeight);
    }*/
}

void YWindow::handleKey(const XKeyEvent &) {
}

YWindow *YWindow::fClickWindow = 0;
Time YWindow::fClickTime = 0;
int YWindow::fClickCount = 0;
XButtonEvent YWindow::fClickEvent = { };
int YWindow::fClickDrag = 0;
unsigned int YWindow::fClickButton = 0;
unsigned int YWindow::fClickButtonDown = 0;

void YWindow::handleButton(const XButtonEvent &button) {
    int x_dif = button.x_root - fClickEvent.x_root;
    int y_dif = button.y_root - fClickEvent.y_root;
    
    unsigned int motionDelta =
        ((x_dif < 0) ? - x_dif : x_dif) +
        ((y_dif < 0) ? - y_dif : y_dif);
    
    if (button.type == ButtonPress) {
        fClickDrag = 0;

        if (fClickWindow != this) {
            fClickWindow = this;
            fClickCount = 1;
        } else {
            if ((button.time - fClickTime < MultiClickTime) &&
                fClickButton == button.button &&
                motionDelta < ClickMotionDistance &&
                button.x >= 0 && button.y >= 0 &&
                button.x < int(width()) && button.y < int(height()))
                fClickCount++;
            else
                fClickCount = 1;
        }
        fClickEvent = button;
        fClickButton = fClickButtonDown = button.button;
        fClickTime = button.time;
    } else if (button.type == ButtonRelease) {
        if ((fClickWindow == this) &&
            !fClickDrag &&
            fClickCount > 0 &&
            fClickButtonDown == button.button &&
            motionDelta < ClickMotionDistance &&
            button.x >= 0 && button.y >= 0 &&
            button.x < int(width()) && button.y < int(height()))
        {
            fClickButtonDown = 0;
            handleClick(fClickEvent, button, fClickCount);
        } else {
            fClickWindow = 0;
            fClickCount = 1;
            fClickButtonDown = 0;
            fClickButton = 0;
            if (fClickDrag)
                handleEndDrag(fClickEvent, button);
        }
    }
}

void YWindow::handleMotion(const XMotionEvent &motion) {
    if (fClickButtonDown) {
        if (fClickDrag) {
            handleDrag(fClickEvent, motion);
        } else {
            int x_dif = motion.x_root - fClickEvent.x_root;
            int y_dif = motion.y_root - fClickEvent.y_root;
            
            unsigned int motionDelta =
                ((x_dif < 0) ? - x_dif : x_dif) +
                ((y_dif < 0) ? - y_dif : y_dif);
            
            if (motionDelta >= ClickMotionDistance) {
                fClickDrag = 1;
                handleBeginDrag(fClickEvent, motion);
            }
        }
    }
}

void YWindow::handleCrossing(const XCrossingEvent &) {
}

void YWindow::handleProperty(const XPropertyEvent &) {
}

void YWindow::handleColormap(const XColormapEvent &) {
}

void YWindow::handleFocus(const XFocusChangeEvent &) {
}

void YWindow::handleClientMessage(const XClientMessageEvent &message) {
    if (message.message_type == _XA_WM_PROTOCOLS 
        && message.format == 32
        && message.data.l[0] == (long)_XA_WM_DELETE_WINDOW)
    {
        handleClose();
    }
}

#if 0
    virtual void handleVisibility(const XVisibilityEvent &visibility);
    virtual void handleCreateWindow(const XCreateWindowEvent &createWindow);
#endif

void YWindow::handleMap(const XMapEvent &) {
    //flags |= wfVisible;
}

void YWindow::handleUnmap(const XUnmapEvent &) {
    if (flags & wfUnmapped) {
        unmapCount--;
        if (unmapCount == 0)
            flags &= ~wfUnmapped;
    }
    else
        flags &= ~wfVisible;
}

void YWindow::handleDestroyWindow(const XDestroyWindowEvent &destroyWindow) {
    if (destroyWindow.window == fHandle)
        flags |= wfDestroyed;
}

void YWindow::handleConfigureRequest(const XConfigureRequestEvent &) {
}

void YWindow::handleMapRequest(const XMapRequestEvent &) {
}

#ifdef SHAPE
void YWindow::handleShapeNotify(const XShapeEvent &) {
}
#endif

void YWindow::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &/*up*/, int /*count*/) {
}

void YWindow::handleBeginDrag(const XButtonEvent &/*down*/, const XMotionEvent &/*motion*/) {
}

void YWindow::handleDrag(const XButtonEvent &/*down*/, const XMotionEvent &/*motion*/) {
}

void YWindow::handleEndDrag(const XButtonEvent &/*down*/, const XButtonEvent &/*up*/) {
}

void YWindow::paint(Graphics &g, int x, int y, unsigned int w, unsigned int h) {
    g.fillRect(x, y, w, h);
}

void YWindow::setGeometry(int x, int y, unsigned int width, unsigned int height) {
    if (x != fX ||
        y != fY ||
        width != fWidth ||
        height != fHeight)
    {
        fX = x;
        fY = y;
        fWidth = width;
        fHeight = height;

        if (flags & wfCreated) {
            XMoveResizeWindow(app->display(),
                              fHandle,
                              fX, fY, fWidth, fHeight);
        }

        configure(fX, fY, fWidth, fHeight);
    }
}

void YWindow::setPosition(int x, int y) {
    if (x != fX || y != fY) {
        fX = x;
        fY = y;

        if (flags & wfCreated)
            XMoveWindow(app->display(), fHandle, fX, fY);

        configure(fX, fY, width(), height());
    }
}

void YWindow::setSize(unsigned int width, unsigned int height) {
    if (width != fWidth || height != fHeight) {
        fWidth = width;
        fHeight = height;

        if (flags & wfCreated)
            XResizeWindow(app->display(), fHandle, fWidth, fHeight);

        configure(x(), y(), fWidth, fHeight);
    }
}

void YWindow::configure(int, int, unsigned int, unsigned int) {
}

void YWindow::setPointer(Cursor pointer) {
    fPointer = pointer;

    if (flags & wfCreated) {
        XSetWindowAttributes attributes;

        attributes.cursor = fPointer;

        XChangeWindowAttributes(app->display(),
                                handle(),
                                CWCursor,
                                &attributes);
    }
}

void YWindow::setGrabPointer(Cursor pointer) {
    XChangeActivePointerGrab(app->display(),
                             ButtonPressMask | PointerMotionMask | ButtonReleaseMask, /* ??? */
                             pointer, CurrentTime);//app->getEventTime());
}

void YWindow::grabKey(int key, unsigned int modifiers) {
    KeyCode keycode = XKeysymToKeycode(app->display(), key);
    if (keycode != 0) {
        XGrabKey(app->display(), keycode, modifiers, handle(), True,
                 GrabModeAsync, GrabModeAsync);
        XGrabKey(app->display(), keycode, modifiers | LockMask, handle(), True,
                 GrabModeAsync, GrabModeAsync);
    }
}

void YWindow::donePopup(YPopupWindow * /*command*/) {
}

void YWindow::handleClose() {
}
