// ----------------------------------------------------------------------------
// Filename:    MainWidget.cpp
// $Date: 2000/09/22 15:00:38 $
// $Revision: 1.15 $
// ----------------------------------------------------------------------------

#include <kstdaccel.h>
#include <kapp.h>
#include <kiconloader.h>
#include <qmessagebox.h>

#include "MainWidget.h"
#include "KTimeclock.h"
#include "DockWidget.h"

// ----------------------------------------------------------------------------
// Specify a global docking widget.
// ----------------------------------------------------------------------------
DockWidget* _dock;

// ----------------------------------------------------------------------------
// Function:    KTimeclockMainWidget (QWidget* parent, const char* name)
// Parameters:  parent  - Handle to parent widget
//              name    - Name of this widget
// ----------------------------------------------------------------------------
// Constructor.  Creates the KTimeclock widget, connects us to it, and then
// initializes the menu/tool/statusbars.
// ----------------------------------------------------------------------------
KTimeclockMainWidget::KTimeclockMainWidget (QWidget*, const char* name)
    : KTMainWindow( name )
{
    ktimeclock = new KTimeclock( this, 0 );
    CHECK_PTR( ktimeclock );
    this->setView( ktimeclock );
    connect( ktimeclock, SIGNAL( timerStarted() ), this, SLOT( startTimer() ) );
    connect( ktimeclock, SIGNAL( timerStopped() ), this, SLOT( stopTimer() ) );
    connect( ktimeclock, SIGNAL( timerState(QString&) ), this, SLOT( timerState(QString&) ) );
    connect( ktimeclock, SIGNAL( itemSelected(QString&) ), this, SLOT( itemSelected(QString&) ) );

    this->initMenu();
    this->initToolbar();
    this->initStatusbar();
    ktimeclock->show();
    this->stopTimer();

    // ------------------------------------------------------------------------
    // Read in any geometry from our last invokation and make that the current
    // geometry.
    // ------------------------------------------------------------------------
    this->_read_geometry();

    // ------------------------------------------------------------------------
    // Create the new dock widget that'll dock us into KPanel.
    // ------------------------------------------------------------------------
    _dock = new DockWidget( this, "dock" );
    CHECK_PTR( _dock );
    _dock->dock();
}

// ----------------------------------------------------------------------------
// Function:    ~KTimeclockMainWidget ()
// ----------------------------------------------------------------------------
// Destructor.  Saves the current state out to disk.
// ----------------------------------------------------------------------------
KTimeclockMainWidget::~KTimeclockMainWidget (void)
{
    ktimeclock->save();
    this->_write_geometry();
}

// ----------------------------------------------------------------------------
// Function:    _read_geometry ()
// ----------------------------------------------------------------------------
// Internal function to read our geometry back in from our config file and to
// reset our geometry back to the way it was last time we ran.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::_read_geometry ()
{
    QString tmp = kapp->getConfig()->readEntry(QString("TopLevelGeometry"));
    if (!tmp.isNull())
    {
        if (tmp.length() == 19)
        {
            int x, y, w, h;
            sscanf( tmp.data(), "%04d:%04d:%04d:%04d", &x, &y, &w, &h );
            this->setGeometry( x, y, w, h );
        }
    }
}

// ----------------------------------------------------------------------------
// Function:    _write_geometry()
// ----------------------------------------------------------------------------
// Internal function to write our geometry out to our config file.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::_write_geometry ()
{
    QString tmp;
    tmp.sprintf( "%04d:%04d:%04d:%04d",
                 this->x(), this->y(),
                 this->width(), this->height() );
    kapp->getConfig()->writeEntry("TopLevelGeometry",tmp);
    kapp->getConfig()->sync();
}

// ----------------------------------------------------------------------------
// Function:    initMenu ()
// ----------------------------------------------------------------------------
// Initializes the menus, sets up the keyboard accelerators, and connects the
// menu items to their appropriate slots.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::initMenu (void)
{
    QPopupMenu *mnuNew = new QPopupMenu;
    CHECK_PTR( mnuNew );
    mnuNew->insertItem( i18n( "&Project" ), MENU_NEWPROJECT );
    mnuNew->connectItem( MENU_NEWPROJECT, ktimeclock, SLOT( AddProject() ) );
    mnuNew->setAccel( CTRL+Key_P, MENU_NEWPROJECT );
    mnuNew->insertItem( i18n( "&Sub-project" ), MENU_NEWSUBPROJECT );
    mnuNew->connectItem( MENU_NEWSUBPROJECT, ktimeclock, SLOT( AddSubProject() ) );
    mnuNew->setAccel( CTRL+Key_S, MENU_NEWSUBPROJECT );
    mnuNew->insertItem( i18n( "&Task" ), MENU_NEWTASK );
    mnuNew->connectItem( MENU_NEWTASK, ktimeclock, SLOT( AddTask() ) );
    mnuNew->setAccel( CTRL+Key_T, MENU_NEWTASK );

    QPopupMenu *mnuFile = new QPopupMenu;
    CHECK_PTR( mnuFile );
    mnuFile->insertItem( i18n( "&New" ), mnuNew );
    mnuFile->insertSeparator();
    mnuFile->insertItem( i18n( "&Quit" ), MENU_QUIT );
    mnuFile->connectItem( MENU_QUIT, this, SLOT( quit() ) );
    mnuFile->setAccel( CTRL+Key_Q, MENU_QUIT );

    QPopupMenu *mnuEdit = new QPopupMenu;
    CHECK_PTR( mnuEdit );
    mnuEdit->insertItem( i18n( "&Edit item" ), MENU_EDIT );
    mnuEdit->connectItem( MENU_EDIT, ktimeclock, SLOT( EditItem() ) );
    mnuEdit->setAccel( CTRL+Key_E, MENU_EDIT );
    mnuEdit->insertItem( i18n( "&Delete item" ), MENU_DELETE );
    mnuEdit->connectItem( MENU_DELETE, ktimeclock, SLOT( DeleteItem() ) );
    mnuEdit->setAccel( CTRL+Key_D, MENU_DELETE );
    mnuEdit->insertItem( i18n( "&Clear item" ), MENU_CLEAR );
    mnuEdit->connectItem( MENU_CLEAR, ktimeclock, SLOT( ClearItem() ) );
    mnuEdit->setAccel( CTRL+Key_C, MENU_CLEAR );
    mnuEdit->insertSeparator();
    mnuEdit->insertItem( i18n( "Clear &all items" ), MENU_CLEAR_ALL );
    mnuEdit->connectItem( MENU_CLEAR_ALL, ktimeclock, SLOT( ClearAllItems() ) );
    mnuEdit->setAccel( CTRL+Key_A, MENU_CLEAR_ALL );
    mnuEdit->insertSeparator();
    mnuEdit->insertItem( i18n( "&Preferences" ), MENU_PREFERENCES );
    mnuEdit->connectItem( MENU_PREFERENCES, ktimeclock, SLOT( Preferences() ) );
    mnuEdit->setAccel( CTRL+Key_P, MENU_PREFERENCES );
    mnuEdit->setItemEnabled( MENU_PREFERENCES, false );

    QPopupMenu *mnuTimer = new QPopupMenu;
    CHECK_PTR( mnuTimer );
    mnuTimer->insertItem( i18n( "&Start timer" ), MENU_STARTTIMER );
    mnuTimer->connectItem( MENU_STARTTIMER, ktimeclock, SLOT( StartTimer() ) );
    mnuTimer->insertItem( i18n( "S&top timer" ), MENU_STOPTIMER );
    mnuTimer->connectItem( MENU_STOPTIMER, ktimeclock, SLOT( StopTimer() ) );

    QPopupMenu *mnuHelp = new QPopupMenu;
    CHECK_PTR( mnuHelp );
    mnuHelp->insertItem( i18n( "&Contents" ), MENU_CONTENTS );
    mnuHelp->connectItem( MENU_CONTENTS, this, SLOT( help() ) );
    mnuHelp->setAccel( Key_F1, MENU_CONTENTS );
    mnuHelp->insertSeparator();
    mnuHelp->insertItem( i18n( "&About" ), MENU_ABOUT );
    mnuHelp->connectItem( MENU_ABOUT, this, SLOT( about() ) );

    menu = this->menuBar();
    CHECK_PTR( menu );
    menu->insertItem( i18n( "&File" ), mnuFile );
    menu->insertItem( i18n( "&Edit" ), mnuEdit );
    menu->insertItem( i18n( "&Timer" ), mnuTimer );
    menu->insertSeparator();
    menu->insertItem( i18n( "&Help" ), mnuHelp );

    menu->show();
    this->setMenu( menu );
}

// ----------------------------------------------------------------------------
// Function:    initToolbar ()
// ----------------------------------------------------------------------------
// Creates our toolbar, adds all of the items to it that we wish to have added,
// and connects the items to their respective slots.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::initToolbar (void)
{
    KIconLoader* loader = kapp->getIconLoader();
    toolbar = new KToolBar( this, "toolbar" );

    toolbar->insertButton( loader->loadIcon( "mini/mini-run.xpm" ),
                           MENU_STARTTIMER,
                           SIGNAL( clicked() ),
                           ktimeclock,
                           SLOT( StartTimer() ),
                           true,
                           i18n( "Start timer" ) );
    toolbar->insertButton( loader->loadIcon( "mini/mini-stop.xpm" ),
                           MENU_STOPTIMER,
                           SIGNAL( clicked() ),
                           ktimeclock,
                           SLOT( StopTimer() ),
                           true,
                           i18n( "Stop timer" ) );
    toolbar->insertSeparator();
    toolbar->insertButton( loader->loadIcon( "mini/mini-question.xpm" ),
                           MENU_CONTENTS,
                           SIGNAL( clicked() ),
                           this,
                           SLOT( help() ),
                           true,
                           i18n( "Help contents" ) );
    toolbar->show();
    this->enableToolBar( KToolBar::Show, this->addToolBar( toolbar ) );

    return;
}

// ----------------------------------------------------------------------------
// Function:    initStatusbar ()
// ----------------------------------------------------------------------------
// Creates the status bar, and sets the size of the items it contains.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::initStatusbar (void)
{
    statusbar = this->statusBar();
    statusbar->setFrameStyle( QFrame::Panel | QFrame::Raised );
    statusbar->setBorderWidth( 2 );
    statusbar->setFontPropagation( QWidget::AllChildren );
    statusbar->setFont( QFont( "helvetica", 10, QFont::Normal ) );

    status_state = new QLabel( statusbar, "status_state" );
    status_state->setFrameStyle( QFrame::Panel | QFrame::Sunken );
    statusbar->insertWidget( status_state, 80, 0 );

    status_selected = new QLabel( statusbar, "status_selected" );
    status_selected->setFrameStyle( QFrame::Panel | QFrame::Sunken );
    statusbar->insertWidget( status_selected, 200, 1 );

    this->setStatusBar( statusbar );

    return;
}

// ----------------------------------------------------------------------------
// Function:    about ()
// ----------------------------------------------------------------------------
// Displays an about box to the user.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::about ()
{
    QMessageBox::about( this,
                        i18n( "About" ),
                        i18n( KTIMECLOCK_VERSION"\n\n(C) 1998-2000 Graham TerMarsch (gtermars@home.com)\n" )
                      );
    return;
}

// ----------------------------------------------------------------------------
// Function:    help ()
// ----------------------------------------------------------------------------
// Displays the on-line help to the user.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::help ()
{
    kapp->invokeHTMLHelp( "ktimeclock/ktimeclock.html", "" );
    return;
}

// ----------------------------------------------------------------------------
// Function:    startTimer ()
// ----------------------------------------------------------------------------
// En/disables menu/toolbar items when the timer is started.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::startTimer ()
{
    toolbar->setItemEnabled( MENU_STARTTIMER, false );
    toolbar->setItemEnabled( MENU_STOPTIMER, true );

    menu->setItemEnabled( MENU_STARTTIMER, false );
    menu->setItemEnabled( MENU_STOPTIMER, true );
}

// ----------------------------------------------------------------------------
// Function:    stopTimer ()
// ----------------------------------------------------------------------------
// En/disables menu/toolbar items when the timer is stopped.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::stopTimer ()
{
    toolbar->setItemEnabled( MENU_STARTTIMER, true );
    toolbar->setItemEnabled( MENU_STOPTIMER, false );

    menu->setItemEnabled( MENU_STARTTIMER, true );
    menu->setItemEnabled( MENU_STOPTIMER, false );
}

// ----------------------------------------------------------------------------
// Function:    timerState (QString& state)
// Parameters:  state   - State which the timer has changed into
// ----------------------------------------------------------------------------
// Sets the current state of the timer into the status bar's display area.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::timerState( QString& state )
{
    status_state->setText( state );
    //statusbar->changeItem( state, 0 );
}

// ----------------------------------------------------------------------------
// Function:    itemSelected (QString& item)
// Parameters:  item    - Name of currently selected item.
// ----------------------------------------------------------------------------
// Sets the currently selected item into the status bar's display area.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::itemSelected( QString& item )
{
    status_selected->setText( item );
    //statusbar->changeItem( item, 1 );
}

// ----------------------------------------------------------------------------
// Function:    quit ()
// ----------------------------------------------------------------------------
// Quits the application, making sure to properly save down the file first.
// ----------------------------------------------------------------------------
void KTimeclockMainWidget::quit ()
{
    // Save out our stuff first.
    ktimeclock->save();
    this->_write_geometry();

    // Then tell the 'kapp' that we'd like to quit.
    kapp->quit();
}
