// ----------------------------------------------------------------------------
// Filename:    KTimeclock.cpp
// $Date: 2000/09/22 15:00:38 $
// $Revision: 1.12 $
// ----------------------------------------------------------------------------

#include <kapp.h>
#include <kfiledialog.h>
#include <qmessagebox.h>
#include <qdir.h>

#include <stdlib.h>         // Needed for getenv()
#include <sys/stat.h>       // Needed for S_IRWXU, mkdir()
#include <dirent.h>         // Needed for opendir(), closedir()

#include "KTimeclock.h"
#include "ItemDialog.h"

#define Inherited KTimeclockData

// ----------------------------------------------------------------------------
// Function:    KTimeclock (QWidget* parent, const char* name)
// Parameters:  parent  - Handle to parent widget
//              name    - Name of this widget
// ----------------------------------------------------------------------------
// Constructor.
// ----------------------------------------------------------------------------
KTimeclock::KTimeclock ( QWidget* parent, const char* name )
	: Inherited( parent, name )
{
	setCaption( "KTimeclock" );

    // ------------------------------------------------------------------------
    // Check to ensure that all of the directories we'll require are present
    // ------------------------------------------------------------------------
    TestDir( "share" );
    TestDir( "share/config" );
    TestDir( "share/apps" );
    TestDir( "share/apps/ktimeclock" );

    // ------------------------------------------------------------------------
    // Load in any existing data file and trigger a visual refresh.
    // ------------------------------------------------------------------------
    if (this->load())
    {
        QMessageBox::warning( this, "Error", "Unable to load data file." );
    }
    treeProjects->triggerUpdate();
}


// ----------------------------------------------------------------------------
// Function:    ~KTimeclock ()
// ----------------------------------------------------------------------------
// Destructor.
// ----------------------------------------------------------------------------
KTimeclock::~KTimeclock ()
{
}

// ----------------------------------------------------------------------------
// Function:    AddProject ()
// ----------------------------------------------------------------------------
// Shows the dialog box asking for information on a new project and adds it to
// the tree view if required.
// ----------------------------------------------------------------------------
void KTimeclock::AddProject ()
{
    // ------------------------------------------------------------------------
    // Display the dialog to get the name of the new project.
    // ------------------------------------------------------------------------
    ItemDialog* dlg = new ItemDialog( this, "dlg" );
    dlg->setCaption( "Add new project" );
    dlg->enableTimeSpent( false );

    if (dlg->exec())
    {
        // --------------------------------------------------------------------
        // If we were actually given a name of a project to create, add it to
        // the tree view.
        // --------------------------------------------------------------------
        const char* project = dlg->name();

        if (project != NULL)
        {
            new KTimeclockListItem( treeProjects,
                                    KTimeclockListItem::ITEM_PROJECT,
                                    project,
                                    0 );
        }
    }

    delete dlg;
}

// ----------------------------------------------------------------------------
// Function:    AddSubProject ()
// ----------------------------------------------------------------------------
// Shows the dialog box asking for information on a new sub-project and adds it
// to the tree view if required.  While we're at it, don't forget to check to
// make sure that the user _has_ selected a project/sub-project to make this
// new one a child of.
// ----------------------------------------------------------------------------
void KTimeclock::AddSubProject ()
{
    // ------------------------------------------------------------------------
    // Get the currently selected item and verify that it's a project; we can't
    // add a sub-project to: 1) nothing, 2) a task
    // ------------------------------------------------------------------------
    KTimeclockListItem* parent = (KTimeclockListItem*)treeProjects->currentItem();
    if ( (parent == NULL) || (KTimeclockListItem::ITEM_PROJECT != parent->type()) )
    {
        QMessageBox::warning( this, "Error", "You must select a project to add a sub-project to.");
        return;
    }

    // ------------------------------------------------------------------------
    // Display the dialog to get the name of the new sub-project.
    // ------------------------------------------------------------------------
    ItemDialog* dlg = new ItemDialog( this, "dlg" );
    dlg->setCaption( "Add new sub-project" );
    dlg->enableTimeSpent( false );

    if (dlg->exec())
    {
        // --------------------------------------------------------------------
        // If we were actually given a name of a sub-project to create, add it
        // to the tree view.
        // --------------------------------------------------------------------
        const char* subproject = dlg->name();

        if (subproject != NULL)
        {
            new KTimeclockListItem( parent,
                                    KTimeclockListItem::ITEM_PROJECT,
                                    subproject,
                                    0 );
        }
    }

    delete dlg;
}

// ----------------------------------------------------------------------------
// Function:    AddTask ()
// ----------------------------------------------------------------------------
// Shows the dialog box asking for information on a new task and adds it to the
// tree view if required.  While we're at it, don't forget to check to make
// sure that the user _has_ selected a project/sub-project to make this new
// task a child of.
// ----------------------------------------------------------------------------
void KTimeclock::AddTask ()
{
    // ------------------------------------------------------------------------
    // Get the currently selected item and verify that it's a project; we can't
    // add a task to: 1) nothing, 2) another task.
    // ------------------------------------------------------------------------
    KTimeclockListItem* parent = (KTimeclockListItem*)treeProjects->currentItem();
    if ( (parent == NULL) || (KTimeclockListItem::ITEM_PROJECT != parent->type()) )
    {
        QMessageBox::warning( this, "Error", "You must select a project to add a sub-project to.");
        return;
    }

    // ------------------------------------------------------------------------
    // Display the dialog to get the name of the new task.
    // ------------------------------------------------------------------------
    ItemDialog* dlg = new ItemDialog( this, "dlg" );
    dlg->setCaption( "Add new task" );
    
    if (dlg->exec())
    {
        // --------------------------------------------------------------------
        // If we were actually given a task to create, add it to the tree view.
        // --------------------------------------------------------------------
        const char* task = dlg->name();

        if (task != NULL)
        {
            int timespent = dlg->timeSpent();
            new KTimeclockListItem( parent,
                                    KTimeclockListItem::ITEM_TASK,
                                    task,
                                    timespent );
        }
    }

    delete dlg;
}

// ----------------------------------------------------------------------------
// Function:    EditItem ()
// ----------------------------------------------------------------------------
// Edits the currently selected item.
// ----------------------------------------------------------------------------
void KTimeclock::EditItem ()
{
    // ------------------------------------------------------------------------
    // Get the current selected item.
    // ------------------------------------------------------------------------
    KTimeclockListItem* selected = (KTimeclockListItem*)treeProjects->currentItem();
    if (select == NULL)
    {
        QMessageBox::warning( this, "Error", "You must select an item you wish to edit." );
        return;
    }

    // ------------------------------------------------------------------------
    // Display the edit dialog and then put the values back into this item.
    // ------------------------------------------------------------------------
    ItemDialog* dlg = new ItemDialog( this, "dlg" );
    dlg->setCaption( "Edit task" );
    dlg->setName( selected->description() );

    if (selected->type() != KTimeclockListItem::ITEM_TASK)
        { dlg->enableTimeSpent( false ); }
    else
        { dlg->setTimeSpent( selected->totalTime() ); }

    if (dlg->exec())
    {
        selected->description( dlg->name() );
        selected->totalTime( dlg->timeSpent() );
    }
}

// ----------------------------------------------------------------------------
// Function:    DeleteItem ()
// ----------------------------------------------------------------------------
// Asks the user for confirmation and deletes the currently selected item.
// ----------------------------------------------------------------------------
void KTimeclock::DeleteItem ()
{
    // ------------------------------------------------------------------------
    // Get the currently selected item.
    // ------------------------------------------------------------------------
    KTimeclockListItem* selected = (KTimeclockListItem*)treeProjects->currentItem();
    if (selected == NULL)
    {
        QMessageBox::warning( this, "Error", "You must select an item you wish to delete." );
        return;
    }

    // ------------------------------------------------------------------------
    // Verify with the user that we really want to delete this item and
    // everything underneath it.  Use a different message depending on whether
    // its a project or a task.
    // ------------------------------------------------------------------------
    QString msg = "Do you really want to delete '";
    msg += selected->description();

    if (KTimeclockListItem::ITEM_PROJECT == selected->type())
        { msg += "' and all sub-projects and tasks?"; }
    else
        { msg += "'?"; }

    if (QMessageBox::Yes ==
        QMessageBox::information( this, "Delete item?", msg, QMessageBox::Yes, QMessageBox::No)
       )
    {
        delete selected;
        this->SelectionChanged( treeProjects->currentItem() );
    }
}

// ----------------------------------------------------------------------------
// Function:    ClearItem ()
// ----------------------------------------------------------------------------
// Clears all time associated with the currently selected item.  Note that this
// update is NOT recursive.
// ----------------------------------------------------------------------------
void KTimeclock::ClearItem ()
{
    // ------------------------------------------------------------------------
    // Get the currently selected item.
    // ------------------------------------------------------------------------
    KTimeclockListItem* selected = (KTimeclockListItem*)treeProjects->currentItem();
    if (selected == NULL)
    {
        QMessageBox::warning( this, "Error", "You must select an item you wish to clear." );
        return;
    }

    // ------------------------------------------------------------------------
    // Clear all of the time associated with this item.
    // ------------------------------------------------------------------------
    selected->totalTime( 0 );
    selected->sessionTime( 0 );
}

// ----------------------------------------------------------------------------
// Function:    ClearAllItems ()
// ----------------------------------------------------------------------------
// Clears all time associated with ALL of the tasks in the project list.
// ----------------------------------------------------------------------------
void KTimeclock::ClearAllItems ()
{
    // Verify with the user that we really want to clear the time for all of
    // the items in the tree view.
    QString msg = "Are you sure you want to clear the time spent for all of the items?";
    if (QMessageBox::Yes ==
        QMessageBox::information( this, "Clear all items?", msg, QMessageBox::Yes, QMessageBox::No)
       )
    {
        KTimeclockListItem* first = (KTimeclockListItem*)treeProjects->firstChild();
        this->_recursiveClear( first );
    }
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
void KTimeclock::_recursiveClear (KTimeclockListItem* item)
{
    if (item != NULL)
    {
        // First, clear all of the time associated with this item.
        item->totalTime( 0 );
        item->sessionTime( 0 );

        // Then, clear all of our children.
        this->_recursiveClear( (KTimeclockListItem*)item->firstChild() );

        // Then, go on to clear our next sibling.
        this->_recursiveClear( (KTimeclockListItem*)item->nextSibling() );
    }
}

// ----------------------------------------------------------------------------
// Function:    Preferences ()
// ----------------------------------------------------------------------------
// Currently does nothing.
// ----------------------------------------------------------------------------
void KTimeclock::Preferences ()
{
}

// ----------------------------------------------------------------------------
// Function:    StartTimer ()
// ----------------------------------------------------------------------------
// Starts the timer and hangs onto it's ID number.  While we're at it, don't
// forget to change our state and let everyone else know about it.
// ----------------------------------------------------------------------------
void KTimeclock::StartTimer()
{
    timerID = startTimer( 1000 );
    emit timerStarted();
    state = "Enabled";
    emit timerState( state );
}

// ----------------------------------------------------------------------------
// Function:    StopTimer ()
// ----------------------------------------------------------------------------
// Stops the timer, changes our state, and lets everyone else know about it.
// ----------------------------------------------------------------------------
void KTimeclock::StopTimer()
{
    killTimer( timerID );
    emit timerStopped();
    state = "Disabled";
    emit timerState( state );
}

// ----------------------------------------------------------------------------
// Function:    timerEvent (QTimerEvent *event)
// ----------------------------------------------------------------------------
// Called each time a timer 'tick' has gone by.  Updates the information on the
// currently selected item and forces a repaint of the tree view.
// ----------------------------------------------------------------------------
void KTimeclock::timerEvent( QTimerEvent* )
{
    KTimeclockListItem* item = (KTimeclockListItem*)treeProjects->currentItem();
    if (item == NULL) { return; }

    // ------------------------------------------------------------------------
    // Only count time for tasks.
    // ------------------------------------------------------------------------
    if (KTimeclockListItem::ITEM_TASK == item->type())
    {
        int tmp;

        // --------------------------------------------------------------------
        // increment item[ session seconds ]
        // --------------------------------------------------------------------
        tmp = item->sessionTime();
        tmp ++;
        item->sessionTime( tmp );

        // --------------------------------------------------------------------
        // increment item[ total seconds ]
        // --------------------------------------------------------------------
        tmp = item->totalTime();
        tmp ++;
        item->totalTime( tmp );
    }
}

// ----------------------------------------------------------------------------
// Function:    save ()
// ----------------------------------------------------------------------------
// Saves all of the timeclock information out to our own data file.  We do this
// by calling off to another function who'll recursively process the tree-view
// data and output it to a stream we provide it.
// ----------------------------------------------------------------------------
int KTimeclock::save()
{
    QString filename = kapp->localkdedir() + "/share/apps/ktimeclock/ktimeclock.txt";

    // ------------------------------------------------------------------------
    // Open the output file
    // ------------------------------------------------------------------------
    QFile outfile( filename );
    int failed = 1;

    if (outfile.open( IO_WriteOnly | IO_Truncate ))
    {
        // --------------------------------------------------------------------
        // Call off to have the tree-view output recursively.
        // --------------------------------------------------------------------
        QTextStream outstream( &outfile );
        KTimeclockListItem* item = (KTimeclockListItem*)this->treeProjects->firstChild();
        this->_recursiveSave( outstream, item, 1 );
        failed = 0;
        outfile.close();
    }
    else
    {
        QMessageBox::warning( this, "Error", "Unable to open data file for saving" );
    }

    return (failed);
}

// ----------------------------------------------------------------------------
// Function:    _recursiveSave (QTextStream& str, KTimeclockListItem* item,
//                              int depth)
// Parameters:  str     - Output stream
//              item    - Item to recursively save (and save under)
//              depth   - Current nesting depth
// ----------------------------------------------------------------------------
// Recursively saves the items held under 'item' out to the given output
// stream.
// ----------------------------------------------------------------------------
void KTimeclock::_recursiveSave( QTextStream& str, KTimeclockListItem* item, int depth )
{
    // ------------------------------------------------------------------------
    // If we were given a bogus item to work with, return immediately.
    // ------------------------------------------------------------------------
    if (item == NULL) { return; }

    // ------------------------------------------------------------------------
    // Output ourselves into the stream
    // ------------------------------------------------------------------------
    str << item->totalTime() << '\t'
        << item->type();
    for (int i=0; i<depth; i++)
        { str << '\t'; }
    str << item->description() << endl;

    // ------------------------------------------------------------------------
    // Output any children we have
    // ------------------------------------------------------------------------
    KTimeclockListItem* child = (KTimeclockListItem*)item->firstChild();
    if (child != NULL)
        { this->_recursiveSave( str, child, (depth + 1) ); }

    // ------------------------------------------------------------------------
    // Then go onto our next sibling (if we have one)
    // ------------------------------------------------------------------------
    KTimeclockListItem* sibling = (KTimeclockListItem*)item->nextSibling();
    if (sibling != NULL)
        { this->_recursiveSave( str, sibling, depth ); }
}


// ----------------------------------------------------------------------------
// Function:    load ()
// ----------------------------------------------------------------------------
// Opens our data file, loads in all of the information, and then calls off to
// have all the information stuffed into our tree-view.  Returns non-zero on
// failure to load data file.
// ----------------------------------------------------------------------------
int KTimeclock::load()
{
    QString filename = kapp->localkdedir() + "/share/apps/ktimeclock/ktimeclock.txt";

    // ------------------------------------------------------------------------
    // Open the input file
    // ------------------------------------------------------------------------
    QFile infile( filename );
    int failed = 1;

    if (infile.open( IO_ReadOnly ))
    {
        // --------------------------------------------------------------------
        // Load eveything line-by-line into a queue which we can then process
        // recursively.
        // --------------------------------------------------------------------
        QTextStream instream( &infile );
        QQueue<QString> queue;
        queue.setAutoDelete( true );
        while (!instream.eof())
            { queue.enqueue( new QString(instream.readLine()) ); }
        this->_loadFromQueue( queue, NULL, 0 );
        failed = 0;
        infile.close();
    }
    else
    {
    }

    return (failed);
}

// ----------------------------------------------------------------------------
// Function:    _loadFromQueue (QQueue<QString)& queue,
//                              KTimeclockListItem* parent, int depth)
// Parameters:  queue   - Reference to queue we should be reading from
//              parent  - Parent item to stuff sub-items into
//              depth   - Current recursion depth
// ----------------------------------------------------------------------------
// Recursively loads the list of items from the given queue and stuffs them
// into our tree-view.
// ----------------------------------------------------------------------------
void KTimeclock::_loadFromQueue( QQueue<QString>& queue, KTimeclockListItem* parent, int depth )
{
    // ------------------------------------------------------------------------
    // Loop forever (we'll know when to kick out later on)
    // ------------------------------------------------------------------------
    while (1)
    {
        // --------------------------------------------------------------------
        // Get the next thing in the queue; if it doesn't exist them kick out
        // --------------------------------------------------------------------
        QString* line = queue.head();
        if (!line) { return; }

        // --------------------------------------------------------------------
        // Get the depth of this item in the queue and kick out if it's at a
        // lower level.
        // --------------------------------------------------------------------
        int currdepth = line->contains( '\t' ) - 1;
        if (currdepth <= depth)
            { return; }

        // --------------------------------------------------------------------
        // Get the bits that we need out of this item and then have it removed
        // and it's memory freed up.
        // --------------------------------------------------------------------
        int index = line->find( '\t' );
        QString tmp = line->mid( 0, index );
        int seconds = tmp.toInt();
        index ++; // Skip past the tab character
        int nextindex = line->find( '\t', index );
        QString type = line->mid( index, (nextindex - index) );
        index = line->findRev( '\t' );
        index ++; // Skip past the tab character
        QString description = line->mid( index, (line->length() - index) );
        queue.remove();

        // --------------------------------------------------------------------
        // Now create the new list item and stick it into the tree structure.
        // --------------------------------------------------------------------
        KTimeclockListItem* item;
        if (parent == NULL)
        {
            item = new KTimeclockListItem( treeProjects,
                                        type.data(), description.data(), seconds
                                        );
        }
        else
        {
            item = new KTimeclockListItem( parent,
                                        type.data(), description.data(), seconds
                                        );
        }

        // --------------------------------------------------------------------
        // Now that we've created this item, see how we're supposed to recurse
        // --------------------------------------------------------------------
        line = queue.head();
        if (!line) { return; }
        int nextdepth = line->contains( '\t' ) - 1;
        if (nextdepth > currdepth)
            { this->_loadFromQueue( queue, item, currdepth ); }
        else if (nextdepth < currdepth)
            { return; }
    } // END while
}

// ----------------------------------------------------------------------------
// Function:    SelectionChanged (QListViewItem* item)
// Parameters:  item    - Newly selected item in the tree-view.
// ----------------------------------------------------------------------------
// Gathers the information we need to emit a signal stating that our selected
// item has changed.
// ----------------------------------------------------------------------------
void KTimeclock::SelectionChanged( QListViewItem* item )
{
    if (item != NULL)
    {
        KTimeclockListItem* castitem = (KTimeclockListItem*)item;
        QString msg;
        msg  = castitem->type();
        msg += ": ";
        msg += castitem->description();
        emit itemSelected( msg );
    }
}

// ----------------------------------------------------------------------------
// Function:    ShowPopup (QListViewItem* item, const QPoint& loc, int col)
// Parameters:  item    - Currently selected item
//              loc     - Co-ordinates of where the mouse was clicked
//              col     - Column clicked in (unused)
// ----------------------------------------------------------------------------
// Displays a pop-up menu whenever an item has had the right mouse-button
// clicked on it.
// ----------------------------------------------------------------------------
void KTimeclock::ShowPopup( QListViewItem* item, const QPoint& loc, int )
{
    if (item != NULL)
    {
        treeProjects->setSelected( item, true );
        mnuPopup->move( loc );
        mnuPopup->show();
    }
}

// ----------------------------------------------------------------------------
// Function:    TestDir (const char* name)
// Parameters:  name    - Path to directory to test for/create in local KDE dir
// ----------------------------------------------------------------------------
// Tests for the existence of a given directory underneath the local KDE
// directory, creating it if it doesn't exist.
// ----------------------------------------------------------------------------
void KTimeclock::TestDir (const char* name)
{
    // ------------------------------------------------------------------------
    // Start off in the local KDE dir; everything we're looking for should be
    // under here.
    // ------------------------------------------------------------------------
    QDir dir( kapp->localkdedir() );

    // ------------------------------------------------------------------------
    // See if this directory exists.
    // ------------------------------------------------------------------------
    if (dir.cd( name ))
    {
    }
    else
    {
        if (!dir.mkdir( name ))
        {
            warning( "Could not create directory '%s'", name );
        }
    }
}
