/*
 * Fireworkx 1.4 - pyrotechnics simulation program
 * Copyright (c) 1999-2005 Rony B Chandran <ronybc@asia.com>
 *
 * From Kerala, INDIA
 *
 * url: http://www.ronybc.8k.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Additional programming: 
 * ------------------------
 * Support for different display color modes: 
 * Jean-Pierre Demailly <Jean-Pierre.Demailly@ujf-grenoble.fr>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "config.h"

#define APPNAME "Fireworkx"
#define VERSION "1.4"

#define WIDTH 680 //640
#define HEIGHT 420 //480
#define SHELLCOUNT 3                   /* 3 or 5  */
#define PIXCOUNT 500                   /* 500     */
#define RNDLIFE0 500                   /* violent */
#define RNDLIFE1 2000                  /* 1200    */
#define MINLIFE0 100                   /* violent */
#define MINLIFE1 500                   /* 500     */
#define POWER 5                        /* 5       */
#define FTWEAK 12                      /* 12      */

void mmx_blur(char *a, int b, int c, int d);
void mmx_glow(char *a, int b, int c, int d, char *e);

static int depth;
static int bigendian;
static int flash_on   = 1;
static int glow_mode  = 1;
static int fps_on     = 0;
static int verbose    = 0;
static int shoot      = 0;
static int width;
static int height;
static int rndlife = RNDLIFE1;
static int minlife = MINLIFE1;
static float flash_fade = 0.99;
static unsigned char *palaka1=NULL;
static unsigned char *palaka2=NULL;
static unsigned char *col_val;
static Display *display;
static Window win;
static XImage *xim=NULL;


typedef struct {
  unsigned int burn;
  float x;
  float y;
  float xv;
  float yv;}firepix;

typedef struct {
  unsigned int cx,cy;
  unsigned int life;
  unsigned int color;
  unsigned int special;
  unsigned int cshift;
  unsigned int vgn,shy;
  float air,flash;
  firepix *fpix; }fireshell;

int rnd(int x) {
  return(random()%x);}   /* causes 'divide-by-zero */

int explode(fireshell *fs){
  float air,adg = 0.001;     /* gravity */
  unsigned int n,c;
  unsigned int h = height;
  unsigned int w = width;
  unsigned int *prgb;
  unsigned char *palaka = palaka1;
  firepix *fp = fs->fpix;
  if(fs->vgn){
    if(--fs->cy == fs->shy){  
      fs->vgn   = 0;
      fs->flash = rnd(30000)+15000;}
    else{  
      fs->flash = 50+(fs->cy - fs->shy)*2;
      prgb=(unsigned int *)(palaka + (fs->cy * w + fs->cx + rnd(5)-2)*4);
     *prgb=(rnd(8)+8)*0x000f0f10;
      return(1);}}    
  if(fs->cshift) --fs->cshift;
  if((fs->cshift+1)%50==0) fs->color = ~fs->color;
  c = fs->color;
  air = fs->air;
  fs->flash *= flash_fade;
  for(n=PIXCOUNT;n;n--){
  if(fp->burn){ --fp->burn; 
  if(fs->special){
  fp->x += fp->xv = fp->xv * air + (float)(rnd(200)-100)/2000;
  fp->y += fp->yv = fp->yv * air + (float)(rnd(200)-100)/2000 + adg; }
  else{
  fp->x += fp->xv = fp->xv * air + (float)(rnd(200)-100)/20000;
  fp->y += fp->yv = fp->yv * air + adg; }
  if(fp->y > h){
  if(rnd(5)==3) {fp->yv *= -0.24; fp->y = h;}
  else fp->burn=0;} /* touch muddy ground :) */
  if(fp->x < w && fp->x > 0 && fp->y < h && fp->y > 0){
     prgb = (unsigned int *)(palaka + ((int)fp->y * w + (int)fp->x)*4);
    *prgb = c; }
  } fp++;
  } return(--fs->life); }

void recycle(fireshell *fs,int x,int y){
  unsigned int n,pixlife;
  firepix *fp = fs->fpix;
  fs->vgn = shoot;
  fs->shy = y;
  fs->cx = x;
  fs->cy = shoot ? height : y ;
  fs->color = (rnd(155)+100) <<16 |
              (rnd(155)+100) <<8  |
               rnd(255);
  fs->life = rnd(rndlife)+minlife;
  fs->air  = 1-(float)(rnd(200))/10000;
  fs->flash   = rnd(30000)+15000; /* million jouls */
  fs->cshift  = !rnd(5) ? 120:0; 
  fs->special = !rnd(10) ? 1:0; 
  if(verbose)
  printf("recycle(): color = %x air = %f life = %d \n",fs->color,fs->air,fs->life);
  pixlife = rnd(fs->life)+fs->life/10+1;    /* ! */
  for(n=0;n<PIXCOUNT;n++){
  fp->burn = rnd(pixlife)+32;
  fp->xv = POWER*(float)(rnd(20000)-10000)/10000;
  fp->yv = sqrt(POWER*POWER - fp->xv * fp->xv) *
               (float)(rnd(20000)-10000)/10000;
  fp->x = x;
  fp->y = y; 
  fp++;             }}

void glow(void){
  unsigned int n,q;
  unsigned int w = width;
  unsigned int h = height;
  unsigned char *pa, *pb, *pm, *po;
  pm = palaka1;
  po = palaka2;
  for(n=0;n<w*4;n++) 
  {pm[n]=0; po[n]=0;}   /* clean first line */
  pm+=n; po+=n; h-=2; 
  pa = pm-(w*4);
  pb = pm+(w*4);
  for(n=4;n<w*h*4-4;n++){
  q    = pm[n-4] + (pm[n]*8) + pm[n+4] + 
         pa[n-4] + pa[n] + pa[n+4] + 
         pb[n-4] + pb[n] + pb[n+4];
  q    -= q>8 ? 8:q;
  pm[n] = q/16;
  q     = q/8;
  if(q>255) q=255;
  po[n] = q;}
  pm+=n; po+=n;
  for(n=0;n<w*4;n++)
  {pm[n]=0; po[n]=0;}}   /* clean last line */

void blur(void){
  unsigned int n,q;
  unsigned int w = width;
  unsigned int h = height;
  unsigned char *pa, *pb, *pm;
  pm = palaka1;
  pm += w*4; h-=2;  /* line 0&h */
  pa = pm-(w*4);
  pb = pm+(w*4);
  for(n=4;n<w*h*4-4;n++){
  q    = pm[n-4] + (pm[n]*8) + pm[n+4] + 
         pa[n-4] + pa[n] + pa[n+4] + 
         pb[n-4] + pb[n] + pb[n+4];
  q    -= q>8 ? 8:q;
  pm[n] = q>>4;}
  pm += n-4;    /* last line */
  for(n=0;n<w*4+4;n++) pm[n]=0;
  pm = palaka1; /* first line */
  for(n=0;n<w*4+4;n++) pm[n]=pm[n+w*4];}

void light_2x2(fireshell *fss){
  unsigned int l,t,n,x,y;
  float s;
  int w = width;
  int h = height;
  unsigned char *dim = palaka2;
  unsigned char *sim = palaka1;
  int nl = w*4;
  fireshell *f;
  if(glow_mode) sim=dim;
  for(y=0;y<h;y+=2){
  for(x=0;x<w;x+=2){
  f = fss; s = 0;
  for(n=SHELLCOUNT;n;n--,f++){
  s += f->flash/(sqrt(1+(f->cx - x)*(f->cx - x)+
                        (f->cy - y)*(f->cy - y)));}
  l = s;

  t = l + sim[0];
  dim[0] = (t > 255 ? 255 : t);	
  t = l + sim[1];
  dim[1] = (t > 255 ? 255 : t);
  t = l + sim[2];
  dim[2] = (t > 255 ? 255 : t);

  t = l + sim[4];
  dim[4] = (t > 255 ? 255 : t);
  t = l + sim[5];
  dim[5] = (t > 255 ? 255 : t);
  t = l + sim[6];
  dim[6] = (t > 255 ? 255 : t);

  t = l + sim[nl+0];
  dim[nl+0] = (t > 255 ? 255 : t);
  t = l + sim[nl+1];
  dim[nl+1] = (t > 255 ? 255 : t);
  t = l + sim[nl+2];
  dim[nl+2] = (t > 255 ? 255 : t);

  t = l + sim[nl+4];
  dim[nl+4] = (t > 255 ? 255 : t);
  t = l + sim[nl+5];
  dim[nl+5] = (t > 255 ? 255 : t);
  t = l + sim[nl+6];
  dim[nl+6] = (t > 255 ? 255 : t);

  sim += 8; dim += 8; } sim += nl; dim += nl;}}

void resize(void){
  XWindowAttributes xwa;
  XGetWindowAttributes (display, win, &xwa);
  xwa.width  -= xwa.width % 2;
  xwa.height -= xwa.height % 2;
  if(xwa.height != height || xwa.width != width) {
  width  = xwa.width;
  height = xwa.height;
  if (verbose)
  printf("sky size: %dx%d ------------------------------\n",width,height);
  if (xim) {
  if (xim->data==(char *)palaka2) xim->data=NULL;  
  XDestroyImage(xim);
  if (palaka2!=palaka1) free(palaka2);
  free(palaka1); 
  }
  palaka1 = NULL;     
  palaka2 = NULL; 
  xim = XCreateImage(display, xwa.visual, xwa.depth, ZPixmap, 0, 0,
		     width, height, 8, 0);
  palaka1 = calloc(xim->height+1,xim->width*4)+8; 
  if(flash_on|glow_mode)
  palaka2 = calloc(xim->height+1,xim->width*4)+8;
  else
  palaka2 = palaka1;
  if (depth>=24)
  xim->data = (char *)palaka2;
  else
  xim->data = calloc(xim->height,xim->bytes_per_line);       
}}

void sniff_events(fireshell *fss){
  XEvent e;
  while (XPending(display)){
  XNextEvent (display, &e);
  if (e.type == ConfigureNotify) resize();
  if (e.type == ButtonPress)     recycle(fss,e.xbutton.x, e.xbutton.y); }}

void put_image(Display *display, Window win, GC gc, XImage *xim){
  int x,y,i,j;
  unsigned char r, g, b;
  i = 0;
  j = 0;
  if (depth==16) {
     if(bigendian)
     for (y=0;y<xim->height; y++)
     for (x=0;x<xim->width; x++) {
     r = palaka2[j++];
     g = palaka2[j++];
     b = palaka2[j++];
     j++;
     xim->data[i++] = (g&224)>>5 | (r&248);
     xim->data[i++] = (b&248)>>3 | (g&28)<<3;
     }
     else
     for (y=0;y<xim->height; y++)
     for (x=0;x<xim->width; x++) {
     r = palaka2[j++];
     g = palaka2[j++];
     b = palaka2[j++];
     j++;
     xim->data[i++] = (b&248)>>3 | (g&28)<<3;
     xim->data[i++] = (g&224)>>5 | (r&248);
     }
  }
  if (depth==15) {
     if(bigendian)
     for (y=0;y<xim->height; y++)
     for (x=0;x<xim->width; x++) {
     r = palaka2[j++];
     g = palaka2[j++];
     b = palaka2[j++];
     j++;
     xim->data[i++] = (g&192)>>6 | (r&248)>>1;
     xim->data[i++] = (b&248)>>3 | (g&56)<<2;
     }
     else
     for (y=0;y<xim->height; y++)
     for (x=0;x<xim->width; x++) {
     r = palaka2[j++];
     g = palaka2[j++];
     b = palaka2[j++];
     j++;
     xim->data[i++] = (b&248)>>3 | (g&56)<<2;
     xim->data[i++] = (g&192)>>6 | (r&248)>>1;
     }
  }
  if (depth==8) {
     for (y=0;y<xim->height; y++)
     for (x=0;x<xim->width; x++) {
     r = palaka2[j++];
     g = palaka2[j++];
     b = palaka2[j++];
     j++;     
     xim->data[i++] = col_val[
       (((7*g)/256)*36)+(((6*r)/256)*6)+((6*b)/256)];
     }
  }
  XPutImage(display,win,gc,xim,0,0,0,0,xim->width,xim->height); 
}

void help_out(){
  printf("usage: fireworkx [-root] [-noflash] [-noglow] [-fps] [-violent] [-verbose]\n\n");
  printf("  -root     use root window\n");
  printf("  -noflash  turn off light effect (runs faster)\n");
  printf("  -noglow   turn off glow  effect (dull blasts..!)\n");
  printf("  -violent  increase frequency of explosions\n");
  printf("  -fps      display frames per second\n");
  printf("  -help     display usage info\n");
  printf("  -verbose  for scientific purposes..!\n\n");}

int main(int argc,char *argv[]){
  unsigned int n,q,ti,tii,sec,ft,fn;
  int inroot=0;
  Visual *vi;
  Colormap cmap;
  XSetWindowAttributes swa;
  GC gc;
  XGCValues gcv;
  firepix *ffpix, *fpix;
  fireshell *ffshell, *fshell;
  fprintf(stderr,"%s %s - pyrotechnics simulation program \n", APPNAME, VERSION);
  fprintf(stderr,"Copyright (c) 1999-2005 Rony B Chandran \n\n");
  fprintf(stderr,"url: http://www.ronybc.8k.com \n\n");
  for(n=1;n<argc;n++){
  if(strcmp(argv[n],"-root")==0) inroot=1;
  else
  if(strcmp(argv[n],"-violent")==0){rndlife=RNDLIFE0;minlife=MINLIFE0;}
  else
  if(strcmp(argv[n],"-noflash")==0) flash_on=0;
  else
  if(strcmp(argv[n],"-noglow") ==0) glow_mode=0;
  else
  if(strcmp(argv[n],"-verbose")==0) verbose=1;
  else
  if(strcmp(argv[n],"-fps")==0) fps_on=1;
  else
  if(strcmp(argv[n],"-shoot")==0) shoot=1;
  else 
  {help_out(); exit(0);}
  }
  if(rndlife==RNDLIFE0) flash_fade=0.98;
  
  display = XOpenDisplay(0);
  if(!display){ 
  fprintf(stderr,"\nError.. Failed to open display \n\n"); exit(0);}

  vi=DefaultVisual(display, DefaultScreen(display));
  depth=DefaultDepth(display, DefaultScreen(display));
  bigendian=(ImageByteOrder(display) == MSBFirst);
  if(depth<=8) {
  fprintf(stderr, "Pseudocolor visual: '%s' will give poor results !\n",APPNAME);}
  if (verbose) {
  fprintf (stderr,"depth = %d \n", depth);
  fprintf (stderr,"bits_per_rgb = %d \n", vi->bits_per_rgb);
  fprintf (stderr,"map_entries = %d \n", vi->map_entries);
  fprintf (stderr,"red_mask = %.6lx \n", vi->red_mask);
  fprintf (stderr,"green_mask = %.6lx \n", vi->green_mask);
  fprintf (stderr,"blue_mask = %.6lx \n", vi->blue_mask);
  }
   
  cmap = XCreateColormap(display, RootWindow(display, DefaultScreen(display)),
                         vi, AllocNone);
  if (depth==8) {
     XColor xc;
     xc.flags = DoRed | DoGreen | DoBlue;
     col_val = malloc(256);
     for (n=0;n<252;n++) {
	xc.red = (unsigned short)(((n%36)/6)*51*257);
	xc.green = (unsigned short)((n/36)*42*257);
	xc.blue = (unsigned short)((n%6)*51*257);
	if (!XAllocColor(display, cmap, &xc)) {
	  fprintf(stderr, "XAllocColor failed for entry %%%d\n", n);
	  col_val[n] = 0;
	} else
	  col_val[n] = xc.pixel;
     }
  }
	
  swa.colormap = cmap;
  swa.border_pixel = 0;
  swa.event_mask = StructureNotifyMask;
  if (inroot)
  win = RootWindow(display, DefaultScreen(display));
  else {	
  win = XCreateWindow(display, RootWindow(display, DefaultScreen(display)),
                      0, 0, WIDTH, HEIGHT, 0, depth, InputOutput, vi,
                      CWBorderPixel|CWColormap|CWEventMask, &swa);
  XSelectInput(display, win, StructureNotifyMask|ButtonPressMask);
  XStoreName(display, win, APPNAME);
  XMapWindow(display, win);
  }

  if (depth == 16 && vi->green_mask==992) depth = 15;

  gc = XCreateGC(display, win, 0, &gcv);

  resize();   /* initialize palakas */ 

  srandom(time(0));
  
  ffpix = malloc(sizeof(firepix) * PIXCOUNT * SHELLCOUNT);
  ffshell = malloc(sizeof(fireshell) * SHELLCOUNT);
  fshell = ffshell;
  fpix = ffpix;
  for (n=0;n<SHELLCOUNT;n++){
  fshell->fpix = fpix;
  recycle (fshell,rnd(width),rnd(height));
  fshell++; 
  fpix += PIXCOUNT; }
  
  fn=0; ft=0; sec=1; ti=time(0);

  while(1) {

    for(q=FTWEAK;q;q--){
    fshell=ffshell;
    for(n=SHELLCOUNT;n;n--){
    if (!explode(fshell)){
         recycle(fshell,rnd(width),rnd(height)); }
         fshell++; }}
#if HAVE_X86_MMX
    if(glow_mode) mmx_glow(palaka1,width,height,8,palaka2);
#else
    if(glow_mode) glow();
#endif
    if(flash_on) light_2x2(ffshell);
    put_image(display,win,gc,xim);
    /*usleep(10000);*/
    XSync(display,0);
    sniff_events(ffshell); 
#if HAVE_X86_MMX
    if(!glow_mode) mmx_blur(palaka1,width,height,8);
#else
    if(!glow_mode) blur();
#endif
    if(fps_on){
    fn++; tii=time(0);
    if(tii!=ti){ ft+=fn;
    printf("fps(avg) = %f  last = %d \n",(float)ft/sec,fn);
    ti=tii; fn=0; sec++;}} }

}
