/*
 * Copyright (c) 2006-2006 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: dagsnap_config.c 4038 2006-04-03 05:14:25Z koryn $
 */

/* File header. */
#include "dagsnap_config.h"

/* Endace headers. */
#include "dag_platform.h"
#include "dagutil.h"


typedef struct
{
	uint32_t magic;
	const char* device;
	const char* outfile_name;
	int32_t dagstream;
	uint32_t data_bytes;
	uint32_t run_seconds;
	uint8_t verbosity;
	uint8_t max_performance;
	
} dagsnap_config_t, * DagsnapConfigHeaderPtr;


static const unsigned int kDagsnapConfigMagicNumber = 85;


/* Internal routines. */
static unsigned int valid_dagsnap_config(DagsnapConfigHeaderPtr header);


/* Implementation of internal routines. */
static unsigned int
valid_dagsnap_config(DagsnapConfigHeaderPtr header)
{
	if (header && (kDagsnapConfigMagicNumber == header->magic))
	{
		return 1;
	}
	
	assert(0); /* Invalid header or magic number. */
	return 0;
}


DagsnapConfigPtr
dagsnap_config_init(void)
{
	DagsnapConfigHeaderPtr bean = (DagsnapConfigHeaderPtr) dagutil_malloc(sizeof(*bean));
	
	memset(bean, 0, sizeof(*bean));
	bean->magic = kDagsnapConfigMagicNumber;
	
	return (DagsnapConfigPtr) bean;
}


DagsnapConfigPtr
dagsnap_config_dispose(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return NULL;
	}
	
	if (header->device)
	{
		dagutil_free((char*) header->device);
		header->device = NULL;
	}
	
	if (header->outfile_name)
	{
		dagutil_free((char*) header->outfile_name);
		header->outfile_name = NULL;
	}
	
	dagutil_free(header);
	return NULL;
}


const char*
dagsnap_config_get_device(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return NULL;
	}
	
	return header->device;
}


void
dagsnap_config_set_device(DagsnapConfigPtr bean, const char* value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	if (header->device);
	{
		dagutil_free((char*) header->device);
		header->device = NULL;
	}
	
	header->device = dagutil_strdup(value);
}


const char*
dagsnap_config_get_outfile_name(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return NULL;
	}
	
	return header->outfile_name;
}


void
dagsnap_config_set_outfile_name(DagsnapConfigPtr bean, const char* value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	if (header->outfile_name);
	{
		dagutil_free((char*) header->outfile_name);
		header->outfile_name = NULL;
	}
	
	header->outfile_name = dagutil_strdup(value);
}


int32_t
dagsnap_config_get_dagstream(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return 0;
	}
	
	return header->dagstream;
}


void
dagsnap_config_set_dagstream(DagsnapConfigPtr bean, int32_t value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	header->dagstream = value;
}


uint32_t
dagsnap_config_get_data_bytes(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return 0;
	}
	
	return header->data_bytes;
}


void
dagsnap_config_set_data_bytes(DagsnapConfigPtr bean, uint32_t value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	header->data_bytes = value;
}


uint32_t
dagsnap_config_get_run_seconds(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return 0;
	}
	
	return header->run_seconds;
}


void
dagsnap_config_set_run_seconds(DagsnapConfigPtr bean, uint32_t value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	header->run_seconds = value;
}


uint8_t
dagsnap_config_get_verbosity(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return 0;
	}
	
	return header->verbosity;
}


void
dagsnap_config_set_verbosity(DagsnapConfigPtr bean, uint8_t value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	header->verbosity = value;
}


uint8_t
dagsnap_config_get_max_performance(DagsnapConfigPtr bean)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return 0;
	}
	
	return header->max_performance;
}


void
dagsnap_config_set_max_performance(DagsnapConfigPtr bean, uint8_t value)
{
	DagsnapConfigHeaderPtr header = (DagsnapConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagsnap_config(header))
	{
		return;
	}
	
	header->max_performance = value;
}
