/*
 * Copyright (c) 2006-2006 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: dagflood_config.c 4038 2006-04-03 05:14:25Z koryn $
 */

/* dagflood headers. */
#include "dagflood_config.h"

/* Endace headers. */
#include "dag_platform.h"
#include "dagutil.h"


typedef struct
{
	uint32_t magic;
	const char* device;
	const char* infile_name;
	int32_t dagstream;
	uint32_t burst_max;
	uint32_t inter_burst_microseconds;
	uint32_t repetitions;
	uint32_t run_seconds;
	uint8_t api;
	uint8_t run_mode;
	uint8_t verbosity;
	uint8_t flush_tx_buffer;
	
} dagflood_config_t, * DagfloodConfigHeaderPtr;


static const unsigned int kDagfloodConfigMagicNumber = 86;


/* Internal routines. */
static unsigned int valid_dagflood_config(DagfloodConfigHeaderPtr header);


/* Implementation of internal routines. */
static unsigned int
valid_dagflood_config(DagfloodConfigHeaderPtr header)
{
	if (header && (kDagfloodConfigMagicNumber == header->magic))
	{
		return 1;
	}
	
	assert(0); /* Invalid header or magic number. */
	return 0;
}


DagfloodConfigPtr
dagflood_config_init(void)
{
	DagfloodConfigHeaderPtr bean = (DagfloodConfigHeaderPtr) dagutil_malloc(sizeof(*bean));
	
	memset(bean, 0, sizeof(*bean));
	bean->magic = kDagfloodConfigMagicNumber;
	
	return (DagfloodConfigPtr) bean;
}


DagfloodConfigPtr
dagflood_config_dispose(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return  NULL;
	}
	
	if (header->device)
	{
		dagutil_free((char*) header->device);
		header->device = NULL;
	}
	
	if (header->infile_name)
	{
		dagutil_free((char*) header->infile_name);
		header->infile_name = NULL;
	}
	
	dagutil_free(header);
	return NULL;
}


const char*
dagflood_config_get_device(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return NULL;
	}
	
	return header->device;
}


void
dagflood_config_set_device(DagfloodConfigPtr bean, const char* value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	if (header->device);
	{
		dagutil_free((char*) header->device);
		header->device = NULL;
	}
	
	header->device = dagutil_strdup(value);
}


const char*
dagflood_config_get_infile_name(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return NULL;
	}
	
	return header->infile_name;
}


void
dagflood_config_set_infile_name(DagfloodConfigPtr bean, const char* value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	if (header->infile_name);
	{
		dagutil_free((char*) header->infile_name);
		header->infile_name = NULL;
	}
	
	header->infile_name = dagutil_strdup(value);
}


int32_t
dagflood_config_get_dagstream(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->dagstream;
}


void
dagflood_config_set_dagstream(DagfloodConfigPtr bean, int32_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->dagstream = value;
}


uint32_t
dagflood_config_get_burst_max(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->burst_max;
}


void
dagflood_config_set_burst_max(DagfloodConfigPtr bean, uint32_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->burst_max = value;
}


uint32_t
dagflood_config_get_inter_burst_microseconds(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->inter_burst_microseconds;
}


void
dagflood_config_set_inter_burst_microseconds(DagfloodConfigPtr bean, uint32_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->inter_burst_microseconds = value;
}


uint32_t
dagflood_config_get_repetitions(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->repetitions;
}


void
dagflood_config_set_repetitions(DagfloodConfigPtr bean, uint32_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->repetitions = value;
}


uint32_t
dagflood_config_get_run_seconds(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->run_seconds;
}


void
dagflood_config_set_run_seconds(DagfloodConfigPtr bean, uint32_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->run_seconds = value;
}


uint8_t
dagflood_config_get_api(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->api;
}


void
dagflood_config_set_api(DagfloodConfigPtr bean, uint8_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->api = value;
}


uint8_t
dagflood_config_get_run_mode(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->run_mode;
}


void
dagflood_config_set_run_mode(DagfloodConfigPtr bean, uint8_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->run_mode = value;
}


uint8_t
dagflood_config_get_verbosity(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->verbosity;
}


void
dagflood_config_set_verbosity(DagfloodConfigPtr bean, uint8_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->verbosity = value;
}


uint8_t
dagflood_config_get_flush_tx_buffer(DagfloodConfigPtr bean)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return 0;
	}
	
	return header->flush_tx_buffer;
}


void
dagflood_config_set_flush_tx_buffer(DagfloodConfigPtr bean, uint8_t value)
{
	DagfloodConfigHeaderPtr header = (DagfloodConfigHeaderPtr) bean;
	
	assert(header);
	
	if (0 == valid_dagflood_config(header))
	{
		return;
	}
	
	header->flush_tx_buffer = value;
}
