#include "../../include/dagapi.h"
#include "../../include/dag_component.h"
#include "../../include/dag_config.h"
#include "../../include/dag_config_util.h"
#include "./token_processing.h"
#include "../../include/dagtoken.h"
#include "../../include/dagutil.h"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>


/* the lex majic still works */
extern char *dagopttext;
//on some versions of yacc or bison this functions are not generated 
//extern char *dagoptget_text  (void);

/* global variables */
/* flag to indicate default should be called on token reset */
uint8_t g_default_on_reset = 1;
/* local functions */

static void set_value_to_lval_enable_port(dag_card_ref_t ref, 
                                          dag_component_t root, int lval, int port_index);
static void set_value_to_lval_multi_port(int port_select,
                                         dag_card_ref_t card_ref, dag_component_t root,
                                         dag_attribute_code_t attr_code, int lval);
static void set_value_to_lval_single_port(dag_card_ref_t card_ref, dag_component_t root,
                                          dag_attribute_code_t attr_code, int lval);
static uint32_t is_change_gpp_slen( int port_select,int iface_count, int gpp_iface_count);

void
dag70ge_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    dag_component_t root = NULL, terf = NULL;
    root = dag_config_get_root_component(card);
    terf = dag_component_get_subcomponent(root, kComponentTerf, 0);

    switch(token)
    {
        case T_DEFAULT:
        {
            dag_config_default(card);
            break;
        }
        case T_GPP_SLEN:
        {
            dag_config_util_set_value_to_lval(card,root, kComponentSRGPP, 0, kUint32AttributeSnaplength, lval);
            break;
        }
        case T_GPP_ALIGN64:
        {
            dag_config_util_set_value_to_lval(card,root, kComponentSRGPP, 0, kBooleanAttributeAlign64, lval);
            break;
        }
        case T_LINK_RESET:
        {
            dag_config_reset(card);
            break;
        }
        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }

        case T_LINK_FCL:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback, lval);
                }
            }
            break;
        }

		case T_LINK_EQL:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, lval);
                }
            }
            break;
        }
        case T_LINK_LASER:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeLaser, lval);
                }
            }
            break;
        }
        case T_ETH_AUTO:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeNic, lval);
                }
            }
            break;
        }
        case T_LINK_SFPPWR:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeSfpPwr,lval);
                }
            }
            break;
        }
        case T_LINK_RKTIOPWR:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeRocketIOPower, lval);
                }
            }
            break;
        }
        case T_TERF_STRIP_16:
        {
           dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip16);
           break; 
        }
        case T_TERF_STRIP_32:
        {
            dag_config_util_set_value_to_lval(card,root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip32);
            break;
        }
        case T_TERF_NOSTRIP:
        {
            dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfNoStrip);
            break;
        }        
    }
}


void
dag71s_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    dag_component_t root_component,gpp;
    root_component = dag_config_get_root_component(card);
    gpp = dag_component_get_subcomponent(root_component, kComponentGpp, 0);
    switch(token)
    {
        case T_DEFAULT:
            dag_config_default(card);
            break;

        case T_LINK_RESET:
            dag_config_reset(card);
            break;

        case T_GPP_SLEN:
            if(gpp) //if GPP is present in this firmware
	    {
		if (lval < 40)
    		{
	    		dagutil_warning ("Snaplength entered is smaller than minimum allowed length of 40 Bytes: %d (setting to 40 Bytes)\n", lval);
	    		lval = 40;
    		}	
    		else if (lval >= 8192)
    		{
	    		dagutil_warning ("Snaplength entered is larger than maximum allowed length of 8 KB: %d (setting to firmware image dependent maximum length)\n", lval);
	    		lval = 8184;
    		}	
				
		dag_config_util_set_value_to_lval_verify(card, root_component, kComponentGpp, 0, kUint32AttributeSnaplength, lval, dagopttext);
	    }
	    else    //Card like 7.1S sometimes donot have a GPP
	    {
                dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, 0, kUint32AttributeHDLCSnaplength, lval);
            }
	    break;

        case T_RAW_SLEN:
            if(gpp) //if GPP is present in this firmware then raw snap length is not available
                printf("Non channelized firmware, raw snap length is not available. \n\n");
            else    //Card like 7.1S sometimes donot have a GPP
                dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, 0, kUint32AttributeRAWSnaplength, lval);
            break;

        case T_GPP_VARLEN:
            if(gpp)
                dag_config_util_set_value_to_lval(card, root_component, kComponentGpp, 0, kBooleanAttributeVarlen, lval);
            else
                printf("No GPP. Option varlen is not supported on this firmware. \n\n");
            break;

        case T_GPP_ALIGN64:
            if(gpp)
                dag_config_util_set_value_to_lval(card, root_component, kComponentGpp, 0, kBooleanAttributeAlign64, lval);
            else
                printf("No GPP. Option align64 is not supported on this firmware. \n\n");
            break;

        case T_TIMESTAMP_HEAD:
        case T_TIMESTAMP_TAIL:
        {
            dag_component_t demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
            attr_uuid_t attr;
            if (!demapper)
                break;
            attr = dag_component_get_attribute_uuid(demapper, kUint32AttributeSonetType);
            if (!attr)
                break;
            if (dag_config_get_uint32_attribute(card, attr) == kSonetTypeChannelized)
            {
                attr = dag_component_get_attribute_uuid(demapper, kBooleanAttributeTimeStampEnd);
                if (strstr(text, "head") != 0)
                {
                    dag_config_set_boolean_attribute(card, attr, 0);
                }
                else if (strstr(text, "tail") != 0)
                {
                    dag_config_set_boolean_attribute(card, attr, 1);
                }
            }
            break;
        }

        case T_ATM_PASS_IDLE:
        {
            int count = 0;
            int i = 0;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kBooleanAttributeIdleCellMode, lval);
                }
            }
        }
        break;

        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }

        case T_GPP_ENABLE_A:
        {
            
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 0);
            assert(port != NULL);
	    //check if the firmware allows setting/unsetting of the active component of this port
	    if (dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,0) == kNullAttributeUuid)
            {
                dagutil_warning("option [dis|en]ablea not supported by this firmware.\n");
		break;
            }
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,0);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_B:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 1);
            assert(port != NULL);
	    //check if the firmware allows setting/unsetting of the active component of this port
	    if (dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,1) == kNullAttributeUuid)
            {
                dagutil_warning("option [dis|en]ableb not supported by this firmware.\n");
		break;
            }
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,1);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_C:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 2);
            assert(port != NULL);
	    //check if the firmware allows setting/unsetting of the active component of this port
	    if (dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,2) == kNullAttributeUuid)
            {
                dagutil_warning("option [dis|en]ablec not supported by this firmware.\n");
		break;
            }
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,2);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_D:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 3);
            assert(port != NULL);
	    //check if the firmware allows setting/unsetting of the active component of this port
	    if (dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,3) == kNullAttributeUuid)
            {
                dagutil_warning("option [dis|en]abled not supported by this firmware.\n");
		break;
            }
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,3);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_PBM_RXONLY:
        {
            dag_component_t pbm_component;
            dag_component_t stream_component;
            attr_uuid_t mem_attribute;
            attr_uuid_t buffer_size_attribute;
            attr_uuid_t stream_count_attribute;
            uint32_t size;
            uint32_t buffer_size;
            int i;
            int stream_count = 0;
            int total_streams = dag_component_get_subcomponent_count_of_type(root_component, kComponentStream);

            pbm_component = dag_component_get_subcomponent(root_component, kComponentPbm, 0);
            assert(pbm_component != NULL);
            
            stream_count_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeRxStreamCount);
            assert(stream_count_attribute != kNullAttributeUuid);
            stream_count = dag_config_get_uint32_attribute(card, stream_count_attribute);

            buffer_size_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeBufferSize);
            assert(buffer_size_attribute != kNullAttributeUuid);
            buffer_size = dag_config_get_uint32_attribute(card, buffer_size_attribute); 

            // De-allocate all memory from the rx/tx streams first
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                dag_config_set_uint32_attribute(card, mem_attribute, 0);
            }

            // Now allocate memory to the rx streams only
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                if((i%2) == 0)
                    size = buffer_size/stream_count;
                else
                    size = 0;

                dag_config_set_uint32_attribute(card, mem_attribute, size);
            }
            break;
        }

        case T_PBM_TXONLY:
        {
            dag_component_t pbm_component;
            dag_component_t stream_component;
            attr_uuid_t mem_attribute;
            attr_uuid_t buffer_size_attribute;
            attr_uuid_t stream_count_attribute;
            uint32_t size;
            uint32_t buffer_size;
            int i;
            int stream_count = 0;
            int total_streams = dag_component_get_subcomponent_count_of_type(root_component, kComponentStream);

            pbm_component = dag_component_get_subcomponent(root_component, kComponentPbm, 0);
            assert(pbm_component != NULL);            
            stream_count_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeTxStreamCount);
            assert(stream_count_attribute != kNullAttributeUuid);
            stream_count = dag_config_get_uint32_attribute(card, stream_count_attribute);

            buffer_size_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeBufferSize);
            assert(buffer_size_attribute != kNullAttributeUuid);                
            buffer_size = dag_config_get_uint32_attribute(card, buffer_size_attribute); 

            // De-allocate all memory from the rx/tx streams first
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                dag_config_set_uint32_attribute(card, mem_attribute, 0);
            }

            // Now allocate memory to the tx streams only
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                if ((i%2) != 0)
                    size = buffer_size/stream_count; 
                else
                    size = 0;
                dag_config_set_uint32_attribute(card, mem_attribute, size);
            }
            break;
        }
       
	case T_PBM_RXTX:
        {
            dag_component_t pbm_component;
            dag_component_t stream_component;
            attr_uuid_t mem_attribute;
            attr_uuid_t buffer_size_attribute;
            uint32_t size;
            uint32_t buffer_size;
            int i;
            int total_streams = dag_component_get_subcomponent_count_of_type(root_component, kComponentStream);

            pbm_component = dag_component_get_subcomponent(root_component, kComponentPbm, 0);
            assert(pbm_component != NULL);            

            buffer_size_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeBufferSize);
            assert(buffer_size_attribute != kNullAttributeUuid);                
            buffer_size = dag_config_get_uint32_attribute(card, buffer_size_attribute); 
			
            // De-allocate all memory from the rx/tx streams first
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                dag_config_set_uint32_attribute(card, mem_attribute, 0);
            }
	   
            // Now allocate memory to the rx and tx streams 
            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                size = buffer_size/total_streams; 
                dag_config_set_uint32_attribute(card, mem_attribute, size);
            }
	    
            break;
        }
	
	case T_LINK_STM0:
        case T_LINK_STM1:
        case T_LINK_STM4:
	case T_LINK_STM16:
	case T_LINK_STM64:
        {
            dag_component_t sonic = NULL;
            int count = 0;
            uint32_t retval;
            attr_uuid_t line_rate_attribute = 0;
            line_rate_t line_rate = kLineRateEthernet10;
	    line_rate_t previous_line_rate;
            int i = 0;

            if (T_LINK_STM0 == token)
                line_rate = kLineRateSTM0;
            else if (T_LINK_STM1 == token)
                line_rate = kLineRateSTM1;
            else if (T_LINK_STM4 == token)
                line_rate = kLineRateSTM4;
	    else if (T_LINK_STM16 == token)
		line_rate = kLineRateSTM16;
	    else if (T_LINK_STM64 == token)
		line_rate = kLineRateSTM64;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    line_rate_attribute = dag_component_get_attribute_uuid(sonic, kUint32AttributeLineRate);
		    previous_line_rate = dag_config_get_uint32_attribute(card, line_rate_attribute);
		    dag_config_set_uint32_attribute(card, line_rate_attribute, line_rate);
		    retval = dag_config_get_uint32_attribute(card, line_rate_attribute);

                    if (retval != line_rate)
		    {
			dag_config_set_uint32_attribute(card, line_rate_attribute, previous_line_rate); 
                        printf("Warning: Unsupported or invalid configuration of port %d for this firmware, card or network mode: %s\n", i, dagopttext);
		    }
                }

            }


            break;
        }


	case T_ETH_10:
        case T_ETH_100:
        case T_ETH_1000:
        {
            dag_component_t sonic = NULL;
            int count = 0;
            uint32_t retval;
            attr_uuid_t line_rate_attribute = 0;
            line_rate_t line_rate = kLineRateEthernet10;
	    line_rate_t previous_line_rate;
            int i = 0;

            if (T_ETH_10 == token)
                line_rate = kLineRateEthernet10;
            else if (T_ETH_100 == token)
                line_rate = kLineRateEthernet100;
            else if (T_ETH_1000 == token)
                line_rate = kLineRateEthernet1000;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    line_rate_attribute = dag_component_get_attribute_uuid(sonic, kUint32AttributeLineRate);
		    previous_line_rate = dag_config_get_uint32_attribute(card, line_rate_attribute);
		    dag_config_set_uint32_attribute(card, line_rate_attribute, line_rate);
		    retval = dag_config_get_uint32_attribute(card, line_rate_attribute);

                    if (retval != line_rate)
		    {
			dag_config_set_uint32_attribute(card, line_rate_attribute, previous_line_rate); 
                        printf("Warning: Unsupported or invalid configuration of port %d for this firmware, card or network mode: %s\n", i, dagopttext);
		    }
                }

            }


            break;
        }

	case T_LINK_OC1:
	case T_LINK_OC3:
	case T_LINK_OC12:
	case T_LINK_OC48: 
        case T_LINK_OC192:
        {
            dag_component_t sonic = NULL;
            int count = 0;
	    uint32_t retval;
            attr_uuid_t line_rate_attribute = 0;
            line_rate_t line_rate = kLineRateOC3c;
	    line_rate_t previous_line_rate;
            int i = 0;

            if (T_LINK_OC1 == token)
                line_rate = kLineRateOC1c;
            else if (T_LINK_OC3 == token)
                line_rate = kLineRateOC3c;
	    else if (T_LINK_OC12 == token)
		line_rate = kLineRateOC12c;
	    else if (T_LINK_OC48 == token)
		line_rate = kLineRateOC48c;
	    else if (T_LINK_OC192 == token)
		line_rate = kLineRateOC192c;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    line_rate_attribute = dag_component_get_attribute_uuid(sonic, kUint32AttributeLineRate);
		    previous_line_rate = dag_config_get_uint32_attribute(card, line_rate_attribute);
                    dag_config_set_uint32_attribute(card, line_rate_attribute, line_rate);
		    retval = dag_config_get_uint32_attribute(card, line_rate_attribute);
		    
		    if (retval != line_rate)
		    {
			dag_config_set_uint32_attribute(card, line_rate_attribute, previous_line_rate);
		    	printf("Warning: Unsupported or invalid configuration of port %d for this firmware, card or network mode: %s\n", i, dagopttext);
		    }
		}
		
            }
            break;                    
        }

        case T_ATM:
        case T_POS:
        {
            dag_component_t demapper = NULL;
            dag_component_t mapper = NULL;
            int count = 0;
            attr_uuid_t network_mode = 0;
            int i = 0;
            network_mode_t nm = kNetworkModePoS;
            attr_uuid_t sonet_type;
            
            if (T_ATM == token)
                nm = kNetworkModeATM;
            else if (T_POS == token)
                nm = kNetworkModePoS;

            demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
            sonet_type = dag_component_get_attribute_uuid(demapper, kUint32AttributeSonetType);
            if (dag_config_get_uint32_attribute(card, sonet_type) != kSonetTypeConcatenated)
            {
                printf("option invalid for this demapper/mapper\n");
            }
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                    if (demapper)
                    {
                        network_mode = dag_component_get_attribute_uuid(demapper, kUint32AttributeNetworkMode);
                        if (network_mode)
                        {
                            dag_config_set_uint32_attribute(card, network_mode, nm);
                        }
                    }
                    else
                    {
                        dagutil_warning("demapper component not found %d!\n", i);
                    }
                }
            }
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentMapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    mapper = dag_component_get_subcomponent(root_component, kComponentMapper, i);
                    if (mapper)
                    {
                        network_mode = dag_component_get_attribute_uuid(mapper, kUint32AttributeNetworkMode);
                        if (network_mode)
                        {
                            dag_config_set_uint32_attribute(card, network_mode, nm);
                        }
                    }
                    else
                    {
                        dagutil_warning("mapper component %d not found!\n", i);
                    }
                }
            }
        }
        break;

        case T_LINK_VC3:
        case T_LINK_VC4:
        case T_LINK_VC4C:
        {
            dag_component_t sonic = NULL;
            attr_uuid_t vc_size = 0;
            int count = 0;
            int i = 0;
            vc_size_t value;

            if (T_LINK_VC4 == token)
                value = kVC4;
            else if (T_LINK_VC4C == token)
                value = kVC4C;
            else
                value = kVC3;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    assert(sonic != NULL);
                    if (sonic)
                    {
                        vc_size = dag_component_get_attribute_uuid(sonic, kUint32AttributeVCSize);
                        assert(vc_size != kNullAttributeUuid);
                        if (vc_size)
                        {
                            dag_config_set_uint32_attribute(card, vc_size, value);
                        }
                        else
                        {
                            dagutil_warning("vc size attribute not found!\n");
                        }
                    }
                    else
                    {
                        dagutil_warning("sonic component %d not found\n", i);
                    }
                }
            }
        }
        break;

        case T_LINK_TU12:
        case T_LINK_TU11:
        {
            dag_component_t sonic = NULL;
            attr_uuid_t tributary_unit = 0;
            int count = 0;
            int i = 0;
            tributary_unit_t value = kTU11;

            if (T_LINK_TU11 == token)
                value = kTU11;
            else if (T_LINK_TU12 == token)
                value = kTU12;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    assert(sonic != NULL);
                    tributary_unit = dag_component_get_attribute_uuid(sonic, kUint32AttributeTributaryUnit);
                    assert(tributary_unit != kNullAttributeUuid);
                    dag_config_set_uint32_attribute(card, tributary_unit, value);
                }
            }
        }
        break;

        case T_SONET_SCRAMBLE:
        {
            int i = 0;
            int count = 0;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentSonic, i, kBooleanAttributeScramble, lval);
                }
            }
        }
        break;

        case T_PBM_OVERLAP:
        {
            dag_config_util_set_value_to_lval(card, root_component, kComponentPbm, 0, kBooleanAttributeOverlap, lval);
            break;
        }

        case T_LINK_LASER:
        {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentOptics, i, kBooleanAttributeLaser, lval);
                }
           }
        }
        break;

        case T_LINK_SFPPWR:
        {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentOptics, i, kBooleanAttributeSfpPwr, lval);
                }
           }
        }
        break;

        case T_LINK_BIT_SYNC:
        case T_LINK_ASYNC1:
        case T_LINK_BYTE_SYNC1:
        case T_LINK_BYTE_SYNC2:
        {
            dag_component_t sonic;
            attr_uuid_t payload_mapping;
            payload_mapping_t value = kPayloadMappingAsync;
            int i = 0;
            int count = 0;

            if (T_LINK_BYTE_SYNC2 == token)
                value = kPayloadMappingByteSync2;
            else if (T_LINK_BYTE_SYNC1 == token)
                value = kPayloadMappingByteSync1;
            else if (T_LINK_BIT_SYNC == token)
                value = kPayloadMappingBitSync;
            else if (T_LINK_ASYNC1 == token)
                value = kPayloadMappingAsync;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    assert(sonic != NULL);
                    if (sonic)
                    {
                        payload_mapping = dag_component_get_attribute_uuid(sonic, kUint32AttributePayloadMapping);
                        assert(payload_mapping != kNullAttributeUuid);
                        dag_config_set_uint32_attribute(card, payload_mapping, value); 
                    }
                }
            }
        }
        break;

        case T_POS_SCRAMBLE:
        {
            dag_component_t demapper;
            dag_component_t mapper;
            attr_uuid_t payload_scramble;
            int i = 0;
            int count = 0;
            uint8_t value;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                    mapper = dag_component_get_subcomponent(root_component, kComponentMapper, i);
                    if (demapper)
                    {
                        payload_scramble = dag_component_get_attribute_uuid(demapper, kBooleanAttributePayloadScramble);
                        assert(payload_scramble != kNullAttributeUuid);
                        if (payload_scramble)
                        {
                            value = !(lval == 0);
                            dag_config_set_boolean_attribute(card, payload_scramble, value); 
                        }
                    }
                    if (mapper)
                    {
                        payload_scramble = dag_component_get_attribute_uuid(mapper, kBooleanAttributePayloadScramble);
                        assert(payload_scramble != kNullAttributeUuid);
                        if (payload_scramble)
                        {
                            value = !(lval == 0);
                            dag_config_set_boolean_attribute(card, payload_scramble, value); 
                        }
                    }
                }
            }
        }
        break;

        case T_LINK_EQL:
        {
            dag_config_util_set_value_to_lval(card, root_component, kComponentPhy, 0, kBooleanAttributeEquipmentLoopback, lval);
        }
        break;

        case T_LINK_FCL:
        {
            dag_config_util_set_value_to_lval(card, root_component, kComponentPhy, 0, kBooleanAttributeFacilityLoopback, lval);
        }
        break;

        case T_SLAVE:
        case T_MASTER:
        {
            dag_component_t port;
            int count;
            attr_uuid_t master_slave;
            int i = 0;
            master_slave_t ms = kSlave;

            if (token == T_SLAVE)
                ms = kSlave;
            else if (token == T_MASTER)
                ms = kMaster;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    port = dag_component_get_subcomponent(root_component, kComponentPort, i);
                    assert(port);
                    if (port)
                    {
                        master_slave = dag_component_get_attribute_uuid(port, kUint32AttributeMasterSlave);
                        assert(master_slave);
                        if (master_slave)
                        {
                            dag_config_set_uint32_attribute(card, master_slave, ms);
                        }
                    }
                }
            }
        }
        break;
        case T_LINK_E1:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeE1);
                }
           }           
           break;
           }
        case T_LINK_E1_CRC:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeE1crc);
                }
           }           
           break;
           }           
        case T_LINK_E1_UNFRAMED:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeE1unframed);
                }
           }           
           break;
           }           
        case T_LINK_T1_ESF:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeT1esf);
                }
           }           
           break;
           }           
        case T_LINK_T1_SF:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeT1sf);
                }
           }           
           break;
           }           
        case T_LINK_T1_UNFRAMED:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentE1T1);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentE1T1, i, kUint32AttributeLineType, kLineTypeT1unframed);
                }
           }           
           break;
           }           
        case T_POS_CRC16:
           {
            int i = 0;
            int count = 0;
            dag_component_t demapper_component;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
			if(count<1){	//for sanity check
				dagutil_warning("No demapper component found in this firmware.\n");
				break;
			}
			demapper_component = dag_component_get_subcomponent(root_component, kComponentDemapper,0);
			
			if (dag_component_get_config_attribute_uuid(demapper_component, kUint32AttributeCrcSelect) == kNullAttributeUuid)
            {
                dagutil_warning("option crc16 is invalid for this demapper.\n");
				break;
            }
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, kCrc16);
                    dag_config_util_set_value_to_lval(card, root_component, kComponentMapper, i, kUint32AttributeCrcSelect, kCrc16);
                }
           }           
           break;
           }          
        case T_POS_CRC32:
           {
            int i = 0;
            int count = 0;
            dag_component_t demapper_component;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
			if(count<1){	//for sanity check
				dagutil_warning("No demapper component found in this firmware.\n");
				break;
			}
			//check if the CRC16 option is valid for this demapper
			//Assumption: only need to check for the demapper at index 0. 

			demapper_component = dag_component_get_subcomponent(root_component, kComponentDemapper,0);
			
			if (dag_component_get_config_attribute_uuid(demapper_component, kUint32AttributeCrcSelect) == kNullAttributeUuid)
            {
                dagutil_warning("option crc32 is invalid for this demapper.\n");
				break;
            }

            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, kCrc32);
                    dag_config_util_set_value_to_lval(card, root_component, kComponentMapper, i, kUint32AttributeCrcSelect, kCrc32);
                }
           }           
           break;
           }          
        case T_LINK_CRC:
           {
            int i = 0;
            int count = 0;
            crc_t crc_type;
			dag_component_t demapper_component;
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
			if(count<1){	//for sanity check
				dagutil_warning("No demapper component found in this firmware.\n");
				break;
			}
			demapper_component = dag_component_get_subcomponent(root_component, kComponentDemapper,0);
			
			if (dag_component_get_config_attribute_uuid(demapper_component, kUint32AttributeCrcSelect) == kNullAttributeUuid)
            {
                dagutil_warning("crc option is not supported by this firmware.\n");
				break;
            }
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    if(lval == 0)
                    {
                        crc_type = kCrcOff;
                    }
                    else
                    {
                        crc_type = kCrc32;
                    }
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, crc_type);
                    dag_config_util_set_value_to_lval(card, root_component, kComponentMapper, i, kUint32AttributeCrcSelect, crc_type);
                        
                }
           }           
           break;
           }
        case T_TERF_STRIP_16:
        {
           dag_config_util_set_value_to_lval(card, root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip16);
           break; 
        }
        break; 
        case T_TERF_STRIP_32:
        {
            dag_config_util_set_value_to_lval(card,root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip32);
            break;
        }
        case T_TERF_NOSTRIP:
        {
            dag_config_util_set_value_to_lval(card, root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfNoStrip);
            break;
        }
        default:
	    
            dagutil_panic("token %s not supported\n", dagopttext);
            break;
    }

}

#if 0
void
dag50s_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    dag_component_t root_component;
    root_component = dag_config_get_root_component(card);

    switch(token)
    {
        case T_DEFAULT:
            dag_config_default(card);
            break;

        case T_LINK_RESET:
            dag_config_reset(card);
            break;

        case T_GPP_SLEN:
            dag_config_util_set_value_to_lval(card, root_component, kComponentGpp, 0, kUint32AttributeSnaplength, lval);
            break;

        case T_GPP_VARLEN:
            dag_config_util_set_value_to_lval(card, root_component, kComponentGpp, 0, kBooleanAttributeVarlen, lval);
            break;

        case T_GPP_ALIGN64:
            dag_config_util_set_value_to_lval(card, root_component, kComponentGpp, 0, kBooleanAttributeAlign64, lval);
            break;

   
        case T_ATM_PASS_IDLE:
        {
            int count = 0;
            int i = 0;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kBooleanAttributeIdleCellMode, lval);
                }
            }
        }
        break;

        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }

        case T_GPP_ENABLE_A:
        {

            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 0);
            assert(port != NULL);
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,0);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_B:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 1);
            assert(port != NULL);
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,1);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_C:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 2);
            assert(port != NULL);
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,2);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_GPP_ENABLE_D:
        {
            dag_component_t port;
            attr_uuid_t active;
            uint8_t val;
            port = dag_component_get_subcomponent(root_component, kComponentPort, 3);
            assert(port != NULL);
            active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive,3);
            assert(active != kNullAttributeUuid);
            val = lval == 1;
            dag_config_set_boolean_attribute(card, active, val);
            break;
        }

        case T_PBM_RXONLY:
        {
            dag_component_t pbm_component;
            dag_component_t stream_component;
            attr_uuid_t mem_attribute;
            attr_uuid_t buffer_size_attribute;
            attr_uuid_t stream_count_attribute;
            uint32_t size;
            uint32_t buffer_size;
            int i;
            int stream_count = 0;
            int total_streams = dag_component_get_subcomponent_count_of_type(root_component, kComponentStream);

            pbm_component = dag_component_get_subcomponent(root_component, kComponentPbm, 0);
            assert(pbm_component != NULL);
            
            stream_count_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeRxStreamCount);
            assert(stream_count_attribute != kNullAttributeUuid);
            stream_count = dag_config_get_uint32_attribute(card, stream_count_attribute);

            buffer_size_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeBufferSize);
            assert(buffer_size_attribute != kNullAttributeUuid);                
            buffer_size = dag_config_get_uint32_attribute(card, buffer_size_attribute); 

            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);



                if((i%2) == 0)
                    size = buffer_size/stream_count;
                else
                    size = 0;

                dag_config_set_uint32_attribute(card, mem_attribute, size);
            }
            break;
        }

        case T_PBM_TXONLY:
        {
            dag_component_t pbm_component;
            dag_component_t stream_component;
            attr_uuid_t mem_attribute;
            attr_uuid_t buffer_size_attribute;
            attr_uuid_t stream_count_attribute;
            uint32_t size;
            uint32_t buffer_size;
            int i;
            int stream_count = 0;
            int total_streams = dag_component_get_subcomponent_count_of_type(root_component, kComponentStream);

            pbm_component = dag_component_get_subcomponent(root_component, kComponentPbm, 0);
            assert(pbm_component != NULL);            
            stream_count_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeTxStreamCount);
            assert(stream_count_attribute != kNullAttributeUuid);
            stream_count = dag_config_get_uint32_attribute(card, stream_count_attribute);

            buffer_size_attribute = dag_component_get_attribute_uuid(pbm_component, kUint32AttributeBufferSize);
            assert(buffer_size_attribute != kNullAttributeUuid);                
            buffer_size = dag_config_get_uint32_attribute(card, buffer_size_attribute); 

            for (i = 0; i < total_streams; i++)
            {
                stream_component = dag_component_get_subcomponent(root_component, kComponentStream, i);
                assert(stream_component != NULL);

                mem_attribute = dag_component_get_attribute_uuid(stream_component, kUint32AttributeMem);
                assert(mem_attribute != kNullAttributeUuid);

                if ((i%2) != 0)
                    size = buffer_size/stream_count; 
                else
                    size = 0;
                dag_config_set_uint32_attribute(card, mem_attribute, size);
            }
            break;
        }
        
        case T_LINK_OC1:
        case T_LINK_OC3:
        case T_LINK_OC12:
        case T_LINK_OC48:
        case T_LINK_OC192:
	{
            dag_component_t deframer = NULL;
            int count = 0;
            attr_uuid_t line_rate_attribute = 0;
            line_rate_t line_rate = kLineRateOC3c;
            int i = 0;
            
	    switch (token)
	    {
	       case T_LINK_OC1:
		dagutil_warning("Line rate OC1c not available\n");
		 break;
	       case T_LINK_OC3:
                 line_rate = kLineRateOC3c;
                 break;
	       case T_LINK_OC12:
                 line_rate = kLineRateOC12c;
		 break;
	       case T_LINK_OC48:
                 line_rate = kLineRateOC48c;
		 break;
	       case T_LINK_OC192:
                 line_rate = kLineRateOC192c;
		 break;
	    }	 
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    deframer = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                    line_rate_attribute = dag_component_get_attribute_uuid(deframer, kUint32AttributeLineRate);
                    dag_config_set_uint32_attribute(card, line_rate_attribute, line_rate);
                }
            }
            break;                    
        }

        case T_ATM:
        case T_POS:
        {
            dag_component_t demapper = NULL;
            dag_component_t mapper = NULL;
            int count = 0;
            attr_uuid_t network_mode = 0;
            int i = 0;
            network_mode_t nm = kNetworkModePoS;
            
            if (T_ATM == token)
	    {
		/* ATM mode not supported in this firmware image version */
		dagutil_warning("ATM not supported in this firmare image version\n");
                /*nm = kNetworkModeATM;*/
	    }
            else if (T_POS == token)
                nm = kNetworkModePoS;

            demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                    if (demapper)
                    {
                        network_mode = dag_component_get_attribute_uuid(demapper, kUint32AttributeNetworkMode);
                        if (network_mode)
                        {
                            dag_config_set_uint32_attribute(card, network_mode, nm);
                        }
                        else
                        {
                            dagutil_warning("network mode attribute not found!\n");
                        }
                    }
                    else
                    {
                        dagutil_warning("demapper component not found %d!\n", i);
                    }
                }
            }
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentMapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    mapper = dag_component_get_subcomponent(root_component, kComponentMapper, i);
                    if (mapper)
                    {
                        network_mode = dag_component_get_attribute_uuid(mapper, kUint32AttributeNetworkMode);
                        if (network_mode)
                        {
                            dag_config_set_uint32_attribute(card, network_mode, nm);
                        }
                        else
                        {
                            dagutil_warning("network mode attribute not found!\n");
                        }
                    }
                    else
                    {
                        dagutil_warning("mapper component %d not found!\n", i);
                    }
                }
            }
        }
        break;

        case T_LINK_VC3:
        case T_LINK_VC4:
        case T_LINK_VC4C:
        {
            dag_component_t sonic = NULL;
            attr_uuid_t vc_size = 0;
            int count = 0;
            int i = 0;
            vc_size_t value;

            if (T_LINK_VC4 == token)
                value = kVC4;
            else if (T_LINK_VC4C == token)
                value = kVC4C;
            else
                value = kVC3;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root_component, kComponentSonic, i);
                    assert(sonic != NULL);
                    if (sonic)
                    {
                        vc_size = dag_component_get_attribute_uuid(sonic, kUint32AttributeVCSize);
                        assert(vc_size != kNullAttributeUuid);
                        if (vc_size)
                        {
                            dag_config_set_uint32_attribute(card, vc_size, value);
                        }
                        else
                        {
                            dagutil_warning("vc size attribute not found!\n");
                        }
                    }
                    else
                    {
                        dagutil_warning("sonic component %d not found\n", i);
                    }
                }
            }
        }
        break;

 
        case T_SONET_SCRAMBLE:
	{
	    int i = 0;
	    int count = 0;    
        count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
	    for (i = 0; i < count; i++)
		{
		   if ((port_select & (1 << i)) == (1 << i))
		   {
			   dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kBooleanAttributeScramble, lval);
		   }
	    }
 	 }
	 break;

       case T_LINK_LASER:
        {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentOptics, i, kBooleanAttributeLaser, lval);
                }
           }
        }
        break;

        case T_LINK_SFPPWR:
        {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentOptics, i, kBooleanAttributeSfpPwr, lval);
                }
           }
        }
        break;
 
        case T_POS_SCRAMBLE:
        {
            dag_component_t demapper;
            dag_component_t mapper;
            attr_uuid_t payload_scramble;
            int i = 0;
            int count = 0;
            uint8_t value;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                    mapper = dag_component_get_subcomponent(root_component, kComponentMapper, i);
                    if (demapper)
                    {
                        payload_scramble = dag_component_get_attribute_uuid(demapper, kBooleanAttributePayloadScramble);
                        assert(payload_scramble != kNullAttributeUuid);
                        if (payload_scramble)
                        {
                            value = !(lval == 0);
                            dag_config_set_boolean_attribute(card, payload_scramble, value); 
                        }
                    }
                    if (mapper)
                    {
                        payload_scramble = dag_component_get_attribute_uuid(mapper, kBooleanAttributePayloadScramble);
                        assert(payload_scramble != kNullAttributeUuid);
                        if (payload_scramble)
                        {
                            value = !(lval == 0);
                            dag_config_set_boolean_attribute(card, payload_scramble, value); 
                        }
                    }
                }
            }
        }
        break;

        case T_LINK_EQL:
        {
            int count;
            int i = 0;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
            for (i = 0; i < count; i++)
        { 
                dag_config_util_set_value_to_lval(card, root_component, kComponentPort, i, kBooleanAttributeEquipmentLoopback, lval);
        }
        }
        break;

        case T_LINK_FCL:
        {            
            int count;
            int i = 0;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
            for (i = 0; i < count; i++)
        { 
                dag_config_util_set_value_to_lval(card, root_component, kComponentPort, i, kBooleanAttributeFacilityLoopback, lval);
        }
        }
        break;

        case T_SLAVE:
        case T_MASTER:
        {
            dag_component_t port;
            int count;
            attr_uuid_t master_slave;
            int i = 0;
            master_slave_t ms = kSlave;

            if (token == T_SLAVE)
                ms = kSlave;
            else if (token == T_MASTER)
                ms = kMaster;

            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    port = dag_component_get_subcomponent(root_component, kComponentPort, i);
                    assert(port);
                    if (port)
                    {
                        master_slave = dag_component_get_attribute_uuid(port, kUint32AttributeMasterSlave);
                        assert(master_slave);
                        if (master_slave)
                        {
                            dag_config_set_uint32_attribute(card, master_slave, ms);
                        }
                    }
                }
            }
        }
        break;
    
        case T_POS_CRC16:
           {
            int i = 0;
            int count = 0;
            dag_component_t demapper;
            attr_uuid_t attr;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                demapper = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
                if ((port_select & (1 << i)) == (1 << i))
                {
                    attr = dag_component_get_attribute_uuid(demapper, kUint32AttributeLineRate);
		    		if (dag_config_get_uint32_attribute(card, attr) != kLineRateOC192c) {
                    	dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, kCrc16);
                    } else {
						dagutil_warning("CRC16 invalid for OC192\n");
                    }
                }
           }           
           break;
           }          
        case T_POS_CRC32:
           {
            int i = 0;
            int count = 0;
            
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, kCrc32);
                }
           }           
           break;
           }          
        case T_LINK_CRC:
           {
            int i = 0;
            int count = 0;
            crc_t crc_type;
            count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    if(lval == 0)
                    {
                        crc_type = kCrcOff;
                    }
                    else
                    {
                        crc_type = kCrc32;
                    }
                    dag_config_util_set_value_to_lval(card, root_component, kComponentDemapper, i, kUint32AttributeCrcSelect, crc_type);
                        
                }
           }           
           break;
           }
        case T_TERF_STRIP_16:
        {
           dag_config_util_set_value_to_lval(card, root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip16);
           break; 
        }
        break; 
        case T_TERF_STRIP_32:
        {
            dag_config_util_set_value_to_lval(card,root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip32);
            break;
        }
        case T_TERF_NOSTRIP:
        {
            dag_config_util_set_value_to_lval(card, root_component, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfNoStrip);
            break;
        }
        default:
            dagutil_panic("token %s not supported\n", dagopttext);
            break;
    }

}
#endif

void 
dag62se_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    int count = 0;
    int i = 0;
    dag_component_t root;
    root = dag_config_get_root_component(card);
    switch(token)
    {
        case T_DEFAULT:
            dag_config_default(card);
            break;

        case T_LINK_RESET:
            dag_config_reset(card);
            break;
        case T_POS:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeEthernetMode, kNetworkModePoS);
                }
            }
            break;
        case T_POS_SCRAMBLE:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributePayloadScramble,lval );
                }
            }
            break;
        case T_LINK_LSFCL:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeLineSideFacilityLoopback,lval );
                }
            }            
            break;
        case T_LINK_LSEQL:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeLineSideEquipmentLoopback,lval );
                }
            }             
            break;
        case T_LINK_FCL:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback,lval );
                }
            }
            break;
        case T_LINK_EQL:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback,lval );
                }
            }            
            break;
        case T_ETH:
        case T_ETH_WAN:
        {
            ethernet_mode_t eth_mode;
            if(lval == 0)
                eth_mode = kEthernetMode10GBase_LR;
            else
                eth_mode = kEthernetMode10GBase_LW;
                
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {/* Default to WAN */
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeEthernetMode, eth_mode);
                }
            }            
            break;
        }
        case T_RAW:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeRAW);
                }
            }
            break;
        case T_LINK_CRC:
        {
            crc_t crc_val;
            if(lval == 0)
                crc_val = kCrcOff;
            else
                crc_val = kCrc32;
                
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect,crc_val );
                }
            }            
            break;
        }
        case T_POS_CRCSTRIP:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeCrcStrip,lval );
                }
            }            
            break;
        case T_POS_MINCHECK:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributePMinCheck,lval );
                }
            }            
            break;
        case T_POS_MAXCHECK:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributePMaxCheck,lval );
                }
            }            
            break;
        case T_POS_MINLEN:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeMinPktLen,lval );
                }
            }
            break;
        case T_POS_MAXLEN:
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeMaxPktLen, lval);
                }
            }
            break;
        case T_STEER:
        {
            steer_t steer_mode = kSteerStream0;
            attr_uuid_t any_attr;
            char * subtext;
            dag_component_t port = dag_component_get_subcomponent(root, kComponentPort, 0);
            
            subtext = strstr(text, "=");
            if(subtext == NULL)
                break;
            subtext++; /* step over the = */
            if(strncmp(subtext, "stream0", 7) == 0)
                steer_mode = kSteerStream0;
            else if(strncmp(subtext, "parity",6) == 0)
                steer_mode = kSteerParity;
            else if(strncmp(subtext, "crc",3) == 0)
                steer_mode = kSteerCrc;
            else if(strncmp(subtext, "iface",5) == 0)
                steer_mode = kSteerIface;

            any_attr = dag_component_get_attribute_uuid(port, kUint32AttributeSteer);
            if(any_attr != kNullAttributeUuid)
                dag_config_set_uint32_attribute(card, any_attr, steer_mode);
            break;
        }
            
        case T_GPP_SLEN:
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kUint32AttributeSnaplength, lval);
            break;
        case T_GPP_VARLEN:
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kBooleanAttributeVarlen, lval);
            break;
        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }
    default:
            dagutil_panic("token %s not supported\n", dagopttext);
            break;
            
    }
}


void 
dag37d_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    int count = 0;
    int i = 0;
    dag_component_t root;
    root = dag_config_get_root_component(card);
    switch(token)
    {
        case T_DEFAULT:
	{
            dag_config_default(card);
            break;
	}

	case T_DEFAULT_DS3_ATM:
	{
            dag_config_default(card);
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeATM);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3Cbit);
                }
            }
            break;
	}

        case T_DEFAULT_DS3_HDLC:
	{
            dag_config_default(card);
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeHDLC);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3Cbit);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeLinkDiscard, 1);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFF00Delete, 0);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect, kCrc32);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback, 1);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, 0);
                }
            }
            break;
	}

        case T_DEFAULT_E3_ATM:
	{
            dag_config_default(card);
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeATM);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3);
                }
            }
            break;
	}

        case T_DEFAULT_E3_HDLC:
	{
            dag_config_default(card);
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeHDLC);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3G751IF);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeLinkDiscard, 1);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFF00Delete, 1);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeDescramble, 0);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect, kCrc16);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback, 1);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, 0);
                }
            }
            break;
	}

        case T_DEFAULT_KENTROX:
	{
            dag_config_default(card);
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeHDLC);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3Cbit);
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeLinkDiscard, 0);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFF00Delete, 0);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect, kCrc16);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback, 1);
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, 0);
                }
            }
            break;
	}

        case T_LINK_RESET:
	{
            dag_config_reset(card);
            break;
	}

        case T_LINK_DISCARD:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeLinkDiscard, lval);
                }
            }
            break;
	}

        case T_LINK_EQL:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, lval);
                }
            }
            break;
	}

        case T_LINK_FCL:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback,lval );
                }
            }
            break;
	}

        case T_LINK_CRC:
        {
            crc_t crc_val;
            if(lval == 0)
                crc_val = kCrcOff;
            else
                crc_val = kCrc32;

            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect,crc_val );
                }
            }
            break;
        }

        case T_POS_CRC16:
        {
            crc_t crc_val = kCrc16;

            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect,crc_val );
                }
            }
            break;
        }

        case T_POS_CRC32:
        {
            crc_t crc_val = kCrc32;
                
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kUint32AttributeCrcSelect,crc_val );
                }
            }
            break;
        }
	case  T_POS_SCRAMBLE:
	{
	    count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributePayloadScramble, lval);
                }
            }
            break;

	}
        case T_ATM_DESCRAMBLE:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeDescramble, lval);
                }
            }
            break;
        }

        case T_HDLC:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeHDLC);
                }
            }
            break;
        }

        case T_ATM:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeNetworkMode, kNetworkModeATM);
                }
            }
            break;
        }

        case T_RX_MONITOR_MODE:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeRXMonitorMode, lval);
                }
            }
            break;
        }

        case T_LINK_DS3_M23:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3m23);
                }
            }
            break;
        }

        case T_LINK_DS3_M23_FF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3m23FF);
                }
            }
            break;
        }

        case T_LINK_DS3_M23_EF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3m23EF);
                }
            }
            break;
        }

        case T_LINK_DS3_M23_IF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3m23IF);
                }
            }
            break;
        }

        case T_LINK_DS3_CBIT:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3Cbit);
                }
            }
            break;
        }

        case T_LINK_DS3_CBIT_FF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode,kFramingModeDs3CbitFF);
                }
            }
            break;
        }

        case T_LINK_DS3_CBIT_EF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3CbitEF);
                }
            }
            break;
        }

        case T_LINK_DS3_CBIT_IF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDs3CbitIF);
                }
            }
            break;
        }

        case T_LINK_DS3_CBIT_PLCP:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDS3CbitPLCP);
                }
            }
            break;
        }

        case T_LINK_DS3_M23_PLCP:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeDS3M23PLCP);
                }
            }
            break;
        }

        case T_LINK_E3:
	case T_LINK_E3_G751:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3);
                }
            }
            break;
        }

        case T_LINK_E3_G751_PLCP:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3G751PLCP);
                }
            }
            break;
        }

        case T_LINK_E3_G751_IF:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3G751IF);
                }
            }
            break;
        }

        case T_LINK_E3_CC:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeFramingMode, kFramingModeE3CC);
                }
            }
            break;
	}

        case T_FF_00_DEL:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kBooleanAttributeFF00Delete, lval);
                }
            }
            break;
	}

        case T_LINK_E3_HDLC_FRACT:
	{
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeHDLCFraction, lval);
                }
            }
            break;
	}

        case T_GPP_SLEN:
	{
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kUint32AttributeSnaplength, lval);
            break;
	}

        case T_GPP_VARLEN:
	{
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kBooleanAttributeVarlen, lval);
            break;
	}

        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }

        default:
            dagutil_panic("token %s not supported\n", dagopttext);
            break;

    }
}

#if 0
void 
dag82x_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    int count = 0;
    int i = 0;
    dag_component_t root= NULL;
    root = dag_config_get_root_component(card);
    switch(token)
    {
        case T_DEFAULT:
            dag_config_default(card);
            break;

        case T_LINK_RESET:
            dag_config_reset(card);
            break;

        case T_PBM_OVERLAP:
        {
            dag_config_util_set_value_to_lval(card, root, kComponentPbm, 0, kBooleanAttributeOverlap, lval);
            break;
        }

        case T_LINK_EQL:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeEquipmentLoopback, lval);
                }
            }
            break;
        }
        case T_LINK_FCL:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentPort, i, kBooleanAttributeFacilityLoopback,lval );
                }
            }
            break;
        }
        case T_PBM_DROP:
        {
            dag_config_util_set_value_to_lval(card, root, kComponentPbm, 0, kBooleanAttributeDrop, lval);
            break;
        }
        /*case T_ETH_10GBASESR:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_SR);
                }
            }
            break;
        }        
        case T_ETH_10GBASELR:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_LR);
                }
            }
            break;
        }        
        case T_ETH_10GBASEER:
        {
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_ER);
                }
            }
            break;
        }        
*/
        case T_STEER:
        {
            dag_component_t steer = dag_component_get_subcomponent(root, kComponentSteering, 0);
            if (steer)
            {
                steer_t steer_mode;
                char* subtext;
                attr_uuid_t attr = 0;
                
                attr = dag_component_get_attribute_uuid(steer, kUint32AttributeSteer);
                subtext = strstr(text, "=");
                if(subtext == NULL)
                    break;
                subtext++; /* step over the = */
                if(strncmp(subtext, "stream0", 7) == 0)
                {
                    steer_mode = kSteerStream0;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
                else if(strncmp(subtext, "colour",6) == 0)
                {
                    steer_mode = kSteerColour;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
                else if(strncmp(subtext, "dsm",3) == 0)
                {
                    steer_mode = kSteerColour;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
            }
            break;
    
        }
        
        case T_GPP_SLEN:
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kUint32AttributeSnaplength, lval);
            break;
        case T_GPP_VARLEN:
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kBooleanAttributeVarlen, lval);
            break;
        case T_GPP_ALIGN64:
            dag_config_util_set_value_to_lval(card, root, kComponentGpp, 0, kBooleanAttributeAlign64, lval);
            break;
        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }  		
        case T_HLB_RANGE:
        {
            dag_config_util_do_hlb_config(card,text);
            break;
		}
        case T_TX_CRC:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentXGMII);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 << i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval(card, root, kComponentXGMII, i, kBooleanAttributeTxCrc, lval);
                }
            }
           break; 
        }
        case T_TERF_STRIP_16:
        {
           dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip16);
           break; 
        }
		
        case T_TERF_STRIP_32:
        {
            dag_config_util_set_value_to_lval(card,root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip32);
            break;
        }
        case T_TERF_NOSTRIP:
        {
            dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfNoStrip);
            break;
        }     
        case T_TR_TERF_NO_DELAY:
        {
	    dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTimeMode, kTrTerfNoDelay);
            break; 
        }

        case T_TR_TERF_RELATIVE:
        {
            dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kUint32AttributeTimeMode, kTrTerfRelative);
            break;
        }
        case T_TR_TERF_RX_ERROR:
        {
            dag_component_t any_component;
            dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorA, lval);
            /* FIXME Temporary fix */
            any_component = dag_component_get_subcomponent(root, kComponentTerf,0);
            if (any_component) 
            {
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorB) ) 
                    dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorB, lval);
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorC) ) 
                    dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorC, lval);
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorD) ) 
                    dag_config_util_set_value_to_lval(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorD, lval);
            }
            dag_config_set_last_error(card, kDagErrNone);
            break;
        }
        default:
            dagutil_panic("token %s not supported\n", dagopttext);
            break;
            
    }
}
#endif


bool
token_proc_is_token_supported(dag_card_ref_t card, int token)
{
	switch (token)
	{
		case T_LINK_LASER:
		{
			//dag_card_t card_type = dag_config_get_card_type(card);
			/* Even though firmware on 4.5 cards have this attribute we don't let the users touch it. */
//			return !(card_type == kDag452z8 || card_type == kDag452z || card_type == kDag454e || card_type == kDag452cf || card_type == kDag452e || card_type == kDag452gf);
			//requested to have this option 
			//NOTE: if Copper module used this option is actually the txenable , but when disabled puts the Marvel chip into reset 
			//which kills at least the SMB bus NOTE: not clear is it only the SMBUS in reset or the whole chip including the receive 
			return true;
		}

		case T_LINK_RKTIOPWR:
		{
			//dag_card_t card_type = dag_config_get_card_type(card);
			/* Even though firmware on 4.5 cards have this attribute we don't let the users touch it. */
			//return !(card_type == kDag452z8 || card_type == kDag452z || card_type == kDag454e || card_type == kDag452cf || card_type == kDag452e || card_type == kDag452gf);
			return true;
		}

		case T_LINK_SFPPWR:
		{
			//dag_card_t card_type = dag_config_get_card_type(card);
			/* Even though firmware on 4.5 cards have this attribute we don't let the users touch it. */
			//return !(card_type == kDag452z8 || card_type == kDag452z || card_type == kDag454e || card_type == kDag452cf || card_type == kDag452e || card_type == kDag452gf);
			return true;
		}

		case T_POS_CRC16:
		{
			dag_card_t card_type = dag_config_get_card_type(card);
			/* This token should only be valid for PoS cards, not the 4.5 Ethernet cards but all crc setting options are tied to the same attribute (kUint32AttributeCrcSelect) */
			return !(card_type == kDag452z8 || card_type == kDag452z || card_type == kDag454e || card_type == kDag452cf || card_type == kDag452e || card_type == kDag452gf);
		}

		default:
			return true;
	}
}

/* process token routine for all sorts of cards. there is not much need to have a specialized 
 per card token processing routine for each card. Ideally all cards token processing needs should
 be handled in the function below. This means that when a new card comes along it will reduce the
 need to add per card specific token processing routines.
 */
void
dagx_process_token_routine(dag_card_ref_t card, int token, const char* text, int lval, int port_select)
{
    dag_component_t root = NULL;
    root = dag_config_get_root_component(card);

	/* Not all cards will support the same tokens  */
	if (!token_proc_is_token_supported(card, token))
	{
		dagutil_panic("Token '%s' not supported on dagconfig for this FW/card\n", dagopttext);
	}

    switch (token)
    {
        case T_DEFAULT:
        {
            dag_config_default(card);
            break;
        }
	    case T_INFINIBAND_TAPMODE:
	    {
	        if(dag_config_set_infiniband_tapmode(card) != kDagErrNone)
	        {
		        dagutil_warning("Not able to configure the Card in Tap mode.\n");
	        }
	        else
		        printf("Configured in Tap Mode\n");     	
	        break;
	    }
	    case T_INFINIBAND_ACTIVEMODE:
	    {
		    if(dag_config_set_infiniband_activemode(card)!= kDagErrNone)
		    {
			    dagutil_warning("Not able to configure the card in Active mode \n");
		    }
		    else
		        printf("Configured in Active Mode \n");
		    break;
	    }
	    case T_INFINIBAND_MONITORMODE:
	    {
		    if(dag_config_set_infiniband_monitormode(card)!= kDagErrNone)
		    {
		    	dagutil_warning("Not able to configure the card in Monitor Mode");
		    }
		    else
		        printf("Configured in Monitor Mode \n");
		    break;
	    }
        case T_INTERFACE_SWAP:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentMux, 0, kBooleanAttributeSwap, lval, dagopttext);
            break;
        }

        case T_SLAVE:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeMasterSlave, kSlave);
            break;
        }
        case T_MASTER:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeMasterSlave, kMaster);
            break;
        }
        case T_HLB_RANGE:
        {
            dag_config_util_do_hlb_config(card,text);
            break;
        }
        case T_DUCK_SYNC_THRESH:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentDUCK, 0, kUint32AttributeDUCKThreshold, lval, dagopttext);
            break;
        }
        case T_DUCK_SET_TO_HOST:
        {
            dag_component_t comp = 0;
            if ((comp = dag_component_get_subcomponent(root, kComponentDUCK, 0)))
            {
                attr_uuid_t attr = 0;
                attr = dag_component_get_attribute_uuid(comp, kNullAttributeDUCKSetToHost);
                dag_config_set_null_attribute(card, attr);
            }
            break;
        }
        case T_PBM_OVERLAP:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentPbm, 0, kBooleanAttributeOverlap, lval, dagopttext);
            break;
        }
        case T_GPP_SLEN:
        {
        int i;
        int total_iface_count = 0;
        dag_component_t gpp;
        int gpp_count;

        /* Check if alteast 1 gpp exists in the card. If not break so that error would be set to NoComponent */
        gpp = dag_component_get_subcomponent(root, kComponentGpp, 0);
        if ( NULL == gpp)
        {
            /* error would be printed outside this switch-case. */
            break;
        }
	    if (lval < 40)
	    {
		    dagutil_warning ("Snaplength entered is smaller than minimum allowed length of 40 Bytes: %d (setting to 40 Bytes)\n", lval);
		    lval = 40;
	    }	
	    else if (lval > 16*1024)
	    {
		    dagutil_warning ("Snaplength entered is larger than maximum allowed length of 16 KB: %d (setting to firmware image dependent maximum length)\n", lval);
		    lval = 16*1024;
	    }

	    gpp_count = dag_component_get_subcomponent_count_of_type(root,kComponentGpp);
	    for (i = 0;i < gpp_count;i++)
        {
            int gpp_iface_count = 0;
            attr_uuid_t interface_counter ;
            gpp = dag_component_get_subcomponent(root,kComponentGpp,i);
            interface_counter = dag_component_get_attribute_uuid(gpp, kUint32AttributeInterfaceCount);
            gpp_iface_count = dag_config_get_uint32_attribute(card, interface_counter);
	        /* check whether the snap length of the current gpp is to be changed depending upon the port_select*/
            if ( is_change_gpp_slen( port_select, total_iface_count, gpp_iface_count) )
            {
                dag_config_util_set_value_to_lval_verify(card,root,kComponentGpp,i,kUint32AttributeSnaplength,lval, dagopttext);
            }
            total_iface_count +=  gpp_iface_count;
        }
	    //set_value_to_lval_single_port(card, root, kUint32AttributeSnaplength, lval);
	    break;
        }
        case T_PBM_DROP:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentPbm, 0, kBooleanAttributeDrop, lval, dagopttext);
            break;
        }
        case T_PBM_RXONLY:
        {
            /* assign all memory to the rx stream */
            attr_uuid_t attr;
            dag_component_t pbm;
            uint32_t buffer_size = 0;
            char text[256];

            pbm = dag_component_get_subcomponent(root, kComponentPbm, 0);
            attr = dag_component_get_attribute_uuid(pbm, kUint32AttributeBufferSize);
            buffer_size = dag_config_get_uint32_attribute(card, attr);
            snprintf(text, 256, "mem=%u:0", buffer_size);
            dag_config_util_domemconfig(card, text);
            break;
        }
            
        case T_PBM_TXONLY:
        {
            attr_uuid_t attr;
            dag_component_t pbm;
            uint32_t buffer_size = 0;
            char text[256];

            pbm = dag_component_get_subcomponent(root, kComponentPbm, 0);
            attr = dag_component_get_attribute_uuid(pbm, kUint32AttributeBufferSize);
            buffer_size = dag_config_get_uint32_attribute(card, attr);
            snprintf(text, 256, "mem=0:%u", buffer_size);
            dag_config_util_domemconfig(card, text);
            break;
        }
            
        case T_PBM_RXTX:
        {
            attr_uuid_t attr;
            dag_component_t pbm;
            uint32_t buffer_size = 0;
            char text[256];

            pbm = dag_component_get_subcomponent(root, kComponentPbm, 0);
            attr = dag_component_get_attribute_uuid(pbm, kUint32AttributeBufferSize);
            buffer_size = dag_config_get_uint32_attribute(card, attr);
            snprintf(text, 256, "mem=%u:%u", buffer_size- PBM_TX_DEFAULT_SIZE, PBM_TX_DEFAULT_SIZE);
            dag_config_util_domemconfig(card, text);
            break;
        }

        case T_GPP_VARLEN:
        {
            set_value_to_lval_single_port(card, root, kBooleanAttributeVarlen, lval);
            break;
        }
        case T_GPP_ALIGN64:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentGpp, 0, kBooleanAttributeAlign64, lval, dagopttext);
            break;
        }

        /* Order Important. Here we have some fun with case fall-through
         * portnum was 4. for each case:
         * A: -4 = 0
         * B: -3 = 1
         * C: -2 = 2
         * D: -1 = 3
         * All this to avoid having the same code four times */
        case T_GPP_ENABLE_A:
        {
                set_value_to_lval_enable_port(card, root, lval, 0);
                break;
        }
        case T_GPP_ENABLE_B:
        {
            dag_component_t gpp1;
            attr_uuid_t active;
            uint8_t val;
            uint8_t retval;
	    /* for infiniband cards- it has 2 GPPs */
	    gpp1 = dag_component_get_subcomponent(root, kComponentGpp, 1);
            
            if (gpp1 == NULL)
	    {
               /* no second gpp . So not a multi-gpp card.*/
                set_value_to_lval_enable_port(card, root, lval, 1);
                break;
            }
	    else
            {
               	 /* This modification done for infiniband cards. It has 2 GPP firmware components instead of one.
		  * Each component of the GPP means one port.More precisely interface count of each GPP component 
		  * is 1 which adds up to two ports.So here enable/disable portb means disabling or enabling
		  * kBooleanAttributeActive (attribute index 0 ) of GPP component (component index 1).Hence this implementation*/
		 if(( active = dag_component_get_indexed_attribute_uuid_of_type(gpp1, kBooleanAttributeActive,0) ) == kNullAttributeUuid)
		 {
			dagutil_warning("option [dis|en]able 1 not supported by this firmware.\n");
			break;
		 }
		 assert(active != kNullAttributeUuid);
		 val = lval == 1;
		 dag_config_set_boolean_attribute(card, active, val);

		 // Perform a read back to confirm that the set has worked.
		 retval = dag_config_get_boolean_attribute(card, active);
		 if (retval != lval)
		     dagutil_warning("disabling of port 1 is not valid for this FW/card.\n");

		 break;
	    }
        }
        case T_GPP_ENABLE_C:
        {
             set_value_to_lval_enable_port(card, root, lval, 2);
             break;
        }
        case T_GPP_ENABLE_D:
        {
            set_value_to_lval_enable_port(card, root, lval, 3);
            break;
        }
        case T_GPP_ENABLE_E:
        {
            set_value_to_lval_enable_port(card, root, lval, 4);
            break;
        }
        case T_GPP_ENABLE_F:
        {
            set_value_to_lval_enable_port(card, root, lval, 5);
            break;
        }
        case T_GPP_ENABLE_G:
        {
            set_value_to_lval_enable_port(card, root, lval, 6);
            break;
        }
        case T_GPP_ENABLE_H:
        {
            set_value_to_lval_enable_port(card, root, lval, 7);
            break;
        }
        case T_GPP_ENABLE_I:
        {
            set_value_to_lval_enable_port(card, root, lval, 8);
            break;
        }
        case T_GPP_ENABLE_J:
        {
            set_value_to_lval_enable_port(card, root, lval, 9);
            break;
        }
        case T_GPP_ENABLE_K:
        {
            set_value_to_lval_enable_port(card, root, lval, 10);
            break;
        }
        case T_GPP_ENABLE_L:
        {
            set_value_to_lval_enable_port(card, root, lval, 11);
            break;
        }
        case T_GPP_ENABLE_M:
        {
            set_value_to_lval_enable_port(card, root, lval, 12);
            break;
        }
        case T_GPP_ENABLE_N:
        {
            set_value_to_lval_enable_port(card, root, lval, 13);
            break;
        }
        case T_GPP_ENABLE_O:
        {
            set_value_to_lval_enable_port(card, root, lval, 14);
            break;
        }
        case T_GPP_ENABLE_P:
        {
            set_value_to_lval_enable_port(card, root, lval, 15);
            break;
        }
        case T_LINK_RESET:
        {
            dag_config_reset(card);
            if(g_default_on_reset)
            {
                dag_config_default(card);
            }
            break;
        }
        case T_PBM_MEMCONFIG:
        {
            dag_config_util_domemconfig(card, text);
            break;
        }
        case T_LINK_TERM75:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeTermination, kTermination75ohm);
            break;
        }
        case T_LINK_TERM100:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeTermination, kTermination100ohm);
            break;
        }
        case T_LINK_TERM120:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeTermination, kTermination120ohm);
            break;
        }
        case T_LINK_TERMEXT:
        {
             set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeTermination, kTerminationExternal);
             break;
        }
        case T_LINK_EQL:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeEquipmentLoopback, lval);
            break;
        }
        case T_HEC_CORRECTION:
        {
             set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeHECCorrection, lval);
             break;
        }
        case T_POS_RXPKTS:
        {
             set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeRxPkts, lval);
             break;
        }
        case T_POS_TXPKTS:
        {
             set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeTxPkts, lval);
             break;
        }
        case T_LINK_LASER:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeLaser, lval);
            break;
        }
        case T_LINK_B8ZS:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeZeroCodeSuppress, kZeroCodeSuppressB8ZS);
            break;
        }
	case T_LINK_AMI:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeZeroCodeSuppress, kZeroCodeSuppressAMI);
            break;
        }
        case T_LINK_CLEAR:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeClear, lval);
            break;
        }
        case T_ETH_AUTO:
        {
	    int count = 0;
	   /*If kComponentXGMII is present we should use RSFaultInhibit attribute.else try Nic.
            *Also note that the value for RSFaultInhibit is inverted to corrospond to NIC/NONIC*/
	    count = dag_component_get_subcomponent_count_of_type(root, kComponentXGMII);
            if (count != 0)
            {
                 set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeRSFaultEnable, lval);
            }
            else
            {
                set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeNic, lval);
            }
            break;
	}
        case T_LINK_SFPPWR:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeSfpPwr, lval);
            break;
        }
        case T_LINK_RKTIOPWR:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeRocketIOPower, lval);
            break;
        }
        case T_TX_CRC:
        {
			dag_component_t comp;
			int count,i = 0;
			attr_uuid_t attr_uuid;
			uint8_t set_value = 0;
			uint32_t retval = 0;
			/*This is a temporary solution to this problem.This code has to be modified when we get active/inactive attribute.*/
			count = dag_config_get_interface_count(card);
			for (i = 0; i < count; i++)
			{
					if((port_select & (1 << i)) == (1 << i))
					{
						attr_uuid =  dag_config_get_indexed_attribute_uuid(card, kUint32AttributeLineRate, i);
						retval = dag_config_get_uint32_attribute(card, attr_uuid);
						if((retval == kLineRateEthernet10GE) || (retval == kLineRateWAN))
						{
							/*set the XGMII value.*/
							set_value = 1;
							comp = dag_component_get_subcomponent(root,kComponentXGMII,i);
							attr_uuid = dag_component_get_attribute_uuid(comp,kBooleanAttributeTxCrc);
							dag_config_set_boolean_attribute(card, attr_uuid,lval);
							if(lval != dag_config_get_boolean_attribute(card,attr_uuid))
							{
								dagutil_warning("Unsupported or invalid configuration of port %d for this firmware, card, network mode or transceiver module: %s\n", i, dagopttext);
							}

						}
					}
			}
			if(!set_value)
			{		
		    		set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeTxCrc, lval);
			}
            		break; 
        }
        case T_TERF_STRIP_16:
        {
           dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip16, dagopttext);
           break; 
        }	
        case T_TERF_STRIP_32:
        {
            dag_config_util_set_value_to_lval_verify(card,root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfStrip32, dagopttext);
            break;
        }
        case T_TERF_NOSTRIP:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kUint32AttributeTerfStripCrc, kTerfNoStrip, dagopttext);
            break;
        }     
        case T_TR_TERF_NO_DELAY:
        {
	        dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kUint32AttributeTimeMode, kTrTerfNoDelay, dagopttext);
            break; 
        }
        case T_TR_TERF_RELATIVE:
        {
            dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kUint32AttributeTimeMode, kTrTerfRelative, dagopttext);
            break;
        }
        case T_TR_TERF_RX_ERROR:
        {
            dag_component_t any_component;
            dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorA, lval, dagopttext);
            /* FIXME Temporary fix */
            any_component = dag_component_get_subcomponent(root, kComponentTerf,0);
            if (any_component) 
            {
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorB) ) 
                    dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorB, lval, dagopttext);
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorC) ) 
                    dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorC, lval, dagopttext);
                if (dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXErrorD) ) 
                    dag_config_util_set_value_to_lval_verify(card, root, kComponentTerf, 0, kBooleanAttributeRXErrorD, lval, dagopttext);
            }
            dag_config_set_last_error(card, kDagErrNone);
            break;
        }
        case T_STEER:
        {
            dag_component_t steer;
            dag_component_t mux;	// NOTE: This component is only applicable to the 3.7ge
            dag_component_t current_component;

            steer = dag_component_get_subcomponent(root, kComponentSteering, 0);
            mux = dag_component_get_subcomponent(root, kComponentMux, 0);
	        current_component = (steer != NULL) ? steer : mux;
            if (current_component)
            {
                steer_t steer_mode;
                char* subtext;
                attr_uuid_t attr = 0;
                attr = dag_component_get_attribute_uuid(current_component, kUint32AttributeSteer);
                subtext = strstr(text, "=");
                if(subtext == NULL)
                    break;
                subtext++; /* step over the = */
                if(strncmp(subtext, "stream0", 7) == 0)
                {
                    steer_mode = kSteerStream0;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
                else if(strncmp(subtext, "colour",6) == 0)
                {
                    steer_mode = kSteerColour;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
                else if(strncmp(subtext, "dsm",3) == 0)
                {
                    steer_mode = kSteerColour;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
                else if(strncmp(subtext, "iface",5) == 0)
                {
                    steer_mode = kSteerIface;
                    dag_config_set_uint32_attribute(card, attr, steer_mode);
                }
            }
            break;
	    }
	    case T_ATM_PLOAD_SCRAMBLE:
	    case T_POS_SCRAMBLE:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributePayloadScramble, lval);
            break;
        }
	    case T_SONET_SCRAMBLE:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeScramble, lval);
            break;
        }
        case T_TIMESTAMP_HEAD:
        case T_TIMESTAMP_TAIL:
        {
            dag_component_t mapper = dag_component_get_subcomponent(root, kComponentMapper, 0);
            attr_uuid_t attr;
            if (!mapper)
                break;
            attr = dag_component_get_attribute_uuid(mapper, kBooleanAttributeTimeStampEnd);
            if (strstr(text, "head") != 0)
            {
                dag_config_set_boolean_attribute(card, attr, 0);
            }
            else if (strstr(text, "tail") != 0)
            {
                dag_config_set_boolean_attribute(card, attr, 1);
            }
            
            break;    
        }
        case T_LINK_OC1:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateOC1c);
            break;
        }
	    case T_LINK_OC3:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateOC3c);
            break;
        }
    	case T_LINK_OC12:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateOC12c);
            break;
        }
	    case T_LINK_OC48:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateOC48c);
            break;
        }
	    case T_LINK_OC192:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateOC192c);
            break;
        }
	    case T_LINK_STM0:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateSTM0);
            break;
        }
	    case T_LINK_STM1:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateSTM1);
            break;
        }
	    case T_LINK_STM4:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateSTM4);
            break;
        }
	    case T_LINK_STM16:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateSTM16);
            break;
        }
	    case T_LINK_STM64:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateSTM64);
            break;
        }
	    case T_ETH_1000:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateEthernet1000);
            break;
        }
	    case T_ETH_100:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateEthernet100);
            break;
        }
	    case T_ETH_10:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateEthernet10);
            break;
        }
	case T_ETH_10G:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineRate, kLineRateEthernet10GE);
            break;
        }
	    case T_ATM:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeNetworkMode, kNetworkModeATM);
            break;
        }
	    case T_POS:
		{
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeNetworkMode, kNetworkModePoS);
            break;
		}
        case T_LINK_FCL:
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeFacilityLoopback, lval);
            break;
        }
        case T_LINK_MODE:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeMode, lval);
            break;
        }
		case T_LINK_E1:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeE1);
            break;
        }
		case T_LINK_E1_UNFRAMED:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeE1unframed);
            break;
        }
		case T_LINK_E1_CRC:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeE1crc);
            break;
        }
		case T_LINK_T1_UNFRAMED:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeT1unframed);
            break;
        }
		case T_LINK_T1_ESF:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeT1esf);
            break;
        }
		case T_LINK_T1_SF:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeT1sf);
            break;
        }
		case T_LINK_T1:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeLineType, kLineTypeT1);
            break;
        }
        case T_SUPPRESS_ERROR:
        {
            dag_config_util_set_value_to_lval_verify(card,root, kComponentGeneral, 0, kBooleanAttributeSuppressError, lval, dagopttext);
            break;
        }
	    case T_SONET: /* Sonet mode: SONET */
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeSonetMode, 1);
            break;
        }
        case T_SDH: /* Sonet mode: SDH */
        {
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeSonetMode, 0);
            break;
        }
        case T_LINK_CRC:
        {
            set_value_to_lval_multi_port(port_select, card, root,  kUint32AttributeCrcSelect, lval ? kCrc32 : kCrcOff);
            break;
        }          
        case T_POS_CRC16:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeCrcSelect, kCrc16);
            break;
        }          
        case T_RAW:
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeNetworkMode, kNetworkModeRAW);
            break;
        case T_POS_CRC32:
        {
            set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeCrcSelect, kCrc32);
            break;
        };
        case T_POS_CRCSTRIP:
            set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeCrcStrip, lval);
            break;
	    
        case T_POS_MINLEN:
          	set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeMinPktLen, lval);
            break;
        case T_POS_MAXLEN:
       		set_value_to_lval_multi_port(port_select, card, root, kUint32AttributeMaxPktLen, lval);
            break;

        case T_POS_MINCHECK:
       		set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributePMinCheck, lval);
            break;

	    case T_IDELAY:
	    {
		    set_value_to_lval_single_port(card, root, kUint32AttributeIDELAY, lval);
            break;	
	    }
 
	    case T_ERROR:
		    dagutil_panic("Unrecognized token: %s\n", dagopttext);
		    break;

        /* tokens copied from dag50s - missed during dagx-ising it */
        case T_ATM_PASS_IDLE:
        {
            int count = 0;
            int i = 0;

            count = dag_component_get_subcomponent_count_of_type(root, kComponentDemapper);
            if (count != 0)
            {
                for (i = 0; i < count; i++)
                {
                    if ((port_select & (1 << i)) == (1 << i))
                    {
                        dag_config_util_set_value_to_lval(card, root, kComponentDemapper, i, kBooleanAttributeIdleCellMode, lval);
                    }
                }
            }
            else // Look for the SONET PP component if there's no Demapper
            {
                count = dag_component_get_subcomponent_count_of_type(root, kComponentSonetPP);
                for (i = 0; i < count; i++)
                {
                    if ((port_select & (1 << i)) == (1 << i))
                    {
                        dag_config_util_set_value_to_lval(card, root, kComponentSonetPP, i, kBooleanAttributeIdleCellMode, lval);
                    }
                }
            }
        }
        break;

        case T_LINK_VC3:
        case T_LINK_VC4:
        case T_LINK_VC4C:
        {
            dag_component_t sonic = NULL;
            attr_uuid_t vc_size = 0;
            int count = 0;
            int i = 0;
            vc_size_t value;

            if (T_LINK_VC4 == token)
                value = kVC4;
            else if (T_LINK_VC4C == token)
                value = kVC4C;
            else
                value = kVC3;

            count = dag_component_get_subcomponent_count_of_type(root, kComponentSonic);
            for (i = 0; i < count; i++)
            {
                if ((port_select & (1 << i)) == (1 << i))
                {
                    sonic = dag_component_get_subcomponent(root, kComponentSonic, i);
                    assert(sonic != NULL);
                    if (sonic)
                    {
                        vc_size = dag_component_get_attribute_uuid(sonic, kUint32AttributeVCSize);
                        assert(vc_size != kNullAttributeUuid);
                        if (vc_size)
                        {
                            dag_config_set_uint32_attribute(card, vc_size, value);
                        }
                        else
                        {
                            dagutil_warning("vc size attribute not found!\n");
                        }
                    }
                    else
                    {
                        dagutil_warning("sonic component %d not found\n", i);
                    }
                }
            }
        }
        break;
      /* tokens copied from dag82x - missed during dagx-ising it */
      case T_ETH_10GBASESR:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval_verify(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_SR, dagopttext);
                }
            }
            break;
        }        
        case T_ETH_10GBASELR:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval_verify(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_LR, dagopttext);
                }
            }
            break;
        }        
        case T_ETH_10GBASEER:
        {
            int count = 0;
            int i = 0;
            count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
            for(i = 0; i < count; i++)
            {
                if((port_select & (1 <<i)) == (1 << i))
                {
                    dag_config_util_set_value_to_lval_verify(card,root, kComponentPort, i, kUint32AttributeEthernetMode, kEthernetMode10GBase_ER, dagopttext);
                }
            }
            break;
        }
        case T_ETH_WAN:
        {
	    dag_component_t xgmii = NULL;
            attr_uuid_t mac_wan_mode = 0;
	    set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeWanMode, lval);
	    xgmii = dag_component_get_subcomponent(root, kComponentXGMII, 0);
 	    mac_wan_mode = dag_component_get_attribute_uuid(xgmii, kBooleanAttributeMacWanMode);
	    if(mac_wan_mode)
	    {
			set_value_to_lval_multi_port(port_select, card, root, kBooleanAttributeMacWanMode, lval);
	    }
	    else
	    {
		/*It is ok if you don't find the mac_wan_mode attribute.It is not there is version 0 of xgmii.
		this version is used in wan version of Dag 8.1sx*/
		if(dag_config_get_last_error(card) == kDagErrNoSuchAttribute)
		  	dag_config_set_last_error(card, kDagErrNone);
	    }
            break;
        }
	default:
            dagutil_panic("Token not supported on dagconfig for this FW/Card: %s\n", dagopttext);
            break;
    }
    //FIXME: temporary added because dag_config_default will return the last error as unsoported component 
	if (dag_config_get_last_error(card) == kDagErrNoSuchAttribute ||
		dag_config_get_last_error(card) == kDagErrNoSuchComponent)
	{
		dag_config_set_last_error(card, kDagErrNone);
	  	if ( token != T_DEFAULT ) 
	  	{
			dagutil_panic("firmware or CSAPI does not support this token %s\n", dagopttext);
	  	};
	}

}

static void 
set_value_to_lval_enable_port(dag_card_ref_t ref, dag_component_t root, int lval, int port_index)
{
    dag_component_t gpp;
    dag_component_t port;
    dag_component_t xgmii;
    attr_uuid_t active;
    uint8_t val;
    uint8_t retval;
    active = kNullAttributeUuid;
    gpp = dag_component_get_subcomponent(root, kComponentGpp, 0);
   
    /* first check for gpp component */
    if (NULL !=  gpp )
    {
        if ( (active = dag_component_get_indexed_attribute_uuid_of_type(gpp, kBooleanAttributeActive, port_index)) == kNullAttributeUuid) 
        {
             dagutil_warning("option [dis|en]able %d not supported by this firmware.\n", port_index);
             /* warning message already printed. Now set the error to kDagErrNone */
             dag_config_set_last_error(ref, kDagErrNone);
             return;
        }
    }
    else
    {
        /* if no gpp, try to find the attribute in port component*/
        port = dag_component_get_subcomponent(root, kComponentPort, port_index);
        if  (NULL != port ) 
        {
            if ((active = dag_component_get_indexed_attribute_uuid_of_type(port, kBooleanAttributeActive, 0)) == kNullAttributeUuid)
            {
                xgmii = dag_component_get_subcomponent(root, kComponentXGMII, port_index);
                if (xgmii != NULL)
                {
                    if ((active = dag_component_get_indexed_attribute_uuid_of_type(xgmii, kBooleanAttributeRxEnable, 0)) == kNullAttributeUuid)
                    {
                        dagutil_warning("option [dis|en]able %d not supported by this firmware.\n", port_index);
                        /* warning message already printed. Now set the error to kDagErrNone */
                        dag_config_set_last_error(ref, kDagErrNone);
                        return;
		    }
		}
            }
	}
    }

    assert(active != kNullAttributeUuid);
    val = lval == 1;
    dag_config_set_boolean_attribute(ref, active, val);

    // Perform a read back to confirm that the set has worked.
    retval = dag_config_get_boolean_attribute(ref, active);
    if (retval != lval)
        dagutil_warning("disabling of port %d is not valid for this FW/card.\n", port_index);

}

/**
 * Set value of given attribute to lval. 
 *
 * Search through all components of the card to find the given
 * attribute. Search stops at first attribute found. Use
 * set_value_to_lval_multi_port if the attribute can be is several
 * instances of the same component.
 *
 * Search is one-level deep only, subcomponents of components are not
 * searched.
 */
static void
set_value_to_lval_single_port(dag_card_ref_t card_ref, dag_component_t root,
                              dag_attribute_code_t attr_code, int lval)
{
    set_value_to_lval_multi_port(1, card_ref, root, attr_code, lval);
}

/**
 * Set value of given attribute to lval. 
 *
 * Search through all components of the card to find the given
 * attribute. We assume there is only one component type having such
 * attribute (but the component can have multiple instances, one for
 * each port). NOTE: multiple component *types* are not handled.
 *
 * Search is one-level deep only, subcomponents of components are not
 * searched.
 */
static void
set_value_to_lval_multi_port(int port_select,
                             dag_card_ref_t card_ref, dag_component_t root,
                             dag_attribute_code_t attr_code, int lval)
{
	int i, count;
	attr_uuid_t attr_uuid;
	uint32_t retval;
	dag_attr_t attr_valuetype;
	uint32_t interface_count = 0;


	count = dag_config_get_attribute_count(card_ref, attr_code);
	if (count == 0)
	{
		dagutil_panic("Token not supported by dagconfig for this FW/Card: %s\n", dagopttext);
		return;
	}
	/*This function assumes that there is only one component type having a particular attribute.However this is not 
	 true in case of line_rate attribute on some images.There are instances where line_rate attribute is reported from
	more than one component types.eg from kComponentPort and kComponentPhy.The variable count is essentially the number
	of ports on the dag card.Since this variable fails in case of the above scenario,we have to properly calculate the line_rate
	attribute.The below code will keep the impact minimal.*/
	interface_count = dag_config_get_interface_count(card_ref);
	if(count > interface_count)
	{
		count = interface_count;	
	}

	for(i = 0; i < count; i++)
	{
		if((port_select & (1 << i)) == (1 << i))
		{
			attr_uuid =  dag_config_get_indexed_attribute_uuid(card_ref, attr_code, i);
			dag_config_util_set_attribute(card_ref, attr_uuid, &lval);

			attr_valuetype = dag_config_get_attribute_valuetype(card_ref, attr_uuid);
			switch (attr_valuetype)
			{
				case kAttributeBoolean:
					retval = dag_config_get_boolean_attribute(card_ref, attr_uuid);
					break;

				case kAttributeUint32:
					retval = dag_config_get_uint32_attribute(card_ref, attr_uuid);
					break;
				
				default:
					dagutil_warning("%s does not support attribute valuetype: %d\n", __FUNCTION__, attr_valuetype);
					return;
			}
			if (retval != lval)
				dagutil_warning("Unsupported or invalid configuration of port %d for this firmware, card, network mode or transceiver module: %s\n", i, dagopttext);

		}
	}
}
/**
 * To check whether the snaplength for the current gpp  has to be changed or not.
 * Introduced to support multiple gpp and to change snaplength per port.
 * Parameters:    port_select - default value will be 0xff meaning all ports.For porta 0x1, portb 0x2 and so on.
  *                 iface_count    - total interface count of the previously calculated gpp(s)
 *                 gpp_iface_count- interface of the current gpp
 */
uint32_t is_change_gpp_slen( int port_select, int iface_count, int gpp_iface_count)
{
    int i;
    for(  i = iface_count; i < iface_count + gpp_iface_count; i++)
    {
        if( port_select & ( 0x1 << i) )
            return 1;
    }
    return 0;
}

void token_proc_set_default_on_reset(uint8_t val)
{
	g_default_on_reset = val;
}
