/*
 * Copyright (c) 2005 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 */

/* For printing card configuration */

#include "dag_config.h"
#include "dag_attribute_codes.h"
#include "dag_component_codes.h"
#include "dag_component.h"
#include "statistics_printing.h"
#include "attribute_print.h"
#include "dagutil.h"
#include <stdio.h>
#include <assert.h>


static attribute_verbosity_mapping_t attr_verbose_array[] =
{
	/*  Link */
	{kBooleanAttributeLink, 0},
	/* Lock */
	{kBooleanAttributeLock, 0},
 	     /* LOS */
 	{kBooleanAttributeLossOfSignal, 0},
             /* LOC */
 	{kBooleanAttributeLossOfClock, 0},
             /* LOF */
 	{kBooleanAttributeLossOfFrame, 0},
             /* LOP */
 	{kBooleanAttributeLossOfPointer, 0},
	/* Fault on transmit or receive */
	{kBooleanAttributeFault, 0},
		/* Fault on tx path */
	{kBooleanAttributeTxFault, 1},
		/* Fault on Rx path */
	{kBooleanAttributeRxFault, 1},
             /* OOF */
 	{kBooleanAttributeOutOfFrame, 1},
             /* FCS_ERR */
 	{kUint64AttributeFCSErrors, 1},
             /* GOOD_PACKET */
 	{kUint64AttributeGoodPackets, 1},
             /* RXF */
    {kUint64AttributeFIFOOverrunCount, 1},
    {kBooleanAttributeRemoteFault, 1},
    {kBooleanAttributePeerLink, 1},
    {kBooleanAttributeCRCError, 1},
    {kBooleanAttributeCounterLatch, 1},
    {kBooleanAttributeSFPTxFaultCurrent, 1},
    {kBooleanAttributeMiniMacLostSync, 1},
    {kBooleanAttributeRemoteErrorCurrent, 1},
    {kBooleanAttributeLossOfFramingCurrent, 1},
    {kBooleanAttributeSFPLossOfSignalCurrent, 1},
    {kBooleanAttributePeerLinkCurrent, 1},
    {kBooleanAttributeLinkCurrent, 1},
    {kBooleanAttributeSFPTxFaultEverHi, 2},
    {kBooleanAttributeCRCErrorEverHi, 2},
    {kBooleanAttributeRemoteErrorEverHi, 2},
    {kBooleanAttributeLossOfFramingEverHi, 2},
    {kBooleanAttributeSFPLossOfSignalEverHi, 2},
    {kBooleanAttributePeerLinkEverHi, 2 },
    {kBooleanAttributeLinkEverHi, 2 },
    {kBooleanAttributeSFPTxFaultEverLo, 2},
    {kBooleanAttributeCRCErrorEverLo, 2},
    {kBooleanAttributeRemoteErrorEverLo, 2},
    {kBooleanAttributeLossOfFramingEverLo, 2},
    {kBooleanAttributeSFPLossOfSignalEverLo, 2},
    {kBooleanAttributePeerLinkEverLo, 2 },
    {kBooleanAttributeLinkEverLo, 2 },
   /*New Statastics printing attributes added for Infiniband cards*/
    {kBooleanAttributeRxLinkUp,0},
    {kBooleanAttributeRxFrameError,0},
    {kBooleanAttributePLLDetect,0},
    {kBooleanAttributeGTPResetDone,0},
    {kBooleanAttributeRxByteAlign,0},
    {kBooleanAttributeRxChannelAlign,0},
    {kBooleanAttributeRxByteAlignZero,2},
    {kBooleanAttributeRxByteAlignOne,2},
    {kBooleanAttributeRxByteAlignTwo,2},
    {kBooleanAttributeRxByteAlignThree,2},
    {kBooleanAttributeRxChannelAlignZero,2},
    {kBooleanAttributeRxChannelAlignOne,2},
    {kBooleanAttributeRxChannelAlignTwo,2},
    {kBooleanAttributeRxChannelAlignThree,2},
    /*Crosspoint switch*/
    {kBooleanAttributeInputLOS,0},
    /* dagconfig-ising dag43ge */
    {kBooleanAttributeAutoNegotiationComplete, 0},
    /* dag3.7T statistics  */
    {kBooleanAttributeReceiveLossOfSignal, 0},
    {kBooleanAttributeFIFOLimitStatus, 0},
    {kBooleanAttributeAlarmSignal, 0},
    {kBooleanAttributeDriverMonitorOutput, 0},
    {kBooleanAttributeLineCodeViolation, 0},
    {kUint32AttributeCableLoss, 0},
    {kBooleanAttributeE1T1Link, 0},
    {kBooleanAttributeE1T1CRCError, 0},
    {kBooleanAttributeE1T1AISError, 0},
    {kBooleanAttributeE1T1FramerError, 0},
    {kBooleanAttributeE1T1Tx1, 0},
    {kBooleanAttributeE1T1Tx0, 0},
    {kBooleanAttributeE1T1Rx1, 0},
    {kBooleanAttributeE1T1Rx0, 0},
    {kBooleanAttributeLossOfCellDelineation, 0}
};
/* List of component codes to print statistics info for
 *
 * Some fimrware modules implement both the (de)framer and
 * (de)mapper and the component type can be any of those
 * below. Hopefully only one of them is present, otherwise we will
 * end up with multiple lines for each framer/mapper. */
static dag_component_code_t comp_codes[] = {
    kComponentPhy,
    kComponentPort,
    kComponentSonetPP,
    kComponentFramer,
    kComponentDeframer,
    kComponentMapper,
    kComponentDemapper,
    kComponentPCS,
    kComponentXGMIIStatistics,
    kComponentXGMII,
    kComponentInfiniBandFramer,
    kComponentGpp,
    kComponentMiniMac
};

//this is for foture use in is attribute printable function 
//it will be used to reduce the amount of attributes to show 
// static dag_attribute_code_t dagx_statistic_attributes_to_show_pos[] = {
// 	/* LOS */
// 	kBooleanAttributeLossOfSignal,
//             /* LOC */
// 	kBooleanAttributeLossOfClock,
//             /* OOF */
// 	kBooleanAttributeOutOfFrame,
//             /* LOF */
// 	kBooleanAttributeLossOfFrame,
//             /* LOP */
// 	kBooleanAttributeLossOfPointer,
//             /* FCS_ERR */
// 	kUint64AttributeFCSErrors,
//             /* GOOD_PACKET */
// 	kUint64AttributeGoodPackets,
//             /* RXF */
// 	kUint64AttributeFIFOOverrunCount,
// };
//
// static dag_attribute_code_t dagx_statistic_attributes_to_show_Eth[] = {
// 	/* LOS */
// 	kBooleanAttributeLossOfSignal,
//             /* LOC */
// 	kBooleanAttributeLossOfClock,
//             /* OOF */
// 	kBooleanAttributeOutOfFrame,
//             /* LOF */
// 	kBooleanAttributeLossOfFrame,
//             /* LOP */
// 	kBooleanAttributeLossOfPointer,
//             /* FCS_ERR */
// 	kUint64AttributeFCSErrors,
//             /* GOOD_PACKET */
// 	kUint64AttributeGoodPackets,
//             /* RXF */
// 	kUint64AttributeFIFOOverrunCount,
// };

		
void dagx_print_statistics_header(dag_card_ref_t card_ref);
void dagx_print_statistics(dag_card_ref_t card_ref,int port);
static bool attribute_is_printable(dag_card_ref_t card_ref, attr_uuid_t attr);

void
print_statistics(dag_card_ref_t card, int port, int vc_index)
{
    switch (dag_config_get_card_type(card))
    {
        case kDag71s:
            dag71s_print_statistics(card, port, vc_index);
            break;
        case kDag70ge:
            dag70ge_print_statistics(card, port);
            break;
        case kDag62:
            dag62se_print_statistics(card, port);
            break;
       case kDag452e:
        case kDag454e:
        case kDag452gf:
        case kDag452cf:
        case kDag452z:
        case kDag452z8:
            dag45ge_print_statistics(card, port);
            break;
        case kDag37d:
            dag37d_print_statistics(card, port);
            break;
        case kDag82x:
        case kDag82x2:
        case kDag82z:
	case kDag52x:
	//case kDag810:
	//    dag82x_print_statistics(card, port);
      //      break;
//        case kDag50s:
//            dag50s_print_statistics(card, port);
//            break;
        default:
            dagx_print_statistics(card,port);
            break;
    }
}

void
print_statistics_header(dag_card_ref_t card)
{
    switch (dag_config_get_card_type(card))
    {
        case kDag71s:
            dag71s_print_statistics_header();
            break;
        case kDag70ge:
            dag70ge_print_statistics_header();
            break;
        case kDag62:
            dag62se_print_statistics_header(card);
             break;
        case kDag452e:
        case kDag454e:
        case kDag452gf:
        case kDag452cf:
        case kDag452z:
        case kDag452z8:
            dag45ge_print_statistics_header();
            break;
        case kDag37d:
            dag37d_print_statistics_header();
            break;
        case kDag82x2:
        case kDag82x:
	case kDag52x:
        case kDag82z:
        //case kDag810:
            //dag82x_print_statistics_header();
            //break;
//        case kDag50s:
//            dag50s_print_statistics_header();
//            break;
        default:
            dagx_print_statistics_header(card);
            break;
    }
}
void
dag45ge_print_statistics_header()
{
    printf("Link  PLink  RFault  LOF  LOS  ");
}

void
dag45ge_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attribute;
    
    root_component = dag_config_get_root_component(card_ref);
    /* Read Port information */
    any_component = dag_component_get_subcomponent(root_component, kComponentPort, port);
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLink);
    printf("%4u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePeerLink);
    printf("%5u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRemoteFault);
    printf("%6u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));
}
void dag62se_print_statistics_header(dag_card_ref_t card_ref)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attr;
    network_mode_t net_mode;
    ethernet_mode_t eth_mode;
    
    root_component = dag_config_get_root_component(card_ref);
    any_component = dag_component_get_subcomponent(root_component, kComponentPort, 0);
    
    any_attr = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
    net_mode = dag_config_get_uint32_attribute(card_ref, any_attr);
    
    any_attr = dag_component_get_attribute_uuid(any_component, kUint32AttributeEthernetMode);
    eth_mode = dag_config_get_uint32_attribute(card_ref, any_attr);
    
    switch(net_mode)
    {
        case kNetworkModePoS:
            /* PoS */
            printf("LOS  LOC  OOF  LOF  LOP  FCS_ERR  GOOD_PACKET  RXF  ");
            break;
        case kNetworkModeEth:
            /* Eth */
            if ( eth_mode == kEthernetMode10GBase_LR)
            {
                /* LAN */
                printf("LOS  LOC  LOF  BER  LFT  RFT  FCS_ERR  BAD_PACKET  GOOD_PACKET  RXF  ");  
            }
            else
            {
                /* WAN */
                printf("LOS  LOC  OOF  LOF  LOP  BER  LFT  RFT  FCS_ERR  BAD_PACKET  GOOD_PACKET  RXF  ");  
            }
                
            break;
        default:
            break;
    }
}

void 
dag62se_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attr;
    network_mode_t net_mode;
    ethernet_mode_t eth_mode;

    root_component = dag_config_get_root_component(card_ref);
    
    /* Read Port information */
    any_component = dag_component_get_subcomponent(root_component, kComponentPort,port);
    /* check to see what type of stats to display */
    any_attr = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
    net_mode = dag_config_get_uint32_attribute(card_ref, any_attr);
    
    any_attr = dag_component_get_attribute_uuid(any_component, kUint32AttributeEthernetMode);
    eth_mode = dag_config_get_uint32_attribute(card_ref, any_attr);

    /* Latch the counters */
    any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeCounterLatch);
    assert(any_attr != kNullAttributeUuid);
    dag_config_set_boolean_attribute(card_ref, any_attr, 1);
    
    
    switch(net_mode)
    {
        case kNetworkModePoS:
            /* LOS */
            any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
            printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
            /* LOC */
            any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfClock);
            printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
            /* OOF */
            any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeOutOfFrame);
            printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
            /* LOF */
            any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
            printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
            /* LOP */
            any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfPointer);
            printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
            /* FCS_ERR */
            any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFCSErrors);
            printf("%7"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));
            /* GOOD_PACKET */
            any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeGoodPackets);
            printf("%12"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));            
            /* RXF */
            any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFIFOOverrunCount);
            printf("%3"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));             
            
            break;
        case kNetworkModeEth:
            if(eth_mode == kEthernetMode10GBase_LR)
            {
                /* LAN */
                /* LOS */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LOC */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfClock);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LOF */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* BER */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeHighBitErrorRateDetected);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LFT */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLocalFault);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));                
                /* RFT */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRemoteFault);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));                
                /* FCS_ERR */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFCSErrors);
                printf("%7"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));
                /* BAD_PACKET */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeBadPackets);
                printf("%11"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));                  
                /* GOOD_PACKET */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeGoodPackets);
                printf("%12"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));            
                /* RXF */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFIFOOverrunCount);
                printf("%3"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));                   
            }
            else
            {
                /* WAN */
                /* LOS */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LOC */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfClock);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
               /* OOF */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeOutOfFrame);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LOF */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LOP */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfPointer);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* BER */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeHighBitErrorRateDetected);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
                /* LFT */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLocalFault);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));                
                /* RFT */
                any_attr = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRemoteFault);
                printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));                
                /* FCS_ERR */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFCSErrors);
                printf("%7"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));
                /* BAD_PACKET */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeBadPackets);
                printf("%11"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));                  
                /* GOOD_PACKET */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeGoodPackets);
                printf("%12"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));            
                /* RXF */
                any_attr = dag_component_get_attribute_uuid(any_component, kUint64AttributeFIFOOverrunCount);
                printf("%3"PRIu64"  ", dag_config_get_uint64_attribute(card_ref, any_attr));                   
            }
            break;
        default:
            break;
    }
}
            
    

void
dag70ge_print_statistics_header()
{
        printf("Port  Link  PLink  RFault  LOF  LOS  ");
}
void
dag70ge_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attribute;

    root_component = dag_config_get_root_component(card_ref);
    /* Read Port information */
    any_component = dag_component_get_subcomponent(root_component, kComponentPort, port);
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLink);
    printf("%4u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePeerLink);
    printf("%5u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRemoteFault);
    printf("%6u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
    printf("%3u", dag_config_get_boolean_attribute(card_ref, any_attribute));

}
void
dag71s_print_statistics_header()
{
    printf("LOS  LOF  OOF  B1  B2  B3  REI  C2  PTR  ");
}

void
dag71s_print_statistics(dag_card_ref_t card_ref, int port, int vc_index)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attribute;
    vc_pointer_state_t ptr_state;

    root_component = dag_config_get_root_component(card_ref);
    
    any_component = dag_component_get_subcomponent(root_component, kComponentSonic, port);
    /* Set the vc to use */
    any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeVCIndex);
    dag_config_set_uint32_attribute(card_ref, any_attribute, vc_index);


    if(vc_index == 0)
    {
        any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
        printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

        any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
        printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

        any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeOutOfFrame);
        printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

        any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeB1Error);
        printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

        any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeB2Error);
        printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));
    }
    else
    {   
        printf("%3s  ", "..");
        printf("%3s  ", "..");
        printf("%3s  ", "..");
        printf("%2s  ", "..");
        printf("%2s  ", "..");
    }

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeB3Error);
    printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeREIError);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeC2PathLabel);
    printf("%02x  ", dag_config_get_uint32_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributePointerState);
    ptr_state = dag_config_get_uint32_attribute(card_ref, any_attribute);
    switch (ptr_state)
    {
        case kPointerStateValid:
            printf ("valid ");
            break;

        case kPointerStateConcatenationIndicator:
            printf ("concat ");
            break;

        case kPointerStateAlarmSignalIndicator:
            printf ("ais ");
            break;

        case kPointerStateLossOfPointer:
            printf ("lop ");
            break;
    }
}

void
dag37d_print_statistics_header()
{
        printf("RLOL  LOS  LOF  RDI  AIS  OOF  ");
}
void
dag37d_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attribute;

    root_component = dag_config_get_root_component(card_ref);
    /* Read Port information */
    any_component = dag_component_get_subcomponent(root_component, kComponentPort, port);
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeReceiveLockError);
    printf("%4u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfSignal);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLossOfFrame);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRemoteDefectIndication);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeAlarmIndicationSignal);
    printf("%3u", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeOutOfFrame);
    printf("%3u", dag_config_get_boolean_attribute(card_ref, any_attribute));
}
#if 0
void
dag82x_print_statistics_header()
{
	printf("Link  TxFault  RxFault  Lock  ");
}
void
dag82x_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t any_component;
    dag_component_t root_component;
    attr_uuid_t any_attribute;

    root_component = dag_config_get_root_component(card_ref);
    if(dag_config_get_card_type(card_ref) == kDag810)
    {
    	/* Read Port information */
    	any_component = dag_component_get_subcomponent(root_component, kComponentPCS, port);
    }
    else
    {
	/* Read Port information */
    	any_component = dag_component_get_subcomponent(root_component, kComponentPort, port);
    }

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLink);
    printf("%4u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));
    
    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeTxFault);
    printf("%7u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRxFault);
    printf("%7u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

    any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLock);
    printf("%4u  ", dag_config_get_boolean_attribute(card_ref, any_attribute));

}
#endif

/* DAG 5.2SXA */
void
dag52sxa_print_statistics_header()
{
    printf("CRCErr  LoS  OoF  LoF  AIS  RDI  REI  LoP  B1  B2  B3  ");
}

void
dag52sxa_print_statistics(dag_card_ref_t card_ref, int port)
{
    dag_component_t root_component;
    dag_component_t port_component;
    dag_component_t deframer_component;
    attr_uuid_t any_attr;

    root_component = dag_config_get_root_component(card_ref);
    port_component = dag_component_get_subcomponent(root_component, kComponentPort, port);;
    deframer_component = dag_component_get_subcomponent(root_component, kComponentDeframer, port);;

   /* CRCErr */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeCRCError);
    printf("%6u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* LOS */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeLossOfSignal);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* OOF */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeOutOfFrame);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* LOF */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeLossOfFrame);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* AIS */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeAlarmIndicationSignal);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* RDI */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeRDIError);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* REI */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeREIError);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* LOP */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeLossOfPointer);
    printf("%3u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* B1 */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeB1Error);
    printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* B2 */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeB2Error);
    printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
    /* B3 */
    any_attr = dag_component_get_attribute_uuid(deframer_component, kBooleanAttributeB3Error);
    printf("%2u  ", dag_config_get_boolean_attribute(card_ref, any_attr));
}

void
dagx_print_statistics_header(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute;
	dag_component_t root;
	dag_component_t comp;
	dag_attribute_code_t code;
	int comp_count = 0;
	int attr_count = 0;
	int i = 0;
	int j = 0;
    dag_component_code_t comp_code;
    int h = 0;

	root = dag_config_get_root_component(card_ref);
    for(h = 0; h < sizeof(comp_codes) / sizeof(comp_codes[0]); h++) {
        /* try each component type */
        comp_code = comp_codes[h];
        comp_count = dag_config_get_recursive_component_count(card_ref, comp_code);//       comp_count = dag_component_get_subcomponent_count_of_type(root, comp_code);
	//we print one header per port(componet)
        for (i = 0; (i < comp_count)&& (i < 1); i++)
        {
            comp = dag_config_get_recursive_indexed_component(card_ref, comp_code,i);//comp = dag_component_get_subcomponent(root, comp_code, i);
	    if(NULL == comp ) break;
            attr_count = dag_component_get_attribute_count(comp);
            for (j = 0; j < attr_count; j++)
            {
                any_attribute = dag_component_get_indexed_attribute_uuid(comp, j);
                code = dag_config_get_attribute_code(any_attribute);
                if (attribute_is_printable(card_ref, any_attribute))
                {
                    attribute_print_header(card_ref, any_attribute);
                }
            }
        }
        if (comp_count > 0) {
        }
    }
}

void
dagx_print_statistics(dag_card_ref_t card_ref, int port ) {
	attr_uuid_t any_attribute;
	dag_component_t root;
	dag_component_t comp;
	dag_attribute_code_t code;
	int comp_count = 0;
	int attr_count = 0;
	int i = 0;
	int j = 0;
    dag_component_code_t comp_code;
    int h = 0;

	root = dag_config_get_root_component(card_ref);
    for(h = 0; h < sizeof(comp_codes) / sizeof(comp_codes[0]); h++) {
        /* try each component type */
        comp_code = comp_codes[h];
        comp_count = dag_config_get_recursive_component_count(card_ref, comp_code);//        comp_count = dag_component_get_subcomponent_count_of_type(root, comp_code);
	
	if( comp_count > 1 )
	{
	    comp_count = port+1;
	    i = port;
	} else
	  i = 0;
	  
        for (; i < comp_count; i++)
        {
            dag_config_state_t comp_state = kStateActive;
            comp = dag_config_get_recursive_indexed_component(card_ref, comp_code,i);
            if ( NULL == comp ) break;
            comp_state =  dag_config_get_component_state(comp);
            attr_count = dag_component_get_attribute_count(comp);
            for (j = 0; j < attr_count; j++)
            {
                any_attribute = dag_component_get_indexed_attribute_uuid(comp, j);
                code = dag_config_get_attribute_code(any_attribute);
                if (attribute_is_printable(card_ref, any_attribute))
                {
                    attribute_print_value(card_ref, any_attribute, comp_state,i);
                }
            }
        }
        if (comp_count > 0) {
        }
    }
}


/*
 * Print only the Boolean status attributes (non-Boolean status is
 * actually a counter and printed in extendedtstatistics.
 */
static bool
attribute_is_printable(dag_card_ref_t card_ref, attr_uuid_t attr)
{
	dag_attr_config_status_t cs;
	dag_attr_t type;
	dag_attribute_code_t code;
    int verbosity = dagutil_get_verbosity();
    int i = 0;
	cs = dag_config_get_attribute_config_status(attr);
	type = dag_config_get_attribute_valuetype(card_ref, attr);
	if (cs != kDagAttrStatus || type != kAttributeBoolean)
	{
		return false;
	}

  //printf("attrname %s\n",dag_config_get_attribute_name(attr)); 
        /* list of exceptions */
        
	code = dag_config_get_attribute_code(attr);
	switch (code)
	{
        case kAttributeInvalid:
            /* dummy placeholder for exceptions */
            return false;

		default:
			break;
	}
    /* verbosity 3 and above print all boolean attribs */
	if(verbosity >=  3)
		 return true;
     /* Check the attribute is to be printed at this verbosity level*/
    for( i = 0; i < ( sizeof (attr_verbose_array) / sizeof(attr_verbose_array[0])) ; i++)
    {
        if (code == attr_verbose_array[i].attribute_code)
        {
            if (attr_verbose_array[i].attribute_verbosity > verbosity)
                return false;
            else
                return true;
        }
    }
    return false;
}

