/*
 * Copyright (c) 2005 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 */

/* For printing card configuration */

#include "../../include/dag_config.h"
#include "../../include/dag_attribute_codes.h"
#include "../../include/dag_component_codes.h"
#include "../../include/dag_component.h"
#include "../../include/dagutil.h"
#include "dag_platform.h"
#include "dag_romutil.h"
#include "dagapi.h"
#include <stdio.h>
#include <assert.h>
#include "statistics_printing.h"	// included for the attribute verbosity array definition

static void port_print_configuration(dag_card_ref_t card_ref);	/* dagx */
static void print_framer_configuration(dag_card_ref_t card_ref);
static void print_demapper_configuration(dag_card_ref_t card_ref);
static void dag71s_print_configuration(dag_card_ref_t card_ref);
static void dag70ge_print_configuration(dag_card_ref_t card_ref);
static void dag62se_print_configuration(dag_card_ref_t card_ref);
static void dag37d_print_configuration(dag_card_ref_t card_ref);
//static void dag810_print_configuration(dag_card_ref_t card_ref);
//static void dag82x_print_configuration(dag_card_ref_t card_ref);
static void pbm_print_configuration(dag_card_ref_t card_ref);
static void gpp_print_configuration(dag_card_ref_t card_ref);
static void terf_print_configuration(dag_card_ref_t card_ref);
static void print_card_information(dag_card_ref_t card_ref);
static void print_card_configuration(dag_card_ref_t card_ref);
static void dsm_print_configuration(dag_card_ref_t card_ref);
static void steering_print_configuration(dag_card_ref_t card_ref);
static void mux_print_configuration(dag_card_ref_t card_ref);
static void print_attribute(dag_card_ref_t card_ref, dag_component_t component,dag_attribute_code_t attr_code);
static void print_attribute_tree(dag_card_ref_t card_ref,dag_component_t component);
//static void duck_print_configuration(dag_card_ref_t card_ref);

typedef struct
{
    const char* mHeader ;
    dag_attribute_code_t mCode;
} header_attr_code_t;

/*The maximux number of confiugrable attributes for Port using dagconfig*/
#define MAX_PORT_CONFIG_ATTRIBUTES 24

static attribute_verbosity_mapping_t port_attr_verbose_array[] =
{
	{kBooleanAttributePhyBistEnable, 1},
	{kBooleanAttributePhyTxClockOff, 1},
	{kBooleanAttributePhyKillRxClock, 1},
	{kUint32AttributePhyRate, 1},
	{kUint32AttributePhyRefSelect, 1},
	{kUint32AttributeConfig, 1},
	{kBooleanAttributeDiscardData, 1},
	{kBooleanAttributeTransmitLockError, 1},
    {kBooleanAttributeWanMode, 1},
    {kBooleanAttributeMacWanMode, 1},
	{kUint32AttributeNetworkMode, 1},
	{kUint32AttributeTransceiverPowerLevelState, 1}
};

header_attr_code_t headers[] =
{
    {"RxFrames", kUint32AttributeRxFrames} ,
    {"RxBytes", kUint64AttributeRxBytes},
    {"TxFrames", kUint32AttributeTxFrames},
    {"TxBytes", kUint64AttributeTxBytes},
    {"CRCErrs", kUint32AttributeCRCErrors},
    {"ShortErrs", kUint32AttributeShortPacketErrors},
    {"LongErrs", kUint32AttributeLongPacketErrors},
    {"ErrBlock", kUint32AttributeErrorBlockCounter},
    {"HighBER", kUint32AttributeBERCounter},
    {"", 0}
};

void
print_attribute_value(dag_card_ref_t card, attr_uuid_t attr)
{
    char format[256];
    int i = 0;
	if (!card)
	{
		dagutil_warning("Bad card\n");
		return;
	}
	if (!attr)
	{
		dagutil_warning("Bad attribute\n");
		return;
	}
    /* match the attribute code with a header so we know how much padding to use */
    while (headers[i].mCode != 0)
    {
        if (headers[i].mCode == dag_config_get_attribute_code(attr))
        {
		sprintf(format, "%%%ld",(long)strlen(headers[i].mHeader));
            break;
        }
        i++;
    }
	switch (dag_config_get_attribute_valuetype(card, attr))
	{
		case kAttributeBoolean:
        strcat(format, PRIu8);
        strcat(format, "  ");
		printf(format, dag_config_get_boolean_attribute(card, attr));
		break;

		case kAttributeUint32:
        strcat(format, PRIu32);
        strcat(format, "  ");
		printf(format, dag_config_get_uint32_attribute(card, attr));
		break;
		
		case kAttributeUint64:
        strcat(format, PRIu64);
        strcat(format, "  ");
		printf(format, dag_config_get_uint64_attribute(card, attr));
		break;

		case kAttributeInt32:
        strcat(format, PRId32);
        strcat(format, "  ");
		printf(format, dag_config_get_int32_attribute(card, attr));
		break;

		case kAttributeInt64:
        strcat(format, PRId64);
        strcat(format, "  ");
		printf(format, dag_config_get_int64_attribute(card, attr));
		break;

		case kAttributeFloat:
        strcat(format, "f");
        strcat(format, "  ");
		printf(format, dag_config_get_float_attribute(card, attr));
		break;

		default:
		dagutil_warning("Don't know how to display this value\n");
	}
}

#if 0
void
duck_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t root;
	dag_component_t duck;
	attr_uuid_t attr;
	uint8_t bool_val;
	uint32_t uint32_val;
	int32_t int32_val;

	root = dag_config_get_root_component(card_ref);
	assert(root);
	duck = dag_component_get_subcomponent(root, kComponentDUCK, 0);
	/* We call the DUCK a DAG Clock because it's more user-friendly */
	if (duck)
	{
		printf("DAG Clock:\n");

		printf("Muxin: ");
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKRS422Input);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "rs422 ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKOverInput);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "over ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKHostInput);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "host ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKAuxInput);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "aux":"");
		}
		printf("\n");
		printf("Muxout: ");
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKRS422Output);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "rs422 ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKOverOutput);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "over ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKHostOutput);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "host ":"");
		}
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKLoop);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%s", bool_val ? "loop":"");
		}
		printf("\n");
		printf("Status: ");
		attr = dag_component_get_attribute_uuid(duck, kBooleanAttributeDUCKSynchronized);
		if (attr)
		{
			bool_val = dag_config_get_boolean_attribute(card_ref, attr);
			printf("%sSynchronized ", bool_val ? "":"Not ");
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKThreshold);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Threshold %.0fns ", uint32_val/ (0x100000000ll/1000000000.0));
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKFailures);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Failures %u ", uint32_val);
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKResyncs);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Resyncs %u", uint32_val);
		}
		printf("\nError: ");
		attr = dag_component_get_attribute_uuid(duck, kInt32AttributeDUCKFrequencyError);
		if (attr)
		{
			int32_val = dag_config_get_int32_attribute(card_ref, attr);
			printf("Freq %.0fppb ", int32_val/(0x100000000ll/1000000000.0));
		}
		attr = dag_component_get_attribute_uuid(duck, kInt32AttributeDUCKPhaseError);
		if (attr)
		{
			int32_val = dag_config_get_int32_attribute(card_ref, attr);
			printf("Phase %.0fns ", int32_val/(0x100000000ll/1000000000.0));
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKWorstFrequencyError);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Worst Freq %.0fppb ", uint32_val/(0x100000000ll/1000000000.0));
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKWorstPhaseError);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Worst Phase %.0fns", uint32_val/(0x100000000ll/1000000000.0));
		}
		printf("\nInput: ");
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKPulses);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Total %u", uint32_val);
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKResyncs);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Bad %u", uint32_val);
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKSinglePulsesMissing);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Single Pulses Missing %u", uint32_val);
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKLongestPulseMissing);
		if (attr)
		{
			uint32_val = dag_config_get_uint32_attribute(card_ref, attr);
			printf("Longest Sequence Missed %u", uint32_val);
		}
		printf("\nCrystal: ");
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKCrystalFrequency);
		if (attr)
		{
			int32_val = dag_config_get_int32_attribute(card_ref, attr);
			printf("Actual %dHz ", int32_val);
		}
		attr = dag_component_get_attribute_uuid(duck, kUint32AttributeDUCKSynthFrequency);
		if (attr)
		{
			int32_val = dag_config_get_int32_attribute(card_ref, attr);
			printf("Synthesized %dHz", int32_val);
		}
		printf("\n");
		attr = dag_component_get_attribute_uuid(duck, kStructAttributeDUCKTimeInfo);
		if (attr)
		{
			union
			{
				duck_time_info_t* ti;
				void* ptr;
			} time_info;
			dag_config_get_struct_attribute(card_ref, attr, &time_info.ptr);
			assert(time_info.ti);
			printf("Start: %s", ctime(&(time_info.ti->mStart)));
			printf("Host: %s", ctime(&(time_info.ti->mEnd)));
		}
		printf("\n");
	}
}
#endif

void
print_card_information(dag_card_ref_t card_ref)
{
   int serial = 0;
   int i = 0;
   char *copro_image_name = NULL;
   dag_component_t root;
   dag_firmware_t active_firmware = 0;

   int dagfd;
   romtab_t *rp;
   attr_uuid_t attr;
   int version;

   /* only dag3.7t and dag3.7t4 need to hold pbi bus */ 
   if (dag_config_get_card_type(card_ref) == kDag37t4 || dag_config_get_card_type(card_ref) == kDag37t)
   {
	dagfd = dag_config_get_card_fd(card_ref);
   	rp = dagrom_init(dagfd, 0,1);
   }
 
   if ((dag_config_get_card_type(card_ref) != kDag82x2) && (dag_config_get_card_type(card_ref) != kDag91x2Tx)) 
   {
	version = dag_firmware_controller_get_version(card_ref);
	if( version == kFirmwareVerNone)
	  return;

	printf("Card Information:\n");

	if( version == 0x2)
	{
	   active_firmware = dag_firmware_get_active(card_ref);
	   dag_firmware_read_serial_id(card_ref, &serial);
	   switch (active_firmware)
	   {	
		case kFirmwareFactory:
		     printf("Firmware: %s (factory)\n", dag_firmware_read_factory_firmware_name(card_ref));
               	     copro_image_name = dag_copro_read_factory_firmware_name(card_ref);
               	     if ( NULL != copro_image_name)
                    	printf("Copro: %s (factory)\n", copro_image_name);
		     break;
		
		case kFirmwareUser1:
		     printf("Firmware: %s (user 1)\n", dag_firmware_read_new_user_firmware_name(card_ref,1));
               	     copro_image_name = dag_copro_read_user_firmware_name(card_ref);
               	     if ( NULL != copro_image_name)
                   	printf("Copro: %s (user)\n", copro_image_name);
		     break;
		case kFirmwareUser2:
		     printf("Firmware: %s (user 2)\n", dag_firmware_read_new_user_firmware_name(card_ref,2));
               	     copro_image_name = dag_copro_read_user_firmware_name(card_ref);
               	     if ( NULL != copro_image_name)
                    	printf("Copro: %s (user)\n", copro_image_name);
		      break;
		case kFirmwareUser3:
		      printf("Firmware: %s (user 3)\n", dag_firmware_read_new_user_firmware_name(card_ref,3));
               	      copro_image_name = dag_copro_read_user_firmware_name(card_ref);
               	      if ( NULL != copro_image_name)
                    	 printf("Copro: %s (user)\n", copro_image_name);
		      break;
		case kFirmwareNone:
			break;
            	default:
                	break;
	    }	
	    printf("Serial : %d\n",serial) ;
	}
	else if ( (version == 0x1) || (version == 0x0) )
	{
	   active_firmware = dag_firmware_get_active(card_ref);
	   dag_firmware_read_serial_id(card_ref, &serial);
	   switch(active_firmware)
	   {
		   case kFirmwareFactory:
               		printf("Firmware: %s (factory)\n", dag_firmware_read_factory_firmware_name(card_ref));
               		copro_image_name = dag_copro_read_factory_firmware_name(card_ref);
               		if ( NULL != copro_image_name)
                    		printf("Copro: %s (factory)\n", copro_image_name);
			break;
		   case kFirmwareUser:
			printf("Firmware: %s (user)\n", dag_firmware_read_user_firmware_name(card_ref));
               		copro_image_name = dag_copro_read_user_firmware_name(card_ref);
               		if ( NULL != copro_image_name)
                    	        printf("Copro: %s (user)\n", copro_image_name);
		        break;
		   case kFirmwareNone:
			break;
            	default:
                	break;
	   }
	   printf("Serial : %d\n",serial) ;
	}
   } else {
     printf("Firmware: none - uses base card firmware\n");
   } 

   root = dag_config_get_root_component(card_ref);
   for(i = 0; i < dag_component_get_subcomponent_count_of_type(root, kComponentPort); i++)
   {
	   dag_component_t port = dag_component_get_subcomponent(root, kComponentXGMII, i);
	   
	   if(!port)
	   {
		 port = dag_component_get_subcomponent(root, kComponentPort, i);
	   }
	   attr = dag_component_get_attribute_uuid(port, kStringAttributeEthernetMACAddress);
	   if(attr != kNullAttributeUuid)
	   {
		   printf("MAC Address %c : %s\n", ('A' + i), dag_config_get_string_attribute(card_ref, attr));
	   }
   }
   printf("\n");
}

		
void
print_configuration(dag_card_ref_t card_ref)
{
	print_card_information(card_ref);
	switch(dag_config_get_card_type(card_ref))
	{
		case kDag37d:
			dag37d_print_configuration(card_ref);
			break;
		case kDag62:
			dag62se_print_configuration(card_ref);
			break;
		case kDag71s:
			dag71s_print_configuration(card_ref);
			break;
		case kDag70ge:
			dag70ge_print_configuration(card_ref);
			break;
		//case kDag82x:
		//case kDag82x2:
		//case kDag52x:
		//case kDag82z:
		//	dag82x_print_configuration(card_ref);
		//	break;
		
		//case kDag810:
		//	dag810_print_configuration(card_ref);
		//	break;
		default:
            /* discover all attributes automatically (dagx) */
			port_print_configuration(card_ref);
            print_framer_configuration(card_ref);
            print_demapper_configuration(card_ref);
			break;
	}
	/* print general component configuration information */
	print_card_configuration(card_ref);
}

static void
pbm_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute;
	dag_component_t root;
	dag_component_t pbm;
    attr_uuid_t attr_bus_type = kNullAttributeUuid;
	//dag_meminfo_t mem_info;
	//int count;

	root = dag_config_get_root_component(card_ref);
	pbm = dag_component_get_subcomponent(root, kComponentPbm, 0);
	if (pbm)
	{
		printf("PCI Burst Manager:\n");
		any_attribute = dag_component_get_attribute_uuid(pbm, kUint32AttributePCIBusSpeed);
		if (any_attribute)
		{
            attr_bus_type = dag_component_get_attribute_uuid(pbm, kUint32AttributePCIBusType);
            printf("%s %s\n",(attr_bus_type == kNullAttributeUuid)?"":dag_config_get_attribute_to_string(card_ref, attr_bus_type), dag_config_get_attribute_to_string(card_ref, any_attribute));
			/* this code is in the config and status api for the attribute 
			switch (dag_config_get_uint32_attribute(card_ref, any_attribute))
			{
			case kPCIBusSpeed33Mhz:
				printf("33Mhz ");
				break;

			case kPCIBusSpeed66Mhz:
				printf("66MHz ");
				break;

			case kPCIBusSpeed100Mhz:
				printf("100MHz ");
				break;

			case kPCIBusSpeed133Mhz:
				printf("133MHz ");
				break;

			case kPCIBusSpeedUnknown:
				printf("unknown bus speed ");
				break;

			case kPCIBusSpeedUnstable:
				printf("unstable bus speed ");
				break;
			}
			*/
		}
		any_attribute = dag_component_get_attribute_uuid(pbm, kUint32AttributeBufferSize);
		if (any_attribute)
			printf("buffer_size=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));

		any_attribute = dag_component_get_attribute_uuid(pbm, kUint32AttributeRxStreamCount);
		if (any_attribute)
			printf("rx_streams=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));

		any_attribute = dag_component_get_attribute_uuid(pbm, kUint32AttributeTxStreamCount);
		if (any_attribute)
			printf("tx_streams=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));

		/* we don't allow the user to change the (no)drop on the 2nd core of the 8.2x */
		if (dag_config_get_card_type(card_ref) == kDag82x2)
		{
			printf("\n\n");
			return;
		}
		any_attribute = dag_component_get_attribute_uuid(pbm, kBooleanAttributeDrop);
		if (any_attribute)
			printf("%sdrop ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

		any_attribute = dag_component_get_attribute_uuid(pbm, kBooleanAttributeOverlap);
		if (any_attribute)
			printf("%soverlap ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

		printf("\n\n");
	}

}

static void
stream_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t mem;
	//attr_uuid_t stream_count;
	dag_component_t root;
	dag_component_t stream;
	//dag_component_t pbm;
	uint32_t value;
	int total_streams = 0;
	int i = 0;


	root = dag_config_get_root_component(card_ref);
	//pbm = dag_component_get_subcomponent(root, kComponentPbm, 0);
	//stream_count = dag_component_get_attribute_uuid(pbm, kUint32AttributeTxStreamCount);
	//total_streams = dag_config_get_uint32_attribute(card_ref, stream_count);
	//stream_count = dag_component_get_attribute_uuid(pbm, kUint32AttributeRxStreamCount);
	//total_streams += dag_config_get_uint32_attribute(card_ref, stream_count);
	total_streams = dag_component_get_subcomponent_count_of_type(root, kComponentStream);
	if (total_streams > 0)
	{
		printf("Memory Streams:\n");
		printf("mem=");
		for (i = 0; i < total_streams; i++)
		{
			if (i > 0)
				printf(":");
			stream = dag_component_get_subcomponent(root, kComponentStream, i);
			assert(stream != NULL);
			mem = dag_component_get_attribute_uuid(stream, kUint32AttributeMem);
			assert(mem != kNullAttributeUuid);
			value = dag_config_get_uint32_attribute(card_ref, mem);
			printf("%u", value);
		}
		printf("\n\n");
	}
}

static void
attribute_print(dag_card_ref_t card, attr_uuid_t attr)
{
	dag_attr_t type;
	dag_attribute_code_t code;
	uint8_t val;
	const char* string;
	char number[40];

	type = dag_config_get_attribute_valuetype(card, attr);
	code = dag_config_get_attribute_code(attr);
	
    switch (code) {
	//TODO: to be moved to the attribute string representation at the moment is on /off in the config and status api 
	case kBooleanAttributeSonetMode :
		val = dag_config_get_boolean_attribute(card, attr);
		printf(" %s", val ?  "sonet" : "sdh");
		break;
	case kBooleanAttributeActive :
//we do not print here the attribute enable deisbale because we need the port number and this function is intended to be used not only port
// interface specific attributes 
// 		i should be index of the parent component of the attribute 
//		printf(" %sable%c", dag_config_get_boolean_attribute(card, attr) == 0 ? "dis":"en", ('a' + i) ); 
		break;
	case kBooleanAttributeRxEnable:
		break;
	case kBooleanAttributeMacWanMode:
	case kBooleanAttributeWanMode:
		val = dag_config_get_boolean_attribute(card, attr);
		printf(" %s", val ?  "wan" : "lan");
		break;
	default:

	/* default print function for the attributes 
		it printts the buoolean functions as yes or no
		the unit 32 is comparing for unimplemented or other bad string conversion 
		not required any more but well 
		leave every thing else as a string printout
	*/
		if (type == kAttributeBoolean)
		{
			uint8_t val = dag_config_get_boolean_attribute(card, attr);
			const char* name = dag_config_get_attribute_name(attr);
			printf(" %s%s", val ? "" : "no", name);
		}
		else if (type == kAttributeUint32)
		{
        		string = dag_config_get_attribute_to_string(card, attr);
        		snprintf(number, 40, "%u", dag_config_get_uint32_attribute(card, attr));

        		if (0 == strcmp(string, number)) /* numbers only */
        		{
            		printf(" %s=%s", dag_config_get_attribute_name(attr), string);
        		}else /* the string representation is not numerical */
        		{
           			printf(" %s", string);
        		}
		} else {
        		string = dag_config_get_attribute_to_string(card, attr);
            		printf(" %s", string);
		};	
    }
}

static bool
attribute_is_printable(dag_card_ref_t card_ref, dag_component_t comp, attr_uuid_t attr)
{
	dag_attribute_code_t code;
	dag_attr_config_status_t cs;
	int verbosity = dagutil_get_verbosity();
	int i;

	dag_component_code_t comp_code;
	comp_code = dag_config_get_component_code(comp);
	code = dag_config_get_attribute_code(attr);
	assert(code != kAttributeInvalid );
	
	cs = dag_config_get_attribute_config_status(attr);
    /* Removing drop_count from port configuration. It will be printed in
     * configuration */
	if (cs == kDagAttrStatus )//&& code != kUint32AttributeDropCount)
	{
		return false;
	}
	/*
	 *Removing the line_rate attribute from the sonet_pp from being displayed.
	 *However line_rate from the port component will be displayed.
	 */
	if(((comp_code == kComponentSonetPP)) && (code == kUint32AttributeLineRate))
	{
		return false;
	}
	if(verbosity > 3) return true;
	
	switch (code)
	{
		case kBooleanAttributeRefreshCache:
		case kBooleanAttributeCounterLatch:
		case kUint32AttributeForceLineRate:
		case kStructAttributeErfMux:
		case kBooleanAttributePayloadScrambleATM:
		case kBooleanAttributePayloadScramblePoS:
		case kBooleanAttributeSfpPwr:
		case kBooleanAttribute10GEthernetScrambling:
		case kBooleanAttributeWISScrambling:
		case kBooleanAttributeGTPPMAReset:
		case kBooleanAttributeTxEnable:
		case kBooleanAttributeTileReset:
		case kBooleanAttributeEnableFullRate:
        		return false;
/* Commenting out because for 71s this function won't get called 
 * It will be called only from port_print_configuration() which is not for 71s
 */
/*
		case kBooleanAttributeRocketIOPower:
		case kBooleanAttributeSfpPwr:
		case kBooleanAttributeLaser:
		{

			dag_card_t type = dag_config_get_card_type(card_ref);
			if (type == kDag452z8 || type == kDag452z || type == kDag452gf || type == kDag452cf || type == kDag452e || type == kDag454e)
			{
				return false;
			}

		}
		case kBooleanAttributeAlign64:
			{
			dag_card_t type = dag_config_get_card_type(card_ref);
			if (type == kDag71s) 
			return false;
			}
*/
		default:
			// Do nothing
			break;
	}

	/* Check the attribute is to be printed at this verbosity level*/
	for( i = 0; i < ( sizeof (port_attr_verbose_array) / sizeof(port_attr_verbose_array[0])) ; i++)
	{
		if (code == port_attr_verbose_array[i].attribute_code)
		{
		if (port_attr_verbose_array[i].attribute_verbosity > verbosity)
			return false;
		else
			return true;
		}
	}

	return true;
}

static void
port_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute;
	dag_component_t root, /*port,*/ comp;
	dag_attribute_code_t code;
	int port_count = 0;
	int comp_count = 0;
	int attr_count = 0;
	int i, j, k;
	/*Array of attribute code just to remove duplicate attributes.*/
	dag_attribute_code_t attribute_codes[MAX_PORT_CONFIG_ATTRIBUTES];
	int attribute_count = 0;
	int loop = 0;
	uint8_t found = 0;
    const dag_component_code_t comp_codes[] = {
    kComponentOptics,
	kComponentPhy,
	kComponentSonetPP,
	kComponentPCS,
	kComponentXGMII,
	kComponentDeframer,
	kComponentPort,
	kComponentMiniMac
    };

	root = dag_config_get_root_component(card_ref);
	port_count = dag_component_get_subcomponent_count_of_type(root, kComponentPort);
	//temp fix for for 5.0S Sonet or in case the card does not have portcomponent
	if( port_count <= 0 ) {
		port_count = dag_component_get_subcomponent_count_of_type(root, kComponentPhy);
	}
	
	for (i = 0; i < port_count; i++)
	{
		printf("Port %c:\n", 'A' + i);
        	/* other "port-like" components */
		printf(" ");
		attribute_count = 0;
        	for(k = 0; k < (sizeof(comp_codes)/sizeof(dag_component_code_t)); k++) 
		{
	            /* try each component type */
        	    code = comp_codes[k];
	            comp_count = dag_component_get_subcomponent_count_of_type(root, code);
        	    if(comp_count != 0)
	            {
			comp = dag_component_get_subcomponent(root, code, i);
        	        if ((comp_count >= i) && (kStateActive == dag_config_get_component_state(comp))) /* current port */ 
			{
                    		attr_count = dag_component_get_attribute_count(comp);

                    		for (j = 0; j < attr_count; j++)
                    		{
                        		any_attribute = dag_component_get_indexed_attribute_uuid(comp, j);
                        		code = dag_config_get_attribute_code(any_attribute);
					if (attribute_is_printable(card_ref,comp,any_attribute))
                        		{
						if(attribute_count == 0)
                                        	{
		                                        attribute_codes[0] = code;
                		                        attribute_count++;
                                		}
						else
						{
                                       			for (loop = 0; loop < attribute_count ; loop++)
                                        		{
                                                		if(attribute_codes[loop] == code)
                                                		{
                                                                	found = 1;
									break;
                                                		}
                                        		}
						}
						if(found != 1)
						{
							attribute_codes[attribute_count] = code;
							attribute_count++;
							attribute_print(card_ref, any_attribute);
						}
						else
						{
							found =0;
						}
                        		}
					
					if (code == kBooleanAttributeActive || code == kBooleanAttributeRxEnable)
					{
						printf(" %sable%c", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "dis":"en", ('a' + i) ); 
					};
                    }
                }
            }
        };

		printf("\n");
	};
	if (port_count > 0)
		printf("\n");
}

static void
print_framer_configuration(dag_card_ref_t card_ref)
{
    attr_uuid_t any_attribute;
    dag_component_t root, /*port,*/ comp;
    int framer_count = 0;
    int attr_count = 0;
    int i, j;
    
    root = dag_config_get_root_component(card_ref);
    framer_count = dag_component_get_subcomponent_count_of_type(root, kComponentFramer);
    if(framer_count != 0) {
           printf("Framer status:\n"); 
            for (i = 0; i < framer_count; i ++) {
                    comp = dag_component_get_subcomponent(root, kComponentFramer, i);
                    attr_count = dag_component_get_attribute_count(comp);

                    for (j = 0; j < attr_count; j++)
                    {
                            any_attribute = dag_component_get_indexed_attribute_uuid(comp, j);

                            if (attribute_is_printable(card_ref,comp,any_attribute))
                            {
                                    attribute_print(card_ref, any_attribute);
                            }

                   }
           }
           printf("\n\n");
    }

}
static void
print_demapper_configuration(dag_card_ref_t card_ref)
{
    attr_uuid_t any_attribute;
    dag_component_t root, /*port,*/ comp;
    int demapper_count = 0;
    int attr_count = 0;
    int i, j;
    
    root = dag_config_get_root_component(card_ref);
    demapper_count = dag_component_get_subcomponent_count_of_type(root, kComponentDemapper);
    if(demapper_count != 0) {
           printf("Demapper status:\n"); 
            for (i = 0; i <demapper_count; i ++) {
                    comp = dag_component_get_subcomponent(root, kComponentDemapper, i);
                    attr_count = dag_component_get_attribute_count(comp);

                    for (j = 0; j < attr_count; j++)
                    {
                            any_attribute = dag_component_get_indexed_attribute_uuid(comp, j);

                            if (attribute_is_printable(card_ref,comp,any_attribute))
                            {
                                    attribute_print(card_ref, any_attribute);
                            }

                   }
           }
           printf("\n\n");
    }
}
static void
gpp_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute;
	dag_component_t root;
	dag_component_t gpp;
	int boolean_attribute_value;
    uint32_t iface_count = 0;/* per GPP */
    uint32_t total_iface_count = 0; /* added to support multiple gpp*/
    int i = 0, j = 0;
    int gpp_count = 0;
    
	root = dag_config_get_root_component(card_ref);
	assert(root);
        /*Modification for InfiniBand Cards - GPP1 - Will be executed only if GPP 1 is present.*/	
    gpp_count =  dag_component_get_subcomponent_count_of_type(root,kComponentGpp);
    for( j = 0; j < gpp_count; j++)
    {
        gpp = dag_component_get_subcomponent(root, kComponentGpp, j);
        if (gpp)
        {
	    if(gpp_count>1)
		    printf("GPP%d:\n",j);
	    else
		    printf("GPP:\n");
            any_attribute = dag_component_get_attribute_uuid(gpp, kBooleanAttributeVarlen);
            if (any_attribute)
            printf("%svarlen ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

            any_attribute = dag_component_get_attribute_uuid(gpp, kUint32AttributeSnaplength);
            if (any_attribute)
                printf("slen=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));

            any_attribute = dag_component_get_attribute_uuid(gpp, kBooleanAttributeAlign64);
            if (any_attribute){
				    boolean_attribute_value = dag_config_get_boolean_attribute(card_ref, any_attribute);
				    printf("%salign64 ",  boolean_attribute_value == 0 ? "no":"");
            }
            printf("\n");
            any_attribute = dag_component_get_attribute_uuid(gpp, kUint32AttributeInterfaceCount);
            if (any_attribute)
            {
                iface_count = dag_config_get_uint32_attribute(card_ref, any_attribute);
            }
            for( i = 0; i < iface_count; i++)
            {
                any_attribute = dag_component_get_indexed_attribute_uuid_of_type( gpp, kUint32AttributeDropCount, i);
                printf("Port %c: drop_count = %u\n",'A'+total_iface_count+i, dag_config_get_uint32_attribute(card_ref, any_attribute));

            }
            total_iface_count += iface_count;
            iface_count = 0;
            printf("\n\n");
	   }
    }

}


static void print_attribute(dag_card_ref_t card_ref, dag_component_t component,dag_attribute_code_t attr_code)
{
	attr_uuid_t any_attribute;
    any_attribute = dag_component_get_attribute_uuid(component, attr_code);
    if (any_attribute) 
        printf("%s: %s\n", dag_config_get_attribute_name(any_attribute), dag_config_get_attribute_to_string(card_ref, any_attribute));
}
static void
terf_print_configuration(dag_card_ref_t card_ref)
{
    attr_uuid_t any_attribute; 
    dag_component_t root;
    dag_component_t terf;

    root = dag_config_get_root_component(card_ref);
    assert(root);
    terf = dag_component_get_subcomponent(root, kComponentTerf, 0);
    if (terf != NULL)
    {
        printf("TERF:\n");
        any_attribute = dag_component_get_attribute_uuid(terf, kUint32AttributeTerfStripCrc);
        if (any_attribute)
            printf("%s\n", dag_config_get_attribute_to_string(card_ref, any_attribute));
        
        print_attribute(card_ref, terf, kBooleanAttributeRXErrorA);
        print_attribute(card_ref, terf, kBooleanAttributeRXErrorB);
        print_attribute(card_ref, terf, kBooleanAttributeRXErrorC);
        print_attribute(card_ref, terf, kBooleanAttributeRXErrorD);
        print_attribute(card_ref, terf, kUint32AttributeTimeMode);
        printf("\n");
    }
}    
	

static void
hlb_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute; 
	dag_component_t root;
	dag_component_t hlb;
	hlb_range_t*	p_range_data;
	void*		ptr;
	uint32_t			loop;
	

	root = dag_config_get_root_component(card_ref);
	assert(root);
	hlb = dag_component_get_subcomponent(root, kComponentHlb, 0);

	if (hlb != NULL)
	{
		printf("Hash Load Balancer:\n");
		any_attribute = dag_component_get_attribute_uuid(hlb, kStructAttributeHlbRange);
		if (any_attribute)
		{
			printf("hlb=");
			if(0==dag_config_get_struct_attribute(card_ref,any_attribute, &ptr))
			{
				p_range_data = ptr;
				for(loop=0;loop<p_range_data->stream_num;loop++)
				{
					if(loop)
						printf(":");
					printf("%d.%0d-%d.%0d",
							p_range_data->stream_range[loop].min/10,
							p_range_data->stream_range[loop].min%10,
							p_range_data->stream_range[loop].max/10,
							p_range_data->stream_range[loop].max%10);
				}
				printf("\n");
			}
		}
	}
}
static void
hat_print_configuration(dag_card_ref_t card_ref)
{
	attr_uuid_t any_attribute; 
	dag_component_t root;
	dag_component_t hlb;
	hat_range_t*	p_range_data;
	void*	  ptr;
	uint32_t  loop;
	
	root = dag_config_get_root_component(card_ref);
	assert(root);
	hlb = dag_component_get_subcomponent(root, kComponentHAT, 0);

	if (hlb != NULL)
	{
		printf("Hash Load Balancer:\n");
		any_attribute = dag_component_get_attribute_uuid(hlb, kStructAttributeHATRange);
		if (any_attribute)
		{
			printf("hat=");
			if(0==dag_config_get_struct_attribute(card_ref,any_attribute, &ptr))
			{
				p_range_data = ptr;
				for(loop=0;loop<p_range_data->bin_num;loop++)
				{
					if(loop)
						printf(":");
					printf("%d.%0d-%d.%0d",
							p_range_data->bin_range[loop].min/10,
							p_range_data->bin_range[loop].min%10,
							p_range_data->bin_range[loop].max/10,
							p_range_data->bin_range[loop].max%10);
				}
				printf("\n");
			}
		}
	}
}



static void
dag62se_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t any_component;
	dag_component_t root_component;
	attr_uuid_t any_attribute;
	int i =0;
	int count = 0;
	network_mode_t net_mode;
	ethernet_mode_t eth_mode;


	root_component = dag_config_get_root_component(card_ref);
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
	if( count > 0 )
	{
		for(i = 0; i < count; i++)
		{
			printf("Port %c: ", 'A' + i);

			any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
			assert(any_component);

			/* Network Mode */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
			assert(any_attribute != kNullAttributeUuid);
			net_mode = dag_config_get_uint32_attribute(card_ref, any_attribute);
			switch(net_mode)
			{
				case kNetworkModePoS:
					printf("POS ");
					break;
				case kNetworkModeEth:
					printf("ETH ");
					break;
				default:
					break;
			}
			/* Ethernet Mode */
			if(net_mode == kNetworkModeEth)
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeEthernetMode);
				assert(any_attribute != kNullAttributeUuid);
				eth_mode = dag_config_get_uint32_attribute(card_ref, any_attribute);
				switch(eth_mode)
				{
					case kEthernetMode10GBase_LR:
						printf("LAN ");
						break;
					case kEthernetMode10GBase_LW:
						printf("WAN ");
						break;
					default:
						break;
				}
			}

			/* crc select */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeCrcSelect);
			assert(any_attribute != kNullAttributeUuid);
			printf("%scrc ", dag_config_get_uint32_attribute(card_ref, any_attribute) == kCrcOff ?"no":"");
		  
			/* crcstrip */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeCrcStrip);
			assert(any_attribute != kNullAttributeUuid);
			printf("%scrcstrip ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			/* eql */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
			assert(any_attribute != kNullAttributeUuid);
			printf("%seql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			/* fcl */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFacilityLoopback);
			assert(any_attribute != kNullAttributeUuid);
			printf("%sfcl ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			/* lseql */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLineSideEquipmentLoopback);
			assert(any_attribute != kNullAttributeUuid);
			printf("%slseql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			/* lsfcl */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLineSideFacilityLoopback);
			assert(any_attribute != kNullAttributeUuid);
			printf("%slsfcl ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			/* pscramble */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePayloadScramble);
			assert(any_attribute != kNullAttributeUuid);
			printf("%spscramble ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			/*pmin */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePMinCheck);
			assert(any_attribute != kNullAttributeUuid);
			printf("%spmin ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			/* pmax */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePMaxCheck);
			assert(any_attribute != kNullAttributeUuid);
			printf("%spmax ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			/* short */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeMinPktLen);
			assert(any_attribute != kNullAttributeUuid);
			printf("short=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));

			/* long */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeMaxPktLen);
			assert(any_attribute != kNullAttributeUuid);
			printf("long=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));
			
			printf("\n");

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSteer);
			if(any_attribute != kNullAttributeUuid)
			{
			   steer_t steer = dag_config_get_uint32_attribute(card_ref, any_attribute);
			   printf("steer=");
			   switch(steer)
			   {
				   case kSteerStream0:
					   printf("stream0 ");
					   break;
				   case kSteerParity:
					   printf("parity ");
					   break;
				   case kSteerCrc:
					   printf("crc ");
					   break;
				   case kSteerIface:
					   printf("iface ");
					   break;
					case kSteerColour:
					   printf("colour/DSM");
					   break;
			   }
			}
		}
		printf("\n");
	}

	if( dag_component_get_subcomponent_count_of_type(root_component, kComponentGpp) > 0 )
	{
		any_component = dag_component_get_subcomponent(root_component, kComponentPort,0);
		assert(any_component);
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeDropCount);
		assert(any_attribute != kNullAttributeUuid);
		printf("drop_count=%u ", dag_config_get_uint32_attribute(card_ref, any_attribute));
		
	}
}

static void
dag70ge_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t any_component;
	dag_component_t root_component;
	attr_uuid_t any_attribute;
	int i =0;
	int count = 0;
	root_component = dag_config_get_root_component(card_ref);
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
	for(i = 0; i < count; i++)
	{
		printf("Port %c: ", 'A' + i);
		
		any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
		assert(any_component);
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeNic);
		assert(any_attribute != kNullAttributeUuid);
		printf("%snic ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
		assert(any_attribute != kNullAttributeUuid);
		printf("%seql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRocketIOPower);
		assert(any_attribute != kNullAttributeUuid);
		printf("%srktiopwr ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLaser);
		assert(any_attribute != kNullAttributeUuid);
		printf("%slaser ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
		
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSfpPwr);
		assert(any_attribute != kNullAttributeUuid);
		printf("%ssfppwr ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSFPDetect);
		assert(any_attribute != kNullAttributeUuid);
		printf("%sdetect ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
		
		printf("\n");
	}
}

static void
dag71s_print_configuration(dag_card_ref_t card_ref)
{
	int i;
	int count;

	dag_component_t any_component = 0;
	dag_component_t root_component = 0;
	attr_uuid_t any_attribute;
	root_component = dag_config_get_root_component(card_ref);
	/* Read the optics information */
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
	
	printf("SFP status:\n");
	for (i = 0; i < count; i++)
	{
		printf("SFP %c: ", 'A' + i);
		any_component = dag_component_get_subcomponent(root_component, kComponentOptics, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeDetect);
			if (dag_config_get_boolean_attribute(card_ref, any_attribute) == 1)
			{
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLaser);
				if (any_attribute)
					printf("%slaser ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSignal);
				if (any_attribute)
					printf("%ssignal ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSfpPwr);
				if (any_attribute)
					printf("%ssfppwr ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			}
			else
			{
				printf("No SFP module.");
			}

			printf("\n");
		}
		else
		{
			printf(" component not found\n");
		}
	}

	printf("\nPort status\n");
	for (i = 0; i < count; i++)
	{
		printf("Port %c: ", 'A' + i);
		any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeCore);
			if (dag_config_get_boolean_attribute(card_ref, any_attribute) == 1)
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLock);
				if (any_attribute)
					printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "lock":"nolock");
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeLineRate);
				if (any_attribute)
					printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFIFOError);
				if (any_attribute)
					printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "fifo_error":"nofifo_error");

				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeMasterSlave);
				if (any_attribute)
					printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeActive);
				if (any_attribute)
					printf("%sable%c ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "dis":"en", ('a' + i) );
			}
			else
			{
				printf("Core off!\n");
			}

			printf("\n");
		}
		else
		{
			printf(" component not found\n");
		}
	}

	printf("\nSONET/SDH status\n");
	for (i = 0; i < count; i++)
	{
		printf("SONET %c: ", 'A' + i);
		any_component = dag_component_get_subcomponent(root_component, kComponentSonic, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeLineRate);
			if (any_attribute)
				printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeVCSize);
			if (any_attribute)
				printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeScramble);
			if (any_attribute)
				printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "scramble":"noscramble");

			any_component = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
			if (any_component)
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
				if (dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeChannelized)
				{
					any_component = dag_component_get_subcomponent(root_component, kComponentSonic, i);

					any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeTributaryUnit);
					if (any_attribute)
						printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

					any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributePayloadMapping);
					if (any_attribute)
						printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));
				}
			}
 
			printf("\n");
		}
		else
		{
			printf(" component not found.\n");
		}
	}

	any_component = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
	if (any_component)
	{
		any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
		if (dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeChannelized)
		{
			printf("\nE1/T1 status\n");
			for (i = 0; i < count; i++)
			{
				printf("E1/T1 %c: ", 'A' + i);
				any_component = dag_component_get_subcomponent(root_component, kComponentE1T1, i);
				if (any_component)
				{
					any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeLineType);
					if (any_attribute)
						printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

					any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeE1T1GenerateAlarmIndication);
					if (any_attribute)
						printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "txais":"notxais");

					printf("\n");
				}
				else
				{
					printf(" component not found.\n");
				}
			}
		}
	}

	//Print the PHY status
	printf("\nPhy status (AMCC Quad Transceiver):\n");
	any_component = dag_component_get_subcomponent(root_component, kComponentPhy, 0);
	if (any_component)
	{
		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
		if (any_attribute)
			printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "eql" : "noeql");

		any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFacilityLoopback);
		if (any_attribute)
			printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "fcl" : "nofcl");

		printf("\n");
	}
	else
	{
		printf(" component not found.\n");
	}

	//Print the mappers and demapper information
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDemapper);
	if (count > 0)
	{
		any_component = dag_component_get_subcomponent(root_component, kComponentDemapper, 0);
		any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
		if (any_attribute)
			printf("\n%s Demapper Status:\n", dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeConcatenated ?
				   "Concatenated" : dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeChannelized ? "Channelized":"invalid" );
	}
	for (i = 0; i < count; i++)
	{
		any_component = dag_component_get_subcomponent(root_component, kComponentDemapper, i);
		if (any_component != NULL)
		{
			sonet_type_t sonet_type;
			//which demapper is it?
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
			sonet_type = dag_config_get_uint32_attribute(card_ref, any_attribute);
			if (kSonetTypeChannelized == sonet_type)
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32Attribute71sChannelizedRevisionID);
				if (any_attribute)
					printf("%s\n", dag_config_get_attribute_to_string(card_ref, any_attribute));

				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeTimeStampEnd);
				if (any_attribute)
					printf("Timestamp measured from %s of packet\n", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "tail":"head");

				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeHDLCSnaplength);
				if (any_attribute)
					printf("HDLC snap length = %d ", dag_config_get_uint32_attribute(card_ref, any_attribute));

                any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeRAWSnaplength);
                if (any_attribute)
					printf("RAW snap length = %d ", dag_config_get_uint32_attribute(card_ref, any_attribute));
                
                printf("\n");

			}
			else if (kSonetTypeConcatenated == sonet_type)
			{				
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
				if (any_attribute)
					printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

				if (dag_config_get_uint32_attribute(card_ref, any_attribute) == kNetworkModePoS)
				{
					any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeCrcSelect);
					if (any_attribute)
						printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));
				}
				else
				{
					any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeIdleCellMode);
					if (any_attribute)
						printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "aidle" : "noaidle");
				}

				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePayloadScramble);
				if (any_attribute)
					printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "pscramble" : "nopscramble");
			}
			printf("\n");
		}			
	}

	printf("\n");

	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentMapper);
	if (count > 0)
	{
		any_component = dag_component_get_subcomponent(root_component, kComponentMapper, 0);
		any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
		if (any_attribute)
			printf("\n%s Mapper Status:\n", dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeConcatenated ?
				"Concatenated" : dag_config_get_uint32_attribute(card_ref, any_attribute) == kSonetTypeChannelized ? "Channelized":"invalid" );
	}
	for (i = 0; i < count; i++)
	{
		any_component = dag_component_get_subcomponent(root_component, kComponentMapper, i);
		if (any_component != NULL)
		{
			sonet_type_t sonet_type;
			//which demapper is it?
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeSonetType);
			sonet_type = dag_config_get_uint32_attribute(card_ref, any_attribute);
			if (kSonetTypeChannelized == sonet_type)
			{
			}
			else if (kSonetTypeConcatenated == sonet_type)
			{
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
				if (any_attribute)
					printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));

				if (dag_config_get_uint32_attribute(card_ref, any_attribute) == kNetworkModePoS)
				{
					any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeCrcSelect);
					if (any_attribute)
						printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));
				}
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePayloadScramble);
				if (any_attribute)
					printf("%s ", dag_config_get_uint32_attribute(card_ref, any_attribute) == 1 ? "pscramble" : "nopscramble");
			}
			printf("\n");
		}
	}
	if (count > 0)
		printf("\n");

}

static void
dag37d_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t any_component;
	dag_component_t root_component;
	attr_uuid_t any_attribute;
	int i =0;
	int count = 0;
	root_component = dag_config_get_root_component(card_ref);
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
	for(i = 0; i < count; i++)
	{
		printf("Port %c: ", 'A' + i);
		
		any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
			if (any_attribute)
				printf("%seql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFacilityLoopback);
			if (any_attribute)
				printf("%sfcl ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");		

			/* Framing Mode */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeFramingMode);
			if (any_attribute)
			{
				switch((framing_mode_t)dag_config_get_uint32_attribute(card_ref, any_attribute))
				{
					case kFramingModeDs3m23:
						printf("ds_m23 ");
						break;
					case kFramingModeDs3m23FF:
						printf("ds_m23_ff ");
						break;
					case kFramingModeDs3m23IF:
						printf("ds_m23_if ");
						break;
					case kFramingModeDs3m23EF:
						printf("ds3_m23_ef ");
						break;
					case kFramingModeDs3Cbit:
						printf("ds3_cbit ");
						break;
					case kFramingModeDs3CbitFF:
						printf("ds3_cbit_ff ");
						break;
					case kFramingModeDs3CbitEF:
						printf("ds3_cbit_ef ");
						break;
					case kFramingModeDs3CbitIF:
						printf("ds3_cbit_if ");
						break;
					case kFramingModeE3:
						printf("e3_g751 ");
						break;
					case kFramingModeDS3CbitPLCP:
						printf("ds3_cbit_plcp ");
						break;
					case kFramingModeDS3M23PLCP:
						printf("ds3_m23_plcp ");
						break;
					case kFramingModeE3G751PLCP:
						printf("e3_g751_plcp ");
						break;
					case kFramingModeE3G751IF:
						printf("e3_g751_if ");
						break;
					case kFramingModeE3CC:
						printf("e3_cc ");
						break;

					default:
						printf("invalid_framingmode ");
				}
			}
					

			/* Network Mode */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
			if (any_attribute)
				printf("%s ", dag_config_get_uint32_attribute(card_ref, any_attribute) == kNetworkModeHDLC ? "hdlc":"atm");

			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeRXMonitorMode);
			if (any_attribute)
				printf("%srx_monitor ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");		 

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeCrcSelect);
			if (any_attribute)
			{
				switch((crc_t)dag_config_get_uint32_attribute(card_ref, any_attribute))
				{
					case kCrcOff:
						printf("crcoff ");
						break;
					case kCrc16:
						printf("crc16 ");
						break;
					case kCrc32:
						printf("crc32 ");
						break;
					case kCrcInvalid:
						printf("invalid");
						break;
				}
			}

			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeDescramble);
			if (any_attribute)
				printf("%sdescramble ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePayloadScramble);	
			if (any_attribute)
				printf("%spscramble ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
	
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLinkDiscard);
			if (any_attribute)
				printf("%sdiscard ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeNetworkMode);
			if (!(dag_config_get_uint32_attribute(card_ref, any_attribute) == kNetworkModeATM))
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFF00Delete);
				if (any_attribute)
					printf("%sff00del ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			}

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeFramingMode);
			if (dag_config_get_uint32_attribute(card_ref, any_attribute) == kFramingModeE3CC)
			{
				any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeHDLCFraction);
				if (any_attribute)
					printf("fract=%d ", dag_config_get_uint32_attribute(card_ref, any_attribute));
			}

		}

		printf("\n");
	}
	printf("\n\n");
}

#if 0
static void
dag810_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t any_component;
	dag_component_t root_component;
	attr_uuid_t any_attribute;
	int i =0;
	int count = 0;
	root_component = dag_config_get_root_component(card_ref);
	
	/* Read the optics information */
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentOptics);
	for (i = 0; i < count; i++)
	{
		printf("Xfp %c: ", 'A' + i);
		any_component = dag_component_get_subcomponent(root_component, kComponentOptics, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeDetect);
			if (dag_config_get_boolean_attribute(card_ref, any_attribute) == 1)
			{
				
				any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSfpPwr);
				if (any_attribute)
					printf("%spwr ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");

			}
			else
			{
				printf("No XFP module.");
			}

			printf("\n");
		}
		else
		{
			printf(" component not found\n");
		}
	}

	/* Read the port information */
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
	for(i = 0; i < count; i++)
	{
		printf("Port %c: ", 'A' + i);
		
		any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
		if (any_component)
		{
			/* Lock */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeLock);
			if (any_attribute)
				printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "lock":"nolock");
				
			/* eql */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
			if (any_attribute)
				printf("%seql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "":"no");
			
			/* fcl */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFacilityLoopback);
			if (any_attribute)
				printf("%sfcl ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "":"no");		
			
		}

		any_component = dag_component_get_subcomponent(root_component, kComponentDeframer, i);
		if (any_component)
		{
			/* Mode select */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeSonetMode);
			if (any_attribute)
				printf("%s ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "sonet" : "sdh");
			
			/* crc select */
			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeCrcSelect);
			if (any_attribute)
				printf("%s ", dag_config_get_attribute_to_string(card_ref, any_attribute));
			
			/* payload scramble (demapper/mapper) */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributePayloadScramble);
			if (any_attribute)
				printf("%spscramble ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "" : "no");
			
			/* scramble (deframer/framer) */
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeScramble);
			if (any_attribute)
				printf("%sscramble ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 1 ? "" : "no");

		}

		printf("\n\n");
	}
	

}
#endif

#if 0
static void
dag82x_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t any_component;
	dag_component_t root_component;
	attr_uuid_t any_attribute;
	int i =0;
	int count = 0;
	root_component = dag_config_get_root_component(card_ref);
	count = dag_component_get_subcomponent_count_of_type(root_component, kComponentPort);
	for(i = 0; i < count; i++)
	{
		printf("Port %c: ", 'A' + i);
		
		any_component = dag_component_get_subcomponent(root_component, kComponentXGMII, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeTxCrc);
			if (any_attribute)
				printf("%stx_crc ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");		
        }

		any_component = dag_component_get_subcomponent(root_component, kComponentPort, i);
		if (any_component)
		{
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeEquipmentLoopback);
			if (any_attribute)
				printf("%seql ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");
			
			any_attribute = dag_component_get_attribute_uuid(any_component, kBooleanAttributeFacilityLoopback);
			if (any_attribute)
				printf("%sfcl ", dag_config_get_boolean_attribute(card_ref, any_attribute) == 0 ? "no":"");		

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeEthernetMode);
			if (any_attribute)
			{
				switch((ethernet_mode_t)dag_config_get_uint32_attribute(card_ref, any_attribute))
				{
					case kEthernetMode10GBase_LR:
						printf("10gbase_lr ");
						break;
					case kEthernetMode10GBase_ER:
						printf("10gbase_er ");
						break;
					case kEthernetMode10GBase_SR:
						printf("10gbase_sr ");
						break;
					default:
						break;
				}
			}

			any_attribute = dag_component_get_attribute_uuid(any_component, kUint32AttributeDropCount);
			if (any_attribute)
				printf("drop_count = %u\n", dag_config_get_uint32_attribute(card_ref, any_attribute));
		}

		printf("\n");
	}

}
#endif

void
steering_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t root_component = dag_config_get_root_component(card_ref);
	dag_component_t steering_component = dag_component_get_subcomponent(root_component, kComponentSteering, 0);
	
	if (steering_component)
	{
		attr_uuid_t steer_value;

		steer_value = dag_component_get_attribute_uuid(steering_component, kUint32AttributeSteer);
		printf("Steering: %s\n", dag_config_get_attribute_to_string(card_ref, steer_value));
		printf("\n");
	}
}
void
dsm_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t root_component = dag_config_get_root_component(card_ref);
	dag_component_t dsm_component = dag_component_get_subcomponent(root_component, kComponentDSM, 0);
	
	if (dsm_component)
	{
		attr_uuid_t attr;

		attr = dag_component_get_attribute_uuid(dsm_component, kBooleanAttributeDSMBypass);
		printf("DSM Bypass: %s\n", dag_config_get_attribute_to_string(card_ref, attr));
		printf("\n");
	}
}
void infiniband_print_mode(dag_card_ref_t card_ref)
{
	if((dag_config_get_card_type(card_ref) == kDag850) || (dag_config_get_card_type(card_ref) == kDag840))
	{
		if (dag_config_get_infiniband_monitormode(card_ref))
		{
			if (dag_config_get_infiniband_tapmode(card_ref))
			{
				printf("Configured in Tap Mode \n");		
			}		
			else
			{
				printf("Configured in Monitor Mode \n");
			}
		}
		else
		{
			printf("Not configured in any mode \n");
		}		
	}
}

// This component and its attributes are only applicable to the 3.7ge
void
mux_print_configuration(dag_card_ref_t card_ref)
{
	dag_component_t root_component = dag_config_get_root_component(card_ref);
	dag_component_t mux_component = dag_component_get_subcomponent(root_component, kComponentMux, 0);
	
	if (mux_component)
	{
		attr_uuid_t attr;

		printf("Mux:\n");
		attr = dag_component_get_attribute_uuid(mux_component, kUint32AttributeSteer);
		if(attr != kNullAttributeUuid)
		{
			steer_t steer = dag_config_get_uint32_attribute(card_ref, attr);
			printf("steer=");
			switch(steer)
			{
				case kSteerStream0:
					printf("stream0 ");
					break;
				case kSteerIface:
					printf("iface ");
					break;

				default:
					printf("invalid ");
					break;
			}
		}

		attr = dag_component_get_attribute_uuid(mux_component, kBooleanAttributeSwap);
		if (attr != kNullAttributeUuid)
			printf("%sifaceswap", dag_config_get_boolean_attribute(card_ref, attr) == 0 ? "no":"");
		
		printf("\n\n");
	}
}

/***********************************************************************
* FUNCTION:	void print_card_configuration(dag_card_ref_t card_ref)
* DESCRIPTION:	This function is intended to provide a general mechanism
* 		to print configruation data for the components present
* 		in any card.
* INPUTS:	card_ref	- the card to print config for
* RETURNS:	none.
* 
* NOTE:		All remaming generic component configuration output will
* 		be moved to here.
***********************************************************************/
void print_card_configuration(dag_card_ref_t card_ref)
{
	int count,ind;
	int dagfd;
	dag_component_t comp;
	attr_uuid_t attr;
	dag_component_t root_component = dag_config_get_root_component(card_ref);

	gpp_print_configuration(card_ref);

	/* Drop Component drop counts */
	if((dag_component_get_subcomponent(root_component, kComponentStreamFeatures,0) != NULL))
	{	
		comp = dag_component_get_subcomponent(root_component, kComponentStreamFeatures,0);
		// get stream_count 
        attr = dag_component_get_attribute_uuid(comp, kUint32AttributeNumberOfStreams);
		dagfd = dag_config_get_card_fd(card_ref);
	    count = dag_rx_get_stream_count(dagfd);
		//dag_config_get_uint32_attribute(card_ref,attr);
	    attr = dag_component_get_attribute_uuid(comp,kBooleanAttributePerStreamDropCounterPresent);
		
		if(dag_config_get_boolean_attribute(card_ref,attr) == 1)
		{
			attr = dag_component_get_attribute_uuid(comp,kBooleanAttributePerStreamDropCounterPresent);
			for(ind = 0 ; ind < count; ind++)
			{
				attr = dag_component_get_indexed_attribute_uuid_of_type(comp, kUint32AttributeStreamDropCount,ind);
				assert(attr != kNullAttributeUuid);
				printf("Stream %d: stream_drop_count = %u\n", ind * 2,
					  dag_config_get_uint32_attribute(card_ref, attr)); 
			}
		}
		else if((count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDrop)) > 0)
        	{
			for (ind = 0; ind < count; ind++)
            		{
				comp = dag_component_get_subcomponent(root_component, kComponentDrop, ind);
				attr = dag_component_get_attribute_uuid(comp, kUint32AttributeStreamDropCount);
				assert(attr != kNullAttributeUuid);
				printf("Stream %d: stream_drop_count = %u\n", ind * 2, dag_config_get_uint32_attribute(card_ref, attr));
            		}
 		        printf("\n");
        	}

	}
	else if ((count = dag_component_get_subcomponent_count_of_type(root_component, kComponentDrop)) > 0) 
	{
	    for (ind = 0; ind < count; ind++)
	    {
		    comp = dag_component_get_subcomponent(root_component, kComponentDrop, ind);
		    attr = dag_component_get_attribute_uuid(comp, kUint32AttributeStreamDropCount);
		    assert(attr != kNullAttributeUuid);
		    printf("Stream %d: stream_drop_count = %u\n", ind * 2, dag_config_get_uint32_attribute(card_ref, attr)); 
	    }
	    printf("\n");
	}
	
	infiniband_print_mode(card_ref);
	dsm_print_configuration(card_ref);
	steering_print_configuration(card_ref);
	pbm_print_configuration(card_ref);
	stream_print_configuration(card_ref);
	terf_print_configuration(card_ref);
	hat_print_configuration(card_ref);
	hlb_print_configuration(card_ref);
	mux_print_configuration(card_ref);
}




/***********************************************************************
* FUNCTION:	void print_card_tree(dag_card_ref_t card_ref)
* DESCRIPTION:	This function is intended to provide a general mechanism
* 		to print tree of the components and attributes and subcomponents presented on the card
* INPUTS:	card_ref	- the card to print config for
		level - how many levels of componets subcomponets to print
		verb - what level of verb
* RETURNS:	none.
* 
* NOTE:		It is used mainy for debuging testing 
***********************************************************************/
void print_card_tree(dag_card_ref_t card_ref, int Level, int verb)
{
	int comp_count,comp_ind;
	dag_component_t component;
	dag_component_t sub_component;
	int subcomp_count,subcomp_ind;
	dag_component_t root_component = dag_config_get_root_component(card_ref);

	if ((comp_count = dag_component_get_subcomponent_count(root_component)) > 0)
	{
		for (comp_ind = 0; comp_ind < comp_count; comp_ind++)
		{
			component = dag_component_get_indexed_subcomponent(root_component, comp_ind);
			if((subcomp_count = dag_component_get_subcomponent_count(component)) > 0)
			{
				for(subcomp_ind = 0;subcomp_ind < subcomp_count; subcomp_ind++)
				{
					sub_component = dag_component_get_indexed_subcomponent(component,subcomp_ind);
					print_attribute_tree(card_ref,sub_component);	
				}	
			}
			print_attribute_tree(card_ref,component);
		}
	}
}

/***********************************************************************
* FUNCTION:	void print_attribute_tree(dag_card_ref_t card_ref,dag_component_t component)
* DESCRIPTION:	This function is takes a component reference and prints out all the 
		attributes of that particular component
* 		
* INPUTS:	card_ref	- the card to print config for
		verb - what level of verb
* RETURNS:	none.
* 
* NOTE:		It is used mainy for debuging testing 
***********************************************************************/
void print_attribute_tree(dag_card_ref_t card_ref,dag_component_t component)
{

	int attr_count,attr_ind;
	attr_uuid_t attr;	
	if ( dagutil_get_verbosity() >= 0 )
	{
		printf("Component: name= %s", dag_config_get_component_name(component) ); 
		printf(", code=%d(%s)", dag_config_get_component_code(component),dag_config_get_component_code_as_string(component));
		printf(", description= %s ", dag_config_get_component_description(component));
		printf(", Attributes: count= %d,", dag_component_get_attribute_count(component));
		printf("Component State= %s \n",dag_config_get_component_state_as_string(component));
	}
			
	attr_count = dag_component_get_attribute_count(component);
	if( attr_count > 0) {
	if( dagutil_get_verbosity() > 1 )
		printf("\tComponent: name   ");
	
	printf("\tAttribute:  name    ,    type   ,    value    ");
			
	if( dagutil_get_verbosity() > 1 )
		printf(", value type         ");
	if( dagutil_get_verbosity() > 0 )
		printf(",    attribute code    " );
		printf(",     description \n");		    
	for(attr_ind = 0; attr_ind < attr_count; attr_ind ++)
	{
		attr = dag_component_get_indexed_attribute_uuid(component, attr_ind);
		assert(attr != kNullAttributeUuid);
		if ( dagutil_get_verbosity() >= 0 )
		{
			if(dagutil_get_verbosity() >= 2)
				printf("\t   %-17s,", dag_config_get_component_name(component) ); 

			printf("\t   %-17s", dag_config_get_attribute_name(attr));
			printf(",  %-8s ", dag_config_get_attribute_type_to_string(attr));
			printf(",  %-10s ", dag_config_get_attribute_to_string(card_ref,attr));
			//value type is generally easy to understand via the component code string representation
			if( dagutil_get_verbosity() > 1 )
				printf(",  %-17s ", dag_config_get_attribute_valuetype_to_string(attr));
			if( dagutil_get_verbosity() > 0 ) 
				printf(",  %-30s ", dag_config_get_attribute_code_as_string(attr));
				printf(", %-30s\n", dag_config_get_attribute_description(attr));
		}
	};
	};
	printf("\n");
};







