/*
 * Copyright (c) 2002-2007 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: ifc.c 15597 2012-03-28 22:21:32Z jomi.gregory $
 */

/* Documentation notes: 
 * --------------------
 *
 * The test cases and the test plan is documented int he IPF and HAT 
 * test plan. Each test case and test doesn't have an in code documentation.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dagcrc.h"
#include "dagclarg.h"
#include "dagapi.h"
#include "dagerf.h"
#include "ifc.h"

#define IP_ADDR_MAKE(a,b,c,d) (((a&0xff)<<24)|((b&0xff)<<16)|((c&0xff)<<8)|(d&0xff))

/* Command Line Arguments constants */
enum
{
	CLA_IFC_LIST
};

/* Structure to hold the information required by the tests */
typedef struct ipf_fields
{
	uint8_t  ifc;
	uint8_t  rectype;
	uint16_t rlen;
	uint16_t wlen;
	uint16_t colour;
	uint8_t  stream;
	uint8_t  hash;
	uint16_t etype;
	uint32_t src_ip;
	uint32_t dst_ip;
	uint16_t ip_prot;
	uint16_t src_port;
	uint16_t dst_port;
	uint8_t  tcp_flags;
	uint8_t  icmp_type;
	uint8_t  icmp_code;
} ipf_fields_t;

/* Internal Function prototypes */
int parse_rec(char *rec, int len, ipf_fields_t *flds);
void incr_counters(int res);

/* Global variables for this test */
char rec_msg[MAX_STR_LEN];
int rec_msg_len = 0;

global_params_t settings;

/* Global counters for final reporting */
uint32_t ifc_total = 0;
uint32_t ifc_ignored = 0;
uint32_t ifc_failed = 0;
uint32_t ifc_passed = 0;
uint32_t ifc_warning = 0;
uint32_t ifc_unknown = 0;

int pkt_cap[MAX_IFC_CNT];

uint8_t expc_ifc_lst[MAX_IFC_CNT];

/* Main initialization of the test.
 * Valid parameters are: 
 * -i N (Expected capture interface values 0 to 3)
 *       May repeat more than once, values are kept in a list
 */
int ifc_init(char *params[], int param_cnt, global_params_t *global_params, test_printf f)
{

	ClArgPtr clarg = NULL;
	FILE* errorfile = NULL;
	int argindex = 0;
	int clarg_result = 0;
	int code = 0;
	int h_val;

	memset(expc_ifc_lst, 0, sizeof(expc_ifc_lst));
	memset(pkt_cap, 0, sizeof(pkt_cap));

	/* Set up the command line options. */
	clarg = dagclarg_init(param_cnt, (const char* const *) params);
	
	/* General options. */
	dagclarg_add_int(clarg, "Indicate the expected interfaces", "--ifc", 'i', "n", &h_val, CLA_IFC_LIST);

	/* Parse the command line options. */
	clarg_result = dagclarg_parse(clarg, errorfile, &argindex, &code);
	while (1 == clarg_result)
	{
		switch (code)
		{
		case CLA_IFC_LIST:
			if ((h_val >= 0) && (h_val < MAX_IFC_CNT))
				expc_ifc_lst[h_val] = 1;
			break;
			
		default:
			if (params[argindex][0] == '-')
			{
				/* Unknown option. */
				dagutil_error("unknown option %s\n", params[argindex]);
				/* print_usage(clarg);*/
				return TEST_FAIL;
			}
			break;
		}
		clarg_result = dagclarg_parse(clarg, errorfile, &argindex, &code);
	}

	/* Check for errors in the arguments */
	if (-1 == clarg_result)
	{
		if (argindex < param_cnt)
		{
			dagutil_error("while processing option %s\n", params[argindex]);
		}
		dagclarg_display_usage(clarg, stderr);
		return TEST_FAIL;
	}

	return TEST_PASS;
}

/* This is the main test function.
 *
 * This function basically calls the correct function for testing
 * and increments the correct counters depending on the test result.
 *
 */
int ifc_run(char *rec, int len, char* lastpkt, struct_protocol_t prot, uint64_t rec_cnt)
{

	ipf_fields_t flds;
	int ret_val = TEST_PASS;
        int len_incr;

	rec_msg_len = 0;

	/* Parse the record and verify that we have a correct record. */
	if ((ret_val = parse_rec(rec, len, &flds)) != TEST_PASS)
	{
		incr_counters(ret_val);
		return ret_val;
	}

	if (expc_ifc_lst[flds.ifc] != 1)
	{
		len_incr = snprintf(&rec_msg[rec_msg_len], MAX_STR_LEN-rec_msg_len, "Captured on unexpected interface (%d).\n", flds.ifc);
		rec_msg_len += len_incr;
		ret_val = TEST_FAIL;
	}

	incr_counters(ret_val);

	return ret_val;
}

int ifc_err_msg(char *buf, int size)
{
    if (size > 0)
    {
	    strncpy(buf, rec_msg, size);
	    buf[size-1] = '\0'; /* Just in case the buffer was too short...*/
    }
	return TEST_PASS;
}

int ifc_final_msg(char *buf, int size)
{
	char msg[MAX_STR_LEN];

	snprintf(msg, MAX_STR_LEN, 
		 "ifc: pass %u\nifc: fail %u\nifc: warning %u\nifc: ignore %u\n",
		 ifc_passed, 
		 ifc_failed, 
		 ifc_warning, 
		 ifc_ignored);

    if (size > 0)
    {
	    strncpy(buf, msg, size);
	    buf[size-1] = '\0';
    }

	/* Test will fail if we had failuirs captures or we didn't capture any packets */
	/* Test will have wanting if we idn't fail, but we encountered wanting during the test, or we ignored packets */
	/* Otherwise, we pass the test */
	if ((ifc_failed > 0) || (ifc_total == 0))
		return TEST_FAIL;
	else if ((ifc_warning > 0) || (ifc_ignored > 0))
		return TEST_WARNING;
	else
		return TEST_PASS;
}

int ifc_cleanup()
{
	return TEST_PASS;
}

int ifc_printf (char *format, ...)
{

	printf ("%s: %s - Test printf\n", __FILE__, __FUNCTION__);

	return TEST_PASS;
}

void incr_counters(int res)
{
	switch (res)
	{
	case TEST_PASS:
		ifc_passed++;
		break;

	case TEST_IGNORE:
		ifc_ignored++;
		break;

	case TEST_FAIL:
		ifc_failed++;
		break;
		
	case TEST_WARNING:
		ifc_warning++;
		break;

	default:
		ifc_unknown++;
		break;
	}
}

int  parse_rec(char *rec, int len, ipf_fields_t *flds)
{
	dag_record_t* drec = (dag_record_t*) rec;
	uint32_t tmp;
	uint16_t color;
	uint16_t stream = 0;
	uint16_t hash = 0;
	uint16_t etype;
	uint8_t  ihl;
	uint8_t *pload;
	uint8_t *tp;
	
	/* We don't handle legacy records, so we ignore them and report that... */
	if (dagerf_is_legacy_type((uint8_t*) rec))
	{
		ifc_ignored++;
		return TEST_IGNORE;
	} 
	
	/* If it's not legacy record than it's modern and we continue... */
	/* We also need at least 20 Bytes to extract the ether-type */
	if (len < 20)
		return TEST_IGNORE;
	else
		len -= 20;

	switch (drec->type) {
	case ERF_TYPE_HDLC_POS:
	case ERF_TYPE_COLOR_HDLC_POS:
	case ERF_TYPE_COLOR_HASH_POS:
		if (drec->type == ERF_TYPE_COLOR_HDLC_POS)
		{
			color = ntohs(drec->lctr);
			stream = color & 0x03;
			color = color >> 2;
			tmp = ntohl(drec->rec.pos.hdlc);
			etype = (uint16_t) (tmp & 0xffff);
		} 
		else if (drec->type == ERF_TYPE_COLOR_HASH_POS)
		{
			color = ntohs(drec->lctr);
			hash = color & 0x0f;
			color = color >> 4;
			tmp = ntohl(drec->rec.pos.hdlc);
			etype = (uint16_t) (tmp & 0xffff);
		} 
		else 
		{
			tmp = ntohl(drec->rec.pos.hdlc);
			color = (uint16_t) ((tmp>>16) & 0xffff);
			stream = color & 0x03;
			color = color >> 2;
			etype = (uint16_t) (tmp & 0xffff);
		}
		flds->ifc = drec->flags.iface;
		flds->rectype = drec->type;
		flds->rlen = ntohs(drec->rlen);
		flds->wlen = ntohs(drec->wlen);
		flds->colour = color;
		flds->stream = stream;
		flds->hash = hash;
		flds->etype = etype;
		
		/* Make sure the payload points to the IP header */
		pload = drec->rec.pos.pload;

		break;
		
	case ERF_TYPE_ETH:
	case ERF_TYPE_COLOR_ETH:
	case ERF_TYPE_COLOR_HASH_ETH:
		if (drec->type == ERF_TYPE_COLOR_ETH)
		{
			color = ntohs(drec->lctr);
			stream = color & 0x03;
			color = color >> 2;
			etype = ntohs(drec->rec.eth.etype);
		} 
		else if (drec->type == ERF_TYPE_COLOR_HASH_ETH)
		{
			color = ntohs(drec->lctr);
			hash = color & 0x0f;
			color = color >> 4;
			etype = ntohs(drec->rec.eth.etype);
		} 
		else 
		{
			tp = &(drec->rec.eth.offset);
			color = ntohs(*((uint16_t*) tp));
			stream = color & 0x03;
			color = color >> 2;
			etype = ntohs(drec->rec.eth.etype);
		}
		pload = drec->rec.eth.pload - 2; /* We want to point to the VLAN label in case we're Q-in-Q */
		while (etype == 0x8100) /* Skip VLAN Tags */
		{
			pload += 4; /* Skip this VLAN */
			etype = ntohs(*pload);
		}
		pload += 2; /* Now we're 2 bytes behind, so we need to atch up */

		flds->ifc = drec->flags.iface;
		flds->rectype = drec->type;
		flds->rlen = ntohs(drec->rlen);
		flds->wlen = ntohs(drec->wlen);
		flds->colour = color;
		flds->stream = stream;
		flds->hash = hash;
		flds->etype = etype;
		break;
		
	default:
		ifc_ignored++;
		return TEST_IGNORE;

	}

	/* We need at least 20 bytes for IP header */
	if (len < 20)
		return TEST_IGNORE;

	if ((etype == 0x0800) || (etype == 0x0021)) /* IP v4 */
	{
		ihl = ((uint8_t) pload[0]) & 0x0f;
		flds->ip_prot = pload[9];
		flds->src_ip = IP_ADDR_MAKE(pload[12], pload[13], pload[14], pload[15]);
		flds->dst_ip = IP_ADDR_MAKE(pload[16], pload[17], pload[18], pload[19]);
		
		len -= 4*ihl;

		pload = pload + 4*ihl;
		if (flds->ip_prot == 1) /* ICMP */
		{
			if (len < 2)
				return TEST_IGNORE;
			flds->icmp_type = pload[0];
			flds->icmp_code = pload[1];
		}
		else if (flds->ip_prot == 6)  /* TCP */
		{
			if (len < 13)
				return TEST_IGNORE;
			flds->src_port = (uint16_t) (pload[0] << 8 | pload[1]);
			flds->dst_port = (uint16_t) (pload[2] << 8 | pload[3]);
			flds->tcp_flags = pload[13];
		}
		else if (flds->ip_prot == 17)  /* UDP */
		{
			if (len < 4)
				return TEST_IGNORE;
			flds->src_port = (uint16_t) (pload[0] << 8 | pload[1]);
			flds->dst_port = (uint16_t) (pload[2] << 8 | pload[3]);
		}
	}
	
	return TEST_PASS;
}
